# Copyright 2021-2022 NVIDIA Corporation.  All rights reserved.
#
# Please refer to the NVIDIA end user license agreement (EULA) associated
# with this source code for terms and conditions that govern your use of
# this software. Any use, reproduction, disclosure, or distribution of
# this software and related documentation outside the terms of the EULA
# is strictly prohibited.
from typing import List, Tuple, Any
from enum import IntEnum
import cython
import ctypes
from libc.stdlib cimport calloc, free
from libc.string cimport memcpy
from libc.stdint cimport int32_t, uint32_t, int64_t, uint64_t
from libc.stddef cimport wchar_t
from libc.limits cimport CHAR_MIN
from libcpp.vector cimport vector
from cpython.buffer cimport PyObject_CheckBuffer, PyObject_GetBuffer, PyBuffer_Release, PyBUF_SIMPLE, PyBUF_ANY_CONTIGUOUS
from cpython.bytes cimport PyBytes_FromStringAndSize
import cuda.cuda

ctypedef unsigned long long signed_char_ptr
ctypedef unsigned long long unsigned_char_ptr
ctypedef unsigned long long char_ptr
ctypedef unsigned long long short_ptr
ctypedef unsigned long long unsigned_short_ptr
ctypedef unsigned long long int_ptr
ctypedef unsigned long long long_int_ptr
ctypedef unsigned long long long_long_int_ptr
ctypedef unsigned long long unsigned_int_ptr
ctypedef unsigned long long unsigned_long_int_ptr
ctypedef unsigned long long unsigned_long_long_int_ptr
ctypedef unsigned long long uint32_t_ptr
ctypedef unsigned long long uint64_t_ptr
ctypedef unsigned long long int32_t_ptr
ctypedef unsigned long long int64_t_ptr
ctypedef unsigned long long unsigned_ptr
ctypedef unsigned long long unsigned_long_long_ptr
ctypedef unsigned long long size_t_ptr
ctypedef unsigned long long float_ptr
ctypedef unsigned long long double_ptr
ctypedef unsigned long long void_ptr

#: Default page-locked allocation flag
cudaHostAllocDefault = ccudart.cudaHostAllocDefault

#: Pinned memory accessible by all CUDA contexts
cudaHostAllocPortable = ccudart.cudaHostAllocPortable

#: Map allocation into device space
cudaHostAllocMapped = ccudart.cudaHostAllocMapped

#: Write-combined memory
cudaHostAllocWriteCombined = ccudart.cudaHostAllocWriteCombined

#: Default host memory registration flag
cudaHostRegisterDefault = ccudart.cudaHostRegisterDefault

#: Pinned memory accessible by all CUDA contexts
cudaHostRegisterPortable = ccudart.cudaHostRegisterPortable

#: Map registered memory into device space
cudaHostRegisterMapped = ccudart.cudaHostRegisterMapped

#: Memory-mapped I/O space
cudaHostRegisterIoMemory = ccudart.cudaHostRegisterIoMemory

#: Memory-mapped read-only
cudaHostRegisterReadOnly = ccudart.cudaHostRegisterReadOnly

#: Default peer addressing enable flag
cudaPeerAccessDefault = ccudart.cudaPeerAccessDefault

#: Default stream flag
cudaStreamDefault = ccudart.cudaStreamDefault

#: Stream does not synchronize with stream 0 (the NULL stream)
cudaStreamNonBlocking = ccudart.cudaStreamNonBlocking

#: Legacy stream handle
#:
#: Stream handle that can be passed as a cudaStream_t to use an implicit
#: stream with legacy synchronization behavior.
#:
#: See details of the \link_sync_behavior
cudaStreamLegacy = ccudart.cudaStreamLegacy

#: Per-thread stream handle
#:
#: Stream handle that can be passed as a cudaStream_t to use an implicit
#: stream with per-thread synchronization behavior.
#:
#: See details of the \link_sync_behavior
cudaStreamPerThread = ccudart.cudaStreamPerThread

#: Default event flag
cudaEventDefault = ccudart.cudaEventDefault

#: Event uses blocking synchronization
cudaEventBlockingSync = ccudart.cudaEventBlockingSync

#: Event will not record timing data
cudaEventDisableTiming = ccudart.cudaEventDisableTiming

#: Event is suitable for interprocess use. cudaEventDisableTiming must be
#: set
cudaEventInterprocess = ccudart.cudaEventInterprocess

#: Default event record flag
cudaEventRecordDefault = ccudart.cudaEventRecordDefault

#: Event is captured in the graph as an external event node when performing
#: stream capture
cudaEventRecordExternal = ccudart.cudaEventRecordExternal

#: Default event wait flag
cudaEventWaitDefault = ccudart.cudaEventWaitDefault

#: Event is captured in the graph as an external event node when performing
#: stream capture
cudaEventWaitExternal = ccudart.cudaEventWaitExternal

#: Device flag - Automatic scheduling
cudaDeviceScheduleAuto = ccudart.cudaDeviceScheduleAuto

#: Device flag - Spin default scheduling
cudaDeviceScheduleSpin = ccudart.cudaDeviceScheduleSpin

#: Device flag - Yield default scheduling
cudaDeviceScheduleYield = ccudart.cudaDeviceScheduleYield

#: Device flag - Use blocking synchronization
cudaDeviceScheduleBlockingSync = ccudart.cudaDeviceScheduleBlockingSync

#: Device flag - Use blocking synchronization [Deprecated]
cudaDeviceBlockingSync = ccudart.cudaDeviceBlockingSync

#: Device schedule flags mask
cudaDeviceScheduleMask = ccudart.cudaDeviceScheduleMask

#: Device flag - Support mapped pinned allocations
cudaDeviceMapHost = ccudart.cudaDeviceMapHost

#: Device flag - Keep local memory allocation after launch
cudaDeviceLmemResizeToMax = ccudart.cudaDeviceLmemResizeToMax

#: Device flags mask
cudaDeviceMask = ccudart.cudaDeviceMask

#: Default CUDA array allocation flag
cudaArrayDefault = ccudart.cudaArrayDefault

#: Must be set in cudaMalloc3DArray to create a layered CUDA array
cudaArrayLayered = ccudart.cudaArrayLayered

#: Must be set in cudaMallocArray or cudaMalloc3DArray in order to bind
#: surfaces to the CUDA array
cudaArraySurfaceLoadStore = ccudart.cudaArraySurfaceLoadStore

#: Must be set in cudaMalloc3DArray to create a cubemap CUDA array
cudaArrayCubemap = ccudart.cudaArrayCubemap

#: Must be set in cudaMallocArray or cudaMalloc3DArray in order to perform
#: texture gather operations on the CUDA array
cudaArrayTextureGather = ccudart.cudaArrayTextureGather

#: Must be set in cudaExternalMemoryGetMappedMipmappedArray if the
#: mipmapped array is used as a color target in a graphics API
cudaArrayColorAttachment = ccudart.cudaArrayColorAttachment

#: Must be set in cudaMallocArray, cudaMalloc3DArray or
#: cudaMallocMipmappedArray in order to create a sparse CUDA array or CUDA
#: mipmapped array
cudaArraySparse = ccudart.cudaArraySparse

#: Must be set in cudaMallocArray, cudaMalloc3DArray or
#: cudaMallocMipmappedArray in order to create a deferred mapping CUDA
#: array or CUDA mipmapped array
cudaArrayDeferredMapping = ccudart.cudaArrayDeferredMapping

#: Automatically enable peer access between remote devices as needed
cudaIpcMemLazyEnablePeerAccess = ccudart.cudaIpcMemLazyEnablePeerAccess

#: Memory can be accessed by any stream on any device
cudaMemAttachGlobal = ccudart.cudaMemAttachGlobal

#: Memory cannot be accessed by any stream on any device
cudaMemAttachHost = ccudart.cudaMemAttachHost

#: Memory can only be accessed by a single stream on the associated device
cudaMemAttachSingle = ccudart.cudaMemAttachSingle

#: Default behavior
cudaOccupancyDefault = ccudart.cudaOccupancyDefault

#: Assume global caching is enabled and cannot be automatically turned off
cudaOccupancyDisableCachingOverride = ccudart.cudaOccupancyDisableCachingOverride

#: Device id that represents the CPU
cudaCpuDeviceId = ccudart.cudaCpuDeviceId

#: Device id that represents an invalid device
cudaInvalidDeviceId = ccudart.cudaInvalidDeviceId

#: If set, each kernel launched as part of
#: :py:obj:`~.cudaLaunchCooperativeKernelMultiDevice` only waits for prior
#: work in the stream corresponding to that GPU to complete before the
#: kernel begins execution.
cudaCooperativeLaunchMultiDeviceNoPreSync = ccudart.cudaCooperativeLaunchMultiDeviceNoPreSync

#: If set, any subsequent work pushed in a stream that participated in a
#: call to :py:obj:`~.cudaLaunchCooperativeKernelMultiDevice` will only
#: wait for the kernel launched on the GPU corresponding to that stream to
#: complete before it begins execution.
cudaCooperativeLaunchMultiDeviceNoPostSync = ccudart.cudaCooperativeLaunchMultiDeviceNoPostSync

#: Indicates that the layered sparse CUDA array or CUDA mipmapped array has
#: a single mip tail region for all layers
cudaArraySparsePropertiesSingleMipTail = ccudart.cudaArraySparsePropertiesSingleMipTail

#: CUDA IPC Handle Size
CUDA_IPC_HANDLE_SIZE = ccudart.CUDA_IPC_HANDLE_SIZE

#: Indicates that the external memory object is a dedicated resource
cudaExternalMemoryDedicated = ccudart.cudaExternalMemoryDedicated

#: When the /p flags parameter of
#: :py:obj:`~.cudaExternalSemaphoreSignalParams` contains this flag, it
#: indicates that signaling an external semaphore object should skip
#: performing appropriate memory synchronization operations over all the
#: external memory objects that are imported as
#: :py:obj:`~.cudaExternalMemoryHandleTypeNvSciBuf`, which otherwise are
#: performed by default to ensure data coherency with other importers of
#: the same NvSciBuf memory objects.
cudaExternalSemaphoreSignalSkipNvSciBufMemSync = ccudart.cudaExternalSemaphoreSignalSkipNvSciBufMemSync

#: When the /p flags parameter of
#: :py:obj:`~.cudaExternalSemaphoreWaitParams` contains this flag, it
#: indicates that waiting an external semaphore object should skip
#: performing appropriate memory synchronization operations over all the
#: external memory objects that are imported as
#: :py:obj:`~.cudaExternalMemoryHandleTypeNvSciBuf`, which otherwise are
#: performed by default to ensure data coherency with other importers of
#: the same NvSciBuf memory objects.
cudaExternalSemaphoreWaitSkipNvSciBufMemSync = ccudart.cudaExternalSemaphoreWaitSkipNvSciBufMemSync

#: When /p flags of :py:obj:`~.cudaDeviceGetNvSciSyncAttributes` is set to
#: this, it indicates that application need signaler specific NvSciSyncAttr
#: to be filled by :py:obj:`~.cudaDeviceGetNvSciSyncAttributes`.
cudaNvSciSyncAttrSignal = ccudart.cudaNvSciSyncAttrSignal

#: When /p flags of :py:obj:`~.cudaDeviceGetNvSciSyncAttributes` is set to
#: this, it indicates that application need waiter specific NvSciSyncAttr
#: to be filled by :py:obj:`~.cudaDeviceGetNvSciSyncAttributes`.
cudaNvSciSyncAttrWait = ccudart.cudaNvSciSyncAttrWait

cudaSurfaceType1D = ccudart.cudaSurfaceType1D

cudaSurfaceType2D = ccudart.cudaSurfaceType2D

cudaSurfaceType3D = ccudart.cudaSurfaceType3D

cudaSurfaceTypeCubemap = ccudart.cudaSurfaceTypeCubemap

cudaSurfaceType1DLayered = ccudart.cudaSurfaceType1DLayered

cudaSurfaceType2DLayered = ccudart.cudaSurfaceType2DLayered

cudaSurfaceTypeCubemapLayered = ccudart.cudaSurfaceTypeCubemapLayered

cudaTextureType1D = ccudart.cudaTextureType1D

cudaTextureType2D = ccudart.cudaTextureType2D

cudaTextureType3D = ccudart.cudaTextureType3D

cudaTextureTypeCubemap = ccudart.cudaTextureTypeCubemap

cudaTextureType1DLayered = ccudart.cudaTextureType1DLayered

cudaTextureType2DLayered = ccudart.cudaTextureType2DLayered

cudaTextureTypeCubemapLayered = ccudart.cudaTextureTypeCubemapLayered

#: CUDA Runtime API Version
CUDART_VERSION = ccudart.CUDART_VERSION

__CUDART_API_VERSION = ccudart.__CUDART_API_VERSION

#: Maximum number of planes per frame
CUDA_EGL_MAX_PLANES = ccudart.CUDA_EGL_MAX_PLANES

class cudaError_t(IntEnum):
    """
    impl_private CUDA error types
    """

    #: The API call returned with no errors. In the case of query calls,
    #: this also means that the operation being queried is complete (see
    #: :py:obj:`~.cudaEventQuery()` and :py:obj:`~.cudaStreamQuery()`).
    cudaSuccess = ccudart.cudaError.cudaSuccess

    #: This indicates that one or more of the parameters passed to the API
    #: call is not within an acceptable range of values.
    cudaErrorInvalidValue = ccudart.cudaError.cudaErrorInvalidValue

    #: The API call failed because it was unable to allocate enough memory
    #: to perform the requested operation.
    cudaErrorMemoryAllocation = ccudart.cudaError.cudaErrorMemoryAllocation

    #: The API call failed because the CUDA driver and runtime could not be
    #: initialized.
    cudaErrorInitializationError = ccudart.cudaError.cudaErrorInitializationError

    #: This indicates that a CUDA Runtime API call cannot be executed
    #: because it is being called during process shut down, at a point in
    #: time after CUDA driver has been unloaded.
    cudaErrorCudartUnloading = ccudart.cudaError.cudaErrorCudartUnloading

    #: This indicates profiler is not initialized for this run. This can
    #: happen when the application is running with external profiling tools
    #: like visual profiler.
    cudaErrorProfilerDisabled = ccudart.cudaError.cudaErrorProfilerDisabled

    #: [Deprecated]
    cudaErrorProfilerNotInitialized = ccudart.cudaError.cudaErrorProfilerNotInitialized

    #: [Deprecated]
    cudaErrorProfilerAlreadyStarted = ccudart.cudaError.cudaErrorProfilerAlreadyStarted

    #: [Deprecated]
    cudaErrorProfilerAlreadyStopped = ccudart.cudaError.cudaErrorProfilerAlreadyStopped

    #: This indicates that a kernel launch is requesting resources that can
    #: never be satisfied by the current device. Requesting more shared
    #: memory per block than the device supports will trigger this error,
    #: as will requesting too many threads or blocks. See
    #: :py:obj:`~.cudaDeviceProp` for more device limitations.
    cudaErrorInvalidConfiguration = ccudart.cudaError.cudaErrorInvalidConfiguration

    #: This indicates that one or more of the pitch-related parameters
    #: passed to the API call is not within the acceptable range for pitch.
    cudaErrorInvalidPitchValue = ccudart.cudaError.cudaErrorInvalidPitchValue

    #: This indicates that the symbol name/identifier passed to the API
    #: call is not a valid name or identifier.
    cudaErrorInvalidSymbol = ccudart.cudaError.cudaErrorInvalidSymbol

    #: This indicates that at least one host pointer passed to the API call
    #: is not a valid host pointer. [Deprecated]
    cudaErrorInvalidHostPointer = ccudart.cudaError.cudaErrorInvalidHostPointer

    #: This indicates that at least one device pointer passed to the API
    #: call is not a valid device pointer. [Deprecated]
    cudaErrorInvalidDevicePointer = ccudart.cudaError.cudaErrorInvalidDevicePointer

    #: This indicates that the texture passed to the API call is not a
    #: valid texture.
    cudaErrorInvalidTexture = ccudart.cudaError.cudaErrorInvalidTexture

    #: This indicates that the texture binding is not valid. This occurs if
    #: you call :py:obj:`~.cudaGetTextureAlignmentOffset()` with an unbound
    #: texture.
    cudaErrorInvalidTextureBinding = ccudart.cudaError.cudaErrorInvalidTextureBinding

    #: This indicates that the channel descriptor passed to the API call is
    #: not valid. This occurs if the format is not one of the formats
    #: specified by :py:obj:`~.cudaChannelFormatKind`, or if one of the
    #: dimensions is invalid.
    cudaErrorInvalidChannelDescriptor = ccudart.cudaError.cudaErrorInvalidChannelDescriptor

    #: This indicates that the direction of the memcpy passed to the API
    #: call is not one of the types specified by
    #: :py:obj:`~.cudaMemcpyKind`.
    cudaErrorInvalidMemcpyDirection = ccudart.cudaError.cudaErrorInvalidMemcpyDirection

    #: This indicated that the user has taken the address of a constant
    #: variable, which was forbidden up until the CUDA 3.1 release.
    #: [Deprecated]
    cudaErrorAddressOfConstant = ccudart.cudaError.cudaErrorAddressOfConstant

    #: This indicated that a texture fetch was not able to be performed.
    #: This was previously used for device emulation of texture operations.
    #: [Deprecated]
    cudaErrorTextureFetchFailed = ccudart.cudaError.cudaErrorTextureFetchFailed

    #: This indicated that a texture was not bound for access. This was
    #: previously used for device emulation of texture operations.
    #: [Deprecated]
    cudaErrorTextureNotBound = ccudart.cudaError.cudaErrorTextureNotBound

    #: This indicated that a synchronization operation had failed. This was
    #: previously used for some device emulation functions. [Deprecated]
    cudaErrorSynchronizationError = ccudart.cudaError.cudaErrorSynchronizationError

    #: This indicates that a non-float texture was being accessed with
    #: linear filtering. This is not supported by CUDA.
    cudaErrorInvalidFilterSetting = ccudart.cudaError.cudaErrorInvalidFilterSetting

    #: This indicates that an attempt was made to read a non-float texture
    #: as a normalized float. This is not supported by CUDA.
    cudaErrorInvalidNormSetting = ccudart.cudaError.cudaErrorInvalidNormSetting

    #: Mixing of device and device emulation code was not allowed.
    #: [Deprecated]
    cudaErrorMixedDeviceExecution = ccudart.cudaError.cudaErrorMixedDeviceExecution

    #: This indicates that the API call is not yet implemented. Production
    #: releases of CUDA will never return this error. [Deprecated]
    cudaErrorNotYetImplemented = ccudart.cudaError.cudaErrorNotYetImplemented

    #: This indicated that an emulated device pointer exceeded the 32-bit
    #: address range. [Deprecated]
    cudaErrorMemoryValueTooLarge = ccudart.cudaError.cudaErrorMemoryValueTooLarge

    #: This indicates that the CUDA driver that the application has loaded
    #: is a stub library. Applications that run with the stub rather than a
    #: real driver loaded will result in CUDA API returning this error.
    cudaErrorStubLibrary = ccudart.cudaError.cudaErrorStubLibrary

    #: This indicates that the installed NVIDIA CUDA driver is older than
    #: the CUDA runtime library. This is not a supported configuration.
    #: Users should install an updated NVIDIA display driver to allow the
    #: application to run.
    cudaErrorInsufficientDriver = ccudart.cudaError.cudaErrorInsufficientDriver

    #: This indicates that the API call requires a newer CUDA driver than
    #: the one currently installed. Users should install an updated NVIDIA
    #: CUDA driver to allow the API call to succeed.
    cudaErrorCallRequiresNewerDriver = ccudart.cudaError.cudaErrorCallRequiresNewerDriver

    #: This indicates that the surface passed to the API call is not a
    #: valid surface.
    cudaErrorInvalidSurface = ccudart.cudaError.cudaErrorInvalidSurface

    #: This indicates that multiple global or constant variables (across
    #: separate CUDA source files in the application) share the same string
    #: name.
    cudaErrorDuplicateVariableName = ccudart.cudaError.cudaErrorDuplicateVariableName

    #: This indicates that multiple textures (across separate CUDA source
    #: files in the application) share the same string name.
    cudaErrorDuplicateTextureName = ccudart.cudaError.cudaErrorDuplicateTextureName

    #: This indicates that multiple surfaces (across separate CUDA source
    #: files in the application) share the same string name.
    cudaErrorDuplicateSurfaceName = ccudart.cudaError.cudaErrorDuplicateSurfaceName

    #: This indicates that all CUDA devices are busy or unavailable at the
    #: current time. Devices are often busy/unavailable due to use of
    #: :py:obj:`~.cudaComputeModeProhibited`,
    #: :py:obj:`~.cudaComputeModeExclusiveProcess`, or when long running
    #: CUDA kernels have filled up the GPU and are blocking new work from
    #: starting. They can also be unavailable due to memory constraints on
    #: a device that already has active CUDA work being performed.
    cudaErrorDevicesUnavailable = ccudart.cudaError.cudaErrorDevicesUnavailable

    #: This indicates that the current context is not compatible with this
    #: the CUDA Runtime. This can only occur if you are using CUDA
    #: Runtime/Driver interoperability and have created an existing Driver
    #: context using the driver API. The Driver context may be incompatible
    #: either because the Driver context was created using an older version
    #: of the API, because the Runtime API call expects a primary driver
    #: context and the Driver context is not primary, or because the Driver
    #: context has been destroyed. Please see :py:obj:`~.Interactions`with
    #: the CUDA Driver API" for more information.
    cudaErrorIncompatibleDriverContext = ccudart.cudaError.cudaErrorIncompatibleDriverContext

    #: The device function being invoked (usually via
    #: :py:obj:`~.cudaLaunchKernel()`) was not previously configured via
    #: the :py:obj:`~.cudaConfigureCall()` function.
    cudaErrorMissingConfiguration = ccudart.cudaError.cudaErrorMissingConfiguration

    #: This indicated that a previous kernel launch failed. This was
    #: previously used for device emulation of kernel launches.
    #: [Deprecated]
    cudaErrorPriorLaunchFailure = ccudart.cudaError.cudaErrorPriorLaunchFailure

    #: This error indicates that a device runtime grid launch did not occur
    #: because the depth of the child grid would exceed the maximum
    #: supported number of nested grid launches.
    cudaErrorLaunchMaxDepthExceeded = ccudart.cudaError.cudaErrorLaunchMaxDepthExceeded

    #: This error indicates that a grid launch did not occur because the
    #: kernel uses file-scoped textures which are unsupported by the device
    #: runtime. Kernels launched via the device runtime only support
    #: textures created with the Texture Object API's.
    cudaErrorLaunchFileScopedTex = ccudart.cudaError.cudaErrorLaunchFileScopedTex

    #: This error indicates that a grid launch did not occur because the
    #: kernel uses file-scoped surfaces which are unsupported by the device
    #: runtime. Kernels launched via the device runtime only support
    #: surfaces created with the Surface Object API's.
    cudaErrorLaunchFileScopedSurf = ccudart.cudaError.cudaErrorLaunchFileScopedSurf

    #: This error indicates that a call to
    #: :py:obj:`~.cudaDeviceSynchronize` made from the device runtime
    #: failed because the call was made at grid depth greater than than
    #: either the default (2 levels of grids) or user specified device
    #: limit :py:obj:`~.cudaLimitDevRuntimeSyncDepth`. To be able to
    #: synchronize on launched grids at a greater depth successfully, the
    #: maximum nested depth at which :py:obj:`~.cudaDeviceSynchronize` will
    #: be called must be specified with the
    #: :py:obj:`~.cudaLimitDevRuntimeSyncDepth` limit to the
    #: :py:obj:`~.cudaDeviceSetLimit` api before the host-side launch of a
    #: kernel using the device runtime. Keep in mind that additional levels
    #: of sync depth require the runtime to reserve large amounts of device
    #: memory that cannot be used for user allocations.
    cudaErrorSyncDepthExceeded = ccudart.cudaError.cudaErrorSyncDepthExceeded

    #: This error indicates that a device runtime grid launch failed
    #: because the launch would exceed the limit
    #: :py:obj:`~.cudaLimitDevRuntimePendingLaunchCount`. For this launch
    #: to proceed successfully, :py:obj:`~.cudaDeviceSetLimit` must be
    #: called to set the :py:obj:`~.cudaLimitDevRuntimePendingLaunchCount`
    #: to be higher than the upper bound of outstanding launches that can
    #: be issued to the device runtime. Keep in mind that raising the limit
    #: of pending device runtime launches will require the runtime to
    #: reserve device memory that cannot be used for user allocations.
    cudaErrorLaunchPendingCountExceeded = ccudart.cudaError.cudaErrorLaunchPendingCountExceeded

    #: The requested device function does not exist or is not compiled for
    #: the proper device architecture.
    cudaErrorInvalidDeviceFunction = ccudart.cudaError.cudaErrorInvalidDeviceFunction

    #: This indicates that no CUDA-capable devices were detected by the
    #: installed CUDA driver.
    cudaErrorNoDevice = ccudart.cudaError.cudaErrorNoDevice

    #: This indicates that the device ordinal supplied by the user does not
    #: correspond to a valid CUDA device or that the action requested is
    #: invalid for the specified device.
    cudaErrorInvalidDevice = ccudart.cudaError.cudaErrorInvalidDevice

    #: This indicates that the device doesn't have a valid Grid License.
    cudaErrorDeviceNotLicensed = ccudart.cudaError.cudaErrorDeviceNotLicensed

    #: By default, the CUDA runtime may perform a minimal set of self-
    #: tests, as well as CUDA driver tests, to establish the validity of
    #: both. Introduced in CUDA 11.2, this error return indicates that at
    #: least one of these tests has failed and the validity of either the
    #: runtime or the driver could not be established.
    cudaErrorSoftwareValidityNotEstablished = ccudart.cudaError.cudaErrorSoftwareValidityNotEstablished

    #: This indicates an internal startup failure in the CUDA runtime.
    cudaErrorStartupFailure = ccudart.cudaError.cudaErrorStartupFailure

    #: This indicates that the device kernel image is invalid.
    cudaErrorInvalidKernelImage = ccudart.cudaError.cudaErrorInvalidKernelImage

    #: This most frequently indicates that there is no context bound to the
    #: current thread. This can also be returned if the context passed to
    #: an API call is not a valid handle (such as a context that has had
    #: :py:obj:`~.cuCtxDestroy()` invoked on it). This can also be returned
    #: if a user mixes different API versions (i.e. 3010 context with 3020
    #: API calls). See :py:obj:`~.cuCtxGetApiVersion()` for more details.
    cudaErrorDeviceUninitialized = ccudart.cudaError.cudaErrorDeviceUninitialized

    #: This indicates that the buffer object could not be mapped.
    cudaErrorMapBufferObjectFailed = ccudart.cudaError.cudaErrorMapBufferObjectFailed

    #: This indicates that the buffer object could not be unmapped.
    cudaErrorUnmapBufferObjectFailed = ccudart.cudaError.cudaErrorUnmapBufferObjectFailed

    #: This indicates that the specified array is currently mapped and thus
    #: cannot be destroyed.
    cudaErrorArrayIsMapped = ccudart.cudaError.cudaErrorArrayIsMapped

    #: This indicates that the resource is already mapped.
    cudaErrorAlreadyMapped = ccudart.cudaError.cudaErrorAlreadyMapped

    #: This indicates that there is no kernel image available that is
    #: suitable for the device. This can occur when a user specifies code
    #: generation options for a particular CUDA source file that do not
    #: include the corresponding device configuration.
    cudaErrorNoKernelImageForDevice = ccudart.cudaError.cudaErrorNoKernelImageForDevice

    #: This indicates that a resource has already been acquired.
    cudaErrorAlreadyAcquired = ccudart.cudaError.cudaErrorAlreadyAcquired

    #: This indicates that a resource is not mapped.
    cudaErrorNotMapped = ccudart.cudaError.cudaErrorNotMapped

    #: This indicates that a mapped resource is not available for access as
    #: an array.
    cudaErrorNotMappedAsArray = ccudart.cudaError.cudaErrorNotMappedAsArray

    #: This indicates that a mapped resource is not available for access as
    #: a pointer.
    cudaErrorNotMappedAsPointer = ccudart.cudaError.cudaErrorNotMappedAsPointer

    #: This indicates that an uncorrectable ECC error was detected during
    #: execution.
    cudaErrorECCUncorrectable = ccudart.cudaError.cudaErrorECCUncorrectable

    #: This indicates that the :py:obj:`~.cudaLimit` passed to the API call
    #: is not supported by the active device.
    cudaErrorUnsupportedLimit = ccudart.cudaError.cudaErrorUnsupportedLimit

    #: This indicates that a call tried to access an exclusive-thread
    #: device that is already in use by a different thread.
    cudaErrorDeviceAlreadyInUse = ccudart.cudaError.cudaErrorDeviceAlreadyInUse

    #: This error indicates that P2P access is not supported across the
    #: given devices.
    cudaErrorPeerAccessUnsupported = ccudart.cudaError.cudaErrorPeerAccessUnsupported

    #: A PTX compilation failed. The runtime may fall back to compiling PTX
    #: if an application does not contain a suitable binary for the current
    #: device.
    cudaErrorInvalidPtx = ccudart.cudaError.cudaErrorInvalidPtx

    #: This indicates an error with the OpenGL or DirectX context.
    cudaErrorInvalidGraphicsContext = ccudart.cudaError.cudaErrorInvalidGraphicsContext

    #: This indicates that an uncorrectable NVLink error was detected
    #: during the execution.
    cudaErrorNvlinkUncorrectable = ccudart.cudaError.cudaErrorNvlinkUncorrectable

    #: This indicates that the PTX JIT compiler library was not found. The
    #: JIT Compiler library is used for PTX compilation. The runtime may
    #: fall back to compiling PTX if an application does not contain a
    #: suitable binary for the current device.
    cudaErrorJitCompilerNotFound = ccudart.cudaError.cudaErrorJitCompilerNotFound

    #: This indicates that the provided PTX was compiled with an
    #: unsupported toolchain. The most common reason for this, is the PTX
    #: was generated by a compiler newer than what is supported by the CUDA
    #: driver and PTX JIT compiler.
    cudaErrorUnsupportedPtxVersion = ccudart.cudaError.cudaErrorUnsupportedPtxVersion

    #: This indicates that the JIT compilation was disabled. The JIT
    #: compilation compiles PTX. The runtime may fall back to compiling PTX
    #: if an application does not contain a suitable binary for the current
    #: device.
    cudaErrorJitCompilationDisabled = ccudart.cudaError.cudaErrorJitCompilationDisabled

    #: This indicates that the provided execution affinity is not supported
    #: by the device.
    cudaErrorUnsupportedExecAffinity = ccudart.cudaError.cudaErrorUnsupportedExecAffinity

    #: This indicates that the device kernel source is invalid.
    cudaErrorInvalidSource = ccudart.cudaError.cudaErrorInvalidSource

    #: This indicates that the file specified was not found.
    cudaErrorFileNotFound = ccudart.cudaError.cudaErrorFileNotFound

    #: This indicates that a link to a shared object failed to resolve.
    cudaErrorSharedObjectSymbolNotFound = ccudart.cudaError.cudaErrorSharedObjectSymbolNotFound

    #: This indicates that initialization of a shared object failed.
    cudaErrorSharedObjectInitFailed = ccudart.cudaError.cudaErrorSharedObjectInitFailed

    #: This error indicates that an OS call failed.
    cudaErrorOperatingSystem = ccudart.cudaError.cudaErrorOperatingSystem

    #: This indicates that a resource handle passed to the API call was not
    #: valid. Resource handles are opaque types like
    #: :py:obj:`~.cudaStream_t` and :py:obj:`~.cudaEvent_t`.
    cudaErrorInvalidResourceHandle = ccudart.cudaError.cudaErrorInvalidResourceHandle

    #: This indicates that a resource required by the API call is not in a
    #: valid state to perform the requested operation.
    cudaErrorIllegalState = ccudart.cudaError.cudaErrorIllegalState

    #: This indicates that a named symbol was not found. Examples of
    #: symbols are global/constant variable names, driver function names,
    #: texture names, and surface names.
    cudaErrorSymbolNotFound = ccudart.cudaError.cudaErrorSymbolNotFound

    #: This indicates that asynchronous operations issued previously have
    #: not completed yet. This result is not actually an error, but must be
    #: indicated differently than :py:obj:`~.cudaSuccess` (which indicates
    #: completion). Calls that may return this value include
    #: :py:obj:`~.cudaEventQuery()` and :py:obj:`~.cudaStreamQuery()`.
    cudaErrorNotReady = ccudart.cudaError.cudaErrorNotReady

    #: The device encountered a load or store instruction on an invalid
    #: memory address. This leaves the process in an inconsistent state and
    #: any further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    cudaErrorIllegalAddress = ccudart.cudaError.cudaErrorIllegalAddress

    #: This indicates that a launch did not occur because it did not have
    #: appropriate resources. Although this error is similar to
    #: :py:obj:`~.cudaErrorInvalidConfiguration`, this error usually
    #: indicates that the user has attempted to pass too many arguments to
    #: the device kernel, or the kernel launch specifies too many threads
    #: for the kernel's register count.
    cudaErrorLaunchOutOfResources = ccudart.cudaError.cudaErrorLaunchOutOfResources

    #: This indicates that the device kernel took too long to execute. This
    #: can only occur if timeouts are enabled - see the device property
    #: :py:obj:`~.kernelExecTimeoutEnabled` for more information. This
    #: leaves the process in an inconsistent state and any further CUDA
    #: work will return the same error. To continue using CUDA, the process
    #: must be terminated and relaunched.
    cudaErrorLaunchTimeout = ccudart.cudaError.cudaErrorLaunchTimeout

    #: This error indicates a kernel launch that uses an incompatible
    #: texturing mode.
    cudaErrorLaunchIncompatibleTexturing = ccudart.cudaError.cudaErrorLaunchIncompatibleTexturing

    #: This error indicates that a call to
    #: :py:obj:`~.cudaDeviceEnablePeerAccess()` is trying to re-enable peer
    #: addressing on from a context which has already had peer addressing
    #: enabled.
    cudaErrorPeerAccessAlreadyEnabled = ccudart.cudaError.cudaErrorPeerAccessAlreadyEnabled

    #: This error indicates that :py:obj:`~.cudaDeviceDisablePeerAccess()`
    #: is trying to disable peer addressing which has not been enabled yet
    #: via :py:obj:`~.cudaDeviceEnablePeerAccess()`.
    cudaErrorPeerAccessNotEnabled = ccudart.cudaError.cudaErrorPeerAccessNotEnabled

    #: This indicates that the user has called
    #: :py:obj:`~.cudaSetValidDevices()`, :py:obj:`~.cudaSetDeviceFlags()`,
    #: :py:obj:`~.cudaD3D9SetDirect3DDevice()`,
    #: :py:obj:`~.cudaD3D10SetDirect3DDevice`,
    #: :py:obj:`~.cudaD3D11SetDirect3DDevice()`, or
    #: :py:obj:`~.cudaVDPAUSetVDPAUDevice()` after initializing the CUDA
    #: runtime by calling non-device management operations (allocating
    #: memory and launching kernels are examples of non-device management
    #: operations). This error can also be returned if using runtime/driver
    #: interoperability and there is an existing :py:obj:`~.CUcontext`
    #: active on the host thread.
    cudaErrorSetOnActiveProcess = ccudart.cudaError.cudaErrorSetOnActiveProcess

    #: This error indicates that the context current to the calling thread
    #: has been destroyed using :py:obj:`~.cuCtxDestroy`, or is a primary
    #: context which has not yet been initialized.
    cudaErrorContextIsDestroyed = ccudart.cudaError.cudaErrorContextIsDestroyed

    #: An assert triggered in device code during kernel execution. The
    #: device cannot be used again. All existing allocations are invalid.
    #: To continue using CUDA, the process must be terminated and
    #: relaunched.
    cudaErrorAssert = ccudart.cudaError.cudaErrorAssert

    #: This error indicates that the hardware resources required to enable
    #: peer access have been exhausted for one or more of the devices
    #: passed to :py:obj:`~.cudaEnablePeerAccess()`.
    cudaErrorTooManyPeers = ccudart.cudaError.cudaErrorTooManyPeers

    #: This error indicates that the memory range passed to
    #: :py:obj:`~.cudaHostRegister()` has already been registered.
    cudaErrorHostMemoryAlreadyRegistered = ccudart.cudaError.cudaErrorHostMemoryAlreadyRegistered

    #: This error indicates that the pointer passed to
    #: :py:obj:`~.cudaHostUnregister()` does not correspond to any
    #: currently registered memory region.
    cudaErrorHostMemoryNotRegistered = ccudart.cudaError.cudaErrorHostMemoryNotRegistered

    #: Device encountered an error in the call stack during kernel
    #: execution, possibly due to stack corruption or exceeding the stack
    #: size limit. This leaves the process in an inconsistent state and any
    #: further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    cudaErrorHardwareStackError = ccudart.cudaError.cudaErrorHardwareStackError

    #: The device encountered an illegal instruction during kernel
    #: execution This leaves the process in an inconsistent state and any
    #: further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    cudaErrorIllegalInstruction = ccudart.cudaError.cudaErrorIllegalInstruction

    #: The device encountered a load or store instruction on a memory
    #: address which is not aligned. This leaves the process in an
    #: inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    cudaErrorMisalignedAddress = ccudart.cudaError.cudaErrorMisalignedAddress

    #: While executing a kernel, the device encountered an instruction
    #: which can only operate on memory locations in certain address spaces
    #: (global, shared, or local), but was supplied a memory address not
    #: belonging to an allowed address space. This leaves the process in an
    #: inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    cudaErrorInvalidAddressSpace = ccudart.cudaError.cudaErrorInvalidAddressSpace

    #: The device encountered an invalid program counter. This leaves the
    #: process in an inconsistent state and any further CUDA work will
    #: return the same error. To continue using CUDA, the process must be
    #: terminated and relaunched.
    cudaErrorInvalidPc = ccudart.cudaError.cudaErrorInvalidPc

    #: An exception occurred on the device while executing a kernel. Common
    #: causes include dereferencing an invalid device pointer and accessing
    #: out of bounds shared memory. Less common cases can be system
    #: specific - more information about these cases can be found in the
    #: system specific user guide. This leaves the process in an
    #: inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    cudaErrorLaunchFailure = ccudart.cudaError.cudaErrorLaunchFailure

    #: This error indicates that the number of blocks launched per grid for
    #: a kernel that was launched via either
    #: :py:obj:`~.cudaLaunchCooperativeKernel` or
    #: :py:obj:`~.cudaLaunchCooperativeKernelMultiDevice` exceeds the
    #: maximum number of blocks as allowed by
    #: :py:obj:`~.cudaOccupancyMaxActiveBlocksPerMultiprocessor` or
    #: :py:obj:`~.cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags`
    #: times the number of multiprocessors as specified by the device
    #: attribute :py:obj:`~.cudaDevAttrMultiProcessorCount`.
    cudaErrorCooperativeLaunchTooLarge = ccudart.cudaError.cudaErrorCooperativeLaunchTooLarge

    #: This error indicates the attempted operation is not permitted.
    cudaErrorNotPermitted = ccudart.cudaError.cudaErrorNotPermitted

    #: This error indicates the attempted operation is not supported on the
    #: current system or device.
    cudaErrorNotSupported = ccudart.cudaError.cudaErrorNotSupported

    #: This error indicates that the system is not yet ready to start any
    #: CUDA work. To continue using CUDA, verify the system configuration
    #: is in a valid state and all required driver daemons are actively
    #: running. More information about this error can be found in the
    #: system specific user guide.
    cudaErrorSystemNotReady = ccudart.cudaError.cudaErrorSystemNotReady

    #: This error indicates that there is a mismatch between the versions
    #: of the display driver and the CUDA driver. Refer to the
    #: compatibility documentation for supported versions.
    cudaErrorSystemDriverMismatch = ccudart.cudaError.cudaErrorSystemDriverMismatch

    #: This error indicates that the system was upgraded to run with
    #: forward compatibility but the visible hardware detected by CUDA does
    #: not support this configuration. Refer to the compatibility
    #: documentation for the supported hardware matrix or ensure that only
    #: supported hardware is visible during initialization via the
    #: CUDA_VISIBLE_DEVICES environment variable.
    cudaErrorCompatNotSupportedOnDevice = ccudart.cudaError.cudaErrorCompatNotSupportedOnDevice

    #: This error indicates that the MPS client failed to connect to the
    #: MPS control daemon or the MPS server.
    cudaErrorMpsConnectionFailed = ccudart.cudaError.cudaErrorMpsConnectionFailed

    #: This error indicates that the remote procedural call between the MPS
    #: server and the MPS client failed.
    cudaErrorMpsRpcFailure = ccudart.cudaError.cudaErrorMpsRpcFailure

    #: This error indicates that the MPS server is not ready to accept new
    #: MPS client requests. This error can be returned when the MPS server
    #: is in the process of recovering from a fatal failure.
    cudaErrorMpsServerNotReady = ccudart.cudaError.cudaErrorMpsServerNotReady

    #: This error indicates that the hardware resources required to create
    #: MPS client have been exhausted.
    cudaErrorMpsMaxClientsReached = ccudart.cudaError.cudaErrorMpsMaxClientsReached

    #: This error indicates the the hardware resources required to device
    #: connections have been exhausted.
    cudaErrorMpsMaxConnectionsReached = ccudart.cudaError.cudaErrorMpsMaxConnectionsReached

    #: The operation is not permitted when the stream is capturing.
    cudaErrorStreamCaptureUnsupported = ccudart.cudaError.cudaErrorStreamCaptureUnsupported

    #: The current capture sequence on the stream has been invalidated due
    #: to a previous error.
    cudaErrorStreamCaptureInvalidated = ccudart.cudaError.cudaErrorStreamCaptureInvalidated

    #: The operation would have resulted in a merge of two independent
    #: capture sequences.
    cudaErrorStreamCaptureMerge = ccudart.cudaError.cudaErrorStreamCaptureMerge

    #: The capture was not initiated in this stream.
    cudaErrorStreamCaptureUnmatched = ccudart.cudaError.cudaErrorStreamCaptureUnmatched

    #: The capture sequence contains a fork that was not joined to the
    #: primary stream.
    cudaErrorStreamCaptureUnjoined = ccudart.cudaError.cudaErrorStreamCaptureUnjoined

    #: A dependency would have been created which crosses the capture
    #: sequence boundary. Only implicit in-stream ordering dependencies are
    #: allowed to cross the boundary.
    cudaErrorStreamCaptureIsolation = ccudart.cudaError.cudaErrorStreamCaptureIsolation

    #: The operation would have resulted in a disallowed implicit
    #: dependency on a current capture sequence from cudaStreamLegacy.
    cudaErrorStreamCaptureImplicit = ccudart.cudaError.cudaErrorStreamCaptureImplicit

    #: The operation is not permitted on an event which was last recorded
    #: in a capturing stream.
    cudaErrorCapturedEvent = ccudart.cudaError.cudaErrorCapturedEvent

    #: A stream capture sequence not initiated with the
    #: :py:obj:`~.cudaStreamCaptureModeRelaxed` argument to
    #: :py:obj:`~.cudaStreamBeginCapture` was passed to
    #: :py:obj:`~.cudaStreamEndCapture` in a different thread.
    cudaErrorStreamCaptureWrongThread = ccudart.cudaError.cudaErrorStreamCaptureWrongThread

    #: This indicates that the wait operation has timed out.
    cudaErrorTimeout = ccudart.cudaError.cudaErrorTimeout

    #: This error indicates that the graph update was not performed because
    #: it included changes which violated constraints specific to
    #: instantiated graph update.
    cudaErrorGraphExecUpdateFailure = ccudart.cudaError.cudaErrorGraphExecUpdateFailure

    #: This indicates that an async error has occurred in a device outside
    #: of CUDA. If CUDA was waiting for an external device's signal before
    #: consuming shared data, the external device signaled an error
    #: indicating that the data is not valid for consumption. This leaves
    #: the process in an inconsistent state and any further CUDA work will
    #: return the same error. To continue using CUDA, the process must be
    #: terminated and relaunched.
    cudaErrorExternalDevice = ccudart.cudaError.cudaErrorExternalDevice

    #: This indicates that an unknown internal error has occurred.
    cudaErrorUnknown = ccudart.cudaError.cudaErrorUnknown

    #: Any unhandled CUDA driver error is added to this value and returned
    #: via the runtime. Production releases of CUDA should not return such
    #: errors. [Deprecated]
    cudaErrorApiFailureBase = ccudart.cudaError.cudaErrorApiFailureBase

class cudaChannelFormatKind(IntEnum):
    """
    Channel format kind
    """

    #: Signed channel format
    cudaChannelFormatKindSigned = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSigned

    #: Unsigned channel format
    cudaChannelFormatKindUnsigned = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsigned

    #: Float channel format
    cudaChannelFormatKindFloat = ccudart.cudaChannelFormatKind.cudaChannelFormatKindFloat

    #: No channel format
    cudaChannelFormatKindNone = ccudart.cudaChannelFormatKind.cudaChannelFormatKindNone

    #: Unsigned 8-bit integers, planar 4:2:0 YUV format
    cudaChannelFormatKindNV12 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindNV12

    #: 1 channel unsigned 8-bit normalized integer
    cudaChannelFormatKindUnsignedNormalized8X1 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedNormalized8X1

    #: 2 channel unsigned 8-bit normalized integer
    cudaChannelFormatKindUnsignedNormalized8X2 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedNormalized8X2

    #: 4 channel unsigned 8-bit normalized integer
    cudaChannelFormatKindUnsignedNormalized8X4 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedNormalized8X4

    #: 1 channel unsigned 16-bit normalized integer
    cudaChannelFormatKindUnsignedNormalized16X1 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedNormalized16X1

    #: 2 channel unsigned 16-bit normalized integer
    cudaChannelFormatKindUnsignedNormalized16X2 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedNormalized16X2

    #: 4 channel unsigned 16-bit normalized integer
    cudaChannelFormatKindUnsignedNormalized16X4 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedNormalized16X4

    #: 1 channel signed 8-bit normalized integer
    cudaChannelFormatKindSignedNormalized8X1 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedNormalized8X1

    #: 2 channel signed 8-bit normalized integer
    cudaChannelFormatKindSignedNormalized8X2 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedNormalized8X2

    #: 4 channel signed 8-bit normalized integer
    cudaChannelFormatKindSignedNormalized8X4 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedNormalized8X4

    #: 1 channel signed 16-bit normalized integer
    cudaChannelFormatKindSignedNormalized16X1 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedNormalized16X1

    #: 2 channel signed 16-bit normalized integer
    cudaChannelFormatKindSignedNormalized16X2 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedNormalized16X2

    #: 4 channel signed 16-bit normalized integer
    cudaChannelFormatKindSignedNormalized16X4 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedNormalized16X4

    #: 4 channel unsigned normalized block-compressed (BC1 compression)
    #: format
    cudaChannelFormatKindUnsignedBlockCompressed1 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed1

    #: 4 channel unsigned normalized block-compressed (BC1 compression)
    #: format with sRGB encoding
    cudaChannelFormatKindUnsignedBlockCompressed1SRGB = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed1SRGB

    #: 4 channel unsigned normalized block-compressed (BC2 compression)
    #: format
    cudaChannelFormatKindUnsignedBlockCompressed2 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed2

    #: 4 channel unsigned normalized block-compressed (BC2 compression)
    #: format with sRGB encoding
    cudaChannelFormatKindUnsignedBlockCompressed2SRGB = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed2SRGB

    #: 4 channel unsigned normalized block-compressed (BC3 compression)
    #: format
    cudaChannelFormatKindUnsignedBlockCompressed3 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed3

    #: 4 channel unsigned normalized block-compressed (BC3 compression)
    #: format with sRGB encoding
    cudaChannelFormatKindUnsignedBlockCompressed3SRGB = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed3SRGB

    #: 1 channel unsigned normalized block-compressed (BC4 compression)
    #: format
    cudaChannelFormatKindUnsignedBlockCompressed4 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed4

    #: 1 channel signed normalized block-compressed (BC4 compression)
    #: format
    cudaChannelFormatKindSignedBlockCompressed4 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedBlockCompressed4

    #: 2 channel unsigned normalized block-compressed (BC5 compression)
    #: format
    cudaChannelFormatKindUnsignedBlockCompressed5 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed5

    #: 2 channel signed normalized block-compressed (BC5 compression)
    #: format
    cudaChannelFormatKindSignedBlockCompressed5 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedBlockCompressed5

    #: 3 channel unsigned half-float block-compressed (BC6H compression)
    #: format
    cudaChannelFormatKindUnsignedBlockCompressed6H = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed6H

    #: 3 channel signed half-float block-compressed (BC6H compression)
    #: format
    cudaChannelFormatKindSignedBlockCompressed6H = ccudart.cudaChannelFormatKind.cudaChannelFormatKindSignedBlockCompressed6H

    #: 4 channel unsigned normalized block-compressed (BC7 compression)
    #: format
    cudaChannelFormatKindUnsignedBlockCompressed7 = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed7

    #: 4 channel unsigned normalized block-compressed (BC7 compression)
    #: format with sRGB encoding
    cudaChannelFormatKindUnsignedBlockCompressed7SRGB = ccudart.cudaChannelFormatKind.cudaChannelFormatKindUnsignedBlockCompressed7SRGB

class cudaMemoryType(IntEnum):
    """
    CUDA memory types
    """

    #: Unregistered memory
    cudaMemoryTypeUnregistered = ccudart.cudaMemoryType.cudaMemoryTypeUnregistered

    #: Host memory
    cudaMemoryTypeHost = ccudart.cudaMemoryType.cudaMemoryTypeHost

    #: Device memory
    cudaMemoryTypeDevice = ccudart.cudaMemoryType.cudaMemoryTypeDevice

    #: Managed memory
    cudaMemoryTypeManaged = ccudart.cudaMemoryType.cudaMemoryTypeManaged

class cudaMemcpyKind(IntEnum):
    """
    CUDA memory copy types
    """

    #: Host -> Host
    cudaMemcpyHostToHost = ccudart.cudaMemcpyKind.cudaMemcpyHostToHost

    #: Host -> Device
    cudaMemcpyHostToDevice = ccudart.cudaMemcpyKind.cudaMemcpyHostToDevice

    #: Device -> Host
    cudaMemcpyDeviceToHost = ccudart.cudaMemcpyKind.cudaMemcpyDeviceToHost

    #: Device -> Device
    cudaMemcpyDeviceToDevice = ccudart.cudaMemcpyKind.cudaMemcpyDeviceToDevice

    #: Direction of the transfer is inferred from the pointer values.
    #: Requires unified virtual addressing
    cudaMemcpyDefault = ccudart.cudaMemcpyKind.cudaMemcpyDefault

class cudaAccessProperty(IntEnum):
    """
    Specifies performance hint with :py:obj:`~.cudaAccessPolicyWindow`
    for hitProp and missProp members.
    """

    #: Normal cache persistence.
    cudaAccessPropertyNormal = ccudart.cudaAccessProperty.cudaAccessPropertyNormal

    #: Streaming access is less likely to persit from cache.
    cudaAccessPropertyStreaming = ccudart.cudaAccessProperty.cudaAccessPropertyStreaming

    #: Persisting access is more likely to persist in cache.
    cudaAccessPropertyPersisting = ccudart.cudaAccessProperty.cudaAccessPropertyPersisting

class cudaStreamCaptureStatus(IntEnum):
    """
    Possible stream capture statuses returned by
    :py:obj:`~.cudaStreamIsCapturing`
    """

    #: Stream is not capturing
    cudaStreamCaptureStatusNone = ccudart.cudaStreamCaptureStatus.cudaStreamCaptureStatusNone

    #: Stream is actively capturing
    cudaStreamCaptureStatusActive = ccudart.cudaStreamCaptureStatus.cudaStreamCaptureStatusActive

    #: Stream is part of a capture sequence that has been invalidated, but
    #: not terminated
    cudaStreamCaptureStatusInvalidated = ccudart.cudaStreamCaptureStatus.cudaStreamCaptureStatusInvalidated

class cudaStreamCaptureMode(IntEnum):
    """
    Possible modes for stream capture thread interactions. For more
    details see :py:obj:`~.cudaStreamBeginCapture` and
    :py:obj:`~.cudaThreadExchangeStreamCaptureMode`
    """
    cudaStreamCaptureModeGlobal = ccudart.cudaStreamCaptureMode.cudaStreamCaptureModeGlobal
    cudaStreamCaptureModeThreadLocal = ccudart.cudaStreamCaptureMode.cudaStreamCaptureModeThreadLocal
    cudaStreamCaptureModeRelaxed = ccudart.cudaStreamCaptureMode.cudaStreamCaptureModeRelaxed

class cudaSynchronizationPolicy(IntEnum):
    """

    """
    cudaSyncPolicyAuto = ccudart.cudaSynchronizationPolicy.cudaSyncPolicyAuto
    cudaSyncPolicySpin = ccudart.cudaSynchronizationPolicy.cudaSyncPolicySpin
    cudaSyncPolicyYield = ccudart.cudaSynchronizationPolicy.cudaSyncPolicyYield
    cudaSyncPolicyBlockingSync = ccudart.cudaSynchronizationPolicy.cudaSyncPolicyBlockingSync

class cudaStreamUpdateCaptureDependenciesFlags(IntEnum):
    """
    Flags for :py:obj:`~.cudaStreamUpdateCaptureDependencies`
    """

    #: Add new nodes to the dependency set
    cudaStreamAddCaptureDependencies = ccudart.cudaStreamUpdateCaptureDependenciesFlags.cudaStreamAddCaptureDependencies

    #: Replace the dependency set with the new nodes
    cudaStreamSetCaptureDependencies = ccudart.cudaStreamUpdateCaptureDependenciesFlags.cudaStreamSetCaptureDependencies

class cudaUserObjectFlags(IntEnum):
    """
    Flags for user objects for graphs
    """

    #: Indicates the destructor execution is not synchronized by any CUDA
    #: handle.
    cudaUserObjectNoDestructorSync = ccudart.cudaUserObjectFlags.cudaUserObjectNoDestructorSync

class cudaUserObjectRetainFlags(IntEnum):
    """
    Flags for retaining user object references for graphs
    """

    #: Transfer references from the caller rather than creating new
    #: references.
    cudaGraphUserObjectMove = ccudart.cudaUserObjectRetainFlags.cudaGraphUserObjectMove

class cudaGraphicsRegisterFlags(IntEnum):
    """
    CUDA graphics interop register flags
    """

    #: Default
    cudaGraphicsRegisterFlagsNone = ccudart.cudaGraphicsRegisterFlags.cudaGraphicsRegisterFlagsNone

    #: CUDA will not write to this resource
    cudaGraphicsRegisterFlagsReadOnly = ccudart.cudaGraphicsRegisterFlags.cudaGraphicsRegisterFlagsReadOnly

    #: CUDA will only write to and will not read from this resource
    cudaGraphicsRegisterFlagsWriteDiscard = ccudart.cudaGraphicsRegisterFlags.cudaGraphicsRegisterFlagsWriteDiscard

    #: CUDA will bind this resource to a surface reference
    cudaGraphicsRegisterFlagsSurfaceLoadStore = ccudart.cudaGraphicsRegisterFlags.cudaGraphicsRegisterFlagsSurfaceLoadStore

    #: CUDA will perform texture gather operations on this resource
    cudaGraphicsRegisterFlagsTextureGather = ccudart.cudaGraphicsRegisterFlags.cudaGraphicsRegisterFlagsTextureGather

class cudaGraphicsMapFlags(IntEnum):
    """
    CUDA graphics interop map flags
    """

    #: Default; Assume resource can be read/written
    cudaGraphicsMapFlagsNone = ccudart.cudaGraphicsMapFlags.cudaGraphicsMapFlagsNone

    #: CUDA will not write to this resource
    cudaGraphicsMapFlagsReadOnly = ccudart.cudaGraphicsMapFlags.cudaGraphicsMapFlagsReadOnly

    #: CUDA will only write to and will not read from this resource
    cudaGraphicsMapFlagsWriteDiscard = ccudart.cudaGraphicsMapFlags.cudaGraphicsMapFlagsWriteDiscard

class cudaGraphicsCubeFace(IntEnum):
    """
    CUDA graphics interop array indices for cube maps
    """

    #: Positive X face of cubemap
    cudaGraphicsCubeFacePositiveX = ccudart.cudaGraphicsCubeFace.cudaGraphicsCubeFacePositiveX

    #: Negative X face of cubemap
    cudaGraphicsCubeFaceNegativeX = ccudart.cudaGraphicsCubeFace.cudaGraphicsCubeFaceNegativeX

    #: Positive Y face of cubemap
    cudaGraphicsCubeFacePositiveY = ccudart.cudaGraphicsCubeFace.cudaGraphicsCubeFacePositiveY

    #: Negative Y face of cubemap
    cudaGraphicsCubeFaceNegativeY = ccudart.cudaGraphicsCubeFace.cudaGraphicsCubeFaceNegativeY

    #: Positive Z face of cubemap
    cudaGraphicsCubeFacePositiveZ = ccudart.cudaGraphicsCubeFace.cudaGraphicsCubeFacePositiveZ

    #: Negative Z face of cubemap
    cudaGraphicsCubeFaceNegativeZ = ccudart.cudaGraphicsCubeFace.cudaGraphicsCubeFaceNegativeZ

class cudaResourceType(IntEnum):
    """
    CUDA resource types
    """

    #: Array resource
    cudaResourceTypeArray = ccudart.cudaResourceType.cudaResourceTypeArray

    #: Mipmapped array resource
    cudaResourceTypeMipmappedArray = ccudart.cudaResourceType.cudaResourceTypeMipmappedArray

    #: Linear resource
    cudaResourceTypeLinear = ccudart.cudaResourceType.cudaResourceTypeLinear

    #: Pitch 2D resource
    cudaResourceTypePitch2D = ccudart.cudaResourceType.cudaResourceTypePitch2D

class cudaResourceViewFormat(IntEnum):
    """
    CUDA texture resource view formats
    """

    #: No resource view format (use underlying resource format)
    cudaResViewFormatNone = ccudart.cudaResourceViewFormat.cudaResViewFormatNone

    #: 1 channel unsigned 8-bit integers
    cudaResViewFormatUnsignedChar1 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedChar1

    #: 2 channel unsigned 8-bit integers
    cudaResViewFormatUnsignedChar2 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedChar2

    #: 4 channel unsigned 8-bit integers
    cudaResViewFormatUnsignedChar4 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedChar4

    #: 1 channel signed 8-bit integers
    cudaResViewFormatSignedChar1 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedChar1

    #: 2 channel signed 8-bit integers
    cudaResViewFormatSignedChar2 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedChar2

    #: 4 channel signed 8-bit integers
    cudaResViewFormatSignedChar4 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedChar4

    #: 1 channel unsigned 16-bit integers
    cudaResViewFormatUnsignedShort1 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedShort1

    #: 2 channel unsigned 16-bit integers
    cudaResViewFormatUnsignedShort2 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedShort2

    #: 4 channel unsigned 16-bit integers
    cudaResViewFormatUnsignedShort4 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedShort4

    #: 1 channel signed 16-bit integers
    cudaResViewFormatSignedShort1 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedShort1

    #: 2 channel signed 16-bit integers
    cudaResViewFormatSignedShort2 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedShort2

    #: 4 channel signed 16-bit integers
    cudaResViewFormatSignedShort4 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedShort4

    #: 1 channel unsigned 32-bit integers
    cudaResViewFormatUnsignedInt1 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedInt1

    #: 2 channel unsigned 32-bit integers
    cudaResViewFormatUnsignedInt2 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedInt2

    #: 4 channel unsigned 32-bit integers
    cudaResViewFormatUnsignedInt4 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedInt4

    #: 1 channel signed 32-bit integers
    cudaResViewFormatSignedInt1 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedInt1

    #: 2 channel signed 32-bit integers
    cudaResViewFormatSignedInt2 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedInt2

    #: 4 channel signed 32-bit integers
    cudaResViewFormatSignedInt4 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedInt4

    #: 1 channel 16-bit floating point
    cudaResViewFormatHalf1 = ccudart.cudaResourceViewFormat.cudaResViewFormatHalf1

    #: 2 channel 16-bit floating point
    cudaResViewFormatHalf2 = ccudart.cudaResourceViewFormat.cudaResViewFormatHalf2

    #: 4 channel 16-bit floating point
    cudaResViewFormatHalf4 = ccudart.cudaResourceViewFormat.cudaResViewFormatHalf4

    #: 1 channel 32-bit floating point
    cudaResViewFormatFloat1 = ccudart.cudaResourceViewFormat.cudaResViewFormatFloat1

    #: 2 channel 32-bit floating point
    cudaResViewFormatFloat2 = ccudart.cudaResourceViewFormat.cudaResViewFormatFloat2

    #: 4 channel 32-bit floating point
    cudaResViewFormatFloat4 = ccudart.cudaResourceViewFormat.cudaResViewFormatFloat4

    #: Block compressed 1
    cudaResViewFormatUnsignedBlockCompressed1 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedBlockCompressed1

    #: Block compressed 2
    cudaResViewFormatUnsignedBlockCompressed2 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedBlockCompressed2

    #: Block compressed 3
    cudaResViewFormatUnsignedBlockCompressed3 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedBlockCompressed3

    #: Block compressed 4 unsigned
    cudaResViewFormatUnsignedBlockCompressed4 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedBlockCompressed4

    #: Block compressed 4 signed
    cudaResViewFormatSignedBlockCompressed4 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedBlockCompressed4

    #: Block compressed 5 unsigned
    cudaResViewFormatUnsignedBlockCompressed5 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedBlockCompressed5

    #: Block compressed 5 signed
    cudaResViewFormatSignedBlockCompressed5 = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedBlockCompressed5

    #: Block compressed 6 unsigned half-float
    cudaResViewFormatUnsignedBlockCompressed6H = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedBlockCompressed6H

    #: Block compressed 6 signed half-float
    cudaResViewFormatSignedBlockCompressed6H = ccudart.cudaResourceViewFormat.cudaResViewFormatSignedBlockCompressed6H

    #: Block compressed 7
    cudaResViewFormatUnsignedBlockCompressed7 = ccudart.cudaResourceViewFormat.cudaResViewFormatUnsignedBlockCompressed7

class cudaFuncAttribute(IntEnum):
    """
    CUDA function attributes that can be set using
    :py:obj:`~.cudaFuncSetAttribute`
    """

    #: Maximum dynamic shared memory size
    cudaFuncAttributeMaxDynamicSharedMemorySize = ccudart.cudaFuncAttribute.cudaFuncAttributeMaxDynamicSharedMemorySize

    #: Preferred shared memory-L1 cache split
    cudaFuncAttributePreferredSharedMemoryCarveout = ccudart.cudaFuncAttribute.cudaFuncAttributePreferredSharedMemoryCarveout
    cudaFuncAttributeMax = ccudart.cudaFuncAttribute.cudaFuncAttributeMax

class cudaFuncCache(IntEnum):
    """
    CUDA function cache configurations
    """

    #: Default function cache configuration, no preference
    cudaFuncCachePreferNone = ccudart.cudaFuncCache.cudaFuncCachePreferNone

    #: Prefer larger shared memory and smaller L1 cache
    cudaFuncCachePreferShared = ccudart.cudaFuncCache.cudaFuncCachePreferShared

    #: Prefer larger L1 cache and smaller shared memory
    cudaFuncCachePreferL1 = ccudart.cudaFuncCache.cudaFuncCachePreferL1

    #: Prefer equal size L1 cache and shared memory
    cudaFuncCachePreferEqual = ccudart.cudaFuncCache.cudaFuncCachePreferEqual

class cudaSharedMemConfig(IntEnum):
    """
    CUDA shared memory configuration
    """
    cudaSharedMemBankSizeDefault = ccudart.cudaSharedMemConfig.cudaSharedMemBankSizeDefault
    cudaSharedMemBankSizeFourByte = ccudart.cudaSharedMemConfig.cudaSharedMemBankSizeFourByte
    cudaSharedMemBankSizeEightByte = ccudart.cudaSharedMemConfig.cudaSharedMemBankSizeEightByte

class cudaSharedCarveout(IntEnum):
    """
    Shared memory carveout configurations. These may be passed to
    cudaFuncSetAttribute
    """

    #: No preference for shared memory or L1 (default)
    cudaSharedmemCarveoutDefault = ccudart.cudaSharedCarveout.cudaSharedmemCarveoutDefault

    #: Prefer maximum available shared memory, minimum L1 cache
    cudaSharedmemCarveoutMaxShared = ccudart.cudaSharedCarveout.cudaSharedmemCarveoutMaxShared

    #: Prefer maximum available L1 cache, minimum shared memory
    cudaSharedmemCarveoutMaxL1 = ccudart.cudaSharedCarveout.cudaSharedmemCarveoutMaxL1

class cudaComputeMode(IntEnum):
    """
    CUDA device compute modes
    """

    #: Default compute mode (Multiple threads can use
    #: :py:obj:`~.cudaSetDevice()` with this device)
    cudaComputeModeDefault = ccudart.cudaComputeMode.cudaComputeModeDefault

    #: Compute-exclusive-thread mode (Only one thread in one process will
    #: be able to use :py:obj:`~.cudaSetDevice()` with this device)
    cudaComputeModeExclusive = ccudart.cudaComputeMode.cudaComputeModeExclusive

    #: Compute-prohibited mode (No threads can use
    #: :py:obj:`~.cudaSetDevice()` with this device)
    cudaComputeModeProhibited = ccudart.cudaComputeMode.cudaComputeModeProhibited

    #: Compute-exclusive-process mode (Many threads in one process will be
    #: able to use :py:obj:`~.cudaSetDevice()` with this device)
    cudaComputeModeExclusiveProcess = ccudart.cudaComputeMode.cudaComputeModeExclusiveProcess

class cudaLimit(IntEnum):
    """
    CUDA Limits
    """

    #: GPU thread stack size
    cudaLimitStackSize = ccudart.cudaLimit.cudaLimitStackSize

    #: GPU printf FIFO size
    cudaLimitPrintfFifoSize = ccudart.cudaLimit.cudaLimitPrintfFifoSize

    #: GPU malloc heap size
    cudaLimitMallocHeapSize = ccudart.cudaLimit.cudaLimitMallocHeapSize

    #: GPU device runtime synchronize depth
    cudaLimitDevRuntimeSyncDepth = ccudart.cudaLimit.cudaLimitDevRuntimeSyncDepth

    #: GPU device runtime pending launch count
    cudaLimitDevRuntimePendingLaunchCount = ccudart.cudaLimit.cudaLimitDevRuntimePendingLaunchCount

    #: A value between 0 and 128 that indicates the maximum fetch
    #: granularity of L2 (in Bytes). This is a hint
    cudaLimitMaxL2FetchGranularity = ccudart.cudaLimit.cudaLimitMaxL2FetchGranularity

    #: A size in bytes for L2 persisting lines cache size
    cudaLimitPersistingL2CacheSize = ccudart.cudaLimit.cudaLimitPersistingL2CacheSize

class cudaMemoryAdvise(IntEnum):
    """
    CUDA Memory Advise values
    """

    #: Data will mostly be read and only occassionally be written to
    cudaMemAdviseSetReadMostly = ccudart.cudaMemoryAdvise.cudaMemAdviseSetReadMostly

    #: Undo the effect of :py:obj:`~.cudaMemAdviseSetReadMostly`
    cudaMemAdviseUnsetReadMostly = ccudart.cudaMemoryAdvise.cudaMemAdviseUnsetReadMostly

    #: Set the preferred location for the data as the specified device
    cudaMemAdviseSetPreferredLocation = ccudart.cudaMemoryAdvise.cudaMemAdviseSetPreferredLocation

    #: Clear the preferred location for the data
    cudaMemAdviseUnsetPreferredLocation = ccudart.cudaMemoryAdvise.cudaMemAdviseUnsetPreferredLocation

    #: Data will be accessed by the specified device, so prevent page
    #: faults as much as possible
    cudaMemAdviseSetAccessedBy = ccudart.cudaMemoryAdvise.cudaMemAdviseSetAccessedBy

    #: Let the Unified Memory subsystem decide on the page faulting policy
    #: for the specified device
    cudaMemAdviseUnsetAccessedBy = ccudart.cudaMemoryAdvise.cudaMemAdviseUnsetAccessedBy

class cudaMemRangeAttribute(IntEnum):
    """
    CUDA range attributes
    """

    #: Whether the range will mostly be read and only occassionally be
    #: written to
    cudaMemRangeAttributeReadMostly = ccudart.cudaMemRangeAttribute.cudaMemRangeAttributeReadMostly

    #: The preferred location of the range
    cudaMemRangeAttributePreferredLocation = ccudart.cudaMemRangeAttribute.cudaMemRangeAttributePreferredLocation

    #: Memory range has :py:obj:`~.cudaMemAdviseSetAccessedBy` set for
    #: specified device
    cudaMemRangeAttributeAccessedBy = ccudart.cudaMemRangeAttribute.cudaMemRangeAttributeAccessedBy

    #: The last location to which the range was prefetched
    cudaMemRangeAttributeLastPrefetchLocation = ccudart.cudaMemRangeAttribute.cudaMemRangeAttributeLastPrefetchLocation

class cudaOutputMode_t(IntEnum):
    """
    CUDA Profiler Output modes
    """

    #: Output mode Key-Value pair format.
    cudaKeyValuePair = ccudart.cudaOutputMode.cudaKeyValuePair

    #: Output mode Comma separated values format.
    cudaCSV = ccudart.cudaOutputMode.cudaCSV

class cudaFlushGPUDirectRDMAWritesOptions(IntEnum):
    """
    CUDA GPUDirect RDMA flush writes APIs supported on the device
    """

    #: :py:obj:`~.cudaDeviceFlushGPUDirectRDMAWrites()` and its CUDA Driver
    #: API counterpart are supported on the device.
    cudaFlushGPUDirectRDMAWritesOptionHost = ccudart.cudaFlushGPUDirectRDMAWritesOptions.cudaFlushGPUDirectRDMAWritesOptionHost

    #: The :py:obj:`~.CU_STREAM_WAIT_VALUE_FLUSH` flag and the
    #: :py:obj:`~.CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES` MemOp are supported
    #: on the CUDA device.
    cudaFlushGPUDirectRDMAWritesOptionMemOps = ccudart.cudaFlushGPUDirectRDMAWritesOptions.cudaFlushGPUDirectRDMAWritesOptionMemOps

class cudaGPUDirectRDMAWritesOrdering(IntEnum):
    """
    CUDA GPUDirect RDMA flush writes ordering features of the device
    """

    #: The device does not natively support ordering of GPUDirect RDMA
    #: writes. :py:obj:`~.cudaFlushGPUDirectRDMAWrites()` can be leveraged
    #: if supported.
    cudaGPUDirectRDMAWritesOrderingNone = ccudart.cudaGPUDirectRDMAWritesOrdering.cudaGPUDirectRDMAWritesOrderingNone

    #: Natively, the device can consistently consume GPUDirect RDMA writes,
    #: although other CUDA devices may not.
    cudaGPUDirectRDMAWritesOrderingOwner = ccudart.cudaGPUDirectRDMAWritesOrdering.cudaGPUDirectRDMAWritesOrderingOwner

    #: Any CUDA device in the system can consistently consume GPUDirect
    #: RDMA writes to this device.
    cudaGPUDirectRDMAWritesOrderingAllDevices = ccudart.cudaGPUDirectRDMAWritesOrdering.cudaGPUDirectRDMAWritesOrderingAllDevices

class cudaFlushGPUDirectRDMAWritesScope(IntEnum):
    """
    CUDA GPUDirect RDMA flush writes scopes
    """

    #: Blocks until remote writes are visible to the CUDA device context
    #: owning the data.
    cudaFlushGPUDirectRDMAWritesToOwner = ccudart.cudaFlushGPUDirectRDMAWritesScope.cudaFlushGPUDirectRDMAWritesToOwner

    #: Blocks until remote writes are visible to all CUDA device contexts.
    cudaFlushGPUDirectRDMAWritesToAllDevices = ccudart.cudaFlushGPUDirectRDMAWritesScope.cudaFlushGPUDirectRDMAWritesToAllDevices

class cudaFlushGPUDirectRDMAWritesTarget(IntEnum):
    """
    CUDA GPUDirect RDMA flush writes targets
    """

    #: Sets the target for :py:obj:`~.cudaDeviceFlushGPUDirectRDMAWrites()`
    #: to the currently active CUDA device context.
    cudaFlushGPUDirectRDMAWritesTargetCurrentDevice = ccudart.cudaFlushGPUDirectRDMAWritesTarget.cudaFlushGPUDirectRDMAWritesTargetCurrentDevice

class cudaDeviceAttr(IntEnum):
    """
    CUDA device attributes
    """

    #: Maximum number of threads per block
    cudaDevAttrMaxThreadsPerBlock = ccudart.cudaDeviceAttr.cudaDevAttrMaxThreadsPerBlock

    #: Maximum block dimension X
    cudaDevAttrMaxBlockDimX = ccudart.cudaDeviceAttr.cudaDevAttrMaxBlockDimX

    #: Maximum block dimension Y
    cudaDevAttrMaxBlockDimY = ccudart.cudaDeviceAttr.cudaDevAttrMaxBlockDimY

    #: Maximum block dimension Z
    cudaDevAttrMaxBlockDimZ = ccudart.cudaDeviceAttr.cudaDevAttrMaxBlockDimZ

    #: Maximum grid dimension X
    cudaDevAttrMaxGridDimX = ccudart.cudaDeviceAttr.cudaDevAttrMaxGridDimX

    #: Maximum grid dimension Y
    cudaDevAttrMaxGridDimY = ccudart.cudaDeviceAttr.cudaDevAttrMaxGridDimY

    #: Maximum grid dimension Z
    cudaDevAttrMaxGridDimZ = ccudart.cudaDeviceAttr.cudaDevAttrMaxGridDimZ

    #: Maximum shared memory available per block in bytes
    cudaDevAttrMaxSharedMemoryPerBlock = ccudart.cudaDeviceAttr.cudaDevAttrMaxSharedMemoryPerBlock

    #: Memory available on device for constant variables in a CUDA C kernel
    #: in bytes
    cudaDevAttrTotalConstantMemory = ccudart.cudaDeviceAttr.cudaDevAttrTotalConstantMemory

    #: Warp size in threads
    cudaDevAttrWarpSize = ccudart.cudaDeviceAttr.cudaDevAttrWarpSize

    #: Maximum pitch in bytes allowed by memory copies
    cudaDevAttrMaxPitch = ccudart.cudaDeviceAttr.cudaDevAttrMaxPitch

    #: Maximum number of 32-bit registers available per block
    cudaDevAttrMaxRegistersPerBlock = ccudart.cudaDeviceAttr.cudaDevAttrMaxRegistersPerBlock

    #: Peak clock frequency in kilohertz
    cudaDevAttrClockRate = ccudart.cudaDeviceAttr.cudaDevAttrClockRate

    #: Alignment requirement for textures
    cudaDevAttrTextureAlignment = ccudart.cudaDeviceAttr.cudaDevAttrTextureAlignment

    #: Device can possibly copy memory and execute a kernel concurrently
    cudaDevAttrGpuOverlap = ccudart.cudaDeviceAttr.cudaDevAttrGpuOverlap

    #: Number of multiprocessors on device
    cudaDevAttrMultiProcessorCount = ccudart.cudaDeviceAttr.cudaDevAttrMultiProcessorCount

    #: Specifies whether there is a run time limit on kernels
    cudaDevAttrKernelExecTimeout = ccudart.cudaDeviceAttr.cudaDevAttrKernelExecTimeout

    #: Device is integrated with host memory
    cudaDevAttrIntegrated = ccudart.cudaDeviceAttr.cudaDevAttrIntegrated

    #: Device can map host memory into CUDA address space
    cudaDevAttrCanMapHostMemory = ccudart.cudaDeviceAttr.cudaDevAttrCanMapHostMemory

    #: Compute mode (See :py:obj:`~.cudaComputeMode` for details)
    cudaDevAttrComputeMode = ccudart.cudaDeviceAttr.cudaDevAttrComputeMode

    #: Maximum 1D texture width
    cudaDevAttrMaxTexture1DWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture1DWidth

    #: Maximum 2D texture width
    cudaDevAttrMaxTexture2DWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DWidth

    #: Maximum 2D texture height
    cudaDevAttrMaxTexture2DHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DHeight

    #: Maximum 3D texture width
    cudaDevAttrMaxTexture3DWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture3DWidth

    #: Maximum 3D texture height
    cudaDevAttrMaxTexture3DHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture3DHeight

    #: Maximum 3D texture depth
    cudaDevAttrMaxTexture3DDepth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture3DDepth

    #: Maximum 2D layered texture width
    cudaDevAttrMaxTexture2DLayeredWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DLayeredWidth

    #: Maximum 2D layered texture height
    cudaDevAttrMaxTexture2DLayeredHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DLayeredHeight

    #: Maximum layers in a 2D layered texture
    cudaDevAttrMaxTexture2DLayeredLayers = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DLayeredLayers

    #: Alignment requirement for surfaces
    cudaDevAttrSurfaceAlignment = ccudart.cudaDeviceAttr.cudaDevAttrSurfaceAlignment

    #: Device can possibly execute multiple kernels concurrently
    cudaDevAttrConcurrentKernels = ccudart.cudaDeviceAttr.cudaDevAttrConcurrentKernels

    #: Device has ECC support enabled
    cudaDevAttrEccEnabled = ccudart.cudaDeviceAttr.cudaDevAttrEccEnabled

    #: PCI bus ID of the device
    cudaDevAttrPciBusId = ccudart.cudaDeviceAttr.cudaDevAttrPciBusId

    #: PCI device ID of the device
    cudaDevAttrPciDeviceId = ccudart.cudaDeviceAttr.cudaDevAttrPciDeviceId

    #: Device is using TCC driver model
    cudaDevAttrTccDriver = ccudart.cudaDeviceAttr.cudaDevAttrTccDriver

    #: Peak memory clock frequency in kilohertz
    cudaDevAttrMemoryClockRate = ccudart.cudaDeviceAttr.cudaDevAttrMemoryClockRate

    #: Global memory bus width in bits
    cudaDevAttrGlobalMemoryBusWidth = ccudart.cudaDeviceAttr.cudaDevAttrGlobalMemoryBusWidth

    #: Size of L2 cache in bytes
    cudaDevAttrL2CacheSize = ccudart.cudaDeviceAttr.cudaDevAttrL2CacheSize

    #: Maximum resident threads per multiprocessor
    cudaDevAttrMaxThreadsPerMultiProcessor = ccudart.cudaDeviceAttr.cudaDevAttrMaxThreadsPerMultiProcessor

    #: Number of asynchronous engines
    cudaDevAttrAsyncEngineCount = ccudart.cudaDeviceAttr.cudaDevAttrAsyncEngineCount

    #: Device shares a unified address space with the host
    cudaDevAttrUnifiedAddressing = ccudart.cudaDeviceAttr.cudaDevAttrUnifiedAddressing

    #: Maximum 1D layered texture width
    cudaDevAttrMaxTexture1DLayeredWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture1DLayeredWidth

    #: Maximum layers in a 1D layered texture
    cudaDevAttrMaxTexture1DLayeredLayers = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture1DLayeredLayers

    #: Maximum 2D texture width if cudaArrayTextureGather is set
    cudaDevAttrMaxTexture2DGatherWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DGatherWidth

    #: Maximum 2D texture height if cudaArrayTextureGather is set
    cudaDevAttrMaxTexture2DGatherHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DGatherHeight

    #: Alternate maximum 3D texture width
    cudaDevAttrMaxTexture3DWidthAlt = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture3DWidthAlt

    #: Alternate maximum 3D texture height
    cudaDevAttrMaxTexture3DHeightAlt = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture3DHeightAlt

    #: Alternate maximum 3D texture depth
    cudaDevAttrMaxTexture3DDepthAlt = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture3DDepthAlt

    #: PCI domain ID of the device
    cudaDevAttrPciDomainId = ccudart.cudaDeviceAttr.cudaDevAttrPciDomainId

    #: Pitch alignment requirement for textures
    cudaDevAttrTexturePitchAlignment = ccudart.cudaDeviceAttr.cudaDevAttrTexturePitchAlignment

    #: Maximum cubemap texture width/height
    cudaDevAttrMaxTextureCubemapWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTextureCubemapWidth

    #: Maximum cubemap layered texture width/height
    cudaDevAttrMaxTextureCubemapLayeredWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTextureCubemapLayeredWidth

    #: Maximum layers in a cubemap layered texture
    cudaDevAttrMaxTextureCubemapLayeredLayers = ccudart.cudaDeviceAttr.cudaDevAttrMaxTextureCubemapLayeredLayers

    #: Maximum 1D surface width
    cudaDevAttrMaxSurface1DWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface1DWidth

    #: Maximum 2D surface width
    cudaDevAttrMaxSurface2DWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface2DWidth

    #: Maximum 2D surface height
    cudaDevAttrMaxSurface2DHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface2DHeight

    #: Maximum 3D surface width
    cudaDevAttrMaxSurface3DWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface3DWidth

    #: Maximum 3D surface height
    cudaDevAttrMaxSurface3DHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface3DHeight

    #: Maximum 3D surface depth
    cudaDevAttrMaxSurface3DDepth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface3DDepth

    #: Maximum 1D layered surface width
    cudaDevAttrMaxSurface1DLayeredWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface1DLayeredWidth

    #: Maximum layers in a 1D layered surface
    cudaDevAttrMaxSurface1DLayeredLayers = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface1DLayeredLayers

    #: Maximum 2D layered surface width
    cudaDevAttrMaxSurface2DLayeredWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface2DLayeredWidth

    #: Maximum 2D layered surface height
    cudaDevAttrMaxSurface2DLayeredHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface2DLayeredHeight

    #: Maximum layers in a 2D layered surface
    cudaDevAttrMaxSurface2DLayeredLayers = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurface2DLayeredLayers

    #: Maximum cubemap surface width
    cudaDevAttrMaxSurfaceCubemapWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurfaceCubemapWidth

    #: Maximum cubemap layered surface width
    cudaDevAttrMaxSurfaceCubemapLayeredWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurfaceCubemapLayeredWidth

    #: Maximum layers in a cubemap layered surface
    cudaDevAttrMaxSurfaceCubemapLayeredLayers = ccudart.cudaDeviceAttr.cudaDevAttrMaxSurfaceCubemapLayeredLayers

    #: Maximum 1D linear texture width
    cudaDevAttrMaxTexture1DLinearWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture1DLinearWidth

    #: Maximum 2D linear texture width
    cudaDevAttrMaxTexture2DLinearWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DLinearWidth

    #: Maximum 2D linear texture height
    cudaDevAttrMaxTexture2DLinearHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DLinearHeight

    #: Maximum 2D linear texture pitch in bytes
    cudaDevAttrMaxTexture2DLinearPitch = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DLinearPitch

    #: Maximum mipmapped 2D texture width
    cudaDevAttrMaxTexture2DMipmappedWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DMipmappedWidth

    #: Maximum mipmapped 2D texture height
    cudaDevAttrMaxTexture2DMipmappedHeight = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture2DMipmappedHeight

    #: Major compute capability version number
    cudaDevAttrComputeCapabilityMajor = ccudart.cudaDeviceAttr.cudaDevAttrComputeCapabilityMajor

    #: Minor compute capability version number
    cudaDevAttrComputeCapabilityMinor = ccudart.cudaDeviceAttr.cudaDevAttrComputeCapabilityMinor

    #: Maximum mipmapped 1D texture width
    cudaDevAttrMaxTexture1DMipmappedWidth = ccudart.cudaDeviceAttr.cudaDevAttrMaxTexture1DMipmappedWidth

    #: Device supports stream priorities
    cudaDevAttrStreamPrioritiesSupported = ccudart.cudaDeviceAttr.cudaDevAttrStreamPrioritiesSupported

    #: Device supports caching globals in L1
    cudaDevAttrGlobalL1CacheSupported = ccudart.cudaDeviceAttr.cudaDevAttrGlobalL1CacheSupported

    #: Device supports caching locals in L1
    cudaDevAttrLocalL1CacheSupported = ccudart.cudaDeviceAttr.cudaDevAttrLocalL1CacheSupported

    #: Maximum shared memory available per multiprocessor in bytes
    cudaDevAttrMaxSharedMemoryPerMultiprocessor = ccudart.cudaDeviceAttr.cudaDevAttrMaxSharedMemoryPerMultiprocessor

    #: Maximum number of 32-bit registers available per multiprocessor
    cudaDevAttrMaxRegistersPerMultiprocessor = ccudart.cudaDeviceAttr.cudaDevAttrMaxRegistersPerMultiprocessor

    #: Device can allocate managed memory on this system
    cudaDevAttrManagedMemory = ccudart.cudaDeviceAttr.cudaDevAttrManagedMemory

    #: Device is on a multi-GPU board
    cudaDevAttrIsMultiGpuBoard = ccudart.cudaDeviceAttr.cudaDevAttrIsMultiGpuBoard

    #: Unique identifier for a group of devices on the same multi-GPU board
    cudaDevAttrMultiGpuBoardGroupID = ccudart.cudaDeviceAttr.cudaDevAttrMultiGpuBoardGroupID

    #: Link between the device and the host supports native atomic
    #: operations
    cudaDevAttrHostNativeAtomicSupported = ccudart.cudaDeviceAttr.cudaDevAttrHostNativeAtomicSupported

    #: Ratio of single precision performance (in floating-point operations
    #: per second) to double precision performance
    cudaDevAttrSingleToDoublePrecisionPerfRatio = ccudart.cudaDeviceAttr.cudaDevAttrSingleToDoublePrecisionPerfRatio

    #: Device supports coherently accessing pageable memory without calling
    #: cudaHostRegister on it
    cudaDevAttrPageableMemoryAccess = ccudart.cudaDeviceAttr.cudaDevAttrPageableMemoryAccess

    #: Device can coherently access managed memory concurrently with the
    #: CPU
    cudaDevAttrConcurrentManagedAccess = ccudart.cudaDeviceAttr.cudaDevAttrConcurrentManagedAccess

    #: Device supports Compute Preemption
    cudaDevAttrComputePreemptionSupported = ccudart.cudaDeviceAttr.cudaDevAttrComputePreemptionSupported

    #: Device can access host registered memory at the same virtual address
    #: as the CPU
    cudaDevAttrCanUseHostPointerForRegisteredMem = ccudart.cudaDeviceAttr.cudaDevAttrCanUseHostPointerForRegisteredMem
    cudaDevAttrReserved92 = ccudart.cudaDeviceAttr.cudaDevAttrReserved92
    cudaDevAttrReserved93 = ccudart.cudaDeviceAttr.cudaDevAttrReserved93
    cudaDevAttrReserved94 = ccudart.cudaDeviceAttr.cudaDevAttrReserved94

    #: Device supports launching cooperative kernels via
    #: :py:obj:`~.cudaLaunchCooperativeKernel`
    cudaDevAttrCooperativeLaunch = ccudart.cudaDeviceAttr.cudaDevAttrCooperativeLaunch

    #: Deprecated, cudaLaunchCooperativeKernelMultiDevice is deprecated.
    cudaDevAttrCooperativeMultiDeviceLaunch = ccudart.cudaDeviceAttr.cudaDevAttrCooperativeMultiDeviceLaunch

    #: The maximum optin shared memory per block. This value may vary by
    #: chip. See :py:obj:`~.cudaFuncSetAttribute`
    cudaDevAttrMaxSharedMemoryPerBlockOptin = ccudart.cudaDeviceAttr.cudaDevAttrMaxSharedMemoryPerBlockOptin

    #: Device supports flushing of outstanding remote writes.
    cudaDevAttrCanFlushRemoteWrites = ccudart.cudaDeviceAttr.cudaDevAttrCanFlushRemoteWrites

    #: Device supports host memory registration via
    #: :py:obj:`~.cudaHostRegister`.
    cudaDevAttrHostRegisterSupported = ccudart.cudaDeviceAttr.cudaDevAttrHostRegisterSupported

    #: Device accesses pageable memory via the host's page tables.
    cudaDevAttrPageableMemoryAccessUsesHostPageTables = ccudart.cudaDeviceAttr.cudaDevAttrPageableMemoryAccessUsesHostPageTables

    #: Host can directly access managed memory on the device without
    #: migration.
    cudaDevAttrDirectManagedMemAccessFromHost = ccudart.cudaDeviceAttr.cudaDevAttrDirectManagedMemAccessFromHost

    #: Maximum number of blocks per multiprocessor
    cudaDevAttrMaxBlocksPerMultiprocessor = ccudart.cudaDeviceAttr.cudaDevAttrMaxBlocksPerMultiprocessor

    #: Maximum L2 persisting lines capacity setting in bytes.
    cudaDevAttrMaxPersistingL2CacheSize = ccudart.cudaDeviceAttr.cudaDevAttrMaxPersistingL2CacheSize

    #: Maximum value of :py:obj:`~.cudaAccessPolicyWindow.num_bytes`.
    cudaDevAttrMaxAccessPolicyWindowSize = ccudart.cudaDeviceAttr.cudaDevAttrMaxAccessPolicyWindowSize

    #: Shared memory reserved by CUDA driver per block in bytes
    cudaDevAttrReservedSharedMemoryPerBlock = ccudart.cudaDeviceAttr.cudaDevAttrReservedSharedMemoryPerBlock

    #: Device supports sparse CUDA arrays and sparse CUDA mipmapped arrays
    cudaDevAttrSparseCudaArraySupported = ccudart.cudaDeviceAttr.cudaDevAttrSparseCudaArraySupported

    #: Device supports using the :py:obj:`~.cudaHostRegister` flag
    #: cudaHostRegisterReadOnly to register memory that must be mapped as
    #: read-only to the GPU
    cudaDevAttrHostRegisterReadOnlySupported = ccudart.cudaDeviceAttr.cudaDevAttrHostRegisterReadOnlySupported

    #: External timeline semaphore interop is supported on the device
    cudaDevAttrTimelineSemaphoreInteropSupported = ccudart.cudaDeviceAttr.cudaDevAttrTimelineSemaphoreInteropSupported

    #: Deprecated, External timeline semaphore interop is supported on the
    #: device
    cudaDevAttrMaxTimelineSemaphoreInteropSupported = ccudart.cudaDeviceAttr.cudaDevAttrMaxTimelineSemaphoreInteropSupported

    #: Device supports using the :py:obj:`~.cudaMallocAsync` and
    #: :py:obj:`~.cudaMemPool` family of APIs
    cudaDevAttrMemoryPoolsSupported = ccudart.cudaDeviceAttr.cudaDevAttrMemoryPoolsSupported

    #: Device supports GPUDirect RDMA APIs, like nvidia_p2p_get_pages (see
    #: https://docs.nvidia.com/cuda/gpudirect-rdma for more information)
    cudaDevAttrGPUDirectRDMASupported = ccudart.cudaDeviceAttr.cudaDevAttrGPUDirectRDMASupported

    #: The returned attribute shall be interpreted as a bitmask, where the
    #: individual bits are listed in the
    #: :py:obj:`~.cudaFlushGPUDirectRDMAWritesOptions` enum
    cudaDevAttrGPUDirectRDMAFlushWritesOptions = ccudart.cudaDeviceAttr.cudaDevAttrGPUDirectRDMAFlushWritesOptions

    #: GPUDirect RDMA writes to the device do not need to be flushed for
    #: consumers within the scope indicated by the returned attribute. See
    #: :py:obj:`~.cudaGPUDirectRDMAWritesOrdering` for the numerical values
    #: returned here.
    cudaDevAttrGPUDirectRDMAWritesOrdering = ccudart.cudaDeviceAttr.cudaDevAttrGPUDirectRDMAWritesOrdering

    #: Handle types supported with mempool based IPC
    cudaDevAttrMemoryPoolSupportedHandleTypes = ccudart.cudaDeviceAttr.cudaDevAttrMemoryPoolSupportedHandleTypes

    #: Device supports deferred mapping CUDA arrays and CUDA mipmapped
    #: arrays
    cudaDevAttrDeferredMappingCudaArraySupported = ccudart.cudaDeviceAttr.cudaDevAttrDeferredMappingCudaArraySupported
    cudaDevAttrMax = ccudart.cudaDeviceAttr.cudaDevAttrMax

class cudaMemPoolAttr(IntEnum):
    """
    CUDA memory pool attributes
    """

    #: (value type = int) Allow cuMemAllocAsync to use memory
    #: asynchronously freed in another streams as long as a stream ordering
    #: dependency of the allocating stream on the free action exists. Cuda
    #: events and null stream interactions can create the required stream
    #: ordered dependencies. (default enabled)
    cudaMemPoolReuseFollowEventDependencies = ccudart.cudaMemPoolAttr.cudaMemPoolReuseFollowEventDependencies

    #: (value type = int) Allow reuse of already completed frees when there
    #: is no dependency between the free and allocation. (default enabled)
    cudaMemPoolReuseAllowOpportunistic = ccudart.cudaMemPoolAttr.cudaMemPoolReuseAllowOpportunistic

    #: (value type = int) Allow cuMemAllocAsync to insert new stream
    #: dependencies in order to establish the stream ordering required to
    #: reuse a piece of memory released by cuFreeAsync (default enabled).
    cudaMemPoolReuseAllowInternalDependencies = ccudart.cudaMemPoolAttr.cudaMemPoolReuseAllowInternalDependencies

    #: (value type = cuuint64_t) Amount of reserved memory in bytes to hold
    #: onto before trying to release memory back to the OS. When more than
    #: the release threshold bytes of memory are held by the memory pool,
    #: the allocator will try to release memory back to the OS on the next
    #: call to stream, event or context synchronize. (default 0)
    cudaMemPoolAttrReleaseThreshold = ccudart.cudaMemPoolAttr.cudaMemPoolAttrReleaseThreshold

    #: (value type = cuuint64_t) Amount of backing memory currently
    #: allocated for the mempool.
    cudaMemPoolAttrReservedMemCurrent = ccudart.cudaMemPoolAttr.cudaMemPoolAttrReservedMemCurrent

    #: (value type = cuuint64_t) High watermark of backing memory allocated
    #: for the mempool since the last time it was reset. High watermark can
    #: only be reset to zero.
    cudaMemPoolAttrReservedMemHigh = ccudart.cudaMemPoolAttr.cudaMemPoolAttrReservedMemHigh

    #: (value type = cuuint64_t) Amount of memory from the pool that is
    #: currently in use by the application.
    cudaMemPoolAttrUsedMemCurrent = ccudart.cudaMemPoolAttr.cudaMemPoolAttrUsedMemCurrent

    #: (value type = cuuint64_t) High watermark of the amount of memory
    #: from the pool that was in use by the application since the last time
    #: it was reset. High watermark can only be reset to zero.
    cudaMemPoolAttrUsedMemHigh = ccudart.cudaMemPoolAttr.cudaMemPoolAttrUsedMemHigh

class cudaMemLocationType(IntEnum):
    """
    Specifies the type of location
    """
    cudaMemLocationTypeInvalid = ccudart.cudaMemLocationType.cudaMemLocationTypeInvalid

    #: Location is a device location, thus id is a device ordinal
    cudaMemLocationTypeDevice = ccudart.cudaMemLocationType.cudaMemLocationTypeDevice

class cudaMemAccessFlags(IntEnum):
    """
    Specifies the memory protection flags for mapping.
    """

    #: Default, make the address range not accessible
    cudaMemAccessFlagsProtNone = ccudart.cudaMemAccessFlags.cudaMemAccessFlagsProtNone

    #: Make the address range read accessible
    cudaMemAccessFlagsProtRead = ccudart.cudaMemAccessFlags.cudaMemAccessFlagsProtRead

    #: Make the address range read-write accessible
    cudaMemAccessFlagsProtReadWrite = ccudart.cudaMemAccessFlags.cudaMemAccessFlagsProtReadWrite

class cudaMemAllocationType(IntEnum):
    """
    Defines the allocation types available
    """
    cudaMemAllocationTypeInvalid = ccudart.cudaMemAllocationType.cudaMemAllocationTypeInvalid

    #: This allocation type is 'pinned', i.e. cannot migrate from its
    #: current location while the application is actively using it
    cudaMemAllocationTypePinned = ccudart.cudaMemAllocationType.cudaMemAllocationTypePinned
    cudaMemAllocationTypeMax = ccudart.cudaMemAllocationType.cudaMemAllocationTypeMax

class cudaMemAllocationHandleType(IntEnum):
    """
    Flags for specifying particular handle types
    """

    #: Does not allow any export mechanism. >
    cudaMemHandleTypeNone = ccudart.cudaMemAllocationHandleType.cudaMemHandleTypeNone

    #: Allows a file descriptor to be used for exporting. Permitted only on
    #: POSIX systems. (int)
    cudaMemHandleTypePosixFileDescriptor = ccudart.cudaMemAllocationHandleType.cudaMemHandleTypePosixFileDescriptor

    #: Allows a Win32 NT handle to be used for exporting. (HANDLE)
    cudaMemHandleTypeWin32 = ccudart.cudaMemAllocationHandleType.cudaMemHandleTypeWin32

    #: Allows a Win32 KMT handle to be used for exporting. (D3DKMT_HANDLE)
    cudaMemHandleTypeWin32Kmt = ccudart.cudaMemAllocationHandleType.cudaMemHandleTypeWin32Kmt

class cudaGraphMemAttributeType(IntEnum):
    """
    Graph memory attributes
    """

    #: (value type = cuuint64_t) Amount of memory, in bytes, currently
    #: associated with graphs.
    cudaGraphMemAttrUsedMemCurrent = ccudart.cudaGraphMemAttributeType.cudaGraphMemAttrUsedMemCurrent

    #: (value type = cuuint64_t) High watermark of memory, in bytes,
    #: associated with graphs since the last time it was reset. High
    #: watermark can only be reset to zero.
    cudaGraphMemAttrUsedMemHigh = ccudart.cudaGraphMemAttributeType.cudaGraphMemAttrUsedMemHigh

    #: (value type = cuuint64_t) Amount of memory, in bytes, currently
    #: allocated for use by the CUDA graphs asynchronous allocator.
    cudaGraphMemAttrReservedMemCurrent = ccudart.cudaGraphMemAttributeType.cudaGraphMemAttrReservedMemCurrent

    #: (value type = cuuint64_t) High watermark of memory, in bytes,
    #: currently allocated for use by the CUDA graphs asynchronous
    #: allocator.
    cudaGraphMemAttrReservedMemHigh = ccudart.cudaGraphMemAttributeType.cudaGraphMemAttrReservedMemHigh

class cudaDeviceP2PAttr(IntEnum):
    """
    CUDA device P2P attributes
    """

    #: A relative value indicating the performance of the link between two
    #: devices
    cudaDevP2PAttrPerformanceRank = ccudart.cudaDeviceP2PAttr.cudaDevP2PAttrPerformanceRank

    #: Peer access is enabled
    cudaDevP2PAttrAccessSupported = ccudart.cudaDeviceP2PAttr.cudaDevP2PAttrAccessSupported

    #: Native atomic operation over the link supported
    cudaDevP2PAttrNativeAtomicSupported = ccudart.cudaDeviceP2PAttr.cudaDevP2PAttrNativeAtomicSupported

    #: Accessing CUDA arrays over the link supported
    cudaDevP2PAttrCudaArrayAccessSupported = ccudart.cudaDeviceP2PAttr.cudaDevP2PAttrCudaArrayAccessSupported

class cudaExternalMemoryHandleType(IntEnum):
    """
    External memory handle types
    """

    #: Handle is an opaque file descriptor
    cudaExternalMemoryHandleTypeOpaqueFd = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeOpaqueFd

    #: Handle is an opaque shared NT handle
    cudaExternalMemoryHandleTypeOpaqueWin32 = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeOpaqueWin32

    #: Handle is an opaque, globally shared handle
    cudaExternalMemoryHandleTypeOpaqueWin32Kmt = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeOpaqueWin32Kmt

    #: Handle is a D3D12 heap object
    cudaExternalMemoryHandleTypeD3D12Heap = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeD3D12Heap

    #: Handle is a D3D12 committed resource
    cudaExternalMemoryHandleTypeD3D12Resource = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeD3D12Resource

    #: Handle is a shared NT handle to a D3D11 resource
    cudaExternalMemoryHandleTypeD3D11Resource = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeD3D11Resource

    #: Handle is a globally shared handle to a D3D11 resource
    cudaExternalMemoryHandleTypeD3D11ResourceKmt = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeD3D11ResourceKmt

    #: Handle is an NvSciBuf object
    cudaExternalMemoryHandleTypeNvSciBuf = ccudart.cudaExternalMemoryHandleType.cudaExternalMemoryHandleTypeNvSciBuf

class cudaExternalSemaphoreHandleType(IntEnum):
    """
    External semaphore handle types
    """

    #: Handle is an opaque file descriptor
    cudaExternalSemaphoreHandleTypeOpaqueFd = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeOpaqueFd

    #: Handle is an opaque shared NT handle
    cudaExternalSemaphoreHandleTypeOpaqueWin32 = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeOpaqueWin32

    #: Handle is an opaque, globally shared handle
    cudaExternalSemaphoreHandleTypeOpaqueWin32Kmt = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeOpaqueWin32Kmt

    #: Handle is a shared NT handle referencing a D3D12 fence object
    cudaExternalSemaphoreHandleTypeD3D12Fence = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeD3D12Fence

    #: Handle is a shared NT handle referencing a D3D11 fence object
    cudaExternalSemaphoreHandleTypeD3D11Fence = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeD3D11Fence

    #: Opaque handle to NvSciSync Object
    cudaExternalSemaphoreHandleTypeNvSciSync = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeNvSciSync

    #: Handle is a shared NT handle referencing a D3D11 keyed mutex object
    cudaExternalSemaphoreHandleTypeKeyedMutex = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeKeyedMutex

    #: Handle is a shared KMT handle referencing a D3D11 keyed mutex object
    cudaExternalSemaphoreHandleTypeKeyedMutexKmt = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeKeyedMutexKmt

    #: Handle is an opaque handle file descriptor referencing a timeline
    #: semaphore
    cudaExternalSemaphoreHandleTypeTimelineSemaphoreFd = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeTimelineSemaphoreFd

    #: Handle is an opaque handle file descriptor referencing a timeline
    #: semaphore
    cudaExternalSemaphoreHandleTypeTimelineSemaphoreWin32 = ccudart.cudaExternalSemaphoreHandleType.cudaExternalSemaphoreHandleTypeTimelineSemaphoreWin32

class cudaCGScope(IntEnum):
    """
    CUDA cooperative group scope
    """

    #: Invalid cooperative group scope
    cudaCGScopeInvalid = ccudart.cudaCGScope.cudaCGScopeInvalid

    #: Scope represented by a grid_group
    cudaCGScopeGrid = ccudart.cudaCGScope.cudaCGScopeGrid

    #: Scope represented by a multi_grid_group
    cudaCGScopeMultiGrid = ccudart.cudaCGScope.cudaCGScopeMultiGrid

class cudaGraphNodeType(IntEnum):
    """
    CUDA Graph node types
    """

    #: GPU kernel node
    cudaGraphNodeTypeKernel = ccudart.cudaGraphNodeType.cudaGraphNodeTypeKernel

    #: Memcpy node
    cudaGraphNodeTypeMemcpy = ccudart.cudaGraphNodeType.cudaGraphNodeTypeMemcpy

    #: Memset node
    cudaGraphNodeTypeMemset = ccudart.cudaGraphNodeType.cudaGraphNodeTypeMemset

    #: Host (executable) node
    cudaGraphNodeTypeHost = ccudart.cudaGraphNodeType.cudaGraphNodeTypeHost

    #: Node which executes an embedded graph
    cudaGraphNodeTypeGraph = ccudart.cudaGraphNodeType.cudaGraphNodeTypeGraph

    #: Empty (no-op) node
    cudaGraphNodeTypeEmpty = ccudart.cudaGraphNodeType.cudaGraphNodeTypeEmpty

    #: External event wait node
    cudaGraphNodeTypeWaitEvent = ccudart.cudaGraphNodeType.cudaGraphNodeTypeWaitEvent

    #: External event record node
    cudaGraphNodeTypeEventRecord = ccudart.cudaGraphNodeType.cudaGraphNodeTypeEventRecord

    #: External semaphore signal node
    cudaGraphNodeTypeExtSemaphoreSignal = ccudart.cudaGraphNodeType.cudaGraphNodeTypeExtSemaphoreSignal

    #: External semaphore wait node
    cudaGraphNodeTypeExtSemaphoreWait = ccudart.cudaGraphNodeType.cudaGraphNodeTypeExtSemaphoreWait

    #: Memory allocation node
    cudaGraphNodeTypeMemAlloc = ccudart.cudaGraphNodeType.cudaGraphNodeTypeMemAlloc

    #: Memory free node
    cudaGraphNodeTypeMemFree = ccudart.cudaGraphNodeType.cudaGraphNodeTypeMemFree
    cudaGraphNodeTypeCount = ccudart.cudaGraphNodeType.cudaGraphNodeTypeCount

class cudaGraphExecUpdateResult(IntEnum):
    """
    CUDA Graph Update error types
    """

    #: The update succeeded
    cudaGraphExecUpdateSuccess = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateSuccess

    #: The update failed for an unexpected reason which is described in the
    #: return value of the function
    cudaGraphExecUpdateError = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateError

    #: The update failed because the topology changed
    cudaGraphExecUpdateErrorTopologyChanged = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateErrorTopologyChanged

    #: The update failed because a node type changed
    cudaGraphExecUpdateErrorNodeTypeChanged = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateErrorNodeTypeChanged

    #: The update failed because the function of a kernel node changed
    #: (CUDA driver < 11.2)
    cudaGraphExecUpdateErrorFunctionChanged = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateErrorFunctionChanged

    #: The update failed because the parameters changed in a way that is
    #: not supported
    cudaGraphExecUpdateErrorParametersChanged = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateErrorParametersChanged

    #: The update failed because something about the node is not supported
    cudaGraphExecUpdateErrorNotSupported = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateErrorNotSupported

    #: The update failed because the function of a kernel node changed in
    #: an unsupported way
    cudaGraphExecUpdateErrorUnsupportedFunctionChange = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateErrorUnsupportedFunctionChange

    #: The update failed because the node attributes changed in a way that
    #: is not supported
    cudaGraphExecUpdateErrorAttributesChanged = ccudart.cudaGraphExecUpdateResult.cudaGraphExecUpdateErrorAttributesChanged

class cudaGetDriverEntryPointFlags(IntEnum):
    """
    Flags to specify search options to be used with
    :py:obj:`~.cudaGetDriverEntryPoint` For more details see
    :py:obj:`~.cuGetProcAddress`
    """

    #: Default search mode for driver symbols.
    cudaEnableDefault = ccudart.cudaGetDriverEntryPointFlags.cudaEnableDefault

    #: Search for legacy versions of driver symbols.
    cudaEnableLegacyStream = ccudart.cudaGetDriverEntryPointFlags.cudaEnableLegacyStream

    #: Search for per-thread versions of driver symbols.
    cudaEnablePerThreadDefaultStream = ccudart.cudaGetDriverEntryPointFlags.cudaEnablePerThreadDefaultStream

class cudaGraphDebugDotFlags(IntEnum):
    """
    CUDA Graph debug write options
    """

    #: Output all debug data as if every debug flag is enabled
    cudaGraphDebugDotFlagsVerbose = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsVerbose

    #: Adds :py:obj:`~.cudaKernelNodeParams` to output
    cudaGraphDebugDotFlagsKernelNodeParams = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsKernelNodeParams

    #: Adds :py:obj:`~.cudaMemcpy3DParms` to output
    cudaGraphDebugDotFlagsMemcpyNodeParams = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsMemcpyNodeParams

    #: Adds :py:obj:`~.cudaMemsetParams` to output
    cudaGraphDebugDotFlagsMemsetNodeParams = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsMemsetNodeParams

    #: Adds :py:obj:`~.cudaHostNodeParams` to output
    cudaGraphDebugDotFlagsHostNodeParams = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsHostNodeParams

    #: Adds cudaEvent_t handle from record and wait nodes to output
    cudaGraphDebugDotFlagsEventNodeParams = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsEventNodeParams

    #: Adds :py:obj:`~.cudaExternalSemaphoreSignalNodeParams` values to
    #: output
    cudaGraphDebugDotFlagsExtSemasSignalNodeParams = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsExtSemasSignalNodeParams

    #: Adds :py:obj:`~.cudaExternalSemaphoreWaitNodeParams` to output
    cudaGraphDebugDotFlagsExtSemasWaitNodeParams = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsExtSemasWaitNodeParams

    #: Adds cudaKernelNodeAttrID values to output
    cudaGraphDebugDotFlagsKernelNodeAttributes = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsKernelNodeAttributes

    #: Adds node handles and every kernel function handle to output
    cudaGraphDebugDotFlagsHandles = ccudart.cudaGraphDebugDotFlags.cudaGraphDebugDotFlagsHandles

class cudaGraphInstantiateFlags(IntEnum):
    """
    Flags for instantiating a graph
    """

    #: Automatically free memory allocated in a graph before relaunching.
    cudaGraphInstantiateFlagAutoFreeOnLaunch = ccudart.cudaGraphInstantiateFlags.cudaGraphInstantiateFlagAutoFreeOnLaunch

    #: Run the graph using the per-node priority attributes rather than the
    #: priority of the stream it is launched into.
    cudaGraphInstantiateFlagUseNodePriority = ccudart.cudaGraphInstantiateFlags.cudaGraphInstantiateFlagUseNodePriority

class cudaStreamAttrID(IntEnum):
    """
    Stream Attributes
    """

    #: Identifier for :py:obj:`~.cudaStreamAttrValue`::accessPolicyWindow.
    cudaStreamAttributeAccessPolicyWindow = ccudart.cudaStreamAttrID.cudaStreamAttributeAccessPolicyWindow

    #: :py:obj:`~.cudaSynchronizationPolicy` for work queued up in this
    #: stream
    cudaStreamAttributeSynchronizationPolicy = ccudart.cudaStreamAttrID.cudaStreamAttributeSynchronizationPolicy

class cudaKernelNodeAttrID(IntEnum):
    """
    Graph kernel node Attributes
    """

    #: Identifier for
    #: :py:obj:`~.cudaKernelNodeAttrValue.accessPolicyWindow`.
    cudaKernelNodeAttributeAccessPolicyWindow = ccudart.cudaKernelNodeAttrID.cudaKernelNodeAttributeAccessPolicyWindow

    #: Allows a kernel node to be cooperative (see
    #: :py:obj:`~.cudaLaunchCooperativeKernel`).
    cudaKernelNodeAttributeCooperative = ccudart.cudaKernelNodeAttrID.cudaKernelNodeAttributeCooperative

    #: Sets the priority of the kernel.
    cudaKernelNodeAttributePriority = ccudart.cudaKernelNodeAttrID.cudaKernelNodeAttributePriority

class cudaGLDeviceList(IntEnum):
    """
    CUDA devices corresponding to the current OpenGL context
    """

    #: The CUDA devices for all GPUs used by the current OpenGL context
    cudaGLDeviceListAll = ccudart.cudaGLDeviceList.cudaGLDeviceListAll

    #: The CUDA devices for the GPUs used by the current OpenGL context in
    #: its currently rendering frame
    cudaGLDeviceListCurrentFrame = ccudart.cudaGLDeviceList.cudaGLDeviceListCurrentFrame

    #: The CUDA devices for the GPUs to be used by the current OpenGL
    #: context in the next frame
    cudaGLDeviceListNextFrame = ccudart.cudaGLDeviceList.cudaGLDeviceListNextFrame

class cudaGLMapFlags(IntEnum):
    """
    CUDA GL Map Flags
    """

    #: Default; Assume resource can be read/written
    cudaGLMapFlagsNone = ccudart.cudaGLMapFlags.cudaGLMapFlagsNone

    #: CUDA kernels will not write to this resource
    cudaGLMapFlagsReadOnly = ccudart.cudaGLMapFlags.cudaGLMapFlagsReadOnly

    #: CUDA kernels will only write to and will not read from this resource
    cudaGLMapFlagsWriteDiscard = ccudart.cudaGLMapFlags.cudaGLMapFlagsWriteDiscard

class cudaRoundMode(IntEnum):
    cudaRoundNearest = ccudart.cudaRoundMode.cudaRoundNearest
    cudaRoundZero = ccudart.cudaRoundMode.cudaRoundZero
    cudaRoundPosInf = ccudart.cudaRoundMode.cudaRoundPosInf
    cudaRoundMinInf = ccudart.cudaRoundMode.cudaRoundMinInf

class cudaSurfaceBoundaryMode(IntEnum):
    """
    CUDA Surface boundary modes
    """

    #: Zero boundary mode
    cudaBoundaryModeZero = ccudart.cudaSurfaceBoundaryMode.cudaBoundaryModeZero

    #: Clamp boundary mode
    cudaBoundaryModeClamp = ccudart.cudaSurfaceBoundaryMode.cudaBoundaryModeClamp

    #: Trap boundary mode
    cudaBoundaryModeTrap = ccudart.cudaSurfaceBoundaryMode.cudaBoundaryModeTrap

class cudaSurfaceFormatMode(IntEnum):
    """
    CUDA Surface format modes
    """

    #: Forced format mode
    cudaFormatModeForced = ccudart.cudaSurfaceFormatMode.cudaFormatModeForced

    #: Auto format mode
    cudaFormatModeAuto = ccudart.cudaSurfaceFormatMode.cudaFormatModeAuto

class cudaTextureAddressMode(IntEnum):
    """
    CUDA texture address modes
    """

    #: Wrapping address mode
    cudaAddressModeWrap = ccudart.cudaTextureAddressMode.cudaAddressModeWrap

    #: Clamp to edge address mode
    cudaAddressModeClamp = ccudart.cudaTextureAddressMode.cudaAddressModeClamp

    #: Mirror address mode
    cudaAddressModeMirror = ccudart.cudaTextureAddressMode.cudaAddressModeMirror

    #: Border address mode
    cudaAddressModeBorder = ccudart.cudaTextureAddressMode.cudaAddressModeBorder

class cudaTextureFilterMode(IntEnum):
    """
    CUDA texture filter modes
    """

    #: Point filter mode
    cudaFilterModePoint = ccudart.cudaTextureFilterMode.cudaFilterModePoint

    #: Linear filter mode
    cudaFilterModeLinear = ccudart.cudaTextureFilterMode.cudaFilterModeLinear

class cudaTextureReadMode(IntEnum):
    """
    CUDA texture read modes
    """

    #: Read texture as specified element type
    cudaReadModeElementType = ccudart.cudaTextureReadMode.cudaReadModeElementType

    #: Read texture as normalized float
    cudaReadModeNormalizedFloat = ccudart.cudaTextureReadMode.cudaReadModeNormalizedFloat

class cudaDataType(IntEnum):
    CUDA_R_16F = ccudart.cudaDataType_t.CUDA_R_16F
    CUDA_C_16F = ccudart.cudaDataType_t.CUDA_C_16F
    CUDA_R_16BF = ccudart.cudaDataType_t.CUDA_R_16BF
    CUDA_C_16BF = ccudart.cudaDataType_t.CUDA_C_16BF
    CUDA_R_32F = ccudart.cudaDataType_t.CUDA_R_32F
    CUDA_C_32F = ccudart.cudaDataType_t.CUDA_C_32F
    CUDA_R_64F = ccudart.cudaDataType_t.CUDA_R_64F
    CUDA_C_64F = ccudart.cudaDataType_t.CUDA_C_64F
    CUDA_R_4I = ccudart.cudaDataType_t.CUDA_R_4I
    CUDA_C_4I = ccudart.cudaDataType_t.CUDA_C_4I
    CUDA_R_4U = ccudart.cudaDataType_t.CUDA_R_4U
    CUDA_C_4U = ccudart.cudaDataType_t.CUDA_C_4U
    CUDA_R_8I = ccudart.cudaDataType_t.CUDA_R_8I
    CUDA_C_8I = ccudart.cudaDataType_t.CUDA_C_8I
    CUDA_R_8U = ccudart.cudaDataType_t.CUDA_R_8U
    CUDA_C_8U = ccudart.cudaDataType_t.CUDA_C_8U
    CUDA_R_16I = ccudart.cudaDataType_t.CUDA_R_16I
    CUDA_C_16I = ccudart.cudaDataType_t.CUDA_C_16I
    CUDA_R_16U = ccudart.cudaDataType_t.CUDA_R_16U
    CUDA_C_16U = ccudart.cudaDataType_t.CUDA_C_16U
    CUDA_R_32I = ccudart.cudaDataType_t.CUDA_R_32I
    CUDA_C_32I = ccudart.cudaDataType_t.CUDA_C_32I
    CUDA_R_32U = ccudart.cudaDataType_t.CUDA_R_32U
    CUDA_C_32U = ccudart.cudaDataType_t.CUDA_C_32U
    CUDA_R_64I = ccudart.cudaDataType_t.CUDA_R_64I
    CUDA_C_64I = ccudart.cudaDataType_t.CUDA_C_64I
    CUDA_R_64U = ccudart.cudaDataType_t.CUDA_R_64U
    CUDA_C_64U = ccudart.cudaDataType_t.CUDA_C_64U

class libraryPropertyType(IntEnum):
    MAJOR_VERSION = ccudart.libraryPropertyType_t.MAJOR_VERSION
    MINOR_VERSION = ccudart.libraryPropertyType_t.MINOR_VERSION
    PATCH_LEVEL = ccudart.libraryPropertyType_t.PATCH_LEVEL

class cudaEglFrameType(IntEnum):
    """
    CUDA EglFrame type - array or pointer
    """

    #: Frame type CUDA array
    cudaEglFrameTypeArray = ccudart.cudaEglFrameType_enum.cudaEglFrameTypeArray

    #: Frame type CUDA pointer
    cudaEglFrameTypePitch = ccudart.cudaEglFrameType_enum.cudaEglFrameTypePitch

class cudaEglResourceLocationFlags(IntEnum):
    """
    Resource location flags- sysmem or vidmem  For CUDA context on
    iGPU, since video and system memory are equivalent - these flags
    will not have an effect on the execution.  For CUDA context on
    dGPU, applications can use the flag
    :py:obj:`~.cudaEglResourceLocationFlags` to give a hint about the
    desired location.  :py:obj:`~.cudaEglResourceLocationSysmem` - the
    frame data is made resident on the system memory to be accessed by
    CUDA.  :py:obj:`~.cudaEglResourceLocationVidmem` - the frame data
    is made resident on the dedicated video memory to be accessed by
    CUDA.  There may be an additional latency due to new allocation and
    data migration, if the frame is produced on a different memory.
    """

    #: Resource location sysmem
    cudaEglResourceLocationSysmem = ccudart.cudaEglResourceLocationFlags_enum.cudaEglResourceLocationSysmem

    #: Resource location vidmem
    cudaEglResourceLocationVidmem = ccudart.cudaEglResourceLocationFlags_enum.cudaEglResourceLocationVidmem

class cudaEglColorFormat(IntEnum):
    """
    CUDA EGL Color Format - The different planar and multiplanar
    formats currently supported for CUDA_EGL interops.
    """

    #: Y, U, V in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYUV420Planar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420Planar

    #: Y, UV in two surfaces (UV as one surface) with VU byte ordering,
    #: width, height ratio same as YUV420Planar.
    cudaEglColorFormatYUV420SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420SemiPlanar

    #: Y, U, V each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = Y height.
    cudaEglColorFormatYUV422Planar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV422Planar

    #: Y, UV in two surfaces with VU byte ordering, width, height ratio
    #: same as YUV422Planar.
    cudaEglColorFormatYUV422SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV422SemiPlanar

    #: R/G/B/A four channels in one surface with BGRA byte ordering.
    cudaEglColorFormatARGB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatARGB

    #: R/G/B/A four channels in one surface with ABGR byte ordering.
    cudaEglColorFormatRGBA = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatRGBA

    #: single luminance channel in one surface.
    cudaEglColorFormatL = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatL

    #: single color channel in one surface.
    cudaEglColorFormatR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatR

    #: Y, U, V in three surfaces, each in a separate surface, U/V width = Y
    #: width, U/V height = Y height.
    cudaEglColorFormatYUV444Planar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV444Planar

    #: Y, UV in two surfaces (UV as one surface) with VU byte ordering,
    #: width, height ratio same as YUV444Planar.
    cudaEglColorFormatYUV444SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV444SemiPlanar

    #: Y, U, V in one surface, interleaved as UYVY in one channel.
    cudaEglColorFormatYUYV422 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUYV422

    #: Y, U, V in one surface, interleaved as YUYV in one channel.
    cudaEglColorFormatUYVY422 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatUYVY422

    #: R/G/B/A four channels in one surface with RGBA byte ordering.
    cudaEglColorFormatABGR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatABGR

    #: R/G/B/A four channels in one surface with ARGB byte ordering.
    cudaEglColorFormatBGRA = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBGRA

    #: Alpha color format - one channel in one surface.
    cudaEglColorFormatA = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatA

    #: R/G color format - two channels in one surface with GR byte ordering
    cudaEglColorFormatRG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatRG

    #: Y, U, V, A four channels in one surface, interleaved as VUYA.
    cudaEglColorFormatAYUV = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatAYUV

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = Y width, U/V height = Y height.
    cudaEglColorFormatYVU444SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU444SemiPlanar

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = 1/2 Y width, U/V height = Y height.
    cudaEglColorFormatYVU422SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU422SemiPlanar

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420SemiPlanar

    #: Y10, V10U10 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = Y width, U/V height = Y height.
    cudaEglColorFormatY10V10U10_444SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_444SemiPlanar

    #: Y10, V10U10 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatY10V10U10_420SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_420SemiPlanar

    #: Y12, V12U12 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = Y width, U/V height = Y height.
    cudaEglColorFormatY12V12U12_444SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12V12U12_444SemiPlanar

    #: Y12, V12U12 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatY12V12U12_420SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12V12U12_420SemiPlanar

    #: Extended Range Y, U, V in one surface, interleaved as YVYU in one
    #: channel.
    cudaEglColorFormatVYUY_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatVYUY_ER

    #: Extended Range Y, U, V in one surface, interleaved as YUYV in one
    #: channel.
    cudaEglColorFormatUYVY_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatUYVY_ER

    #: Extended Range Y, U, V in one surface, interleaved as UYVY in one
    #: channel.
    cudaEglColorFormatYUYV_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUYV_ER

    #: Extended Range Y, U, V in one surface, interleaved as VYUY in one
    #: channel.
    cudaEglColorFormatYVYU_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVYU_ER

    #: Extended Range Y, U, V, A four channels in one surface, interleaved
    #: as AVUY.
    cudaEglColorFormatYUVA_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUVA_ER

    #: Extended Range Y, U, V, A four channels in one surface, interleaved
    #: as VUYA.
    cudaEglColorFormatAYUV_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatAYUV_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = Y width, U/V
    #: height = Y height.
    cudaEglColorFormatYUV444Planar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV444Planar_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = Y height.
    cudaEglColorFormatYUV422Planar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV422Planar_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    cudaEglColorFormatYUV420Planar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420Planar_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = Y width, U/V height = Y height.
    cudaEglColorFormatYUV444SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV444SemiPlanar_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = 1/2 Y width, U/V height = Y height.
    cudaEglColorFormatYUV422SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV422SemiPlanar_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYUV420SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420SemiPlanar_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = Y width, U/V
    #: height = Y height.
    cudaEglColorFormatYVU444Planar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU444Planar_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = Y height.
    cudaEglColorFormatYVU422Planar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU422Planar_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420Planar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420Planar_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = Y width, U/V height = Y height.
    cudaEglColorFormatYVU444SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU444SemiPlanar_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = 1/2 Y width, U/V height = Y height.
    cudaEglColorFormatYVU422SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU422SemiPlanar_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420SemiPlanar_ER

    #: Bayer format - one channel in one surface with interleaved RGGB
    #: ordering.
    cudaEglColorFormatBayerRGGB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerRGGB

    #: Bayer format - one channel in one surface with interleaved BGGR
    #: ordering.
    cudaEglColorFormatBayerBGGR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerBGGR

    #: Bayer format - one channel in one surface with interleaved GRBG
    #: ordering.
    cudaEglColorFormatBayerGRBG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerGRBG

    #: Bayer format - one channel in one surface with interleaved GBRG
    #: ordering.
    cudaEglColorFormatBayerGBRG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerGBRG

    #: Bayer10 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    cudaEglColorFormatBayer10RGGB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer10RGGB

    #: Bayer10 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    cudaEglColorFormatBayer10BGGR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer10BGGR

    #: Bayer10 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    cudaEglColorFormatBayer10GRBG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer10GRBG

    #: Bayer10 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    cudaEglColorFormatBayer10GBRG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer10GBRG

    #: Bayer12 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12RGGB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12RGGB

    #: Bayer12 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12BGGR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12BGGR

    #: Bayer12 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12GRBG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12GRBG

    #: Bayer12 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12GBRG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12GBRG

    #: Bayer14 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    cudaEglColorFormatBayer14RGGB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer14RGGB

    #: Bayer14 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    cudaEglColorFormatBayer14BGGR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer14BGGR

    #: Bayer14 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    cudaEglColorFormatBayer14GRBG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer14GRBG

    #: Bayer14 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    cudaEglColorFormatBayer14GBRG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer14GBRG

    #: Bayer20 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    cudaEglColorFormatBayer20RGGB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer20RGGB

    #: Bayer20 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    cudaEglColorFormatBayer20BGGR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer20BGGR

    #: Bayer20 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    cudaEglColorFormatBayer20GRBG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer20GRBG

    #: Bayer20 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    cudaEglColorFormatBayer20GBRG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer20GBRG

    #: Y, V, U in three surfaces, each in a separate surface, U/V width = Y
    #: width, U/V height = Y height.
    cudaEglColorFormatYVU444Planar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU444Planar

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = Y height.
    cudaEglColorFormatYVU422Planar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU422Planar

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420Planar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420Planar

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved RGGB ordering and mapped to opaque integer
    #: datatype.
    cudaEglColorFormatBayerIspRGGB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerIspRGGB

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved BGGR ordering and mapped to opaque integer
    #: datatype.
    cudaEglColorFormatBayerIspBGGR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerIspBGGR

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved GRBG ordering and mapped to opaque integer
    #: datatype.
    cudaEglColorFormatBayerIspGRBG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerIspGRBG

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved GBRG ordering and mapped to opaque integer
    #: datatype.
    cudaEglColorFormatBayerIspGBRG = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerIspGBRG

    #: Bayer format - one channel in one surface with interleaved BCCR
    #: ordering.
    cudaEglColorFormatBayerBCCR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerBCCR

    #: Bayer format - one channel in one surface with interleaved RCCB
    #: ordering.
    cudaEglColorFormatBayerRCCB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerRCCB

    #: Bayer format - one channel in one surface with interleaved CRBC
    #: ordering.
    cudaEglColorFormatBayerCRBC = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerCRBC

    #: Bayer format - one channel in one surface with interleaved CBRC
    #: ordering.
    cudaEglColorFormatBayerCBRC = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayerCBRC

    #: Bayer10 format - one channel in one surface with interleaved CCCC
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    cudaEglColorFormatBayer10CCCC = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer10CCCC

    #: Bayer12 format - one channel in one surface with interleaved BCCR
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12BCCR = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12BCCR

    #: Bayer12 format - one channel in one surface with interleaved RCCB
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12RCCB = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12RCCB

    #: Bayer12 format - one channel in one surface with interleaved CRBC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12CRBC = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12CRBC

    #: Bayer12 format - one channel in one surface with interleaved CBRC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12CBRC = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12CBRC

    #: Bayer12 format - one channel in one surface with interleaved CCCC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    cudaEglColorFormatBayer12CCCC = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatBayer12CCCC

    #: Color format for single Y plane.
    cudaEglColorFormatY = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY

    #: Y, UV in two surfaces (UV as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    cudaEglColorFormatYUV420SemiPlanar_2020 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420SemiPlanar_2020

    #: Y, VU in two surfaces (VU as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420SemiPlanar_2020 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420SemiPlanar_2020

    #: Y, U, V in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYUV420Planar_2020 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420Planar_2020

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420Planar_2020 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420Planar_2020

    #: Y, UV in two surfaces (UV as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    cudaEglColorFormatYUV420SemiPlanar_709 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420SemiPlanar_709

    #: Y, VU in two surfaces (VU as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420SemiPlanar_709 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420SemiPlanar_709

    #: Y, U, V in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYUV420Planar_709 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUV420Planar_709

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatYVU420Planar_709 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVU420Planar_709

    #: Y10, V10U10 in two surfaces (VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = 1/2 Y height.
    cudaEglColorFormatY10V10U10_420SemiPlanar_709 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_420SemiPlanar_709

    #: Y10, V10U10 in two surfaces (VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = 1/2 Y height.
    cudaEglColorFormatY10V10U10_420SemiPlanar_2020 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_420SemiPlanar_2020

    #: Y10, V10U10 in two surfaces (VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    cudaEglColorFormatY10V10U10_422SemiPlanar_2020 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_422SemiPlanar_2020

    #: Y10, V10U10 in two surfaces (VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    cudaEglColorFormatY10V10U10_422SemiPlanar = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_422SemiPlanar

    #: Y10, V10U10 in two surfaces (VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    cudaEglColorFormatY10V10U10_422SemiPlanar_709 = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_422SemiPlanar_709

    #: Extended Range Color format for single Y plane.
    cudaEglColorFormatY_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY_ER

    #: Extended Range Color format for single Y plane.
    cudaEglColorFormatY_709_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY_709_ER

    #: Extended Range Color format for single Y10 plane.
    cudaEglColorFormatY10_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10_ER

    #: Extended Range Color format for single Y10 plane.
    cudaEglColorFormatY10_709_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10_709_ER

    #: Extended Range Color format for single Y12 plane.
    cudaEglColorFormatY12_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12_ER

    #: Extended Range Color format for single Y12 plane.
    cudaEglColorFormatY12_709_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12_709_ER

    #: Y, U, V, A four channels in one surface, interleaved as AVUY.
    cudaEglColorFormatYUVA = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYUVA

    #: Y, U, V in one surface, interleaved as YVYU in one channel.
    cudaEglColorFormatYVYU = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatYVYU

    #: Y, U, V in one surface, interleaved as VYUY in one channel.
    cudaEglColorFormatVYUY = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatVYUY

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatY10V10U10_420SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_420SemiPlanar_ER

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatY10V10U10_420SemiPlanar_709_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_420SemiPlanar_709_ER

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    cudaEglColorFormatY10V10U10_444SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_444SemiPlanar_ER

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    cudaEglColorFormatY10V10U10_444SemiPlanar_709_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY10V10U10_444SemiPlanar_709_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatY12V12U12_420SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12V12U12_420SemiPlanar_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    cudaEglColorFormatY12V12U12_420SemiPlanar_709_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12V12U12_420SemiPlanar_709_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    cudaEglColorFormatY12V12U12_444SemiPlanar_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12V12U12_444SemiPlanar_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    cudaEglColorFormatY12V12U12_444SemiPlanar_709_ER = ccudart.cudaEglColorFormat_enum.cudaEglColorFormatY12V12U12_444SemiPlanar_709_ER

cdef class cudaArray_t:
    """

    CUDA array

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaArray_t>init_value
        else:
            self._ptr = <ccudart.cudaArray_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaArray_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaArray_const_t:
    """

    CUDA array (as source copy argument)

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaArray_const_t>init_value
        else:
            self._ptr = <ccudart.cudaArray_const_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaArray_const_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaMipmappedArray_t:
    """

    CUDA mipmapped array

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaMipmappedArray_t>init_value
        else:
            self._ptr = <ccudart.cudaMipmappedArray_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaMipmappedArray_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaMipmappedArray_const_t:
    """

    CUDA mipmapped array (as source argument)

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaMipmappedArray_const_t>init_value
        else:
            self._ptr = <ccudart.cudaMipmappedArray_const_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaMipmappedArray_const_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaGraphicsResource_t:
    """

    CUDA graphics resource types

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaGraphicsResource_t>init_value
        else:
            self._ptr = <ccudart.cudaGraphicsResource_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaGraphicsResource_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaExternalMemory_t:
    """

    CUDA external memory

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaExternalMemory_t>init_value
        else:
            self._ptr = <ccudart.cudaExternalMemory_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaExternalMemory_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaExternalSemaphore_t:
    """

    CUDA external semaphore

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaExternalSemaphore_t>init_value
        else:
            self._ptr = <ccudart.cudaExternalSemaphore_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaExternalSemaphore_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLImageKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.EGLImageKHR>init_value
        else:
            self._ptr = <ccudart.EGLImageKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLImageKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLStreamKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.EGLStreamKHR>init_value
        else:
            self._ptr = <ccudart.EGLStreamKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLStreamKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLSyncKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.EGLSyncKHR>init_value
        else:
            self._ptr = <ccudart.EGLSyncKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLSyncKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaHostFn_t:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaHostFn_t>init_value
        else:
            self._ptr = <ccudart.cudaHostFn_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaHostFn_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaStreamCallback_t:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <ccudart.cudaStreamCallback_t>init_value
        else:
            self._ptr = <ccudart.cudaStreamCallback_t *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<cudaStreamCallback_t ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class dim3:
    """

    Attributes
    ----------
    x : unsigned int

    y : unsigned int

    z : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.dim3 *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['x : ' + str(self.x)]
            str_list += ['y : ' + str(self.y)]
            str_list += ['z : ' + str(self.z)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def x(self):
        return self._ptr[0].x
    @x.setter
    def x(self, unsigned int x):
        pass
        self._ptr[0].x = x
    @property
    def y(self):
        return self._ptr[0].y
    @y.setter
    def y(self, unsigned int y):
        pass
        self._ptr[0].y = y
    @property
    def z(self):
        return self._ptr[0].z
    @z.setter
    def z(self, unsigned int z):
        pass
        self._ptr[0].z = z

cdef class cudaChannelFormatDesc:
    """
    CUDA Channel format descriptor

    Attributes
    ----------
    x : int
        x
    y : int
        y
    z : int
        z
    w : int
        w
    f : cudaChannelFormatKind
        Channel format kind

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaChannelFormatDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['x : ' + str(self.x)]
            str_list += ['y : ' + str(self.y)]
            str_list += ['z : ' + str(self.z)]
            str_list += ['w : ' + str(self.w)]
            str_list += ['f : ' + str(self.f)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def x(self):
        return self._ptr[0].x
    @x.setter
    def x(self, int x):
        pass
        self._ptr[0].x = x
    @property
    def y(self):
        return self._ptr[0].y
    @y.setter
    def y(self, int y):
        pass
        self._ptr[0].y = y
    @property
    def z(self):
        return self._ptr[0].z
    @z.setter
    def z(self, int z):
        pass
        self._ptr[0].z = z
    @property
    def w(self):
        return self._ptr[0].w
    @w.setter
    def w(self, int w):
        pass
        self._ptr[0].w = w
    @property
    def f(self):
        return cudaChannelFormatKind(self._ptr[0].f)
    @f.setter
    def f(self, f not None : cudaChannelFormatKind):
        pass
        self._ptr[0].f = f.value

cdef class _cudaArraySparseProperties_tileExtent_s:
    """

    Attributes
    ----------
    width : unsigned int

    height : unsigned int

    depth : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaArraySparseProperties *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['width : ' + str(self.width)]
            str_list += ['height : ' + str(self.height)]
            str_list += ['depth : ' + str(self.depth)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def width(self):
        return self._ptr[0].tileExtent.width
    @width.setter
    def width(self, unsigned int width):
        pass
        self._ptr[0].tileExtent.width = width
    @property
    def height(self):
        return self._ptr[0].tileExtent.height
    @height.setter
    def height(self, unsigned int height):
        pass
        self._ptr[0].tileExtent.height = height
    @property
    def depth(self):
        return self._ptr[0].tileExtent.depth
    @depth.setter
    def depth(self, unsigned int depth):
        pass
        self._ptr[0].tileExtent.depth = depth

cdef class cudaArraySparseProperties:
    """
    Sparse CUDA array and CUDA mipmapped array properties

    Attributes
    ----------
    tileExtent : _cudaArraySparseProperties_tileExtent_s

    miptailFirstLevel : unsigned int
        First mip level at which the mip tail begins
    miptailSize : unsigned long long
        Total size of the mip tail.
    flags : unsigned int
        Flags will either be zero or cudaArraySparsePropertiesSingleMipTail
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaArraySparseProperties *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._tileExtent = _cudaArraySparseProperties_tileExtent_s(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['tileExtent :\n' + '\n'.join(['    ' + line for line in str(self.tileExtent).splitlines()])]
            str_list += ['miptailFirstLevel : ' + str(self.miptailFirstLevel)]
            str_list += ['miptailSize : ' + str(self.miptailSize)]
            str_list += ['flags : ' + str(self.flags)]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def tileExtent(self):
        return self._tileExtent
    @tileExtent.setter
    def tileExtent(self, tileExtent not None : _cudaArraySparseProperties_tileExtent_s):
        pass
        for _attr in dir(tileExtent):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._tileExtent, _attr, getattr(tileExtent, _attr))
    @property
    def miptailFirstLevel(self):
        return self._ptr[0].miptailFirstLevel
    @miptailFirstLevel.setter
    def miptailFirstLevel(self, unsigned int miptailFirstLevel):
        pass
        self._ptr[0].miptailFirstLevel = miptailFirstLevel
    @property
    def miptailSize(self):
        return self._ptr[0].miptailSize
    @miptailSize.setter
    def miptailSize(self, unsigned long long miptailSize):
        pass
        self._ptr[0].miptailSize = miptailSize
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        pass
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        pass
        self._ptr[0].reserved = reserved

cdef class cudaArrayMemoryRequirements:
    """
    CUDA array and CUDA mipmapped array memory requirements

    Attributes
    ----------
    size : size_t
        Total size of the array.
    alignment : size_t
        Alignment necessary for mapping the array.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaArrayMemoryRequirements *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['size : ' + str(self.size)]
            str_list += ['alignment : ' + str(self.alignment)]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, size_t size):
        pass
        self._ptr[0].size = size
    @property
    def alignment(self):
        return self._ptr[0].alignment
    @alignment.setter
    def alignment(self, size_t alignment):
        pass
        self._ptr[0].alignment = alignment
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        pass
        self._ptr[0].reserved = reserved

cdef class cudaPitchedPtr:
    """
    CUDA Pitched memory pointer  ::make_cudaPitchedPtr

    Attributes
    ----------
    ptr : Any
        Pointer to allocated memory
    pitch : size_t
        Pitch of allocated memory in bytes
    xsize : size_t
        Logical width of allocation in elements
    ysize : size_t
        Logical height of allocation in elements

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaPitchedPtr *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['ptr : ' + hex(self.ptr)]
            str_list += ['pitch : ' + str(self.pitch)]
            str_list += ['xsize : ' + str(self.xsize)]
            str_list += ['ysize : ' + str(self.ysize)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def ptr(self):
        return <void_ptr>self._ptr[0].ptr
    @ptr.setter
    def ptr(self, ptr):
        _cptr = utils.HelperInputVoidPtr(ptr)
        self._ptr[0].ptr = <void*><void_ptr>_cptr.cptr
    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, size_t pitch):
        pass
        self._ptr[0].pitch = pitch
    @property
    def xsize(self):
        return self._ptr[0].xsize
    @xsize.setter
    def xsize(self, size_t xsize):
        pass
        self._ptr[0].xsize = xsize
    @property
    def ysize(self):
        return self._ptr[0].ysize
    @ysize.setter
    def ysize(self, size_t ysize):
        pass
        self._ptr[0].ysize = ysize

cdef class cudaExtent:
    """
    CUDA extent  ::make_cudaExtent

    Attributes
    ----------
    width : size_t
        Width in elements when referring to array memory, in bytes when
        referring to linear memory
    height : size_t
        Height in elements
    depth : size_t
        Depth in elements

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExtent *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['width : ' + str(self.width)]
            str_list += ['height : ' + str(self.height)]
            str_list += ['depth : ' + str(self.depth)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        pass
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        pass
        self._ptr[0].height = height
    @property
    def depth(self):
        return self._ptr[0].depth
    @depth.setter
    def depth(self, size_t depth):
        pass
        self._ptr[0].depth = depth

cdef class cudaPos:
    """
    CUDA 3D position  ::make_cudaPos

    Attributes
    ----------
    x : size_t
        x
    y : size_t
        y
    z : size_t
        z

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaPos *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['x : ' + str(self.x)]
            str_list += ['y : ' + str(self.y)]
            str_list += ['z : ' + str(self.z)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def x(self):
        return self._ptr[0].x
    @x.setter
    def x(self, size_t x):
        pass
        self._ptr[0].x = x
    @property
    def y(self):
        return self._ptr[0].y
    @y.setter
    def y(self, size_t y):
        pass
        self._ptr[0].y = y
    @property
    def z(self):
        return self._ptr[0].z
    @z.setter
    def z(self, size_t z):
        pass
        self._ptr[0].z = z

cdef class cudaMemcpy3DParms:
    """
    CUDA 3D memory copying parameters

    Attributes
    ----------
    srcArray : cudaArray_t
        Source memory address
    srcPos : cudaPos
        Source position offset
    srcPtr : cudaPitchedPtr
        Pitched source memory address
    dstArray : cudaArray_t
        Destination memory address
    dstPos : cudaPos
        Destination position offset
    dstPtr : cudaPitchedPtr
        Pitched destination memory address
    extent : cudaExtent
        Requested memory copy size
    kind : cudaMemcpyKind
        Type of transfer

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemcpy3DParms *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcArray = cudaArray_t(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._srcPos = cudaPos(_ptr=<void_ptr>&self._ptr[0].srcPos)
        self._srcPtr = cudaPitchedPtr(_ptr=<void_ptr>&self._ptr[0].srcPtr)
        self._dstArray = cudaArray_t(_ptr=<void_ptr>&self._ptr[0].dstArray)
        self._dstPos = cudaPos(_ptr=<void_ptr>&self._ptr[0].dstPos)
        self._dstPtr = cudaPitchedPtr(_ptr=<void_ptr>&self._ptr[0].dstPtr)
        self._extent = cudaExtent(_ptr=<void_ptr>&self._ptr[0].extent)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['srcArray : ' + str(self.srcArray)]
            str_list += ['srcPos :\n' + '\n'.join(['    ' + line for line in str(self.srcPos).splitlines()])]
            str_list += ['srcPtr :\n' + '\n'.join(['    ' + line for line in str(self.srcPtr).splitlines()])]
            str_list += ['dstArray : ' + str(self.dstArray)]
            str_list += ['dstPos :\n' + '\n'.join(['    ' + line for line in str(self.dstPos).splitlines()])]
            str_list += ['dstPtr :\n' + '\n'.join(['    ' + line for line in str(self.dstPtr).splitlines()])]
            str_list += ['extent :\n' + '\n'.join(['    ' + line for line in str(self.extent).splitlines()])]
            str_list += ['kind : ' + str(self.kind)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef ccudart.cudaArray_t csrcArray
        if srcArray is None:
            csrcArray = <ccudart.cudaArray_t><void_ptr>0
        elif isinstance(srcArray, (cudaArray_t)):
            psrcArray = int(srcArray)
            csrcArray = <ccudart.cudaArray_t><void_ptr>psrcArray
        else:
            psrcArray = int(cudaArray_t(srcArray))
            csrcArray = <ccudart.cudaArray_t><void_ptr>psrcArray
        self._srcArray._ptr[0] = csrcArray
    @property
    def srcPos(self):
        return self._srcPos
    @srcPos.setter
    def srcPos(self, srcPos not None : cudaPos):
        pass
        for _attr in dir(srcPos):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._srcPos, _attr, getattr(srcPos, _attr))
    @property
    def srcPtr(self):
        return self._srcPtr
    @srcPtr.setter
    def srcPtr(self, srcPtr not None : cudaPitchedPtr):
        pass
        for _attr in dir(srcPtr):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._srcPtr, _attr, getattr(srcPtr, _attr))
    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef ccudart.cudaArray_t cdstArray
        if dstArray is None:
            cdstArray = <ccudart.cudaArray_t><void_ptr>0
        elif isinstance(dstArray, (cudaArray_t)):
            pdstArray = int(dstArray)
            cdstArray = <ccudart.cudaArray_t><void_ptr>pdstArray
        else:
            pdstArray = int(cudaArray_t(dstArray))
            cdstArray = <ccudart.cudaArray_t><void_ptr>pdstArray
        self._dstArray._ptr[0] = cdstArray
    @property
    def dstPos(self):
        return self._dstPos
    @dstPos.setter
    def dstPos(self, dstPos not None : cudaPos):
        pass
        for _attr in dir(dstPos):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._dstPos, _attr, getattr(dstPos, _attr))
    @property
    def dstPtr(self):
        return self._dstPtr
    @dstPtr.setter
    def dstPtr(self, dstPtr not None : cudaPitchedPtr):
        pass
        for _attr in dir(dstPtr):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._dstPtr, _attr, getattr(dstPtr, _attr))
    @property
    def extent(self):
        return self._extent
    @extent.setter
    def extent(self, extent not None : cudaExtent):
        pass
        for _attr in dir(extent):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._extent, _attr, getattr(extent, _attr))
    @property
    def kind(self):
        return cudaMemcpyKind(self._ptr[0].kind)
    @kind.setter
    def kind(self, kind not None : cudaMemcpyKind):
        pass
        self._ptr[0].kind = kind.value

cdef class cudaMemcpy3DPeerParms:
    """
    CUDA 3D cross-device memory copying parameters

    Attributes
    ----------
    srcArray : cudaArray_t
        Source memory address
    srcPos : cudaPos
        Source position offset
    srcPtr : cudaPitchedPtr
        Pitched source memory address
    srcDevice : int
        Source device
    dstArray : cudaArray_t
        Destination memory address
    dstPos : cudaPos
        Destination position offset
    dstPtr : cudaPitchedPtr
        Pitched destination memory address
    dstDevice : int
        Destination device
    extent : cudaExtent
        Requested memory copy size

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemcpy3DPeerParms *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcArray = cudaArray_t(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._srcPos = cudaPos(_ptr=<void_ptr>&self._ptr[0].srcPos)
        self._srcPtr = cudaPitchedPtr(_ptr=<void_ptr>&self._ptr[0].srcPtr)
        self._dstArray = cudaArray_t(_ptr=<void_ptr>&self._ptr[0].dstArray)
        self._dstPos = cudaPos(_ptr=<void_ptr>&self._ptr[0].dstPos)
        self._dstPtr = cudaPitchedPtr(_ptr=<void_ptr>&self._ptr[0].dstPtr)
        self._extent = cudaExtent(_ptr=<void_ptr>&self._ptr[0].extent)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['srcArray : ' + str(self.srcArray)]
            str_list += ['srcPos :\n' + '\n'.join(['    ' + line for line in str(self.srcPos).splitlines()])]
            str_list += ['srcPtr :\n' + '\n'.join(['    ' + line for line in str(self.srcPtr).splitlines()])]
            str_list += ['srcDevice : ' + str(self.srcDevice)]
            str_list += ['dstArray : ' + str(self.dstArray)]
            str_list += ['dstPos :\n' + '\n'.join(['    ' + line for line in str(self.dstPos).splitlines()])]
            str_list += ['dstPtr :\n' + '\n'.join(['    ' + line for line in str(self.dstPtr).splitlines()])]
            str_list += ['dstDevice : ' + str(self.dstDevice)]
            str_list += ['extent :\n' + '\n'.join(['    ' + line for line in str(self.extent).splitlines()])]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef ccudart.cudaArray_t csrcArray
        if srcArray is None:
            csrcArray = <ccudart.cudaArray_t><void_ptr>0
        elif isinstance(srcArray, (cudaArray_t)):
            psrcArray = int(srcArray)
            csrcArray = <ccudart.cudaArray_t><void_ptr>psrcArray
        else:
            psrcArray = int(cudaArray_t(srcArray))
            csrcArray = <ccudart.cudaArray_t><void_ptr>psrcArray
        self._srcArray._ptr[0] = csrcArray
    @property
    def srcPos(self):
        return self._srcPos
    @srcPos.setter
    def srcPos(self, srcPos not None : cudaPos):
        pass
        for _attr in dir(srcPos):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._srcPos, _attr, getattr(srcPos, _attr))
    @property
    def srcPtr(self):
        return self._srcPtr
    @srcPtr.setter
    def srcPtr(self, srcPtr not None : cudaPitchedPtr):
        pass
        for _attr in dir(srcPtr):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._srcPtr, _attr, getattr(srcPtr, _attr))
    @property
    def srcDevice(self):
        return self._ptr[0].srcDevice
    @srcDevice.setter
    def srcDevice(self, int srcDevice):
        pass
        self._ptr[0].srcDevice = srcDevice
    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef ccudart.cudaArray_t cdstArray
        if dstArray is None:
            cdstArray = <ccudart.cudaArray_t><void_ptr>0
        elif isinstance(dstArray, (cudaArray_t)):
            pdstArray = int(dstArray)
            cdstArray = <ccudart.cudaArray_t><void_ptr>pdstArray
        else:
            pdstArray = int(cudaArray_t(dstArray))
            cdstArray = <ccudart.cudaArray_t><void_ptr>pdstArray
        self._dstArray._ptr[0] = cdstArray
    @property
    def dstPos(self):
        return self._dstPos
    @dstPos.setter
    def dstPos(self, dstPos not None : cudaPos):
        pass
        for _attr in dir(dstPos):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._dstPos, _attr, getattr(dstPos, _attr))
    @property
    def dstPtr(self):
        return self._dstPtr
    @dstPtr.setter
    def dstPtr(self, dstPtr not None : cudaPitchedPtr):
        pass
        for _attr in dir(dstPtr):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._dstPtr, _attr, getattr(dstPtr, _attr))
    @property
    def dstDevice(self):
        return self._ptr[0].dstDevice
    @dstDevice.setter
    def dstDevice(self, int dstDevice):
        pass
        self._ptr[0].dstDevice = dstDevice
    @property
    def extent(self):
        return self._extent
    @extent.setter
    def extent(self, extent not None : cudaExtent):
        pass
        for _attr in dir(extent):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._extent, _attr, getattr(extent, _attr))

cdef class cudaMemsetParams:
    """
    CUDA Memset node parameters

    Attributes
    ----------
    dst : Any
        Destination device pointer
    pitch : size_t
        Pitch of destination device pointer. Unused if height is 1
    value : unsigned int
        Value to be set
    elementSize : unsigned int
        Size of each element in bytes. Must be 1, 2, or 4.
    width : size_t
        Width of the row in elements
    height : size_t
        Number of rows

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemsetParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['dst : ' + hex(self.dst)]
            str_list += ['pitch : ' + str(self.pitch)]
            str_list += ['value : ' + str(self.value)]
            str_list += ['elementSize : ' + str(self.elementSize)]
            str_list += ['width : ' + str(self.width)]
            str_list += ['height : ' + str(self.height)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def dst(self):
        return <void_ptr>self._ptr[0].dst
    @dst.setter
    def dst(self, dst):
        _cdst = utils.HelperInputVoidPtr(dst)
        self._ptr[0].dst = <void*><void_ptr>_cdst.cptr
    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, size_t pitch):
        pass
        self._ptr[0].pitch = pitch
    @property
    def value(self):
        return self._ptr[0].value
    @value.setter
    def value(self, unsigned int value):
        pass
        self._ptr[0].value = value
    @property
    def elementSize(self):
        return self._ptr[0].elementSize
    @elementSize.setter
    def elementSize(self, unsigned int elementSize):
        pass
        self._ptr[0].elementSize = elementSize
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        pass
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        pass
        self._ptr[0].height = height

cdef class cudaAccessPolicyWindow:
    """
    Specifies an access policy for a window, a contiguous extent of
    memory beginning at base_ptr and ending at base_ptr + num_bytes.
    Partition into many segments and assign segments such that. sum of
    "hit segments" / window == approx. ratio. sum of "miss segments" /
    window == approx 1-ratio. Segments and ratio specifications are
    fitted to the capabilities of the architecture. Accesses in a hit
    segment apply the hitProp access policy. Accesses in a miss segment
    apply the missProp access policy.

    Attributes
    ----------
    base_ptr : Any
        Starting address of the access policy window. CUDA driver may align
        it.
    num_bytes : size_t
        Size in bytes of the window policy. CUDA driver may restrict the
        maximum size and alignment.
    hitRatio : float
        hitRatio specifies percentage of lines assigned hitProp, rest are
        assigned missProp.
    hitProp : cudaAccessProperty
        ::CUaccessProperty set for hit.
    missProp : cudaAccessProperty
        ::CUaccessProperty set for miss. Must be either NORMAL or
        STREAMING.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaAccessPolicyWindow *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['base_ptr : ' + hex(self.base_ptr)]
            str_list += ['num_bytes : ' + str(self.num_bytes)]
            str_list += ['hitRatio : ' + str(self.hitRatio)]
            str_list += ['hitProp : ' + str(self.hitProp)]
            str_list += ['missProp : ' + str(self.missProp)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def base_ptr(self):
        return <void_ptr>self._ptr[0].base_ptr
    @base_ptr.setter
    def base_ptr(self, base_ptr):
        _cbase_ptr = utils.HelperInputVoidPtr(base_ptr)
        self._ptr[0].base_ptr = <void*><void_ptr>_cbase_ptr.cptr
    @property
    def num_bytes(self):
        return self._ptr[0].num_bytes
    @num_bytes.setter
    def num_bytes(self, size_t num_bytes):
        pass
        self._ptr[0].num_bytes = num_bytes
    @property
    def hitRatio(self):
        return self._ptr[0].hitRatio
    @hitRatio.setter
    def hitRatio(self, float hitRatio):
        pass
        self._ptr[0].hitRatio = hitRatio
    @property
    def hitProp(self):
        return cudaAccessProperty(self._ptr[0].hitProp)
    @hitProp.setter
    def hitProp(self, hitProp not None : cudaAccessProperty):
        pass
        self._ptr[0].hitProp = hitProp.value
    @property
    def missProp(self):
        return cudaAccessProperty(self._ptr[0].missProp)
    @missProp.setter
    def missProp(self, missProp not None : cudaAccessProperty):
        pass
        self._ptr[0].missProp = missProp.value

cdef class cudaHostNodeParams:
    """
    CUDA host node parameters

    Attributes
    ----------
    fn : cudaHostFn_t
        The function to call when the node executes
    userData : Any
        Argument to pass to the function

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaHostNodeParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._fn = cudaHostFn_t(_ptr=<void_ptr>&self._ptr[0].fn)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['fn : ' + str(self.fn)]
            str_list += ['userData : ' + hex(self.userData)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def fn(self):
        return self._fn
    @fn.setter
    def fn(self, fn):
        cdef ccudart.cudaHostFn_t cfn
        if fn is None:
            cfn = <ccudart.cudaHostFn_t><void_ptr>0
        elif isinstance(fn, (cudaHostFn_t)):
            pfn = int(fn)
            cfn = <ccudart.cudaHostFn_t><void_ptr>pfn
        else:
            pfn = int(cudaHostFn_t(fn))
            cfn = <ccudart.cudaHostFn_t><void_ptr>pfn
        self._fn._ptr[0] = cfn
    @property
    def userData(self):
        return <void_ptr>self._ptr[0].userData
    @userData.setter
    def userData(self, userData):
        _cuserData = utils.HelperInputVoidPtr(userData)
        self._ptr[0].userData = <void*><void_ptr>_cuserData.cptr

cdef class _cudaResourceDesc_res_res_array_s:
    """

    Attributes
    ----------
    array : cudaArray_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaResourceDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        self._array = cudaArray_t(_ptr=<void_ptr>&self._ptr[0].res.array.array)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['array : ' + str(self.array)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def array(self):
        return self._array
    @array.setter
    def array(self, array):
        cdef ccudart.cudaArray_t carray
        if array is None:
            carray = <ccudart.cudaArray_t><void_ptr>0
        elif isinstance(array, (cudaArray_t)):
            parray = int(array)
            carray = <ccudart.cudaArray_t><void_ptr>parray
        else:
            parray = int(cudaArray_t(array))
            carray = <ccudart.cudaArray_t><void_ptr>parray
        self._array._ptr[0] = carray

cdef class _cudaResourceDesc_res_res_mipmap_s:
    """

    Attributes
    ----------
    mipmap : cudaMipmappedArray_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaResourceDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        self._mipmap = cudaMipmappedArray_t(_ptr=<void_ptr>&self._ptr[0].res.mipmap.mipmap)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['mipmap : ' + str(self.mipmap)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def mipmap(self):
        return self._mipmap
    @mipmap.setter
    def mipmap(self, mipmap):
        cdef ccudart.cudaMipmappedArray_t cmipmap
        if mipmap is None:
            cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>0
        elif isinstance(mipmap, (cudaMipmappedArray_t)):
            pmipmap = int(mipmap)
            cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmap
        else:
            pmipmap = int(cudaMipmappedArray_t(mipmap))
            cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmap
        self._mipmap._ptr[0] = cmipmap

cdef class _cudaResourceDesc_res_res_linear_s:
    """

    Attributes
    ----------
    devPtr : void

    desc : cudaChannelFormatDesc

    sizeInBytes : size_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaResourceDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        self._desc = cudaChannelFormatDesc(_ptr=<void_ptr>&self._ptr[0].res.linear.desc)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['devPtr : ' + hex(self.devPtr)]
            str_list += ['desc :\n' + '\n'.join(['    ' + line for line in str(self.desc).splitlines()])]
            str_list += ['sizeInBytes : ' + str(self.sizeInBytes)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def devPtr(self):
        return <void_ptr>self._ptr[0].res.linear.devPtr
    @devPtr.setter
    def devPtr(self, devPtr):
        _cdevPtr = utils.HelperInputVoidPtr(devPtr)
        self._ptr[0].res.linear.devPtr = <void*><void_ptr>_cdevPtr.cptr
    @property
    def desc(self):
        return self._desc
    @desc.setter
    def desc(self, desc not None : cudaChannelFormatDesc):
        pass
        for _attr in dir(desc):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._desc, _attr, getattr(desc, _attr))
    @property
    def sizeInBytes(self):
        return self._ptr[0].res.linear.sizeInBytes
    @sizeInBytes.setter
    def sizeInBytes(self, size_t sizeInBytes):
        pass
        self._ptr[0].res.linear.sizeInBytes = sizeInBytes

cdef class _cudaResourceDesc_res_res_pitch2D_s:
    """

    Attributes
    ----------
    devPtr : void

    desc : cudaChannelFormatDesc

    width : size_t

    height : size_t

    pitchInBytes : size_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaResourceDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        self._desc = cudaChannelFormatDesc(_ptr=<void_ptr>&self._ptr[0].res.pitch2D.desc)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['devPtr : ' + hex(self.devPtr)]
            str_list += ['desc :\n' + '\n'.join(['    ' + line for line in str(self.desc).splitlines()])]
            str_list += ['width : ' + str(self.width)]
            str_list += ['height : ' + str(self.height)]
            str_list += ['pitchInBytes : ' + str(self.pitchInBytes)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def devPtr(self):
        return <void_ptr>self._ptr[0].res.pitch2D.devPtr
    @devPtr.setter
    def devPtr(self, devPtr):
        _cdevPtr = utils.HelperInputVoidPtr(devPtr)
        self._ptr[0].res.pitch2D.devPtr = <void*><void_ptr>_cdevPtr.cptr
    @property
    def desc(self):
        return self._desc
    @desc.setter
    def desc(self, desc not None : cudaChannelFormatDesc):
        pass
        for _attr in dir(desc):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._desc, _attr, getattr(desc, _attr))
    @property
    def width(self):
        return self._ptr[0].res.pitch2D.width
    @width.setter
    def width(self, size_t width):
        pass
        self._ptr[0].res.pitch2D.width = width
    @property
    def height(self):
        return self._ptr[0].res.pitch2D.height
    @height.setter
    def height(self, size_t height):
        pass
        self._ptr[0].res.pitch2D.height = height
    @property
    def pitchInBytes(self):
        return self._ptr[0].res.pitch2D.pitchInBytes
    @pitchInBytes.setter
    def pitchInBytes(self, size_t pitchInBytes):
        pass
        self._ptr[0].res.pitch2D.pitchInBytes = pitchInBytes

cdef class _cudaResourceDesc_res_u:
    """

    Attributes
    ----------
    array : _cudaResourceDesc_res_res_array_s

    mipmap : _cudaResourceDesc_res_res_mipmap_s

    linear : _cudaResourceDesc_res_res_linear_s

    pitch2D : _cudaResourceDesc_res_res_pitch2D_s


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaResourceDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        self._array = _cudaResourceDesc_res_res_array_s(_ptr=<void_ptr>self._ptr)
        self._mipmap = _cudaResourceDesc_res_res_mipmap_s(_ptr=<void_ptr>self._ptr)
        self._linear = _cudaResourceDesc_res_res_linear_s(_ptr=<void_ptr>self._ptr)
        self._pitch2D = _cudaResourceDesc_res_res_pitch2D_s(_ptr=<void_ptr>self._ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['array :\n' + '\n'.join(['    ' + line for line in str(self.array).splitlines()])]
            str_list += ['mipmap :\n' + '\n'.join(['    ' + line for line in str(self.mipmap).splitlines()])]
            str_list += ['linear :\n' + '\n'.join(['    ' + line for line in str(self.linear).splitlines()])]
            str_list += ['pitch2D :\n' + '\n'.join(['    ' + line for line in str(self.pitch2D).splitlines()])]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def array(self):
        return self._array
    @array.setter
    def array(self, array not None : _cudaResourceDesc_res_res_array_s):
        pass
        for _attr in dir(array):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._array, _attr, getattr(array, _attr))
    @property
    def mipmap(self):
        return self._mipmap
    @mipmap.setter
    def mipmap(self, mipmap not None : _cudaResourceDesc_res_res_mipmap_s):
        pass
        for _attr in dir(mipmap):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._mipmap, _attr, getattr(mipmap, _attr))
    @property
    def linear(self):
        return self._linear
    @linear.setter
    def linear(self, linear not None : _cudaResourceDesc_res_res_linear_s):
        pass
        for _attr in dir(linear):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._linear, _attr, getattr(linear, _attr))
    @property
    def pitch2D(self):
        return self._pitch2D
    @pitch2D.setter
    def pitch2D(self, pitch2D not None : _cudaResourceDesc_res_res_pitch2D_s):
        pass
        for _attr in dir(pitch2D):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._pitch2D, _attr, getattr(pitch2D, _attr))

cdef class cudaResourceDesc:
    """
    CUDA resource descriptor

    Attributes
    ----------
    resType : cudaResourceType
        Resource type
    res : _cudaResourceDesc_res_u


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaResourceDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._res = _cudaResourceDesc_res_u(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['resType : ' + str(self.resType)]
            str_list += ['res :\n' + '\n'.join(['    ' + line for line in str(self.res).splitlines()])]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def resType(self):
        return cudaResourceType(self._ptr[0].resType)
    @resType.setter
    def resType(self, resType not None : cudaResourceType):
        pass
        self._ptr[0].resType = resType.value
    @property
    def res(self):
        return self._res
    @res.setter
    def res(self, res not None : _cudaResourceDesc_res_u):
        pass
        for _attr in dir(res):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._res, _attr, getattr(res, _attr))

cdef class cudaResourceViewDesc:
    """
    CUDA resource view descriptor

    Attributes
    ----------
    format : cudaResourceViewFormat
        Resource view format
    width : size_t
        Width of the resource view
    height : size_t
        Height of the resource view
    depth : size_t
        Depth of the resource view
    firstMipmapLevel : unsigned int
        First defined mipmap level
    lastMipmapLevel : unsigned int
        Last defined mipmap level
    firstLayer : unsigned int
        First layer index
    lastLayer : unsigned int
        Last layer index

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaResourceViewDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['format : ' + str(self.format)]
            str_list += ['width : ' + str(self.width)]
            str_list += ['height : ' + str(self.height)]
            str_list += ['depth : ' + str(self.depth)]
            str_list += ['firstMipmapLevel : ' + str(self.firstMipmapLevel)]
            str_list += ['lastMipmapLevel : ' + str(self.lastMipmapLevel)]
            str_list += ['firstLayer : ' + str(self.firstLayer)]
            str_list += ['lastLayer : ' + str(self.lastLayer)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def format(self):
        return cudaResourceViewFormat(self._ptr[0].format)
    @format.setter
    def format(self, format not None : cudaResourceViewFormat):
        pass
        self._ptr[0].format = format.value
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        pass
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        pass
        self._ptr[0].height = height
    @property
    def depth(self):
        return self._ptr[0].depth
    @depth.setter
    def depth(self, size_t depth):
        pass
        self._ptr[0].depth = depth
    @property
    def firstMipmapLevel(self):
        return self._ptr[0].firstMipmapLevel
    @firstMipmapLevel.setter
    def firstMipmapLevel(self, unsigned int firstMipmapLevel):
        pass
        self._ptr[0].firstMipmapLevel = firstMipmapLevel
    @property
    def lastMipmapLevel(self):
        return self._ptr[0].lastMipmapLevel
    @lastMipmapLevel.setter
    def lastMipmapLevel(self, unsigned int lastMipmapLevel):
        pass
        self._ptr[0].lastMipmapLevel = lastMipmapLevel
    @property
    def firstLayer(self):
        return self._ptr[0].firstLayer
    @firstLayer.setter
    def firstLayer(self, unsigned int firstLayer):
        pass
        self._ptr[0].firstLayer = firstLayer
    @property
    def lastLayer(self):
        return self._ptr[0].lastLayer
    @lastLayer.setter
    def lastLayer(self, unsigned int lastLayer):
        pass
        self._ptr[0].lastLayer = lastLayer

cdef class cudaPointerAttributes:
    """
    CUDA pointer attributes

    Attributes
    ----------
    type : cudaMemoryType
        The type of memory - cudaMemoryTypeUnregistered,
        cudaMemoryTypeHost, cudaMemoryTypeDevice or cudaMemoryTypeManaged.
    device : int
        The device against which the memory was allocated or registered. If
        the memory type is cudaMemoryTypeDevice then this identifies the
        device on which the memory referred physically resides. If the
        memory type is cudaMemoryTypeHost or::cudaMemoryTypeManaged then
        this identifies the device which was current when the memory was
        allocated or registered (and if that device is deinitialized then
        this allocation will vanish with that device's state).
    devicePointer : Any
        The address which may be dereferenced on the current device to
        access the memory or NULL if no such address exists.
    hostPointer : Any
        The address which may be dereferenced on the host to access the
        memory or NULL if no such address exists.  CUDA doesn't check if
        unregistered memory is allocated so this field may contain invalid
        pointer if an invalid pointer has been passed to CUDA.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaPointerAttributes *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['type : ' + str(self.type)]
            str_list += ['device : ' + str(self.device)]
            str_list += ['devicePointer : ' + hex(self.devicePointer)]
            str_list += ['hostPointer : ' + hex(self.hostPointer)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def type(self):
        return cudaMemoryType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : cudaMemoryType):
        pass
        self._ptr[0].type = type.value
    @property
    def device(self):
        return self._ptr[0].device
    @device.setter
    def device(self, int device):
        pass
        self._ptr[0].device = device
    @property
    def devicePointer(self):
        return <void_ptr>self._ptr[0].devicePointer
    @devicePointer.setter
    def devicePointer(self, devicePointer):
        _cdevicePointer = utils.HelperInputVoidPtr(devicePointer)
        self._ptr[0].devicePointer = <void*><void_ptr>_cdevicePointer.cptr
    @property
    def hostPointer(self):
        return <void_ptr>self._ptr[0].hostPointer
    @hostPointer.setter
    def hostPointer(self, hostPointer):
        _chostPointer = utils.HelperInputVoidPtr(hostPointer)
        self._ptr[0].hostPointer = <void*><void_ptr>_chostPointer.cptr

cdef class cudaFuncAttributes:
    """
    CUDA function attributes

    Attributes
    ----------
    sharedSizeBytes : size_t
        The size in bytes of statically-allocated shared memory per block
        required by this function. This does not include dynamically-
        allocated shared memory requested by the user at runtime.
    constSizeBytes : size_t
        The size in bytes of user-allocated constant memory required by
        this function.
    localSizeBytes : size_t
        The size in bytes of local memory used by each thread of this
        function.
    maxThreadsPerBlock : int
        The maximum number of threads per block, beyond which a launch of
        the function would fail. This number depends on both the function
        and the device on which the function is currently loaded.
    numRegs : int
        The number of registers used by each thread of this function.
    ptxVersion : int
        The PTX virtual architecture version for which the function was
        compiled. This value is the major PTX version * 10 + the minor PTX
        version, so a PTX version 1.3 function would return the value 13.
    binaryVersion : int
        The binary architecture version for which the function was
        compiled. This value is the major binary version * 10 + the minor
        binary version, so a binary version 1.3 function would return the
        value 13.
    cacheModeCA : int
        The attribute to indicate whether the function has been compiled
        with user specified option "-Xptxas --dlcm=ca" set.
    maxDynamicSharedSizeBytes : int
        The maximum size in bytes of dynamic shared memory per block for
        this function. Any launch must have a dynamic shared memory size
        smaller than this value.
    preferredShmemCarveout : int
        On devices where the L1 cache and shared memory use the same
        hardware resources, this sets the shared memory carveout
        preference, in percent of the maximum shared memory. Refer to
        cudaDevAttrMaxSharedMemoryPerMultiprocessor. This is only a hint,
        and the driver can choose a different ratio if required to execute
        the function. See cudaFuncSetAttribute

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaFuncAttributes *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['sharedSizeBytes : ' + str(self.sharedSizeBytes)]
            str_list += ['constSizeBytes : ' + str(self.constSizeBytes)]
            str_list += ['localSizeBytes : ' + str(self.localSizeBytes)]
            str_list += ['maxThreadsPerBlock : ' + str(self.maxThreadsPerBlock)]
            str_list += ['numRegs : ' + str(self.numRegs)]
            str_list += ['ptxVersion : ' + str(self.ptxVersion)]
            str_list += ['binaryVersion : ' + str(self.binaryVersion)]
            str_list += ['cacheModeCA : ' + str(self.cacheModeCA)]
            str_list += ['maxDynamicSharedSizeBytes : ' + str(self.maxDynamicSharedSizeBytes)]
            str_list += ['preferredShmemCarveout : ' + str(self.preferredShmemCarveout)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def sharedSizeBytes(self):
        return self._ptr[0].sharedSizeBytes
    @sharedSizeBytes.setter
    def sharedSizeBytes(self, size_t sharedSizeBytes):
        pass
        self._ptr[0].sharedSizeBytes = sharedSizeBytes
    @property
    def constSizeBytes(self):
        return self._ptr[0].constSizeBytes
    @constSizeBytes.setter
    def constSizeBytes(self, size_t constSizeBytes):
        pass
        self._ptr[0].constSizeBytes = constSizeBytes
    @property
    def localSizeBytes(self):
        return self._ptr[0].localSizeBytes
    @localSizeBytes.setter
    def localSizeBytes(self, size_t localSizeBytes):
        pass
        self._ptr[0].localSizeBytes = localSizeBytes
    @property
    def maxThreadsPerBlock(self):
        return self._ptr[0].maxThreadsPerBlock
    @maxThreadsPerBlock.setter
    def maxThreadsPerBlock(self, int maxThreadsPerBlock):
        pass
        self._ptr[0].maxThreadsPerBlock = maxThreadsPerBlock
    @property
    def numRegs(self):
        return self._ptr[0].numRegs
    @numRegs.setter
    def numRegs(self, int numRegs):
        pass
        self._ptr[0].numRegs = numRegs
    @property
    def ptxVersion(self):
        return self._ptr[0].ptxVersion
    @ptxVersion.setter
    def ptxVersion(self, int ptxVersion):
        pass
        self._ptr[0].ptxVersion = ptxVersion
    @property
    def binaryVersion(self):
        return self._ptr[0].binaryVersion
    @binaryVersion.setter
    def binaryVersion(self, int binaryVersion):
        pass
        self._ptr[0].binaryVersion = binaryVersion
    @property
    def cacheModeCA(self):
        return self._ptr[0].cacheModeCA
    @cacheModeCA.setter
    def cacheModeCA(self, int cacheModeCA):
        pass
        self._ptr[0].cacheModeCA = cacheModeCA
    @property
    def maxDynamicSharedSizeBytes(self):
        return self._ptr[0].maxDynamicSharedSizeBytes
    @maxDynamicSharedSizeBytes.setter
    def maxDynamicSharedSizeBytes(self, int maxDynamicSharedSizeBytes):
        pass
        self._ptr[0].maxDynamicSharedSizeBytes = maxDynamicSharedSizeBytes
    @property
    def preferredShmemCarveout(self):
        return self._ptr[0].preferredShmemCarveout
    @preferredShmemCarveout.setter
    def preferredShmemCarveout(self, int preferredShmemCarveout):
        pass
        self._ptr[0].preferredShmemCarveout = preferredShmemCarveout

cdef class cudaMemLocation:
    """
    Specifies a memory location.  To specify a gpu, set type =
    cudaMemLocationTypeDevice and set id = the gpu's device ordinal.

    Attributes
    ----------
    type : cudaMemLocationType
        Specifies the location type, which modifies the meaning of id.
    id : int
        identifier for a given this location's ::CUmemLocationType.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemLocation *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['type : ' + str(self.type)]
            str_list += ['id : ' + str(self.id)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def type(self):
        return cudaMemLocationType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : cudaMemLocationType):
        pass
        self._ptr[0].type = type.value
    @property
    def id(self):
        return self._ptr[0].id
    @id.setter
    def id(self, int id):
        pass
        self._ptr[0].id = id

cdef class cudaMemAccessDesc:
    """
    Memory access descriptor

    Attributes
    ----------
    location : cudaMemLocation
        Location on which the request is to change it's accessibility
    flags : cudaMemAccessFlags
        ::CUmemProt accessibility flags to set on the request

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemAccessDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = cudaMemLocation(_ptr=<void_ptr>&self._ptr[0].location)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            str_list += ['flags : ' + str(self.flags)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : cudaMemLocation):
        pass
        for _attr in dir(location):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._location, _attr, getattr(location, _attr))
    @property
    def flags(self):
        return cudaMemAccessFlags(self._ptr[0].flags)
    @flags.setter
    def flags(self, flags not None : cudaMemAccessFlags):
        pass
        self._ptr[0].flags = flags.value

cdef class cudaMemPoolProps:
    """
    Specifies the properties of allocations made from the pool.

    Attributes
    ----------
    allocType : cudaMemAllocationType
        Allocation type. Currently must be specified as
        cudaMemAllocationTypePinned
    handleTypes : cudaMemAllocationHandleType
        Handle types that will be supported by allocations from the pool.
    location : cudaMemLocation
        Location allocations should reside.
    win32SecurityAttributes : Any
        Windows-specific LPSECURITYATTRIBUTES required when
        cudaMemHandleTypeWin32 is specified. This security attribute
        defines the scope of which exported allocations may be tranferred
        to other processes. In all other cases, this field is required to
        be zero.
    reserved : bytes
        reserved for future use, must be 0

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemPoolProps *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = cudaMemLocation(_ptr=<void_ptr>&self._ptr[0].location)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['allocType : ' + str(self.allocType)]
            str_list += ['handleTypes : ' + str(self.handleTypes)]
            str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            str_list += ['win32SecurityAttributes : ' + hex(self.win32SecurityAttributes)]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def allocType(self):
        return cudaMemAllocationType(self._ptr[0].allocType)
    @allocType.setter
    def allocType(self, allocType not None : cudaMemAllocationType):
        pass
        self._ptr[0].allocType = allocType.value
    @property
    def handleTypes(self):
        return cudaMemAllocationHandleType(self._ptr[0].handleTypes)
    @handleTypes.setter
    def handleTypes(self, handleTypes not None : cudaMemAllocationHandleType):
        pass
        self._ptr[0].handleTypes = handleTypes.value
    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : cudaMemLocation):
        pass
        for _attr in dir(location):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._location, _attr, getattr(location, _attr))
    @property
    def win32SecurityAttributes(self):
        return <void_ptr>self._ptr[0].win32SecurityAttributes
    @win32SecurityAttributes.setter
    def win32SecurityAttributes(self, win32SecurityAttributes):
        _cwin32SecurityAttributes = utils.HelperInputVoidPtr(win32SecurityAttributes)
        self._ptr[0].win32SecurityAttributes = <void*><void_ptr>_cwin32SecurityAttributes.cptr
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].reserved[i] = b

cdef class cudaMemPoolPtrExportData:
    """
    Opaque data for exporting a pool allocation

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemPoolPtrExportData *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].reserved[i] = b

cdef class cudaMemAllocNodeParams:
    """
    Memory allocation node parameters

    Attributes
    ----------
    poolProps : cudaMemPoolProps
        in: location where the allocation should reside (specified in
        ::location). ::handleTypes must be cudaMemHandleTypeNone. IPC is
        not supported. in: array of memory access descriptors. Used to
        describe peer GPU access
    accessDescs : cudaMemAccessDesc
        in: number of memory access descriptors. Must not exceed the number
        of GPUs.
    accessDescCount : size_t
        in: Number of `accessDescs`s
    bytesize : size_t
        in: size in bytes of the requested allocation
    dptr : Any
        out: address of the allocation returned by CUDA

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaMemAllocNodeParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._poolProps = cudaMemPoolProps(_ptr=<void_ptr>&self._ptr[0].poolProps)
    def __dealloc__(self):
        if self._accessDescs is not NULL:
            free(self._accessDescs)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['poolProps :\n' + '\n'.join(['    ' + line for line in str(self.poolProps).splitlines()])]
            str_list += ['accessDescs : ' + str(self.accessDescs)]
            str_list += ['accessDescCount : ' + str(self.accessDescCount)]
            str_list += ['bytesize : ' + str(self.bytesize)]
            str_list += ['dptr : ' + hex(self.dptr)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def poolProps(self):
        return self._poolProps
    @poolProps.setter
    def poolProps(self, poolProps not None : cudaMemPoolProps):
        pass
        for _attr in dir(poolProps):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._poolProps, _attr, getattr(poolProps, _attr))
    @property
    def accessDescs(self):
        arrs = [<void_ptr>self._ptr[0].accessDescs + x*sizeof(ccudart.cudaMemAccessDesc) for x in range(self._accessDescs_length)]
        return [cudaMemAccessDesc(_ptr=arr) for arr in arrs]
    @accessDescs.setter
    def accessDescs(self, val):
        if len(val) == 0:
            free(self._accessDescs)
            self._accessDescs_length = 0
            self._ptr[0].accessDescs = NULL
        else:
            if self._accessDescs_length != <size_t>len(val):
                free(self._accessDescs)
                self._accessDescs = <ccudart.cudaMemAccessDesc*> calloc(len(val), sizeof(ccudart.cudaMemAccessDesc))
                if self._accessDescs is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccudart.cudaMemAccessDesc)))
                self._accessDescs_length = <size_t>len(val)
                self._ptr[0].accessDescs = self._accessDescs
            for idx in range(len(val)):
                memcpy(&self._accessDescs[idx], (<cudaMemAccessDesc>val[idx])._ptr, sizeof(ccudart.cudaMemAccessDesc))
    @property
    def accessDescCount(self):
        return self._ptr[0].accessDescCount
    @accessDescCount.setter
    def accessDescCount(self, size_t accessDescCount):
        pass
        self._ptr[0].accessDescCount = accessDescCount
    @property
    def bytesize(self):
        return self._ptr[0].bytesize
    @bytesize.setter
    def bytesize(self, size_t bytesize):
        pass
        self._ptr[0].bytesize = bytesize
    @property
    def dptr(self):
        return <void_ptr>self._ptr[0].dptr
    @dptr.setter
    def dptr(self, dptr):
        _cdptr = utils.HelperInputVoidPtr(dptr)
        self._ptr[0].dptr = <void*><void_ptr>_cdptr.cptr

cdef class CUuuid_st:
    """

    Attributes
    ----------
    bytes : bytes
        < CUDA definition of UUID

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.CUuuid_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['bytes : ' + str(self.bytes.hex())]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def bytes(self):
        return PyBytes_FromStringAndSize(self._ptr[0].bytes, 16)

cdef class cudaDeviceProp:
    """
    CUDA device properties

    Attributes
    ----------
    name : List[char]
        ASCII string identifying device
    uuid : cudaUUID_t
        16-byte unique identifier
    luid : bytes
        8-byte locally unique identifier. Value is undefined on TCC and
        non-Windows platforms
    luidDeviceNodeMask : unsigned int
        LUID device node mask. Value is undefined on TCC and non-Windows
        platforms
    totalGlobalMem : size_t
        Global memory available on device in bytes
    sharedMemPerBlock : size_t
        Shared memory available per block in bytes
    regsPerBlock : int
        32-bit registers available per block
    warpSize : int
        Warp size in threads
    memPitch : size_t
        Maximum pitch in bytes allowed by memory copies
    maxThreadsPerBlock : int
        Maximum number of threads per block
    maxThreadsDim : List[int]
        Maximum size of each dimension of a block
    maxGridSize : List[int]
        Maximum size of each dimension of a grid
    clockRate : int
        Clock frequency in kilohertz
    totalConstMem : size_t
        Constant memory available on device in bytes
    major : int
        Major compute capability
    minor : int
        Minor compute capability
    textureAlignment : size_t
        Alignment requirement for textures
    texturePitchAlignment : size_t
        Pitch alignment requirement for texture references bound to pitched
        memory
    deviceOverlap : int
        Device can concurrently copy memory and execute a kernel.
        Deprecated. Use instead asyncEngineCount.
    multiProcessorCount : int
        Number of multiprocessors on device
    kernelExecTimeoutEnabled : int
        Specified whether there is a run time limit on kernels
    integrated : int
        Device is integrated as opposed to discrete
    canMapHostMemory : int
        Device can map host memory with
        cudaHostAlloc/cudaHostGetDevicePointer
    computeMode : int
        Compute mode (See cudaComputeMode)
    maxTexture1D : int
        Maximum 1D texture size
    maxTexture1DMipmap : int
        Maximum 1D mipmapped texture size
    maxTexture1DLinear : int
        Deprecated, do not use. Use cudaDeviceGetTexture1DLinearMaxWidth()
        or cuDeviceGetTexture1DLinearMaxWidth() instead.
    maxTexture2D : List[int]
        Maximum 2D texture dimensions
    maxTexture2DMipmap : List[int]
        Maximum 2D mipmapped texture dimensions
    maxTexture2DLinear : List[int]
        Maximum dimensions (width, height, pitch) for 2D textures bound to
        pitched memory
    maxTexture2DGather : List[int]
        Maximum 2D texture dimensions if texture gather operations have to
        be performed
    maxTexture3D : List[int]
        Maximum 3D texture dimensions
    maxTexture3DAlt : List[int]
        Maximum alternate 3D texture dimensions
    maxTextureCubemap : int
        Maximum Cubemap texture dimensions
    maxTexture1DLayered : List[int]
        Maximum 1D layered texture dimensions
    maxTexture2DLayered : List[int]
        Maximum 2D layered texture dimensions
    maxTextureCubemapLayered : List[int]
        Maximum Cubemap layered texture dimensions
    maxSurface1D : int
        Maximum 1D surface size
    maxSurface2D : List[int]
        Maximum 2D surface dimensions
    maxSurface3D : List[int]
        Maximum 3D surface dimensions
    maxSurface1DLayered : List[int]
        Maximum 1D layered surface dimensions
    maxSurface2DLayered : List[int]
        Maximum 2D layered surface dimensions
    maxSurfaceCubemap : int
        Maximum Cubemap surface dimensions
    maxSurfaceCubemapLayered : List[int]
        Maximum Cubemap layered surface dimensions
    surfaceAlignment : size_t
        Alignment requirements for surfaces
    concurrentKernels : int
        Device can possibly execute multiple kernels concurrently
    ECCEnabled : int
        Device has ECC support enabled
    pciBusID : int
        PCI bus ID of the device
    pciDeviceID : int
        PCI device ID of the device
    pciDomainID : int
        PCI domain ID of the device
    tccDriver : int
        1 if device is a Tesla device using TCC driver, 0 otherwise
    asyncEngineCount : int
        Number of asynchronous engines
    unifiedAddressing : int
        Device shares a unified address space with the host
    memoryClockRate : int
        Peak memory clock frequency in kilohertz
    memoryBusWidth : int
        Global memory bus width in bits
    l2CacheSize : int
        Size of L2 cache in bytes
    persistingL2CacheMaxSize : int
        Device's maximum l2 persisting lines capacity setting in bytes
    maxThreadsPerMultiProcessor : int
        Maximum resident threads per multiprocessor
    streamPrioritiesSupported : int
        Device supports stream priorities
    globalL1CacheSupported : int
        Device supports caching globals in L1
    localL1CacheSupported : int
        Device supports caching locals in L1
    sharedMemPerMultiprocessor : size_t
        Shared memory available per multiprocessor in bytes
    regsPerMultiprocessor : int
        32-bit registers available per multiprocessor
    managedMemory : int
        Device supports allocating managed memory on this system
    isMultiGpuBoard : int
        Device is on a multi-GPU board
    multiGpuBoardGroupID : int
        Unique identifier for a group of devices on the same multi-GPU
        board
    hostNativeAtomicSupported : int
        Link between the device and the host supports native atomic
        operations
    singleToDoublePrecisionPerfRatio : int
        Ratio of single precision performance (in floating-point operations
        per second) to double precision performance
    pageableMemoryAccess : int
        Device supports coherently accessing pageable memory without
        calling cudaHostRegister on it
    concurrentManagedAccess : int
        Device can coherently access managed memory concurrently with the
        CPU
    computePreemptionSupported : int
        Device supports Compute Preemption
    canUseHostPointerForRegisteredMem : int
        Device can access host registered memory at the same virtual
        address as the CPU
    cooperativeLaunch : int
        Device supports launching cooperative kernels via
        cudaLaunchCooperativeKernel
    cooperativeMultiDeviceLaunch : int
        Deprecated, cudaLaunchCooperativeKernelMultiDevice is deprecated.
    sharedMemPerBlockOptin : size_t
        Per device maximum shared memory per block usable by special opt in
    pageableMemoryAccessUsesHostPageTables : int
        Device accesses pageable memory via the host's page tables
    directManagedMemAccessFromHost : int
        Host can directly access managed memory on the device without
        migration.
    maxBlocksPerMultiProcessor : int
        Maximum number of resident blocks per multiprocessor
    accessPolicyMaxWindowSize : int
        The maximum value of cudaAccessPolicyWindow::num_bytes.
    reservedSharedMemPerBlock : size_t
        Shared memory reserved by CUDA driver per block in bytes

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaDeviceProp *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._uuid = cudaUUID_t(_ptr=<void_ptr>&self._ptr[0].uuid)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['name : ' + self.name.decode('utf-8')]
            str_list += ['uuid :\n' + '\n'.join(['    ' + line for line in str(self.uuid).splitlines()])]
            str_list += ['luid : ' + self.luid.hex()]
            str_list += ['luidDeviceNodeMask : ' + str(self.luidDeviceNodeMask)]
            str_list += ['totalGlobalMem : ' + str(self.totalGlobalMem)]
            str_list += ['sharedMemPerBlock : ' + str(self.sharedMemPerBlock)]
            str_list += ['regsPerBlock : ' + str(self.regsPerBlock)]
            str_list += ['warpSize : ' + str(self.warpSize)]
            str_list += ['memPitch : ' + str(self.memPitch)]
            str_list += ['maxThreadsPerBlock : ' + str(self.maxThreadsPerBlock)]
            str_list += ['maxThreadsDim : ' + str(self.maxThreadsDim)]
            str_list += ['maxGridSize : ' + str(self.maxGridSize)]
            str_list += ['clockRate : ' + str(self.clockRate)]
            str_list += ['totalConstMem : ' + str(self.totalConstMem)]
            str_list += ['major : ' + str(self.major)]
            str_list += ['minor : ' + str(self.minor)]
            str_list += ['textureAlignment : ' + str(self.textureAlignment)]
            str_list += ['texturePitchAlignment : ' + str(self.texturePitchAlignment)]
            str_list += ['deviceOverlap : ' + str(self.deviceOverlap)]
            str_list += ['multiProcessorCount : ' + str(self.multiProcessorCount)]
            str_list += ['kernelExecTimeoutEnabled : ' + str(self.kernelExecTimeoutEnabled)]
            str_list += ['integrated : ' + str(self.integrated)]
            str_list += ['canMapHostMemory : ' + str(self.canMapHostMemory)]
            str_list += ['computeMode : ' + str(self.computeMode)]
            str_list += ['maxTexture1D : ' + str(self.maxTexture1D)]
            str_list += ['maxTexture1DMipmap : ' + str(self.maxTexture1DMipmap)]
            str_list += ['maxTexture1DLinear : ' + str(self.maxTexture1DLinear)]
            str_list += ['maxTexture2D : ' + str(self.maxTexture2D)]
            str_list += ['maxTexture2DMipmap : ' + str(self.maxTexture2DMipmap)]
            str_list += ['maxTexture2DLinear : ' + str(self.maxTexture2DLinear)]
            str_list += ['maxTexture2DGather : ' + str(self.maxTexture2DGather)]
            str_list += ['maxTexture3D : ' + str(self.maxTexture3D)]
            str_list += ['maxTexture3DAlt : ' + str(self.maxTexture3DAlt)]
            str_list += ['maxTextureCubemap : ' + str(self.maxTextureCubemap)]
            str_list += ['maxTexture1DLayered : ' + str(self.maxTexture1DLayered)]
            str_list += ['maxTexture2DLayered : ' + str(self.maxTexture2DLayered)]
            str_list += ['maxTextureCubemapLayered : ' + str(self.maxTextureCubemapLayered)]
            str_list += ['maxSurface1D : ' + str(self.maxSurface1D)]
            str_list += ['maxSurface2D : ' + str(self.maxSurface2D)]
            str_list += ['maxSurface3D : ' + str(self.maxSurface3D)]
            str_list += ['maxSurface1DLayered : ' + str(self.maxSurface1DLayered)]
            str_list += ['maxSurface2DLayered : ' + str(self.maxSurface2DLayered)]
            str_list += ['maxSurfaceCubemap : ' + str(self.maxSurfaceCubemap)]
            str_list += ['maxSurfaceCubemapLayered : ' + str(self.maxSurfaceCubemapLayered)]
            str_list += ['surfaceAlignment : ' + str(self.surfaceAlignment)]
            str_list += ['concurrentKernels : ' + str(self.concurrentKernels)]
            str_list += ['ECCEnabled : ' + str(self.ECCEnabled)]
            str_list += ['pciBusID : ' + str(self.pciBusID)]
            str_list += ['pciDeviceID : ' + str(self.pciDeviceID)]
            str_list += ['pciDomainID : ' + str(self.pciDomainID)]
            str_list += ['tccDriver : ' + str(self.tccDriver)]
            str_list += ['asyncEngineCount : ' + str(self.asyncEngineCount)]
            str_list += ['unifiedAddressing : ' + str(self.unifiedAddressing)]
            str_list += ['memoryClockRate : ' + str(self.memoryClockRate)]
            str_list += ['memoryBusWidth : ' + str(self.memoryBusWidth)]
            str_list += ['l2CacheSize : ' + str(self.l2CacheSize)]
            str_list += ['persistingL2CacheMaxSize : ' + str(self.persistingL2CacheMaxSize)]
            str_list += ['maxThreadsPerMultiProcessor : ' + str(self.maxThreadsPerMultiProcessor)]
            str_list += ['streamPrioritiesSupported : ' + str(self.streamPrioritiesSupported)]
            str_list += ['globalL1CacheSupported : ' + str(self.globalL1CacheSupported)]
            str_list += ['localL1CacheSupported : ' + str(self.localL1CacheSupported)]
            str_list += ['sharedMemPerMultiprocessor : ' + str(self.sharedMemPerMultiprocessor)]
            str_list += ['regsPerMultiprocessor : ' + str(self.regsPerMultiprocessor)]
            str_list += ['managedMemory : ' + str(self.managedMemory)]
            str_list += ['isMultiGpuBoard : ' + str(self.isMultiGpuBoard)]
            str_list += ['multiGpuBoardGroupID : ' + str(self.multiGpuBoardGroupID)]
            str_list += ['hostNativeAtomicSupported : ' + str(self.hostNativeAtomicSupported)]
            str_list += ['singleToDoublePrecisionPerfRatio : ' + str(self.singleToDoublePrecisionPerfRatio)]
            str_list += ['pageableMemoryAccess : ' + str(self.pageableMemoryAccess)]
            str_list += ['concurrentManagedAccess : ' + str(self.concurrentManagedAccess)]
            str_list += ['computePreemptionSupported : ' + str(self.computePreemptionSupported)]
            str_list += ['canUseHostPointerForRegisteredMem : ' + str(self.canUseHostPointerForRegisteredMem)]
            str_list += ['cooperativeLaunch : ' + str(self.cooperativeLaunch)]
            str_list += ['cooperativeMultiDeviceLaunch : ' + str(self.cooperativeMultiDeviceLaunch)]
            str_list += ['sharedMemPerBlockOptin : ' + str(self.sharedMemPerBlockOptin)]
            str_list += ['pageableMemoryAccessUsesHostPageTables : ' + str(self.pageableMemoryAccessUsesHostPageTables)]
            str_list += ['directManagedMemAccessFromHost : ' + str(self.directManagedMemAccessFromHost)]
            str_list += ['maxBlocksPerMultiProcessor : ' + str(self.maxBlocksPerMultiProcessor)]
            str_list += ['accessPolicyMaxWindowSize : ' + str(self.accessPolicyMaxWindowSize)]
            str_list += ['reservedSharedMemPerBlock : ' + str(self.reservedSharedMemPerBlock)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def name(self):
        return self._ptr[0].name
    @name.setter
    def name(self, name):
        pass
        self._ptr[0].name = name
    @property
    def uuid(self):
        return self._uuid
    @uuid.setter
    def uuid(self, uuid not None : cudaUUID_t):
        pass
        for _attr in dir(uuid):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._uuid, _attr, getattr(uuid, _attr))
    @property
    def luid(self):
        return PyBytes_FromStringAndSize(self._ptr[0].luid, 8)
    @luid.setter
    def luid(self, luid):
        if len(luid) != 8:
            raise ValueError("luid length must be 8, is " + str(len(luid)))
        if CHAR_MIN == 0:
            for i, b in enumerate(luid):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].luid[i] = b
        else:
            for i, b in enumerate(luid):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].luid[i] = b
    @property
    def luidDeviceNodeMask(self):
        return self._ptr[0].luidDeviceNodeMask
    @luidDeviceNodeMask.setter
    def luidDeviceNodeMask(self, unsigned int luidDeviceNodeMask):
        pass
        self._ptr[0].luidDeviceNodeMask = luidDeviceNodeMask
    @property
    def totalGlobalMem(self):
        return self._ptr[0].totalGlobalMem
    @totalGlobalMem.setter
    def totalGlobalMem(self, size_t totalGlobalMem):
        pass
        self._ptr[0].totalGlobalMem = totalGlobalMem
    @property
    def sharedMemPerBlock(self):
        return self._ptr[0].sharedMemPerBlock
    @sharedMemPerBlock.setter
    def sharedMemPerBlock(self, size_t sharedMemPerBlock):
        pass
        self._ptr[0].sharedMemPerBlock = sharedMemPerBlock
    @property
    def regsPerBlock(self):
        return self._ptr[0].regsPerBlock
    @regsPerBlock.setter
    def regsPerBlock(self, int regsPerBlock):
        pass
        self._ptr[0].regsPerBlock = regsPerBlock
    @property
    def warpSize(self):
        return self._ptr[0].warpSize
    @warpSize.setter
    def warpSize(self, int warpSize):
        pass
        self._ptr[0].warpSize = warpSize
    @property
    def memPitch(self):
        return self._ptr[0].memPitch
    @memPitch.setter
    def memPitch(self, size_t memPitch):
        pass
        self._ptr[0].memPitch = memPitch
    @property
    def maxThreadsPerBlock(self):
        return self._ptr[0].maxThreadsPerBlock
    @maxThreadsPerBlock.setter
    def maxThreadsPerBlock(self, int maxThreadsPerBlock):
        pass
        self._ptr[0].maxThreadsPerBlock = maxThreadsPerBlock
    @property
    def maxThreadsDim(self):
        return self._ptr[0].maxThreadsDim
    @maxThreadsDim.setter
    def maxThreadsDim(self, maxThreadsDim):
        pass
        self._ptr[0].maxThreadsDim = maxThreadsDim
    @property
    def maxGridSize(self):
        return self._ptr[0].maxGridSize
    @maxGridSize.setter
    def maxGridSize(self, maxGridSize):
        pass
        self._ptr[0].maxGridSize = maxGridSize
    @property
    def clockRate(self):
        return self._ptr[0].clockRate
    @clockRate.setter
    def clockRate(self, int clockRate):
        pass
        self._ptr[0].clockRate = clockRate
    @property
    def totalConstMem(self):
        return self._ptr[0].totalConstMem
    @totalConstMem.setter
    def totalConstMem(self, size_t totalConstMem):
        pass
        self._ptr[0].totalConstMem = totalConstMem
    @property
    def major(self):
        return self._ptr[0].major
    @major.setter
    def major(self, int major):
        pass
        self._ptr[0].major = major
    @property
    def minor(self):
        return self._ptr[0].minor
    @minor.setter
    def minor(self, int minor):
        pass
        self._ptr[0].minor = minor
    @property
    def textureAlignment(self):
        return self._ptr[0].textureAlignment
    @textureAlignment.setter
    def textureAlignment(self, size_t textureAlignment):
        pass
        self._ptr[0].textureAlignment = textureAlignment
    @property
    def texturePitchAlignment(self):
        return self._ptr[0].texturePitchAlignment
    @texturePitchAlignment.setter
    def texturePitchAlignment(self, size_t texturePitchAlignment):
        pass
        self._ptr[0].texturePitchAlignment = texturePitchAlignment
    @property
    def deviceOverlap(self):
        return self._ptr[0].deviceOverlap
    @deviceOverlap.setter
    def deviceOverlap(self, int deviceOverlap):
        pass
        self._ptr[0].deviceOverlap = deviceOverlap
    @property
    def multiProcessorCount(self):
        return self._ptr[0].multiProcessorCount
    @multiProcessorCount.setter
    def multiProcessorCount(self, int multiProcessorCount):
        pass
        self._ptr[0].multiProcessorCount = multiProcessorCount
    @property
    def kernelExecTimeoutEnabled(self):
        return self._ptr[0].kernelExecTimeoutEnabled
    @kernelExecTimeoutEnabled.setter
    def kernelExecTimeoutEnabled(self, int kernelExecTimeoutEnabled):
        pass
        self._ptr[0].kernelExecTimeoutEnabled = kernelExecTimeoutEnabled
    @property
    def integrated(self):
        return self._ptr[0].integrated
    @integrated.setter
    def integrated(self, int integrated):
        pass
        self._ptr[0].integrated = integrated
    @property
    def canMapHostMemory(self):
        return self._ptr[0].canMapHostMemory
    @canMapHostMemory.setter
    def canMapHostMemory(self, int canMapHostMemory):
        pass
        self._ptr[0].canMapHostMemory = canMapHostMemory
    @property
    def computeMode(self):
        return self._ptr[0].computeMode
    @computeMode.setter
    def computeMode(self, int computeMode):
        pass
        self._ptr[0].computeMode = computeMode
    @property
    def maxTexture1D(self):
        return self._ptr[0].maxTexture1D
    @maxTexture1D.setter
    def maxTexture1D(self, int maxTexture1D):
        pass
        self._ptr[0].maxTexture1D = maxTexture1D
    @property
    def maxTexture1DMipmap(self):
        return self._ptr[0].maxTexture1DMipmap
    @maxTexture1DMipmap.setter
    def maxTexture1DMipmap(self, int maxTexture1DMipmap):
        pass
        self._ptr[0].maxTexture1DMipmap = maxTexture1DMipmap
    @property
    def maxTexture1DLinear(self):
        return self._ptr[0].maxTexture1DLinear
    @maxTexture1DLinear.setter
    def maxTexture1DLinear(self, int maxTexture1DLinear):
        pass
        self._ptr[0].maxTexture1DLinear = maxTexture1DLinear
    @property
    def maxTexture2D(self):
        return self._ptr[0].maxTexture2D
    @maxTexture2D.setter
    def maxTexture2D(self, maxTexture2D):
        pass
        self._ptr[0].maxTexture2D = maxTexture2D
    @property
    def maxTexture2DMipmap(self):
        return self._ptr[0].maxTexture2DMipmap
    @maxTexture2DMipmap.setter
    def maxTexture2DMipmap(self, maxTexture2DMipmap):
        pass
        self._ptr[0].maxTexture2DMipmap = maxTexture2DMipmap
    @property
    def maxTexture2DLinear(self):
        return self._ptr[0].maxTexture2DLinear
    @maxTexture2DLinear.setter
    def maxTexture2DLinear(self, maxTexture2DLinear):
        pass
        self._ptr[0].maxTexture2DLinear = maxTexture2DLinear
    @property
    def maxTexture2DGather(self):
        return self._ptr[0].maxTexture2DGather
    @maxTexture2DGather.setter
    def maxTexture2DGather(self, maxTexture2DGather):
        pass
        self._ptr[0].maxTexture2DGather = maxTexture2DGather
    @property
    def maxTexture3D(self):
        return self._ptr[0].maxTexture3D
    @maxTexture3D.setter
    def maxTexture3D(self, maxTexture3D):
        pass
        self._ptr[0].maxTexture3D = maxTexture3D
    @property
    def maxTexture3DAlt(self):
        return self._ptr[0].maxTexture3DAlt
    @maxTexture3DAlt.setter
    def maxTexture3DAlt(self, maxTexture3DAlt):
        pass
        self._ptr[0].maxTexture3DAlt = maxTexture3DAlt
    @property
    def maxTextureCubemap(self):
        return self._ptr[0].maxTextureCubemap
    @maxTextureCubemap.setter
    def maxTextureCubemap(self, int maxTextureCubemap):
        pass
        self._ptr[0].maxTextureCubemap = maxTextureCubemap
    @property
    def maxTexture1DLayered(self):
        return self._ptr[0].maxTexture1DLayered
    @maxTexture1DLayered.setter
    def maxTexture1DLayered(self, maxTexture1DLayered):
        pass
        self._ptr[0].maxTexture1DLayered = maxTexture1DLayered
    @property
    def maxTexture2DLayered(self):
        return self._ptr[0].maxTexture2DLayered
    @maxTexture2DLayered.setter
    def maxTexture2DLayered(self, maxTexture2DLayered):
        pass
        self._ptr[0].maxTexture2DLayered = maxTexture2DLayered
    @property
    def maxTextureCubemapLayered(self):
        return self._ptr[0].maxTextureCubemapLayered
    @maxTextureCubemapLayered.setter
    def maxTextureCubemapLayered(self, maxTextureCubemapLayered):
        pass
        self._ptr[0].maxTextureCubemapLayered = maxTextureCubemapLayered
    @property
    def maxSurface1D(self):
        return self._ptr[0].maxSurface1D
    @maxSurface1D.setter
    def maxSurface1D(self, int maxSurface1D):
        pass
        self._ptr[0].maxSurface1D = maxSurface1D
    @property
    def maxSurface2D(self):
        return self._ptr[0].maxSurface2D
    @maxSurface2D.setter
    def maxSurface2D(self, maxSurface2D):
        pass
        self._ptr[0].maxSurface2D = maxSurface2D
    @property
    def maxSurface3D(self):
        return self._ptr[0].maxSurface3D
    @maxSurface3D.setter
    def maxSurface3D(self, maxSurface3D):
        pass
        self._ptr[0].maxSurface3D = maxSurface3D
    @property
    def maxSurface1DLayered(self):
        return self._ptr[0].maxSurface1DLayered
    @maxSurface1DLayered.setter
    def maxSurface1DLayered(self, maxSurface1DLayered):
        pass
        self._ptr[0].maxSurface1DLayered = maxSurface1DLayered
    @property
    def maxSurface2DLayered(self):
        return self._ptr[0].maxSurface2DLayered
    @maxSurface2DLayered.setter
    def maxSurface2DLayered(self, maxSurface2DLayered):
        pass
        self._ptr[0].maxSurface2DLayered = maxSurface2DLayered
    @property
    def maxSurfaceCubemap(self):
        return self._ptr[0].maxSurfaceCubemap
    @maxSurfaceCubemap.setter
    def maxSurfaceCubemap(self, int maxSurfaceCubemap):
        pass
        self._ptr[0].maxSurfaceCubemap = maxSurfaceCubemap
    @property
    def maxSurfaceCubemapLayered(self):
        return self._ptr[0].maxSurfaceCubemapLayered
    @maxSurfaceCubemapLayered.setter
    def maxSurfaceCubemapLayered(self, maxSurfaceCubemapLayered):
        pass
        self._ptr[0].maxSurfaceCubemapLayered = maxSurfaceCubemapLayered
    @property
    def surfaceAlignment(self):
        return self._ptr[0].surfaceAlignment
    @surfaceAlignment.setter
    def surfaceAlignment(self, size_t surfaceAlignment):
        pass
        self._ptr[0].surfaceAlignment = surfaceAlignment
    @property
    def concurrentKernels(self):
        return self._ptr[0].concurrentKernels
    @concurrentKernels.setter
    def concurrentKernels(self, int concurrentKernels):
        pass
        self._ptr[0].concurrentKernels = concurrentKernels
    @property
    def ECCEnabled(self):
        return self._ptr[0].ECCEnabled
    @ECCEnabled.setter
    def ECCEnabled(self, int ECCEnabled):
        pass
        self._ptr[0].ECCEnabled = ECCEnabled
    @property
    def pciBusID(self):
        return self._ptr[0].pciBusID
    @pciBusID.setter
    def pciBusID(self, int pciBusID):
        pass
        self._ptr[0].pciBusID = pciBusID
    @property
    def pciDeviceID(self):
        return self._ptr[0].pciDeviceID
    @pciDeviceID.setter
    def pciDeviceID(self, int pciDeviceID):
        pass
        self._ptr[0].pciDeviceID = pciDeviceID
    @property
    def pciDomainID(self):
        return self._ptr[0].pciDomainID
    @pciDomainID.setter
    def pciDomainID(self, int pciDomainID):
        pass
        self._ptr[0].pciDomainID = pciDomainID
    @property
    def tccDriver(self):
        return self._ptr[0].tccDriver
    @tccDriver.setter
    def tccDriver(self, int tccDriver):
        pass
        self._ptr[0].tccDriver = tccDriver
    @property
    def asyncEngineCount(self):
        return self._ptr[0].asyncEngineCount
    @asyncEngineCount.setter
    def asyncEngineCount(self, int asyncEngineCount):
        pass
        self._ptr[0].asyncEngineCount = asyncEngineCount
    @property
    def unifiedAddressing(self):
        return self._ptr[0].unifiedAddressing
    @unifiedAddressing.setter
    def unifiedAddressing(self, int unifiedAddressing):
        pass
        self._ptr[0].unifiedAddressing = unifiedAddressing
    @property
    def memoryClockRate(self):
        return self._ptr[0].memoryClockRate
    @memoryClockRate.setter
    def memoryClockRate(self, int memoryClockRate):
        pass
        self._ptr[0].memoryClockRate = memoryClockRate
    @property
    def memoryBusWidth(self):
        return self._ptr[0].memoryBusWidth
    @memoryBusWidth.setter
    def memoryBusWidth(self, int memoryBusWidth):
        pass
        self._ptr[0].memoryBusWidth = memoryBusWidth
    @property
    def l2CacheSize(self):
        return self._ptr[0].l2CacheSize
    @l2CacheSize.setter
    def l2CacheSize(self, int l2CacheSize):
        pass
        self._ptr[0].l2CacheSize = l2CacheSize
    @property
    def persistingL2CacheMaxSize(self):
        return self._ptr[0].persistingL2CacheMaxSize
    @persistingL2CacheMaxSize.setter
    def persistingL2CacheMaxSize(self, int persistingL2CacheMaxSize):
        pass
        self._ptr[0].persistingL2CacheMaxSize = persistingL2CacheMaxSize
    @property
    def maxThreadsPerMultiProcessor(self):
        return self._ptr[0].maxThreadsPerMultiProcessor
    @maxThreadsPerMultiProcessor.setter
    def maxThreadsPerMultiProcessor(self, int maxThreadsPerMultiProcessor):
        pass
        self._ptr[0].maxThreadsPerMultiProcessor = maxThreadsPerMultiProcessor
    @property
    def streamPrioritiesSupported(self):
        return self._ptr[0].streamPrioritiesSupported
    @streamPrioritiesSupported.setter
    def streamPrioritiesSupported(self, int streamPrioritiesSupported):
        pass
        self._ptr[0].streamPrioritiesSupported = streamPrioritiesSupported
    @property
    def globalL1CacheSupported(self):
        return self._ptr[0].globalL1CacheSupported
    @globalL1CacheSupported.setter
    def globalL1CacheSupported(self, int globalL1CacheSupported):
        pass
        self._ptr[0].globalL1CacheSupported = globalL1CacheSupported
    @property
    def localL1CacheSupported(self):
        return self._ptr[0].localL1CacheSupported
    @localL1CacheSupported.setter
    def localL1CacheSupported(self, int localL1CacheSupported):
        pass
        self._ptr[0].localL1CacheSupported = localL1CacheSupported
    @property
    def sharedMemPerMultiprocessor(self):
        return self._ptr[0].sharedMemPerMultiprocessor
    @sharedMemPerMultiprocessor.setter
    def sharedMemPerMultiprocessor(self, size_t sharedMemPerMultiprocessor):
        pass
        self._ptr[0].sharedMemPerMultiprocessor = sharedMemPerMultiprocessor
    @property
    def regsPerMultiprocessor(self):
        return self._ptr[0].regsPerMultiprocessor
    @regsPerMultiprocessor.setter
    def regsPerMultiprocessor(self, int regsPerMultiprocessor):
        pass
        self._ptr[0].regsPerMultiprocessor = regsPerMultiprocessor
    @property
    def managedMemory(self):
        return self._ptr[0].managedMemory
    @managedMemory.setter
    def managedMemory(self, int managedMemory):
        pass
        self._ptr[0].managedMemory = managedMemory
    @property
    def isMultiGpuBoard(self):
        return self._ptr[0].isMultiGpuBoard
    @isMultiGpuBoard.setter
    def isMultiGpuBoard(self, int isMultiGpuBoard):
        pass
        self._ptr[0].isMultiGpuBoard = isMultiGpuBoard
    @property
    def multiGpuBoardGroupID(self):
        return self._ptr[0].multiGpuBoardGroupID
    @multiGpuBoardGroupID.setter
    def multiGpuBoardGroupID(self, int multiGpuBoardGroupID):
        pass
        self._ptr[0].multiGpuBoardGroupID = multiGpuBoardGroupID
    @property
    def hostNativeAtomicSupported(self):
        return self._ptr[0].hostNativeAtomicSupported
    @hostNativeAtomicSupported.setter
    def hostNativeAtomicSupported(self, int hostNativeAtomicSupported):
        pass
        self._ptr[0].hostNativeAtomicSupported = hostNativeAtomicSupported
    @property
    def singleToDoublePrecisionPerfRatio(self):
        return self._ptr[0].singleToDoublePrecisionPerfRatio
    @singleToDoublePrecisionPerfRatio.setter
    def singleToDoublePrecisionPerfRatio(self, int singleToDoublePrecisionPerfRatio):
        pass
        self._ptr[0].singleToDoublePrecisionPerfRatio = singleToDoublePrecisionPerfRatio
    @property
    def pageableMemoryAccess(self):
        return self._ptr[0].pageableMemoryAccess
    @pageableMemoryAccess.setter
    def pageableMemoryAccess(self, int pageableMemoryAccess):
        pass
        self._ptr[0].pageableMemoryAccess = pageableMemoryAccess
    @property
    def concurrentManagedAccess(self):
        return self._ptr[0].concurrentManagedAccess
    @concurrentManagedAccess.setter
    def concurrentManagedAccess(self, int concurrentManagedAccess):
        pass
        self._ptr[0].concurrentManagedAccess = concurrentManagedAccess
    @property
    def computePreemptionSupported(self):
        return self._ptr[0].computePreemptionSupported
    @computePreemptionSupported.setter
    def computePreemptionSupported(self, int computePreemptionSupported):
        pass
        self._ptr[0].computePreemptionSupported = computePreemptionSupported
    @property
    def canUseHostPointerForRegisteredMem(self):
        return self._ptr[0].canUseHostPointerForRegisteredMem
    @canUseHostPointerForRegisteredMem.setter
    def canUseHostPointerForRegisteredMem(self, int canUseHostPointerForRegisteredMem):
        pass
        self._ptr[0].canUseHostPointerForRegisteredMem = canUseHostPointerForRegisteredMem
    @property
    def cooperativeLaunch(self):
        return self._ptr[0].cooperativeLaunch
    @cooperativeLaunch.setter
    def cooperativeLaunch(self, int cooperativeLaunch):
        pass
        self._ptr[0].cooperativeLaunch = cooperativeLaunch
    @property
    def cooperativeMultiDeviceLaunch(self):
        return self._ptr[0].cooperativeMultiDeviceLaunch
    @cooperativeMultiDeviceLaunch.setter
    def cooperativeMultiDeviceLaunch(self, int cooperativeMultiDeviceLaunch):
        pass
        self._ptr[0].cooperativeMultiDeviceLaunch = cooperativeMultiDeviceLaunch
    @property
    def sharedMemPerBlockOptin(self):
        return self._ptr[0].sharedMemPerBlockOptin
    @sharedMemPerBlockOptin.setter
    def sharedMemPerBlockOptin(self, size_t sharedMemPerBlockOptin):
        pass
        self._ptr[0].sharedMemPerBlockOptin = sharedMemPerBlockOptin
    @property
    def pageableMemoryAccessUsesHostPageTables(self):
        return self._ptr[0].pageableMemoryAccessUsesHostPageTables
    @pageableMemoryAccessUsesHostPageTables.setter
    def pageableMemoryAccessUsesHostPageTables(self, int pageableMemoryAccessUsesHostPageTables):
        pass
        self._ptr[0].pageableMemoryAccessUsesHostPageTables = pageableMemoryAccessUsesHostPageTables
    @property
    def directManagedMemAccessFromHost(self):
        return self._ptr[0].directManagedMemAccessFromHost
    @directManagedMemAccessFromHost.setter
    def directManagedMemAccessFromHost(self, int directManagedMemAccessFromHost):
        pass
        self._ptr[0].directManagedMemAccessFromHost = directManagedMemAccessFromHost
    @property
    def maxBlocksPerMultiProcessor(self):
        return self._ptr[0].maxBlocksPerMultiProcessor
    @maxBlocksPerMultiProcessor.setter
    def maxBlocksPerMultiProcessor(self, int maxBlocksPerMultiProcessor):
        pass
        self._ptr[0].maxBlocksPerMultiProcessor = maxBlocksPerMultiProcessor
    @property
    def accessPolicyMaxWindowSize(self):
        return self._ptr[0].accessPolicyMaxWindowSize
    @accessPolicyMaxWindowSize.setter
    def accessPolicyMaxWindowSize(self, int accessPolicyMaxWindowSize):
        pass
        self._ptr[0].accessPolicyMaxWindowSize = accessPolicyMaxWindowSize
    @property
    def reservedSharedMemPerBlock(self):
        return self._ptr[0].reservedSharedMemPerBlock
    @reservedSharedMemPerBlock.setter
    def reservedSharedMemPerBlock(self, size_t reservedSharedMemPerBlock):
        pass
        self._ptr[0].reservedSharedMemPerBlock = reservedSharedMemPerBlock

cdef class cudaIpcEventHandle_st:
    """
    CUDA IPC event handle

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaIpcEventHandle_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        if CHAR_MIN == 0:
            for i, b in enumerate(reserved):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].reserved[i] = b
        else:
            for i, b in enumerate(reserved):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].reserved[i] = b

cdef class cudaIpcMemHandle_st:
    """
    CUDA IPC memory handle

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaIpcMemHandle_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        if CHAR_MIN == 0:
            for i, b in enumerate(reserved):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].reserved[i] = b
        else:
            for i, b in enumerate(reserved):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].reserved[i] = b

cdef class _cudaExternalMemoryHandleDesc_handle_handle_win32_s:
    """

    Attributes
    ----------
    handle : void

    name : void


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalMemoryHandleDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['handle : ' + hex(self.handle)]
            str_list += ['name : ' + hex(self.name)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def handle(self):
        return <void_ptr>self._ptr[0].handle.win32.handle
    @handle.setter
    def handle(self, handle):
        _chandle = utils.HelperInputVoidPtr(handle)
        self._ptr[0].handle.win32.handle = <void*><void_ptr>_chandle.cptr
    @property
    def name(self):
        return <void_ptr>self._ptr[0].handle.win32.name
    @name.setter
    def name(self, name):
        _cname = utils.HelperInputVoidPtr(name)
        self._ptr[0].handle.win32.name = <void*><void_ptr>_cname.cptr

cdef class _cudaExternalMemoryHandleDesc_handle_u:
    """

    Attributes
    ----------
    fd : int

    win32 : _cudaExternalMemoryHandleDesc_handle_handle_win32_s

    nvSciBufObject : void


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalMemoryHandleDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        self._win32 = _cudaExternalMemoryHandleDesc_handle_handle_win32_s(_ptr=<void_ptr>self._ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['fd : ' + str(self.fd)]
            str_list += ['win32 :\n' + '\n'.join(['    ' + line for line in str(self.win32).splitlines()])]
            str_list += ['nvSciBufObject : ' + hex(self.nvSciBufObject)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def fd(self):
        return self._ptr[0].handle.fd
    @fd.setter
    def fd(self, int fd):
        pass
        self._ptr[0].handle.fd = fd
    @property
    def win32(self):
        return self._win32
    @win32.setter
    def win32(self, win32 not None : _cudaExternalMemoryHandleDesc_handle_handle_win32_s):
        pass
        for _attr in dir(win32):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._win32, _attr, getattr(win32, _attr))
    @property
    def nvSciBufObject(self):
        return <void_ptr>self._ptr[0].handle.nvSciBufObject
    @nvSciBufObject.setter
    def nvSciBufObject(self, nvSciBufObject):
        _cnvSciBufObject = utils.HelperInputVoidPtr(nvSciBufObject)
        self._ptr[0].handle.nvSciBufObject = <void*><void_ptr>_cnvSciBufObject.cptr

cdef class cudaExternalMemoryHandleDesc:
    """
    External memory handle descriptor

    Attributes
    ----------
    type : cudaExternalMemoryHandleType
        Type of the handle
    handle : _cudaExternalMemoryHandleDesc_handle_u

    size : unsigned long long
        Size of the memory allocation
    flags : unsigned int
        Flags must either be zero or cudaExternalMemoryDedicated

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalMemoryHandleDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._handle = _cudaExternalMemoryHandleDesc_handle_u(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['type : ' + str(self.type)]
            str_list += ['handle :\n' + '\n'.join(['    ' + line for line in str(self.handle).splitlines()])]
            str_list += ['size : ' + str(self.size)]
            str_list += ['flags : ' + str(self.flags)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def type(self):
        return cudaExternalMemoryHandleType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : cudaExternalMemoryHandleType):
        pass
        self._ptr[0].type = type.value
    @property
    def handle(self):
        return self._handle
    @handle.setter
    def handle(self, handle not None : _cudaExternalMemoryHandleDesc_handle_u):
        pass
        for _attr in dir(handle):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._handle, _attr, getattr(handle, _attr))
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, unsigned long long size):
        pass
        self._ptr[0].size = size
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        pass
        self._ptr[0].flags = flags

cdef class cudaExternalMemoryBufferDesc:
    """
    External memory buffer descriptor

    Attributes
    ----------
    offset : unsigned long long
        Offset into the memory object where the buffer's base is
    size : unsigned long long
        Size of the buffer
    flags : unsigned int
        Flags reserved for future use. Must be zero.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalMemoryBufferDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['offset : ' + str(self.offset)]
            str_list += ['size : ' + str(self.size)]
            str_list += ['flags : ' + str(self.flags)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        pass
        self._ptr[0].offset = offset
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, unsigned long long size):
        pass
        self._ptr[0].size = size
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        pass
        self._ptr[0].flags = flags

cdef class cudaExternalMemoryMipmappedArrayDesc:
    """
    External memory mipmap descriptor

    Attributes
    ----------
    offset : unsigned long long
        Offset into the memory object where the base level of the mipmap
        chain is.
    formatDesc : cudaChannelFormatDesc
        Format of base level of the mipmap chain
    extent : cudaExtent
        Dimensions of base level of the mipmap chain
    flags : unsigned int
        Flags associated with CUDA mipmapped arrays. See
        cudaMallocMipmappedArray
    numLevels : unsigned int
        Total number of levels in the mipmap chain

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalMemoryMipmappedArrayDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._formatDesc = cudaChannelFormatDesc(_ptr=<void_ptr>&self._ptr[0].formatDesc)
        self._extent = cudaExtent(_ptr=<void_ptr>&self._ptr[0].extent)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['offset : ' + str(self.offset)]
            str_list += ['formatDesc :\n' + '\n'.join(['    ' + line for line in str(self.formatDesc).splitlines()])]
            str_list += ['extent :\n' + '\n'.join(['    ' + line for line in str(self.extent).splitlines()])]
            str_list += ['flags : ' + str(self.flags)]
            str_list += ['numLevels : ' + str(self.numLevels)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        pass
        self._ptr[0].offset = offset
    @property
    def formatDesc(self):
        return self._formatDesc
    @formatDesc.setter
    def formatDesc(self, formatDesc not None : cudaChannelFormatDesc):
        pass
        for _attr in dir(formatDesc):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._formatDesc, _attr, getattr(formatDesc, _attr))
    @property
    def extent(self):
        return self._extent
    @extent.setter
    def extent(self, extent not None : cudaExtent):
        pass
        for _attr in dir(extent):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._extent, _attr, getattr(extent, _attr))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        pass
        self._ptr[0].flags = flags
    @property
    def numLevels(self):
        return self._ptr[0].numLevels
    @numLevels.setter
    def numLevels(self, unsigned int numLevels):
        pass
        self._ptr[0].numLevels = numLevels

cdef class _cudaExternalSemaphoreHandleDesc_handle_handle_win32_s:
    """

    Attributes
    ----------
    handle : void

    name : void


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreHandleDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['handle : ' + hex(self.handle)]
            str_list += ['name : ' + hex(self.name)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def handle(self):
        return <void_ptr>self._ptr[0].handle.win32.handle
    @handle.setter
    def handle(self, handle):
        _chandle = utils.HelperInputVoidPtr(handle)
        self._ptr[0].handle.win32.handle = <void*><void_ptr>_chandle.cptr
    @property
    def name(self):
        return <void_ptr>self._ptr[0].handle.win32.name
    @name.setter
    def name(self, name):
        _cname = utils.HelperInputVoidPtr(name)
        self._ptr[0].handle.win32.name = <void*><void_ptr>_cname.cptr

cdef class _cudaExternalSemaphoreHandleDesc_handle_u:
    """

    Attributes
    ----------
    fd : int

    win32 : _cudaExternalSemaphoreHandleDesc_handle_handle_win32_s

    nvSciSyncObj : void


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreHandleDesc *>_ptr
    def __init__(self, void_ptr _ptr):
        self._win32 = _cudaExternalSemaphoreHandleDesc_handle_handle_win32_s(_ptr=<void_ptr>self._ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['fd : ' + str(self.fd)]
            str_list += ['win32 :\n' + '\n'.join(['    ' + line for line in str(self.win32).splitlines()])]
            str_list += ['nvSciSyncObj : ' + hex(self.nvSciSyncObj)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def fd(self):
        return self._ptr[0].handle.fd
    @fd.setter
    def fd(self, int fd):
        pass
        self._ptr[0].handle.fd = fd
    @property
    def win32(self):
        return self._win32
    @win32.setter
    def win32(self, win32 not None : _cudaExternalSemaphoreHandleDesc_handle_handle_win32_s):
        pass
        for _attr in dir(win32):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._win32, _attr, getattr(win32, _attr))
    @property
    def nvSciSyncObj(self):
        return <void_ptr>self._ptr[0].handle.nvSciSyncObj
    @nvSciSyncObj.setter
    def nvSciSyncObj(self, nvSciSyncObj):
        _cnvSciSyncObj = utils.HelperInputVoidPtr(nvSciSyncObj)
        self._ptr[0].handle.nvSciSyncObj = <void*><void_ptr>_cnvSciSyncObj.cptr

cdef class cudaExternalSemaphoreHandleDesc:
    """
    External semaphore handle descriptor

    Attributes
    ----------
    type : cudaExternalSemaphoreHandleType
        Type of the handle
    handle : _cudaExternalSemaphoreHandleDesc_handle_u

    flags : unsigned int
        Flags reserved for the future. Must be zero.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalSemaphoreHandleDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._handle = _cudaExternalSemaphoreHandleDesc_handle_u(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['type : ' + str(self.type)]
            str_list += ['handle :\n' + '\n'.join(['    ' + line for line in str(self.handle).splitlines()])]
            str_list += ['flags : ' + str(self.flags)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def type(self):
        return cudaExternalSemaphoreHandleType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : cudaExternalSemaphoreHandleType):
        pass
        self._ptr[0].type = type.value
    @property
    def handle(self):
        return self._handle
    @handle.setter
    def handle(self, handle not None : _cudaExternalSemaphoreHandleDesc_handle_u):
        pass
        for _attr in dir(handle):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._handle, _attr, getattr(handle, _attr))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        pass
        self._ptr[0].flags = flags

cdef class _cudaExternalSemaphoreSignalParams_params_params_fence_s:
    """

    Attributes
    ----------
    value : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreSignalParams *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['value : ' + str(self.value)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def value(self):
        return self._ptr[0].params.fence.value
    @value.setter
    def value(self, unsigned long long value):
        pass
        self._ptr[0].params.fence.value = value

cdef class _cudaExternalSemaphoreSignalParams_params_params_nvSciSync_u:
    """

    Attributes
    ----------
    fence : void

    reserved : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreSignalParams *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['fence : ' + hex(self.fence)]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def fence(self):
        return <void_ptr>self._ptr[0].params.nvSciSync.fence
    @fence.setter
    def fence(self, fence):
        _cfence = utils.HelperInputVoidPtr(fence)
        self._ptr[0].params.nvSciSync.fence = <void*><void_ptr>_cfence.cptr
    @property
    def reserved(self):
        return self._ptr[0].params.nvSciSync.reserved
    @reserved.setter
    def reserved(self, unsigned long long reserved):
        pass
        self._ptr[0].params.nvSciSync.reserved = reserved

cdef class _cudaExternalSemaphoreSignalParams_params_params_keyedMutex_s:
    """

    Attributes
    ----------
    key : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreSignalParams *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['key : ' + str(self.key)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def key(self):
        return self._ptr[0].params.keyedMutex.key
    @key.setter
    def key(self, unsigned long long key):
        pass
        self._ptr[0].params.keyedMutex.key = key

cdef class _cudaExternalSemaphoreSignalParams_params_s:
    """

    Attributes
    ----------
    fence : _cudaExternalSemaphoreSignalParams_params_params_fence_s

    nvSciSync : _cudaExternalSemaphoreSignalParams_params_params_nvSciSync_u

    keyedMutex : _cudaExternalSemaphoreSignalParams_params_params_keyedMutex_s

    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreSignalParams *>_ptr
    def __init__(self, void_ptr _ptr):
        self._fence = _cudaExternalSemaphoreSignalParams_params_params_fence_s(_ptr=<void_ptr>self._ptr)
        self._nvSciSync = _cudaExternalSemaphoreSignalParams_params_params_nvSciSync_u(_ptr=<void_ptr>self._ptr)
        self._keyedMutex = _cudaExternalSemaphoreSignalParams_params_params_keyedMutex_s(_ptr=<void_ptr>self._ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['fence :\n' + '\n'.join(['    ' + line for line in str(self.fence).splitlines()])]
            str_list += ['nvSciSync :\n' + '\n'.join(['    ' + line for line in str(self.nvSciSync).splitlines()])]
            str_list += ['keyedMutex :\n' + '\n'.join(['    ' + line for line in str(self.keyedMutex).splitlines()])]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def fence(self):
        return self._fence
    @fence.setter
    def fence(self, fence not None : _cudaExternalSemaphoreSignalParams_params_params_fence_s):
        pass
        for _attr in dir(fence):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._fence, _attr, getattr(fence, _attr))
    @property
    def nvSciSync(self):
        return self._nvSciSync
    @nvSciSync.setter
    def nvSciSync(self, nvSciSync not None : _cudaExternalSemaphoreSignalParams_params_params_nvSciSync_u):
        pass
        for _attr in dir(nvSciSync):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._nvSciSync, _attr, getattr(nvSciSync, _attr))
    @property
    def keyedMutex(self):
        return self._keyedMutex
    @keyedMutex.setter
    def keyedMutex(self, keyedMutex not None : _cudaExternalSemaphoreSignalParams_params_params_keyedMutex_s):
        pass
        for _attr in dir(keyedMutex):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._keyedMutex, _attr, getattr(keyedMutex, _attr))
    @property
    def reserved(self):
        return self._ptr[0].params.reserved
    @reserved.setter
    def reserved(self, reserved):
        pass
        self._ptr[0].params.reserved = reserved

cdef class cudaExternalSemaphoreSignalParams:
    """
    External semaphore signal parameters, compatible with driver type

    Attributes
    ----------
    params : _cudaExternalSemaphoreSignalParams_params_s

    flags : unsigned int
        Only when cudaExternalSemaphoreSignalParams is used to signal a
        cudaExternalSemaphore_t of type
        cudaExternalSemaphoreHandleTypeNvSciSync, the valid flag is
        cudaExternalSemaphoreSignalSkipNvSciBufMemSync: which indicates
        that while signaling the cudaExternalSemaphore_t, no memory
        synchronization operations should be performed for any external
        memory object imported as cudaExternalMemoryHandleTypeNvSciBuf. For
        all other types of cudaExternalSemaphore_t, flags must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalSemaphoreSignalParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._params = _cudaExternalSemaphoreSignalParams_params_s(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['params :\n' + '\n'.join(['    ' + line for line in str(self.params).splitlines()])]
            str_list += ['flags : ' + str(self.flags)]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def params(self):
        return self._params
    @params.setter
    def params(self, params not None : _cudaExternalSemaphoreSignalParams_params_s):
        pass
        for _attr in dir(params):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._params, _attr, getattr(params, _attr))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        pass
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        pass
        self._ptr[0].reserved = reserved

cdef class _cudaExternalSemaphoreWaitParams_params_params_fence_s:
    """

    Attributes
    ----------
    value : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreWaitParams *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['value : ' + str(self.value)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def value(self):
        return self._ptr[0].params.fence.value
    @value.setter
    def value(self, unsigned long long value):
        pass
        self._ptr[0].params.fence.value = value

cdef class _cudaExternalSemaphoreWaitParams_params_params_nvSciSync_u:
    """

    Attributes
    ----------
    fence : void

    reserved : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreWaitParams *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['fence : ' + hex(self.fence)]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def fence(self):
        return <void_ptr>self._ptr[0].params.nvSciSync.fence
    @fence.setter
    def fence(self, fence):
        _cfence = utils.HelperInputVoidPtr(fence)
        self._ptr[0].params.nvSciSync.fence = <void*><void_ptr>_cfence.cptr
    @property
    def reserved(self):
        return self._ptr[0].params.nvSciSync.reserved
    @reserved.setter
    def reserved(self, unsigned long long reserved):
        pass
        self._ptr[0].params.nvSciSync.reserved = reserved

cdef class _cudaExternalSemaphoreWaitParams_params_params_keyedMutex_s:
    """

    Attributes
    ----------
    key : unsigned long long

    timeoutMs : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreWaitParams *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['key : ' + str(self.key)]
            str_list += ['timeoutMs : ' + str(self.timeoutMs)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def key(self):
        return self._ptr[0].params.keyedMutex.key
    @key.setter
    def key(self, unsigned long long key):
        pass
        self._ptr[0].params.keyedMutex.key = key
    @property
    def timeoutMs(self):
        return self._ptr[0].params.keyedMutex.timeoutMs
    @timeoutMs.setter
    def timeoutMs(self, unsigned int timeoutMs):
        pass
        self._ptr[0].params.keyedMutex.timeoutMs = timeoutMs

cdef class _cudaExternalSemaphoreWaitParams_params_s:
    """

    Attributes
    ----------
    fence : _cudaExternalSemaphoreWaitParams_params_params_fence_s

    nvSciSync : _cudaExternalSemaphoreWaitParams_params_params_nvSciSync_u

    keyedMutex : _cudaExternalSemaphoreWaitParams_params_params_keyedMutex_s

    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaExternalSemaphoreWaitParams *>_ptr
    def __init__(self, void_ptr _ptr):
        self._fence = _cudaExternalSemaphoreWaitParams_params_params_fence_s(_ptr=<void_ptr>self._ptr)
        self._nvSciSync = _cudaExternalSemaphoreWaitParams_params_params_nvSciSync_u(_ptr=<void_ptr>self._ptr)
        self._keyedMutex = _cudaExternalSemaphoreWaitParams_params_params_keyedMutex_s(_ptr=<void_ptr>self._ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['fence :\n' + '\n'.join(['    ' + line for line in str(self.fence).splitlines()])]
            str_list += ['nvSciSync :\n' + '\n'.join(['    ' + line for line in str(self.nvSciSync).splitlines()])]
            str_list += ['keyedMutex :\n' + '\n'.join(['    ' + line for line in str(self.keyedMutex).splitlines()])]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def fence(self):
        return self._fence
    @fence.setter
    def fence(self, fence not None : _cudaExternalSemaphoreWaitParams_params_params_fence_s):
        pass
        for _attr in dir(fence):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._fence, _attr, getattr(fence, _attr))
    @property
    def nvSciSync(self):
        return self._nvSciSync
    @nvSciSync.setter
    def nvSciSync(self, nvSciSync not None : _cudaExternalSemaphoreWaitParams_params_params_nvSciSync_u):
        pass
        for _attr in dir(nvSciSync):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._nvSciSync, _attr, getattr(nvSciSync, _attr))
    @property
    def keyedMutex(self):
        return self._keyedMutex
    @keyedMutex.setter
    def keyedMutex(self, keyedMutex not None : _cudaExternalSemaphoreWaitParams_params_params_keyedMutex_s):
        pass
        for _attr in dir(keyedMutex):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._keyedMutex, _attr, getattr(keyedMutex, _attr))
    @property
    def reserved(self):
        return self._ptr[0].params.reserved
    @reserved.setter
    def reserved(self, reserved):
        pass
        self._ptr[0].params.reserved = reserved

cdef class cudaExternalSemaphoreWaitParams:
    """
    External semaphore wait parameters, compatible with driver type

    Attributes
    ----------
    params : _cudaExternalSemaphoreWaitParams_params_s

    flags : unsigned int
        Only when cudaExternalSemaphoreSignalParams is used to signal a
        cudaExternalSemaphore_t of type
        cudaExternalSemaphoreHandleTypeNvSciSync, the valid flag is
        cudaExternalSemaphoreSignalSkipNvSciBufMemSync: which indicates
        that while waiting for the cudaExternalSemaphore_t, no memory
        synchronization operations should be performed for any external
        memory object imported as cudaExternalMemoryHandleTypeNvSciBuf. For
        all other types of cudaExternalSemaphore_t, flags must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalSemaphoreWaitParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._params = _cudaExternalSemaphoreWaitParams_params_s(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['params :\n' + '\n'.join(['    ' + line for line in str(self.params).splitlines()])]
            str_list += ['flags : ' + str(self.flags)]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def params(self):
        return self._params
    @params.setter
    def params(self, params not None : _cudaExternalSemaphoreWaitParams_params_s):
        pass
        for _attr in dir(params):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._params, _attr, getattr(params, _attr))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        pass
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        pass
        self._ptr[0].reserved = reserved

cdef class cudaKernelNodeParams:
    """
    CUDA GPU kernel node parameters

    Attributes
    ----------
    func : Any
        Kernel to launch
    gridDim : dim3
        Grid dimensions
    blockDim : dim3
        Block dimensions
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    kernelParams : Any
        Array of pointers to individual kernel arguments
    extra : Any
        Pointer to kernel arguments in the "extra" format

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaKernelNodeParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._gridDim = dim3(_ptr=<void_ptr>&self._ptr[0].gridDim)
        self._blockDim = dim3(_ptr=<void_ptr>&self._ptr[0].blockDim)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['func : ' + hex(self.func)]
            str_list += ['gridDim :\n' + '\n'.join(['    ' + line for line in str(self.gridDim).splitlines()])]
            str_list += ['blockDim :\n' + '\n'.join(['    ' + line for line in str(self.blockDim).splitlines()])]
            str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            str_list += ['kernelParams : ' + str(self.kernelParams)]
            str_list += ['extra : ' + str(self.extra)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def func(self):
        return <void_ptr>self._ptr[0].func
    @func.setter
    def func(self, func):
        _cfunc = utils.HelperInputVoidPtr(func)
        self._ptr[0].func = <void*><void_ptr>_cfunc.cptr
    @property
    def gridDim(self):
        return self._gridDim
    @gridDim.setter
    def gridDim(self, gridDim not None : dim3):
        pass
        for _attr in dir(gridDim):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._gridDim, _attr, getattr(gridDim, _attr))
    @property
    def blockDim(self):
        return self._blockDim
    @blockDim.setter
    def blockDim(self, blockDim not None : dim3):
        pass
        for _attr in dir(blockDim):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._blockDim, _attr, getattr(blockDim, _attr))
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        pass
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def kernelParams(self):
        return <void_ptr>self._ptr[0].kernelParams
    @kernelParams.setter
    def kernelParams(self, kernelParams):
        self._ckernelParams = utils.HelperKernelParams(kernelParams)
        self._ptr[0].kernelParams = <void**><void_ptr>self._ckernelParams.ckernelParams
    @property
    def extra(self):
        return <void_ptr>self._ptr[0].extra
    @extra.setter
    def extra(self, void_ptr extra):
        self._ptr[0].extra = <void**>extra

cdef class cudaExternalSemaphoreSignalNodeParams:
    """
    External semaphore signal node parameters

    Attributes
    ----------
    extSemArray : cudaExternalSemaphore_t
        Array of external semaphore handles.
    paramsArray : cudaExternalSemaphoreSignalParams
        Array of external semaphore signal parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalSemaphoreSignalNodeParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['extSemArray : ' + str(self.extSemArray)]
            str_list += ['paramsArray : ' + str(self.paramsArray)]
            str_list += ['numExtSems : ' + str(self.numExtSems)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(ccudart.cudaExternalSemaphore_t) for x in range(self._extSemArray_length)]
        return [cudaExternalSemaphore_t(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <ccudart.cudaExternalSemaphore_t*> calloc(len(val), sizeof(ccudart.cudaExternalSemaphore_t))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphore_t)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<cudaExternalSemaphore_t>val[idx])._ptr[0]
    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(ccudart.cudaExternalSemaphoreSignalParams) for x in range(self._paramsArray_length)]
        return [cudaExternalSemaphoreSignalParams(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <ccudart.cudaExternalSemaphoreSignalParams*> calloc(len(val), sizeof(ccudart.cudaExternalSemaphoreSignalParams))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphoreSignalParams)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                memcpy(&self._paramsArray[idx], (<cudaExternalSemaphoreSignalParams>val[idx])._ptr, sizeof(ccudart.cudaExternalSemaphoreSignalParams))
    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        pass
        self._ptr[0].numExtSems = numExtSems

cdef class cudaExternalSemaphoreWaitNodeParams:
    """
    External semaphore wait node parameters

    Attributes
    ----------
    extSemArray : cudaExternalSemaphore_t
        Array of external semaphore handles.
    paramsArray : cudaExternalSemaphoreWaitParams
        Array of external semaphore wait parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaExternalSemaphoreWaitNodeParams *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['extSemArray : ' + str(self.extSemArray)]
            str_list += ['paramsArray : ' + str(self.paramsArray)]
            str_list += ['numExtSems : ' + str(self.numExtSems)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(ccudart.cudaExternalSemaphore_t) for x in range(self._extSemArray_length)]
        return [cudaExternalSemaphore_t(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <ccudart.cudaExternalSemaphore_t*> calloc(len(val), sizeof(ccudart.cudaExternalSemaphore_t))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphore_t)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<cudaExternalSemaphore_t>val[idx])._ptr[0]
    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(ccudart.cudaExternalSemaphoreWaitParams) for x in range(self._paramsArray_length)]
        return [cudaExternalSemaphoreWaitParams(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <ccudart.cudaExternalSemaphoreWaitParams*> calloc(len(val), sizeof(ccudart.cudaExternalSemaphoreWaitParams))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphoreWaitParams)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                memcpy(&self._paramsArray[idx], (<cudaExternalSemaphoreWaitParams>val[idx])._ptr, sizeof(ccudart.cudaExternalSemaphoreWaitParams))
    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        pass
        self._ptr[0].numExtSems = numExtSems

cdef class cudaStreamAttrValue:
    """
    Stream attributes union used with
    cudaStreamSetAttribute/cudaStreamGetAttribute

    Attributes
    ----------
    accessPolicyWindow : cudaAccessPolicyWindow

    syncPolicy : cudaSynchronizationPolicy


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaStreamAttrValue *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._accessPolicyWindow = cudaAccessPolicyWindow(_ptr=<void_ptr>&self._ptr[0].accessPolicyWindow)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['accessPolicyWindow :\n' + '\n'.join(['    ' + line for line in str(self.accessPolicyWindow).splitlines()])]
            str_list += ['syncPolicy : ' + str(self.syncPolicy)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def accessPolicyWindow(self):
        return self._accessPolicyWindow
    @accessPolicyWindow.setter
    def accessPolicyWindow(self, accessPolicyWindow not None : cudaAccessPolicyWindow):
        pass
        for _attr in dir(accessPolicyWindow):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._accessPolicyWindow, _attr, getattr(accessPolicyWindow, _attr))
    @property
    def syncPolicy(self):
        return cudaSynchronizationPolicy(self._ptr[0].syncPolicy)
    @syncPolicy.setter
    def syncPolicy(self, syncPolicy not None : cudaSynchronizationPolicy):
        pass
        self._ptr[0].syncPolicy = syncPolicy.value

cdef class cudaKernelNodeAttrValue:
    """
    Graph kernel node attributes union, used with
    cudaGraphKernelNodeSetAttribute/cudaGraphKernelNodeGetAttribute

    Attributes
    ----------
    accessPolicyWindow : cudaAccessPolicyWindow
        Attribute ::CUaccessPolicyWindow.
    cooperative : int

    priority : int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaKernelNodeAttrValue *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._accessPolicyWindow = cudaAccessPolicyWindow(_ptr=<void_ptr>&self._ptr[0].accessPolicyWindow)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['accessPolicyWindow :\n' + '\n'.join(['    ' + line for line in str(self.accessPolicyWindow).splitlines()])]
            str_list += ['cooperative : ' + str(self.cooperative)]
            str_list += ['priority : ' + str(self.priority)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def accessPolicyWindow(self):
        return self._accessPolicyWindow
    @accessPolicyWindow.setter
    def accessPolicyWindow(self, accessPolicyWindow not None : cudaAccessPolicyWindow):
        pass
        for _attr in dir(accessPolicyWindow):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._accessPolicyWindow, _attr, getattr(accessPolicyWindow, _attr))
    @property
    def cooperative(self):
        return self._ptr[0].cooperative
    @cooperative.setter
    def cooperative(self, int cooperative):
        pass
        self._ptr[0].cooperative = cooperative
    @property
    def priority(self):
        return self._ptr[0].priority
    @priority.setter
    def priority(self, int priority):
        pass
        self._ptr[0].priority = priority

cdef class cudaTextureDesc:
    """
    CUDA texture descriptor

    Attributes
    ----------
    addressMode : List[cudaTextureAddressMode]
        Texture address mode for up to 3 dimensions
    filterMode : cudaTextureFilterMode
        Texture filter mode
    readMode : cudaTextureReadMode
        Texture read mode
    sRGB : int
        Perform sRGB->linear conversion during texture read
    borderColor : List[float]
        Texture Border Color
    normalizedCoords : int
        Indicates whether texture reads are normalized or not
    maxAnisotropy : unsigned int
        Limit to the anisotropy ratio
    mipmapFilterMode : cudaTextureFilterMode
        Mipmap filter mode
    mipmapLevelBias : float
        Offset applied to the supplied mipmap level
    minMipmapLevelClamp : float
        Lower end of the mipmap level range to clamp access to
    maxMipmapLevelClamp : float
        Upper end of the mipmap level range to clamp access to
    disableTrilinearOptimization : int
        Disable any trilinear filtering optimizations.
    seamlessCubemap : int
        Enable seamless cube map filtering.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaTextureDesc *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['addressMode : ' + str(self.addressMode)]
            str_list += ['filterMode : ' + str(self.filterMode)]
            str_list += ['readMode : ' + str(self.readMode)]
            str_list += ['sRGB : ' + str(self.sRGB)]
            str_list += ['borderColor : ' + str(self.borderColor)]
            str_list += ['normalizedCoords : ' + str(self.normalizedCoords)]
            str_list += ['maxAnisotropy : ' + str(self.maxAnisotropy)]
            str_list += ['mipmapFilterMode : ' + str(self.mipmapFilterMode)]
            str_list += ['mipmapLevelBias : ' + str(self.mipmapLevelBias)]
            str_list += ['minMipmapLevelClamp : ' + str(self.minMipmapLevelClamp)]
            str_list += ['maxMipmapLevelClamp : ' + str(self.maxMipmapLevelClamp)]
            str_list += ['disableTrilinearOptimization : ' + str(self.disableTrilinearOptimization)]
            str_list += ['seamlessCubemap : ' + str(self.seamlessCubemap)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def addressMode(self):
        return [cudaTextureAddressMode(_x) for _x in list(self._ptr[0].addressMode)]
    @addressMode.setter
    def addressMode(self, addressMode):
        self._ptr[0].addressMode = [_x.value for _x in addressMode]
    @property
    def filterMode(self):
        return cudaTextureFilterMode(self._ptr[0].filterMode)
    @filterMode.setter
    def filterMode(self, filterMode not None : cudaTextureFilterMode):
        pass
        self._ptr[0].filterMode = filterMode.value
    @property
    def readMode(self):
        return cudaTextureReadMode(self._ptr[0].readMode)
    @readMode.setter
    def readMode(self, readMode not None : cudaTextureReadMode):
        pass
        self._ptr[0].readMode = readMode.value
    @property
    def sRGB(self):
        return self._ptr[0].sRGB
    @sRGB.setter
    def sRGB(self, int sRGB):
        pass
        self._ptr[0].sRGB = sRGB
    @property
    def borderColor(self):
        return self._ptr[0].borderColor
    @borderColor.setter
    def borderColor(self, borderColor):
        pass
        self._ptr[0].borderColor = borderColor
    @property
    def normalizedCoords(self):
        return self._ptr[0].normalizedCoords
    @normalizedCoords.setter
    def normalizedCoords(self, int normalizedCoords):
        pass
        self._ptr[0].normalizedCoords = normalizedCoords
    @property
    def maxAnisotropy(self):
        return self._ptr[0].maxAnisotropy
    @maxAnisotropy.setter
    def maxAnisotropy(self, unsigned int maxAnisotropy):
        pass
        self._ptr[0].maxAnisotropy = maxAnisotropy
    @property
    def mipmapFilterMode(self):
        return cudaTextureFilterMode(self._ptr[0].mipmapFilterMode)
    @mipmapFilterMode.setter
    def mipmapFilterMode(self, mipmapFilterMode not None : cudaTextureFilterMode):
        pass
        self._ptr[0].mipmapFilterMode = mipmapFilterMode.value
    @property
    def mipmapLevelBias(self):
        return self._ptr[0].mipmapLevelBias
    @mipmapLevelBias.setter
    def mipmapLevelBias(self, float mipmapLevelBias):
        pass
        self._ptr[0].mipmapLevelBias = mipmapLevelBias
    @property
    def minMipmapLevelClamp(self):
        return self._ptr[0].minMipmapLevelClamp
    @minMipmapLevelClamp.setter
    def minMipmapLevelClamp(self, float minMipmapLevelClamp):
        pass
        self._ptr[0].minMipmapLevelClamp = minMipmapLevelClamp
    @property
    def maxMipmapLevelClamp(self):
        return self._ptr[0].maxMipmapLevelClamp
    @maxMipmapLevelClamp.setter
    def maxMipmapLevelClamp(self, float maxMipmapLevelClamp):
        pass
        self._ptr[0].maxMipmapLevelClamp = maxMipmapLevelClamp
    @property
    def disableTrilinearOptimization(self):
        return self._ptr[0].disableTrilinearOptimization
    @disableTrilinearOptimization.setter
    def disableTrilinearOptimization(self, int disableTrilinearOptimization):
        pass
        self._ptr[0].disableTrilinearOptimization = disableTrilinearOptimization
    @property
    def seamlessCubemap(self):
        return self._ptr[0].seamlessCubemap
    @seamlessCubemap.setter
    def seamlessCubemap(self, int seamlessCubemap):
        pass
        self._ptr[0].seamlessCubemap = seamlessCubemap

cdef class cudaEglPlaneDesc_st:
    """
    CUDA EGL Plane Descriptor - structure defining each plane of a CUDA
    EGLFrame

    Attributes
    ----------
    width : unsigned int
        Width of plane
    height : unsigned int
        Height of plane
    depth : unsigned int
        Depth of plane
    pitch : unsigned int
        Pitch of plane
    numChannels : unsigned int
        Number of channels for the plane
    channelDesc : cudaChannelFormatDesc
        Channel Format Descriptor
    reserved : List[unsigned int]
        Reserved for future use

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaEglPlaneDesc_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._channelDesc = cudaChannelFormatDesc(_ptr=<void_ptr>&self._ptr[0].channelDesc)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['width : ' + str(self.width)]
            str_list += ['height : ' + str(self.height)]
            str_list += ['depth : ' + str(self.depth)]
            str_list += ['pitch : ' + str(self.pitch)]
            str_list += ['numChannels : ' + str(self.numChannels)]
            str_list += ['channelDesc :\n' + '\n'.join(['    ' + line for line in str(self.channelDesc).splitlines()])]
            str_list += ['reserved : ' + str(self.reserved)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, unsigned int width):
        pass
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, unsigned int height):
        pass
        self._ptr[0].height = height
    @property
    def depth(self):
        return self._ptr[0].depth
    @depth.setter
    def depth(self, unsigned int depth):
        pass
        self._ptr[0].depth = depth
    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, unsigned int pitch):
        pass
        self._ptr[0].pitch = pitch
    @property
    def numChannels(self):
        return self._ptr[0].numChannels
    @numChannels.setter
    def numChannels(self, unsigned int numChannels):
        pass
        self._ptr[0].numChannels = numChannels
    @property
    def channelDesc(self):
        return self._channelDesc
    @channelDesc.setter
    def channelDesc(self, channelDesc not None : cudaChannelFormatDesc):
        pass
        for _attr in dir(channelDesc):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._channelDesc, _attr, getattr(channelDesc, _attr))
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        pass
        self._ptr[0].reserved = reserved

cdef class _cudaEglFrame_cudaEglFrame_cudaEglFrame_st_frame_u:
    """

    Attributes
    ----------
    pArray : List[cudaArray_t]

    pPitch : List[cudaPitchedPtr]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <ccudart.cudaEglFrame_st *>_ptr
    def __init__(self, void_ptr _ptr):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['pArray : ' + str(self.pArray)]
            str_list += ['pPitch :\n' + '\n'.join(['    ' + line for line in str(self.pPitch).splitlines()])]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def pArray(self):
        return [cudaArray_t(init_value=<void_ptr>_pArray) for _pArray in self._ptr[0].frame.pArray]
    @pArray.setter
    def pArray(self, pArray : List[cudaArray_t]):
        if len(pArray) != 3:
            raise IndexError('not enough values found during array assignment, expected 3, got', len(pArray))
        pArray = [int(_pArray) for _pArray in pArray]
        for _idx, _pArray in enumerate(pArray):
            self._ptr[0].frame.pArray[_idx] = <ccudart.cudaArray_t><void_ptr>_pArray
    @property
    def pPitch(self):
        out_pPitch = [cudaPitchedPtr() for _pPitch in self._ptr[0].frame.pPitch]
        for _idx in range(len(out_pPitch)):
            memcpy(<ccudart.cudaPitchedPtr*><void_ptr>out_pPitch[_idx].getPtr(), &self._ptr[0].frame.pPitch[_idx], sizeof(ccudart.cudaPitchedPtr))
        return out_pPitch
    @pPitch.setter
    def pPitch(self, pPitch : List[cudaPitchedPtr]):
        if len(pPitch) != 3:
            raise IndexError('not enough values found during array assignment, expected 3, got', len(pPitch))
        for _idx in range(len(pPitch)):
            memcpy(&self._ptr[0].frame.pPitch[_idx], <ccudart.cudaPitchedPtr*><void_ptr>pPitch[_idx].getPtr(), sizeof(ccudart.cudaPitchedPtr))

cdef class cudaEglFrame_st:
    """
    CUDA EGLFrame Descriptor - structure defining one frame of EGL.
    Each frame may contain one or more planes depending on whether the
    surface is Multiplanar or not. Each plane of EGLFrame is
    represented by cudaEglPlaneDesc which is defined as:
    typedefstructcudaEglPlaneDesc_st unsignedintwidth;
    unsignedintheight; unsignedintdepth; unsignedintpitch;
    unsignedintnumChannels; structcudaChannelFormatDescchannelDesc;
    unsignedintreserved[4]; cudaEglPlaneDesc;

    Attributes
    ----------
    frame : _cudaEglFrame_cudaEglFrame_cudaEglFrame_st_frame_u

    planeDesc : List[cudaEglPlaneDesc]
        CUDA EGL Plane Descriptor cudaEglPlaneDesc
    planeCount : unsigned int
        Number of planes
    frameType : cudaEglFrameType
        Array or Pitch
    eglColorFormat : cudaEglColorFormat
        CUDA EGL Color Format

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaEglFrame_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._frame = _cudaEglFrame_cudaEglFrame_cudaEglFrame_st_frame_u(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            str_list += ['frame :\n' + '\n'.join(['    ' + line for line in str(self.frame).splitlines()])]
            str_list += ['planeDesc :\n' + '\n'.join(['    ' + line for line in str(self.planeDesc).splitlines()])]
            str_list += ['planeCount : ' + str(self.planeCount)]
            str_list += ['frameType : ' + str(self.frameType)]
            str_list += ['eglColorFormat : ' + str(self.eglColorFormat)]
            return '\n'.join(str_list)
        else:
            return ''

    @property
    def frame(self):
        return self._frame
    @frame.setter
    def frame(self, frame not None : _cudaEglFrame_cudaEglFrame_cudaEglFrame_st_frame_u):
        pass
        for _attr in dir(frame):
            if _attr == 'getPtr':
                continue
            if not _attr.startswith('_'):
                setattr(self._frame, _attr, getattr(frame, _attr))
    @property
    def planeDesc(self):
        out_planeDesc = [cudaEglPlaneDesc() for _planeDesc in self._ptr[0].planeDesc]
        for _idx in range(len(out_planeDesc)):
            memcpy(<ccudart.cudaEglPlaneDesc*><void_ptr>out_planeDesc[_idx].getPtr(), &self._ptr[0].planeDesc[_idx], sizeof(ccudart.cudaEglPlaneDesc))
        return out_planeDesc
    @planeDesc.setter
    def planeDesc(self, planeDesc : List[cudaEglPlaneDesc]):
        if len(planeDesc) != 3:
            raise IndexError('not enough values found during array assignment, expected 3, got', len(planeDesc))
        for _idx in range(len(planeDesc)):
            memcpy(&self._ptr[0].planeDesc[_idx], <ccudart.cudaEglPlaneDesc*><void_ptr>planeDesc[_idx].getPtr(), sizeof(ccudart.cudaEglPlaneDesc))
    @property
    def planeCount(self):
        return self._ptr[0].planeCount
    @planeCount.setter
    def planeCount(self, unsigned int planeCount):
        pass
        self._ptr[0].planeCount = planeCount
    @property
    def frameType(self):
        return cudaEglFrameType(self._ptr[0].frameType)
    @frameType.setter
    def frameType(self, frameType not None : cudaEglFrameType):
        pass
        self._ptr[0].frameType = frameType.value
    @property
    def eglColorFormat(self):
        return cudaEglColorFormat(self._ptr[0].eglColorFormat)
    @eglColorFormat.setter
    def eglColorFormat(self, eglColorFormat not None : cudaEglColorFormat):
        pass
        self._ptr[0].eglColorFormat = eglColorFormat.value

cdef class GLenum:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.GLenum *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<GLenum ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class GLuint:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.GLuint *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<GLuint ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLint:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.EGLint *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<EGLint ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpDevice:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.VdpDevice *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpDevice ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpGetProcAddress:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.VdpGetProcAddress *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpGetProcAddress ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpVideoSurface:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.VdpVideoSurface *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpVideoSurface ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpOutputSurface:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.VdpOutputSurface *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpOutputSurface ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaSurfaceObject_t:
    """

    An opaque value that represents a CUDA Surface object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaSurfaceObject_t *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<cudaSurfaceObject_t ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cudaTextureObject_t:
    """

    An opaque value that represents a CUDA texture object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <ccudart.cudaTextureObject_t *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<cudaTextureObject_t ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr


@cython.embedsignature(True)
def cudaVDPAUGetDevice(vdpDevice, vdpGetProcAddress):
    """ Gets the CUDA device associated with a VdpDevice.

    Returns the CUDA device associated with a VdpDevice, if applicable.

    Parameters
    ----------
    vdpDevice : :py:obj:`~.VdpDevice`
        A VdpDevice handle
    vdpGetProcAddress : :py:obj:`~.VdpGetProcAddress`
        VDPAU's VdpGetProcAddress function pointer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`
    device : int
        Returns the device associated with vdpDevice, or -1 if the device
        associated with vdpDevice is not a compute device.

    See Also
    --------
    :py:obj:`~.cudaVDPAUSetVDPAUDevice`, :py:obj:`~.cuVDPAUGetDevice`
    """
    cdef ccudart.VdpGetProcAddress *cvdpGetProcAddress
    if vdpGetProcAddress is None:
        cvdpGetProcAddress = <ccudart.VdpGetProcAddress*><void_ptr>NULL
    elif isinstance(vdpGetProcAddress, (VdpGetProcAddress)):
        pvdpGetProcAddress = vdpGetProcAddress.getPtr()
        cvdpGetProcAddress = <ccudart.VdpGetProcAddress*><void_ptr>pvdpGetProcAddress
    elif isinstance(vdpGetProcAddress, (int)):
        cvdpGetProcAddress = <ccudart.VdpGetProcAddress*><void_ptr>vdpGetProcAddress
    else:
        raise TypeError("Argument 'vdpGetProcAddress' is not instance of type (expected <class 'int, cudart.VdpGetProcAddress'>, found " + str(type(vdpGetProcAddress)))

    cdef ccudart.VdpDevice cvdpDevice
    if vdpDevice is None:
        cvdpDevice = <ccudart.VdpDevice><void_ptr>0
    elif isinstance(vdpDevice, (VdpDevice)):
        pvdpDevice = int(vdpDevice)
        cvdpDevice = <ccudart.VdpDevice><void_ptr>pvdpDevice
    else:
        pvdpDevice = int(VdpDevice(vdpDevice))
        cvdpDevice = <ccudart.VdpDevice><void_ptr>pvdpDevice

    cdef int device = 0
    err = ccudart.cudaVDPAUGetDevice(&device, cvdpDevice, cvdpGetProcAddress)
    return (cudaError_t(err), device)

@cython.embedsignature(True)
def cudaVDPAUSetVDPAUDevice(int device, vdpDevice, vdpGetProcAddress):
    """ Sets a CUDA device to use VDPAU interoperability.

    Records `vdpDevice` as the VdpDevice for VDPAU interoperability with
    the CUDA device `device` and sets `device` as the current device for
    the calling host thread.

    If `device` has already been initialized then this call will fail with
    the error :py:obj:`~.cudaErrorSetOnActiveProcess`. In this case it is
    necessary to reset `device` using :py:obj:`~.cudaDeviceReset()` before
    VDPAU interoperability on `device` may be enabled.

    Parameters
    ----------
    device : int
        Device to use for VDPAU interoperability
    vdpDevice : :py:obj:`~.VdpDevice`
        The VdpDevice to interoperate with
    vdpGetProcAddress : :py:obj:`~.VdpGetProcAddress`
        VDPAU's VdpGetProcAddress function pointer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorSetOnActiveProcess`

    See Also
    --------
    :py:obj:`~.cudaGraphicsVDPAURegisterVideoSurface`, :py:obj:`~.cudaGraphicsVDPAURegisterOutputSurface`, :py:obj:`~.cudaDeviceReset`
    """
    cdef ccudart.VdpGetProcAddress *cvdpGetProcAddress
    if vdpGetProcAddress is None:
        cvdpGetProcAddress = <ccudart.VdpGetProcAddress*><void_ptr>NULL
    elif isinstance(vdpGetProcAddress, (VdpGetProcAddress)):
        pvdpGetProcAddress = vdpGetProcAddress.getPtr()
        cvdpGetProcAddress = <ccudart.VdpGetProcAddress*><void_ptr>pvdpGetProcAddress
    elif isinstance(vdpGetProcAddress, (int)):
        cvdpGetProcAddress = <ccudart.VdpGetProcAddress*><void_ptr>vdpGetProcAddress
    else:
        raise TypeError("Argument 'vdpGetProcAddress' is not instance of type (expected <class 'int, cudart.VdpGetProcAddress'>, found " + str(type(vdpGetProcAddress)))

    cdef ccudart.VdpDevice cvdpDevice
    if vdpDevice is None:
        cvdpDevice = <ccudart.VdpDevice><void_ptr>0
    elif isinstance(vdpDevice, (VdpDevice)):
        pvdpDevice = int(vdpDevice)
        cvdpDevice = <ccudart.VdpDevice><void_ptr>pvdpDevice
    else:
        pvdpDevice = int(VdpDevice(vdpDevice))
        cvdpDevice = <ccudart.VdpDevice><void_ptr>pvdpDevice

    err = ccudart.cudaVDPAUSetVDPAUDevice(device, cvdpDevice, cvdpGetProcAddress)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsVDPAURegisterVideoSurface(vdpSurface, unsigned int flags):
    """ Register a VdpVideoSurface object.

    Registers the VdpVideoSurface specified by `vdpSurface` for access by
    CUDA. A handle to the registered object is returned as `resource`. The
    surface's intended usage is specified using `flags`, as follows:

    - :py:obj:`~.cudaGraphicsMapFlagsNone`: Specifies no hints about how
      this resource will be used. It is therefore assumed that this
      resource will be read from and written to by CUDA. This is the
      default value.

    - :py:obj:`~.cudaGraphicsMapFlagsReadOnly`: Specifies that CUDA will
      not write to this resource.

    - :py:obj:`~.cudaGraphicsMapFlagsWriteDiscard`: Specifies that CUDA
      will not read from this resource and will write over the entire
      contents of the resource, so none of the data previously stored in
      the resource will be preserved.

    Parameters
    ----------
    vdpSurface : :py:obj:`~.VdpVideoSurface`
        VDPAU object to be registered
    flags : unsigned int
        Map flags

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`
    resource : :py:obj:`~.cudaGraphicsResource`
        Pointer to the returned object handle

    See Also
    --------
    :py:obj:`~.cudaVDPAUSetVDPAUDevice`, :py:obj:`~.cudaGraphicsUnregisterResource`, :py:obj:`~.cudaGraphicsSubResourceGetMappedArray`, :py:obj:`~.cuGraphicsVDPAURegisterVideoSurface`
    """
    cdef ccudart.VdpVideoSurface cvdpSurface
    if vdpSurface is None:
        cvdpSurface = <ccudart.VdpVideoSurface><void_ptr>0
    elif isinstance(vdpSurface, (VdpVideoSurface)):
        pvdpSurface = int(vdpSurface)
        cvdpSurface = <ccudart.VdpVideoSurface><void_ptr>pvdpSurface
    else:
        pvdpSurface = int(VdpVideoSurface(vdpSurface))
        cvdpSurface = <ccudart.VdpVideoSurface><void_ptr>pvdpSurface

    cdef cudaGraphicsResource_t resource = cudaGraphicsResource_t()
    err = ccudart.cudaGraphicsVDPAURegisterVideoSurface(resource._ptr, cvdpSurface, flags)
    return (cudaError_t(err), resource)

@cython.embedsignature(True)
def cudaGraphicsVDPAURegisterOutputSurface(vdpSurface, unsigned int flags):
    """ Register a VdpOutputSurface object.

    Registers the VdpOutputSurface specified by `vdpSurface` for access by
    CUDA. A handle to the registered object is returned as `resource`. The
    surface's intended usage is specified using `flags`, as follows:

    - :py:obj:`~.cudaGraphicsMapFlagsNone`: Specifies no hints about how
      this resource will be used. It is therefore assumed that this
      resource will be read from and written to by CUDA. This is the
      default value.

    - :py:obj:`~.cudaGraphicsMapFlagsReadOnly`: Specifies that CUDA will
      not write to this resource.

    - :py:obj:`~.cudaGraphicsMapFlagsWriteDiscard`: Specifies that CUDA
      will not read from this resource and will write over the entire
      contents of the resource, so none of the data previously stored in
      the resource will be preserved.

    Parameters
    ----------
    vdpSurface : :py:obj:`~.VdpOutputSurface`
        VDPAU object to be registered
    flags : unsigned int
        Map flags

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`
    resource : :py:obj:`~.cudaGraphicsResource`
        Pointer to the returned object handle

    See Also
    --------
    :py:obj:`~.cudaVDPAUSetVDPAUDevice`, :py:obj:`~.cudaGraphicsUnregisterResource`, :py:obj:`~.cudaGraphicsSubResourceGetMappedArray`, :py:obj:`~.cuGraphicsVDPAURegisterOutputSurface`
    """
    cdef ccudart.VdpOutputSurface cvdpSurface
    if vdpSurface is None:
        cvdpSurface = <ccudart.VdpOutputSurface><void_ptr>0
    elif isinstance(vdpSurface, (VdpOutputSurface)):
        pvdpSurface = int(vdpSurface)
        cvdpSurface = <ccudart.VdpOutputSurface><void_ptr>pvdpSurface
    else:
        pvdpSurface = int(VdpOutputSurface(vdpSurface))
        cvdpSurface = <ccudart.VdpOutputSurface><void_ptr>pvdpSurface

    cdef cudaGraphicsResource_t resource = cudaGraphicsResource_t()
    err = ccudart.cudaGraphicsVDPAURegisterOutputSurface(resource._ptr, cvdpSurface, flags)
    return (cudaError_t(err), resource)

@cython.embedsignature(True)
def cudaGLGetDevices(unsigned int cudaDeviceCount, deviceList not None : cudaGLDeviceList):
    """ Gets the CUDA devices associated with the current OpenGL context.

    Returns in `*pCudaDeviceCount` the number of CUDA-compatible devices
    corresponding to the current OpenGL context. Also returns in
    `*pCudaDevices` at most `cudaDeviceCount` of the CUDA-compatible
    devices corresponding to the current OpenGL context. If any of the GPUs
    being used by the current OpenGL context are not CUDA capable then the
    call will return cudaErrorNoDevice.

    Parameters
    ----------
    cudaDeviceCount : unsigned int
        The size of the output device array `pCudaDevices`
    deviceList : cudaGLDeviceList
        The set of devices to return. This set may be cudaGLDeviceListAll
        for all devices, cudaGLDeviceListCurrentFrame for the devices used
        to render the current frame (in SLI), or cudaGLDeviceListNextFrame
        for the devices used to render the next frame (in SLI).

    Returns
    -------
    cudaError_t
        cudaSuccess
        cudaErrorNoDevice
        cudaErrorInvalidGraphicsContext
        cudaErrorUnknown
    pCudaDeviceCount : unsigned int
        Returned number of CUDA devices corresponding to the current OpenGL
        context
    pCudaDevices : List[int]
        Returned CUDA devices corresponding to the current OpenGL context

    See Also
    --------
    ~.cudaGraphicsUnregisterResource
    ~.cudaGraphicsMapResources
    ~.cudaGraphicsSubResourceGetMappedArray
    ~.cudaGraphicsResourceGetMappedPointer
    ~.cuGLGetDevices

    Notes
    -----
    This function is not supported on Mac OS X.

    """
    cdef unsigned int pCudaDeviceCount = 0
    cdef int* cpCudaDevices = NULL
    pypCudaDevices = []
    if cudaDeviceCount != 0:
        cpCudaDevices = <int*>calloc(cudaDeviceCount, sizeof(int))
        if cpCudaDevices is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(cudaDeviceCount) + 'x' + str(sizeof(int)))
    cdef ccudart.cudaGLDeviceList cdeviceList = deviceList.value
    err = ccudart.cudaGLGetDevices(&pCudaDeviceCount, cpCudaDevices, cudaDeviceCount, cdeviceList)
    if cudaError_t(err) == cudaError_t(0):
        pypCudaDevices = [<void_ptr>cpCudaDevices[idx] for idx in range(cudaDeviceCount)]
    if cpCudaDevices is not NULL:
        free(cpCudaDevices)
    return (cudaError_t(err), pCudaDeviceCount, pypCudaDevices)

@cython.embedsignature(True)
def cudaGraphicsGLRegisterImage(image, target, unsigned int flags):
    """ Register an OpenGL texture or renderbuffer object.

    Registers the texture or renderbuffer object specified by `image` for
    access by CUDA. A handle to the registered object is returned as
    `resource`.

    `target` must match the type of the object, and must be one of
    :py:obj:`~.GL_TEXTURE_2D`, :py:obj:`~.GL_TEXTURE_RECTANGLE`,
    :py:obj:`~.GL_TEXTURE_CUBE_MAP`, :py:obj:`~.GL_TEXTURE_3D`,
    :py:obj:`~.GL_TEXTURE_2D_ARRAY`, or :py:obj:`~.GL_RENDERBUFFER`.

    The register flags `flags` specify the intended usage, as follows:

    - :py:obj:`~.cudaGraphicsRegisterFlagsNone`: Specifies no hints about
      how this resource will be used. It is therefore assumed that this
      resource will be read from and written to by CUDA. This is the
      default value.

    - :py:obj:`~.cudaGraphicsRegisterFlagsReadOnly`: Specifies that CUDA
      will not write to this resource.

    - :py:obj:`~.cudaGraphicsRegisterFlagsWriteDiscard`: Specifies that
      CUDA will not read from this resource and will write over the entire
      contents of the resource, so none of the data previously stored in
      the resource will be preserved.

    - :py:obj:`~.cudaGraphicsRegisterFlagsSurfaceLoadStore`: Specifies that
      CUDA will bind this resource to a surface reference.

    - :py:obj:`~.cudaGraphicsRegisterFlagsTextureGather`: Specifies that
      CUDA will perform texture gather operations on this resource.

    The following image formats are supported. For brevity's sake, the list
    is abbreviated. For ex., {GL_R, GL_RG} X {8, 16} would expand to the
    following 4 formats {GL_R8, GL_R16, GL_RG8, GL_RG16} :

    - GL_RED, GL_RG, GL_RGBA, GL_LUMINANCE, GL_ALPHA, GL_LUMINANCE_ALPHA,
      GL_INTENSITY

    - {GL_R, GL_RG, GL_RGBA} X {8, 16, 16F, 32F, 8UI, 16UI, 32UI, 8I, 16I,
      32I}

    - {GL_LUMINANCE, GL_ALPHA, GL_LUMINANCE_ALPHA, GL_INTENSITY} X {8, 16,
      16F_ARB, 32F_ARB, 8UI_EXT, 16UI_EXT, 32UI_EXT, 8I_EXT, 16I_EXT,
      32I_EXT}

    The following image classes are currently disallowed:

    - Textures with borders

    - Multisampled renderbuffers

    Parameters
    ----------
    image : :py:obj:`~.GLuint`
        name of texture or renderbuffer object to be registered
    target : :py:obj:`~.GLenum`
        Identifies the type of object specified by `image`
    flags : unsigned int
        Register flags

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`
    resource : :py:obj:`~.cudaGraphicsResource`
        Pointer to the returned object handle

    See Also
    --------
    :py:obj:`~.cudaGraphicsUnregisterResource`, :py:obj:`~.cudaGraphicsMapResources`, :py:obj:`~.cudaGraphicsSubResourceGetMappedArray`, :py:obj:`~.cuGraphicsGLRegisterImage`
    """
    cdef ccudart.GLenum ctarget
    if target is None:
        ctarget = <ccudart.GLenum><void_ptr>0
    elif isinstance(target, (GLenum)):
        ptarget = int(target)
        ctarget = <ccudart.GLenum><void_ptr>ptarget
    else:
        ptarget = int(GLenum(target))
        ctarget = <ccudart.GLenum><void_ptr>ptarget

    cdef ccudart.GLuint cimage
    if image is None:
        cimage = <ccudart.GLuint><void_ptr>0
    elif isinstance(image, (GLuint)):
        pimage = int(image)
        cimage = <ccudart.GLuint><void_ptr>pimage
    else:
        pimage = int(GLuint(image))
        cimage = <ccudart.GLuint><void_ptr>pimage

    cdef cudaGraphicsResource_t resource = cudaGraphicsResource_t()
    err = ccudart.cudaGraphicsGLRegisterImage(resource._ptr, cimage, ctarget, flags)
    return (cudaError_t(err), resource)

@cython.embedsignature(True)
def cudaGraphicsGLRegisterBuffer(buffer, unsigned int flags):
    """ Registers an OpenGL buffer object.

    Registers the buffer object specified by `buffer` for access by CUDA. A
    handle to the registered object is returned as `resource`. The register
    flags `flags` specify the intended usage, as follows:

    - :py:obj:`~.cudaGraphicsRegisterFlagsNone`: Specifies no hints about
      how this resource will be used. It is therefore assumed that this
      resource will be read from and written to by CUDA. This is the
      default value.

    - :py:obj:`~.cudaGraphicsRegisterFlagsReadOnly`: Specifies that CUDA
      will not write to this resource.

    - :py:obj:`~.cudaGraphicsRegisterFlagsWriteDiscard`: Specifies that
      CUDA will not read from this resource and will write over the entire
      contents of the resource, so none of the data previously stored in
      the resource will be preserved.

    Parameters
    ----------
    buffer : :py:obj:`~.GLuint`
        name of buffer object to be registered
    flags : unsigned int
        Register flags

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`
    resource : :py:obj:`~.cudaGraphicsResource`
        Pointer to the returned object handle

    See Also
    --------
    :py:obj:`~.cudaGraphicsUnregisterResource`, :py:obj:`~.cudaGraphicsMapResources`, :py:obj:`~.cudaGraphicsResourceGetMappedPointer`, :py:obj:`~.cuGraphicsGLRegisterBuffer`
    """
    cdef ccudart.GLuint cbuffer
    if buffer is None:
        cbuffer = <ccudart.GLuint><void_ptr>0
    elif isinstance(buffer, (GLuint)):
        pbuffer = int(buffer)
        cbuffer = <ccudart.GLuint><void_ptr>pbuffer
    else:
        pbuffer = int(GLuint(buffer))
        cbuffer = <ccudart.GLuint><void_ptr>pbuffer

    cdef cudaGraphicsResource_t resource = cudaGraphicsResource_t()
    err = ccudart.cudaGraphicsGLRegisterBuffer(resource._ptr, cbuffer, flags)
    return (cudaError_t(err), resource)

@cython.embedsignature(True)
def cudaDeviceReset():
    """ Destroy all allocations and reset all state on the current device in the current process.

    Explicitly destroys and cleans up all resources associated with the
    current device in the current process. It is the caller's
    responsibility to ensure that the resources are not accessed or passed
    in subsequent API calls and doing so will result in undefined behavior.
    These resources include CUDA types such as :py:obj:`~.cudaStream_t`,
    :py:obj:`~.cudaEvent_t`, :py:obj:`~.cudaArray_t`,
    :py:obj:`~.cudaMipmappedArray_t`, :py:obj:`~.cudaTextureObject_t`,
    :py:obj:`~.cudaSurfaceObject_t`, :py:obj:`~.textureReference`,
    :py:obj:`~.surfaceReference`, :py:obj:`~.cudaExternalMemory_t`,
    :py:obj:`~.cudaExternalSemaphore_t` and
    :py:obj:`~.cudaGraphicsResource_t`. Any subsequent API call to this
    device will reinitialize the device.

    Note that this function will reset the device immediately. It is the
    caller's responsibility to ensure that the device is not being accessed
    by any other host threads from the process when this function is
    called.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaDeviceSynchronize`
    """
    err = ccudart.cudaDeviceReset()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceSynchronize():
    """ Wait for compute device to finish.

    Blocks until the device has completed all preceding requested tasks.
    :py:obj:`~.cudaDeviceSynchronize()` returns an error if one of the
    preceding tasks has failed. If the
    :py:obj:`~.cudaDeviceScheduleBlockingSync` flag was set for this
    device, the host thread will block until the device has finished its
    work.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaDeviceReset`, :py:obj:`~.cuCtxSynchronize`
    """
    with nogil:
        err = ccudart.cudaDeviceSynchronize()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceSetLimit(limit not None : cudaLimit, size_t value):
    """ Set resource limits.

    Setting `limit` to `value` is a request by the application to update
    the current limit maintained by the device. The driver is free to
    modify the requested value to meet h/w requirements (this could be
    clamping to minimum or maximum values, rounding up to nearest element
    size, etc). The application can use :py:obj:`~.cudaDeviceGetLimit()` to
    find out exactly what the limit has been set to.

    Setting each :py:obj:`~.cudaLimit` has its own specific restrictions,
    so each is discussed here.

    - :py:obj:`~.cudaLimitStackSize` controls the stack size in bytes of
      each GPU thread.

    - :py:obj:`~.cudaLimitPrintfFifoSize` controls the size in bytes of the
      shared FIFO used by the :py:obj:`~.printf()` device system call.
      Setting :py:obj:`~.cudaLimitPrintfFifoSize` must not be performed
      after launching any kernel that uses the :py:obj:`~.printf()` device
      system call - in such case :py:obj:`~.cudaErrorInvalidValue` will be
      returned.

    - :py:obj:`~.cudaLimitMallocHeapSize` controls the size in bytes of the
      heap used by the :py:obj:`~.malloc()` and :py:obj:`~.free()` device
      system calls. Setting :py:obj:`~.cudaLimitMallocHeapSize` must not be
      performed after launching any kernel that uses the
      :py:obj:`~.malloc()` or :py:obj:`~.free()` device system calls - in
      such case :py:obj:`~.cudaErrorInvalidValue` will be returned.

    - :py:obj:`~.cudaLimitDevRuntimeSyncDepth` controls the maximum nesting
      depth of a grid at which a thread can safely call
      :py:obj:`~.cudaDeviceSynchronize()`. Setting this limit must be
      performed before any launch of a kernel that uses the device runtime
      and calls :py:obj:`~.cudaDeviceSynchronize()` above the default sync
      depth, two levels of grids. Calls to
      :py:obj:`~.cudaDeviceSynchronize()` will fail with error code
      :py:obj:`~.cudaErrorSyncDepthExceeded` if the limitation is violated.
      This limit can be set smaller than the default or up the maximum
      launch depth of 24. When setting this limit, keep in mind that
      additional levels of sync depth require the runtime to reserve large
      amounts of device memory which can no longer be used for user
      allocations. If these reservations of device memory fail,
      :py:obj:`~.cudaDeviceSetLimit` will return
      :py:obj:`~.cudaErrorMemoryAllocation`, and the limit can be reset to
      a lower value. This limit is only applicable to devices of compute
      capability 3.5 and higher. Attempting to set this limit on devices of
      compute capability less than 3.5 will result in the error
      :py:obj:`~.cudaErrorUnsupportedLimit` being returned.

    - :py:obj:`~.cudaLimitDevRuntimePendingLaunchCount` controls the
      maximum number of outstanding device runtime launches that can be
      made from the current device. A grid is outstanding from the point of
      launch up until the grid is known to have been completed. Device
      runtime launches which violate this limitation fail and return
      :py:obj:`~.cudaErrorLaunchPendingCountExceeded` when
      :py:obj:`~.cudaGetLastError()` is called after launch. If more
      pending launches than the default (2048 launches) are needed for a
      module using the device runtime, this limit can be increased. Keep in
      mind that being able to sustain additional pending launches will
      require the runtime to reserve larger amounts of device memory
      upfront which can no longer be used for allocations. If these
      reservations fail, :py:obj:`~.cudaDeviceSetLimit` will return
      :py:obj:`~.cudaErrorMemoryAllocation`, and the limit can be reset to
      a lower value. This limit is only applicable to devices of compute
      capability 3.5 and higher. Attempting to set this limit on devices of
      compute capability less than 3.5 will result in the error
      :py:obj:`~.cudaErrorUnsupportedLimit` being returned.

    - :py:obj:`~.cudaLimitMaxL2FetchGranularity` controls the L2 cache
      fetch granularity. Values can range from 0B to 128B. This is purely a
      performance hint and it can be ignored or clamped depending on the
      platform.

    - :py:obj:`~.cudaLimitPersistingL2CacheSize` controls size in bytes
      available for persisting L2 cache. This is purely a performance hint
      and it can be ignored or clamped depending on the platform.

    Parameters
    ----------
    limit : :py:obj:`~.cudaLimit`
        Limit to set
    value : size_t
        Size of limit

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorUnsupportedLimit`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`

    See Also
    --------
    :py:obj:`~.cudaDeviceGetLimit`, :py:obj:`~.cuCtxSetLimit`
    """
    cdef ccudart.cudaLimit climit = limit.value
    err = ccudart.cudaDeviceSetLimit(climit, value)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceGetLimit(limit not None : cudaLimit):
    """ Returns resource limits.

    Returns in `*pValue` the current size of `limit`. The supported
    :py:obj:`~.cudaLimit` values are:

    - :py:obj:`~.cudaLimitStackSize`: stack size in bytes of each GPU
      thread;

    - :py:obj:`~.cudaLimitPrintfFifoSize`: size in bytes of the shared FIFO
      used by the :py:obj:`~.printf()` device system call.

    - :py:obj:`~.cudaLimitMallocHeapSize`: size in bytes of the heap used
      by the :py:obj:`~.malloc()` and :py:obj:`~.free()` device system
      calls;

    - :py:obj:`~.cudaLimitDevRuntimeSyncDepth`: maximum grid depth at which
      a thread can isssue the device runtime call
      :py:obj:`~.cudaDeviceSynchronize()` to wait on child grid launches to
      complete.

    - :py:obj:`~.cudaLimitDevRuntimePendingLaunchCount`: maximum number of
      outstanding device runtime launches.

    - :py:obj:`~.cudaLimitMaxL2FetchGranularity`: L2 cache fetch
      granularity.

    - :py:obj:`~.cudaLimitPersistingL2CacheSize`: Persisting L2 cache size
      in bytes

    Parameters
    ----------
    limit : :py:obj:`~.cudaLimit`
        Limit to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorUnsupportedLimit`, :py:obj:`~.cudaErrorInvalidValue`
    pValue : int
        Returned size of the limit

    See Also
    --------
    :py:obj:`~.cudaDeviceSetLimit`, :py:obj:`~.cuCtxGetLimit`
    """
    cdef size_t pValue = 0
    cdef ccudart.cudaLimit climit = limit.value
    err = ccudart.cudaDeviceGetLimit(&pValue, climit)
    return (cudaError_t(err), pValue)

@cython.embedsignature(True)
def cudaDeviceGetTexture1DLinearMaxWidth(fmtDesc : cudaChannelFormatDesc, int device):
    """ Returns the maximum number of elements allocatable in a 1D linear texture for a given element size.

    Returns in `maxWidthInElements` the maximum number of elements
    allocatable in a 1D linear texture for given format descriptor
    `fmtDesc`.

    Parameters
    ----------
    fmtDesc : :py:obj:`~.cudaChannelFormatDesc`
        Texture format description.
    None : int
        None

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorUnsupportedLimit`, :py:obj:`~.cudaErrorInvalidValue`
    maxWidthInElements : int
        Returns maximum number of texture elements allocatable for given
        `fmtDesc`.

    See Also
    --------
    :py:obj:`~.cuDeviceGetMaxTexture1DLinear`,
    """
    cdef size_t maxWidthInElements = 0
    cdef ccudart.cudaChannelFormatDesc* cfmtDesc_ptr = fmtDesc._ptr if fmtDesc != None else NULL
    err = ccudart.cudaDeviceGetTexture1DLinearMaxWidth(&maxWidthInElements, cfmtDesc_ptr, device)
    return (cudaError_t(err), maxWidthInElements)

@cython.embedsignature(True)
def cudaDeviceGetCacheConfig():
    """ Returns the preferred cache configuration for the current device.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this returns through `pCacheConfig` the preferred cache
    configuration for the current device. This is only a preference. The
    runtime will use the requested configuration if possible, but it is
    free to choose a different configuration if required to execute
    functions.

    This will return a `pCacheConfig` of
    :py:obj:`~.cudaFuncCachePreferNone` on devices where the size of the L1
    cache and shared memory are fixed.

    The supported cache configurations are:

    - :py:obj:`~.cudaFuncCachePreferNone`: no preference for shared memory
      or L1 (default)

    - :py:obj:`~.cudaFuncCachePreferShared`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.cudaFuncCachePreferL1`: prefer larger L1 cache and smaller
      shared memory

    - :py:obj:`~.cudaFuncCachePreferEqual`: prefer equal size L1 cache and
      shared memory

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`
    pCacheConfig : :py:obj:`~.cudaFuncCache`
        Returned cache configuration

    See Also
    --------
    :py:obj:`~.cudaDeviceSetCacheConfig`, :py:obj:`~.cudaFuncSetCacheConfig (C API)`, cudaFuncSetCacheConfig (C++ API), :py:obj:`~.cuCtxGetCacheConfig`
    """
    cdef ccudart.cudaFuncCache pCacheConfig
    err = ccudart.cudaDeviceGetCacheConfig(&pCacheConfig)
    return (cudaError_t(err), cudaFuncCache(pCacheConfig))

@cython.embedsignature(True)
def cudaDeviceGetStreamPriorityRange():
    """ Returns numerical values that correspond to the least and greatest stream priorities.

    Returns in `*leastPriority` and `*greatestPriority` the numerical
    values that correspond to the least and greatest stream priorities
    respectively. Stream priorities follow a convention where lower numbers
    imply greater priorities. The range of meaningful stream priorities is
    given by [`*greatestPriority`, `*leastPriority`]. If the user attempts
    to create a stream with a priority value that is outside the the
    meaningful range as specified by this API, the priority is
    automatically clamped down or up to either `*leastPriority` or
    `*greatestPriority` respectively. See
    :py:obj:`~.cudaStreamCreateWithPriority` for details on creating a
    priority stream. A NULL may be passed in for `*leastPriority` or
    `*greatestPriority` if the value is not desired.

    This function will return '0' in both `*leastPriority` and
    `*greatestPriority` if the current context's device does not support
    stream priorities (see :py:obj:`~.cudaDeviceGetAttribute`).

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`
    leastPriority : int
        Pointer to an int in which the numerical value for least stream
        priority is returned
    greatestPriority : int
        Pointer to an int in which the numerical value for greatest stream
        priority is returned

    See Also
    --------
    :py:obj:`~.cudaStreamCreateWithPriority`, :py:obj:`~.cudaStreamGetPriority`, :py:obj:`~.cuCtxGetStreamPriorityRange`
    """
    cdef int leastPriority = 0
    cdef int greatestPriority = 0
    err = ccudart.cudaDeviceGetStreamPriorityRange(&leastPriority, &greatestPriority)
    return (cudaError_t(err), leastPriority, greatestPriority)

@cython.embedsignature(True)
def cudaDeviceSetCacheConfig(cacheConfig not None : cudaFuncCache):
    """ Sets the preferred cache configuration for the current device.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this sets through `cacheConfig` the preferred cache
    configuration for the current device. This is only a preference. The
    runtime will use the requested configuration if possible, but it is
    free to choose a different configuration if required to execute the
    function. Any function preference set via
    :py:obj:`~.cudaFuncSetCacheConfig (C API)` or cudaFuncSetCacheConfig
    (C++ API) will be preferred over this device-wide setting. Setting the
    device-wide cache configuration to :py:obj:`~.cudaFuncCachePreferNone`
    will cause subsequent kernel launches to prefer to not change the cache
    configuration unless required to launch the kernel.

    This setting does nothing on devices where the size of the L1 cache and
    shared memory are fixed.

    Launching a kernel with a different preference than the most recent
    preference setting may insert a device-side synchronization point.

    The supported cache configurations are:

    - :py:obj:`~.cudaFuncCachePreferNone`: no preference for shared memory
      or L1 (default)

    - :py:obj:`~.cudaFuncCachePreferShared`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.cudaFuncCachePreferL1`: prefer larger L1 cache and smaller
      shared memory

    - :py:obj:`~.cudaFuncCachePreferEqual`: prefer equal size L1 cache and
      shared memory

    Parameters
    ----------
    cacheConfig : :py:obj:`~.cudaFuncCache`
        Requested cache configuration

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaDeviceGetCacheConfig`, :py:obj:`~.cudaFuncSetCacheConfig (C API)`, cudaFuncSetCacheConfig (C++ API), :py:obj:`~.cuCtxSetCacheConfig`
    """
    cdef ccudart.cudaFuncCache ccacheConfig = cacheConfig.value
    err = ccudart.cudaDeviceSetCacheConfig(ccacheConfig)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceGetSharedMemConfig():
    """ Returns the shared memory configuration for the current device.

    This function will return in `pConfig` the current size of shared
    memory banks on the current device. On devices with configurable shared
    memory banks, :py:obj:`~.cudaDeviceSetSharedMemConfig` can be used to
    change this setting, so that all subsequent kernel launches will by
    default use the new bank size. When
    :py:obj:`~.cudaDeviceGetSharedMemConfig` is called on devices without
    configurable shared memory, it will return the fixed bank size of the
    hardware.

    The returned bank configurations can be either:

    - :py:obj:`~.cudaSharedMemBankSizeFourByte` - shared memory bank width
      is four bytes.

    - :py:obj:`~.cudaSharedMemBankSizeEightByte` - shared memory bank width
      is eight bytes.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pConfig : :py:obj:`~.cudaSharedMemConfig`
        Returned cache configuration

    See Also
    --------
    :py:obj:`~.cudaDeviceSetCacheConfig`, :py:obj:`~.cudaDeviceGetCacheConfig`, :py:obj:`~.cudaDeviceSetSharedMemConfig`, :py:obj:`~.cudaFuncSetCacheConfig`, :py:obj:`~.cuCtxGetSharedMemConfig`
    """
    cdef ccudart.cudaSharedMemConfig pConfig
    err = ccudart.cudaDeviceGetSharedMemConfig(&pConfig)
    return (cudaError_t(err), cudaSharedMemConfig(pConfig))

@cython.embedsignature(True)
def cudaDeviceSetSharedMemConfig(config not None : cudaSharedMemConfig):
    """ Sets the shared memory configuration for the current device.

    On devices with configurable shared memory banks, this function will
    set the shared memory bank size which is used for all subsequent kernel
    launches. Any per-function setting of shared memory set via
    :py:obj:`~.cudaFuncSetSharedMemConfig` will override the device wide
    setting.

    Changing the shared memory configuration between launches may introduce
    a device side synchronization point.

    Changing the shared memory bank size will not increase shared memory
    usage or affect occupancy of kernels, but may have major effects on
    performance. Larger bank sizes will allow for greater potential
    bandwidth to shared memory, but will change what kinds of accesses to
    shared memory will result in bank conflicts.

    This function will do nothing on devices with fixed shared memory bank
    size.

    The supported bank configurations are:

    - :py:obj:`~.cudaSharedMemBankSizeDefault`: set bank width the device
      default (currently, four bytes)

    - :py:obj:`~.cudaSharedMemBankSizeFourByte`: set shared memory bank
      width to be four bytes natively.

    - :py:obj:`~.cudaSharedMemBankSizeEightByte`: set shared memory bank
      width to be eight bytes natively.

    Parameters
    ----------
    config : :py:obj:`~.cudaSharedMemConfig`
        Requested cache configuration

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaDeviceSetCacheConfig`, :py:obj:`~.cudaDeviceGetCacheConfig`, :py:obj:`~.cudaDeviceGetSharedMemConfig`, :py:obj:`~.cudaFuncSetCacheConfig`, :py:obj:`~.cuCtxSetSharedMemConfig`
    """
    cdef ccudart.cudaSharedMemConfig cconfig = config.value
    err = ccudart.cudaDeviceSetSharedMemConfig(cconfig)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceGetByPCIBusId(char* pciBusId):
    """ Returns a handle to a compute device.

    Returns in `*device` a device ordinal given a PCI bus ID string.

    where `domain`, `bus`, `device`, and `function` are all hexadecimal
    values

    Parameters
    ----------
    pciBusId : bytes
        String in one of the following forms:

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`
    device : int
        Returned device ordinal

    See Also
    --------
    :py:obj:`~.cudaDeviceGetPCIBusId`, :py:obj:`~.cuDeviceGetByPCIBusId`
    """
    cdef int device = 0
    err = ccudart.cudaDeviceGetByPCIBusId(&device, pciBusId)
    return (cudaError_t(err), device)

@cython.embedsignature(True)
def cudaDeviceGetPCIBusId(int length, int device):
    """ Returns a PCI Bus Id string for the device.

    Returns an ASCII string identifying the device `dev` in the NULL-
    terminated string pointed to by `pciBusId`. `length` specifies the
    maximum length of the string that may be returned.

    where `domain`, `bus`, `device`, and `function` are all hexadecimal
    values. pciBusId should be large enough to store 13 characters
    including the NULL-terminator.

    Parameters
    ----------
    length : int
        Maximum length of string to store in `name`
    device : int
        Device to get identifier string for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`
    pciBusId : bytes
        Returned identifier string for the device in the following format

    See Also
    --------
    :py:obj:`~.cudaDeviceGetByPCIBusId`, :py:obj:`~.cuDeviceGetPCIBusId`
    """
    pypciBusId = b" " * length
    cdef char* pciBusId = pypciBusId
    err = ccudart.cudaDeviceGetPCIBusId(pciBusId, length, device)
    return (cudaError_t(err), pypciBusId)

@cython.embedsignature(True)
def cudaIpcGetEventHandle(event):
    """ Gets an interprocess handle for a previously allocated event.

    Takes as input a previously allocated event. This event must have been
    created with the :py:obj:`~.cudaEventInterprocess` and
    :py:obj:`~.cudaEventDisableTiming` flags set. This opaque handle may be
    copied into other processes and opened with
    :py:obj:`~.cudaIpcOpenEventHandle` to allow efficient hardware
    synchronization between GPU work in different processes.

    After the event has been been opened in the importing process,
    :py:obj:`~.cudaEventRecord`, :py:obj:`~.cudaEventSynchronize`,
    :py:obj:`~.cudaStreamWaitEvent` and :py:obj:`~.cudaEventQuery` may be
    used in either process. Performing operations on the imported event
    after the exported event has been freed with
    :py:obj:`~.cudaEventDestroy` will result in undefined behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux operating systems. IPC functionality is not
    supported on Tegra platforms.

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event allocated with :py:obj:`~.cudaEventInterprocess` and
        :py:obj:`~.cudaEventDisableTiming` flags.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorMemoryAllocation`, :py:obj:`~.cudaErrorMapBufferObjectFailed`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`
    handle : :py:obj:`~.cudaIpcEventHandle_t`
        Pointer to a user allocated cudaIpcEventHandle in which to return
        the opaque event handle

    See Also
    --------
    :py:obj:`~.cudaEventCreate`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaIpcOpenEventHandle`, :py:obj:`~.cudaIpcGetMemHandle`, :py:obj:`~.cudaIpcOpenMemHandle`, :py:obj:`~.cudaIpcCloseMemHandle`, :py:obj:`~.cuIpcGetEventHandle`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    cdef cudaIpcEventHandle_t handle = cudaIpcEventHandle_t()
    err = ccudart.cudaIpcGetEventHandle(handle._ptr, cevent)
    return (cudaError_t(err), handle)

@cython.embedsignature(True)
def cudaIpcOpenEventHandle(handle not None : cudaIpcEventHandle_t):
    """ Opens an interprocess event handle for use in the current process.

    Opens an interprocess event handle exported from another process with
    :py:obj:`~.cudaIpcGetEventHandle`. This function returns a
    :py:obj:`~.cudaEvent_t` that behaves like a locally created event with
    the :py:obj:`~.cudaEventDisableTiming` flag specified. This event must
    be freed with :py:obj:`~.cudaEventDestroy`.

    Performing operations on the imported event after the exported event
    has been freed with :py:obj:`~.cudaEventDestroy` will result in
    undefined behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux operating systems. IPC functionality is not
    supported on Tegra platforms.

    Parameters
    ----------
    handle : :py:obj:`~.cudaIpcEventHandle_t`
        Interprocess handle to open

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorMapBufferObjectFailed`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorDeviceUninitialized`
    event : :py:obj:`~.cudaEvent_t`
        Returns the imported event

    See Also
    --------
    :py:obj:`~.cudaEventCreate`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaIpcGetEventHandle`, :py:obj:`~.cudaIpcGetMemHandle`, :py:obj:`~.cudaIpcOpenMemHandle`, :py:obj:`~.cudaIpcCloseMemHandle`, :py:obj:`~.cuIpcOpenEventHandle`
    """
    cdef cudaEvent_t event = cudaEvent_t()
    err = ccudart.cudaIpcOpenEventHandle(<ccudart.cudaEvent_t*>event._ptr, handle._ptr[0])
    return (cudaError_t(err), event)

@cython.embedsignature(True)
def cudaIpcGetMemHandle(devPtr):
    """ Gets an interprocess memory handle for an existing device memory allocation.

    Takes a pointer to the base of an existing device memory allocation
    created with :py:obj:`~.cudaMalloc` and exports it for use in another
    process. This is a lightweight operation and may be called multiple
    times on an allocation without adverse effects.

    If a region of memory is freed with :py:obj:`~.cudaFree` and a
    subsequent call to :py:obj:`~.cudaMalloc` returns memory with the same
    device address, :py:obj:`~.cudaIpcGetMemHandle` will return a unique
    handle for the new memory.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux operating systems. IPC functionality is not
    supported on Tegra platforms.

    Parameters
    ----------
    devPtr : Any
        Base pointer to previously allocated device memory

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorMemoryAllocation`, :py:obj:`~.cudaErrorMapBufferObjectFailed`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`
    handle : :py:obj:`~.cudaIpcMemHandle_t`
        Pointer to user allocated :py:obj:`~.cudaIpcMemHandle` to return
        the handle in.

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaIpcGetEventHandle`, :py:obj:`~.cudaIpcOpenEventHandle`, :py:obj:`~.cudaIpcOpenMemHandle`, :py:obj:`~.cudaIpcCloseMemHandle`, :py:obj:`~.cuIpcGetMemHandle`
    """
    cdef cudaIpcMemHandle_t handle = cudaIpcMemHandle_t()
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    err = ccudart.cudaIpcGetMemHandle(handle._ptr, cdevPtr_ptr)
    return (cudaError_t(err), handle)

@cython.embedsignature(True)
def cudaIpcOpenMemHandle(handle not None : cudaIpcMemHandle_t, unsigned int flags):
    """ Opens an interprocess memory handle exported from another process and returns a device pointer usable in the local process.

    Maps memory exported from another process with
    :py:obj:`~.cudaIpcGetMemHandle` into the current device address space.
    For contexts on different devices :py:obj:`~.cudaIpcOpenMemHandle` can
    attempt to enable peer access between the devices as if the user called
    :py:obj:`~.cudaDeviceEnablePeerAccess`. This behavior is controlled by
    the :py:obj:`~.cudaIpcMemLazyEnablePeerAccess` flag.
    :py:obj:`~.cudaDeviceCanAccessPeer` can determine if a mapping is
    possible.

    :py:obj:`~.cudaIpcOpenMemHandle` can open handles to devices that may
    not be visible in the process calling the API.

    Contexts that may open :py:obj:`~.cudaIpcMemHandles` are restricted in
    the following way. :py:obj:`~.cudaIpcMemHandles` from each device in a
    given process may only be opened by one context per device per other
    process.

    If the memory handle has already been opened by the current context,
    the reference count on the handle is incremented by 1 and the existing
    device pointer is returned.

    Memory returned from :py:obj:`~.cudaIpcOpenMemHandle` must be freed
    with :py:obj:`~.cudaIpcCloseMemHandle`.

    Calling :py:obj:`~.cudaFree` on an exported memory region before
    calling :py:obj:`~.cudaIpcCloseMemHandle` in the importing context will
    result in undefined behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux operating systems. IPC functionality is not
    supported on Tegra platforms.

    Parameters
    ----------
    handle : :py:obj:`~.cudaIpcMemHandle_t`
        :py:obj:`~.cudaIpcMemHandle` to open
    flags : unsigned int
        Flags for this operation. Must be specified as
        :py:obj:`~.cudaIpcMemLazyEnablePeerAccess`

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorMapBufferObjectFailed`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorDeviceUninitialized`, :py:obj:`~.cudaErrorTooManyPeers`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`
    devPtr : Any
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaIpcGetEventHandle`, :py:obj:`~.cudaIpcOpenEventHandle`, :py:obj:`~.cudaIpcGetMemHandle`, :py:obj:`~.cudaIpcCloseMemHandle`, :py:obj:`~.cudaDeviceEnablePeerAccess`, :py:obj:`~.cudaDeviceCanAccessPeer`, :py:obj:`~.cuIpcOpenMemHandle`

    Notes
    -----
    No guarantees are made about the address returned in `*devPtr`. 
     In particular, multiple processes may not receive the same address for the same `handle`.
    """
    cdef void_ptr devPtr = 0
    err = ccudart.cudaIpcOpenMemHandle(<void**>&devPtr, handle._ptr[0], flags)
    return (cudaError_t(err), devPtr)

@cython.embedsignature(True)
def cudaIpcCloseMemHandle(devPtr):
    """ Attempts to close memory mapped with cudaIpcOpenMemHandle.

    Decrements the reference count of the memory returnd by
    :py:obj:`~.cudaIpcOpenMemHandle` by 1. When the reference count reaches
    0, this API unmaps the memory. The original allocation in the exporting
    process as well as imported mappings in other processes will be
    unaffected.

    Any resources used to enable peer access will be freed if this is the
    last mapping using them.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux operating systems. IPC functionality is not
    supported on Tegra platforms.

    Parameters
    ----------
    devPtr : Any
        Device pointer returned by :py:obj:`~.cudaIpcOpenMemHandle`

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorMapBufferObjectFailed`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaIpcGetEventHandle`, :py:obj:`~.cudaIpcOpenEventHandle`, :py:obj:`~.cudaIpcGetMemHandle`, :py:obj:`~.cudaIpcOpenMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`
    """
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    err = ccudart.cudaIpcCloseMemHandle(cdevPtr_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceFlushGPUDirectRDMAWrites(target not None : cudaFlushGPUDirectRDMAWritesTarget, scope not None : cudaFlushGPUDirectRDMAWritesScope):
    """ Blocks until remote writes are visible to the specified scope.

    Blocks until remote writes to the target context via mappings created
    through GPUDirect RDMA APIs, like nvidia_p2p_get_pages (see
    https://docs.nvidia.com/cuda/gpudirect-rdma for more information), are
    visible to the specified scope.

    If the scope equals or lies within the scope indicated by
    :py:obj:`~.cudaDevAttrGPUDirectRDMAWritesOrdering`, the call will be a
    no-op and can be safely omitted for performance. This can be determined
    by comparing the numerical values between the two enums, with smaller
    scopes having smaller values.

    Users may query support for this API via
    :py:obj:`~.cudaDevAttrGPUDirectRDMAFlushWritesOptions`.

    Parameters
    ----------
    target : :py:obj:`~.cudaFlushGPUDirectRDMAWritesTarget`
        The target of the operation, see cudaFlushGPUDirectRDMAWritesTarget
    scope : :py:obj:`~.cudaFlushGPUDirectRDMAWritesScope`
        The scope of the operation, see cudaFlushGPUDirectRDMAWritesScope

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorNotSupported`,

    See Also
    --------
    :py:obj:`~.cuFlushGPUDirectRDMAWrites`
    """
    cdef ccudart.cudaFlushGPUDirectRDMAWritesTarget ctarget = target.value
    cdef ccudart.cudaFlushGPUDirectRDMAWritesScope cscope = scope.value
    err = ccudart.cudaDeviceFlushGPUDirectRDMAWrites(ctarget, cscope)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaThreadExit():
    """ Exit and clean up from CUDA launches.

    [Deprecated]

    Note that this function is deprecated because its name does not reflect
    its behavior. Its functionality is identical to the non-deprecated
    function :py:obj:`~.cudaDeviceReset()`, which should be used instead.

    Explicitly destroys all cleans up all resources associated with the
    current device in the current process. Any subsequent API call to this
    device will reinitialize the device.

    Note that this function will reset the device immediately. It is the
    caller's responsibility to ensure that the device is not being accessed
    by any other host threads from the process when this function is
    called.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaDeviceReset`
    """
    err = ccudart.cudaThreadExit()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaThreadSynchronize():
    """ Wait for compute device to finish.

    [Deprecated]

    Note that this function is deprecated because its name does not reflect
    its behavior. Its functionality is similar to the non-deprecated
    function :py:obj:`~.cudaDeviceSynchronize()`, which should be used
    instead.

    Blocks until the device has completed all preceding requested tasks.
    :py:obj:`~.cudaThreadSynchronize()` returns an error if one of the
    preceding tasks has failed. If the
    :py:obj:`~.cudaDeviceScheduleBlockingSync` flag was set for this
    device, the host thread will block until the device has finished its
    work.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaDeviceSynchronize`
    """
    err = ccudart.cudaThreadSynchronize()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaThreadSetLimit(limit not None : cudaLimit, size_t value):
    """ Set resource limits.

    [Deprecated]

    Note that this function is deprecated because its name does not reflect
    its behavior. Its functionality is identical to the non-deprecated
    function :py:obj:`~.cudaDeviceSetLimit()`, which should be used
    instead.

    Setting `limit` to `value` is a request by the application to update
    the current limit maintained by the device. The driver is free to
    modify the requested value to meet h/w requirements (this could be
    clamping to minimum or maximum values, rounding up to nearest element
    size, etc). The application can use :py:obj:`~.cudaThreadGetLimit()` to
    find out exactly what the limit has been set to.

    Setting each :py:obj:`~.cudaLimit` has its own specific restrictions,
    so each is discussed here.

    - :py:obj:`~.cudaLimitStackSize` controls the stack size of each GPU
      thread.

    - :py:obj:`~.cudaLimitPrintfFifoSize` controls the size of the shared
      FIFO used by the :py:obj:`~.printf()` device system call. Setting
      :py:obj:`~.cudaLimitPrintfFifoSize` must be performed before
      launching any kernel that uses the :py:obj:`~.printf()` device system
      call, otherwise :py:obj:`~.cudaErrorInvalidValue` will be returned.

    - :py:obj:`~.cudaLimitMallocHeapSize` controls the size of the heap
      used by the :py:obj:`~.malloc()` and :py:obj:`~.free()` device system
      calls. Setting :py:obj:`~.cudaLimitMallocHeapSize` must be performed
      before launching any kernel that uses the :py:obj:`~.malloc()` or
      :py:obj:`~.free()` device system calls, otherwise
      :py:obj:`~.cudaErrorInvalidValue` will be returned.

    Parameters
    ----------
    limit : :py:obj:`~.cudaLimit`
        Limit to set
    value : size_t
        Size in bytes of limit

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorUnsupportedLimit`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaDeviceSetLimit`
    """
    cdef ccudart.cudaLimit climit = limit.value
    err = ccudart.cudaThreadSetLimit(climit, value)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaThreadGetLimit(limit not None : cudaLimit):
    """ Returns resource limits.

    [Deprecated]

    Note that this function is deprecated because its name does not reflect
    its behavior. Its functionality is identical to the non-deprecated
    function :py:obj:`~.cudaDeviceGetLimit()`, which should be used
    instead.

    Returns in `*pValue` the current size of `limit`. The supported
    :py:obj:`~.cudaLimit` values are:

    - :py:obj:`~.cudaLimitStackSize`: stack size of each GPU thread;

    - :py:obj:`~.cudaLimitPrintfFifoSize`: size of the shared FIFO used by
      the :py:obj:`~.printf()` device system call.

    - :py:obj:`~.cudaLimitMallocHeapSize`: size of the heap used by the
      :py:obj:`~.malloc()` and :py:obj:`~.free()` device system calls;

    Parameters
    ----------
    limit : :py:obj:`~.cudaLimit`
        Limit to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorUnsupportedLimit`, :py:obj:`~.cudaErrorInvalidValue`
    pValue : int
        Returned size in bytes of limit

    See Also
    --------
    :py:obj:`~.cudaDeviceGetLimit`
    """
    cdef size_t pValue = 0
    cdef ccudart.cudaLimit climit = limit.value
    err = ccudart.cudaThreadGetLimit(&pValue, climit)
    return (cudaError_t(err), pValue)

@cython.embedsignature(True)
def cudaGetLastError():
    """ Returns the last error from a runtime call.

    Returns the last error that has been produced by any of the runtime
    calls in the same host thread and resets it to :py:obj:`~.cudaSuccess`.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorMissingConfiguration`, :py:obj:`~.cudaErrorMemoryAllocation`, :py:obj:`~.cudaErrorInitializationError`, :py:obj:`~.cudaErrorLaunchFailure`, :py:obj:`~.cudaErrorLaunchTimeout`, :py:obj:`~.cudaErrorLaunchOutOfResources`, :py:obj:`~.cudaErrorInvalidDeviceFunction`, :py:obj:`~.cudaErrorInvalidConfiguration`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidSymbol`, :py:obj:`~.cudaErrorUnmapBufferObjectFailed`, :py:obj:`~.cudaErrorInvalidDevicePointer`, :py:obj:`~.cudaErrorInvalidTexture`, :py:obj:`~.cudaErrorInvalidTextureBinding`, :py:obj:`~.cudaErrorInvalidChannelDescriptor`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`, :py:obj:`~.cudaErrorInvalidFilterSetting`, :py:obj:`~.cudaErrorInvalidNormSetting`, :py:obj:`~.cudaErrorUnknown`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorInsufficientDriver`, :py:obj:`~.cudaErrorNoDevice`, :py:obj:`~.cudaErrorSetOnActiveProcess`, :py:obj:`~.cudaErrorStartupFailure`, :py:obj:`~.cudaErrorInvalidPtx`, :py:obj:`~.cudaErrorUnsupportedPtxVersion`, :py:obj:`~.cudaErrorNoKernelImageForDevice`, :py:obj:`~.cudaErrorJitCompilerNotFound`, :py:obj:`~.cudaErrorJitCompilationDisabled`

    See Also
    --------
    :py:obj:`~.cudaPeekAtLastError`, :py:obj:`~.cudaGetErrorName`, :py:obj:`~.cudaGetErrorString`, :py:obj:`~.cudaError`
    """
    err = ccudart.cudaGetLastError()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaPeekAtLastError():
    """ Returns the last error from a runtime call.

    Returns the last error that has been produced by any of the runtime
    calls in the same host thread. Note that this call does not reset the
    error to :py:obj:`~.cudaSuccess` like :py:obj:`~.cudaGetLastError()`.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorMissingConfiguration`, :py:obj:`~.cudaErrorMemoryAllocation`, :py:obj:`~.cudaErrorInitializationError`, :py:obj:`~.cudaErrorLaunchFailure`, :py:obj:`~.cudaErrorLaunchTimeout`, :py:obj:`~.cudaErrorLaunchOutOfResources`, :py:obj:`~.cudaErrorInvalidDeviceFunction`, :py:obj:`~.cudaErrorInvalidConfiguration`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidSymbol`, :py:obj:`~.cudaErrorUnmapBufferObjectFailed`, :py:obj:`~.cudaErrorInvalidDevicePointer`, :py:obj:`~.cudaErrorInvalidTexture`, :py:obj:`~.cudaErrorInvalidTextureBinding`, :py:obj:`~.cudaErrorInvalidChannelDescriptor`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`, :py:obj:`~.cudaErrorInvalidFilterSetting`, :py:obj:`~.cudaErrorInvalidNormSetting`, :py:obj:`~.cudaErrorUnknown`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorInsufficientDriver`, :py:obj:`~.cudaErrorNoDevice`, :py:obj:`~.cudaErrorSetOnActiveProcess`, :py:obj:`~.cudaErrorStartupFailure`, :py:obj:`~.cudaErrorInvalidPtx`, :py:obj:`~.cudaErrorUnsupportedPtxVersion`, :py:obj:`~.cudaErrorNoKernelImageForDevice`, :py:obj:`~.cudaErrorJitCompilerNotFound`, :py:obj:`~.cudaErrorJitCompilationDisabled`

    See Also
    --------
    :py:obj:`~.cudaGetLastError`, :py:obj:`~.cudaGetErrorName`, :py:obj:`~.cudaGetErrorString`, :py:obj:`~.cudaError`
    """
    err = ccudart.cudaPeekAtLastError()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGetErrorName(error not None : cudaError_t):
    """ Returns the string representation of an error code enum name.

    Returns a string containing the name of an error code in the enum. If
    the error code is not recognized, "unrecognized error code" is
    returned.

    Parameters
    ----------
    error : :py:obj:`~.cudaError_t`
        Error code to convert to string

    Returns
    -------
    cudaError_t.cudaSuccess
        cudaError_t.cudaSuccess
    bytes
        `char*` pointer to a NULL-terminated string

    See Also
    --------
    :py:obj:`~.cudaGetErrorString`, :py:obj:`~.cudaGetLastError`, :py:obj:`~.cudaPeekAtLastError`, :py:obj:`~.cudaError`, :py:obj:`~.cuGetErrorName`
    """
    cdef ccudart.cudaError_t cerror = error.value
    err = ccudart.cudaGetErrorName(cerror)
    return (cudaError_t.cudaSuccess, err)

@cython.embedsignature(True)
def cudaGetErrorString(error not None : cudaError_t):
    """ Returns the description string for an error code.

    Returns the description string for an error code. If the error code is
    not recognized, "unrecognized error code" is returned.

    Parameters
    ----------
    error : :py:obj:`~.cudaError_t`
        Error code to convert to string

    Returns
    -------
    cudaError_t.cudaSuccess
        cudaError_t.cudaSuccess
    bytes
        `char*` pointer to a NULL-terminated string

    See Also
    --------
    :py:obj:`~.cudaGetErrorName`, :py:obj:`~.cudaGetLastError`, :py:obj:`~.cudaPeekAtLastError`, :py:obj:`~.cudaError`, :py:obj:`~.cuGetErrorString`
    """
    cdef ccudart.cudaError_t cerror = error.value
    err = ccudart.cudaGetErrorString(cerror)
    return (cudaError_t.cudaSuccess, err)

@cython.embedsignature(True)
def cudaGetDeviceCount():
    """ Returns the number of compute-capable devices.

    Returns in `*count` the number of devices with compute capability
    greater or equal to 2.0 that are available for execution.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`
    count : int
        Returns the number of devices with compute capability greater or
        equal to 2.0

    See Also
    --------
    :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaGetDeviceProperties`, :py:obj:`~.cudaChooseDevice`, :py:obj:`~.cuDeviceGetCount`
    """
    cdef int count = 0
    err = ccudart.cudaGetDeviceCount(&count)
    return (cudaError_t(err), count)

@cython.embedsignature(True)
def cudaGetDeviceProperties(int device):
    """ Returns information about the compute-device.

    Returns in `*prop` the properties of device `dev`. The
    :py:obj:`~.cudaDeviceProp` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.name[256]` is an ASCII string identifying the device;

    - :py:obj:`~.uuid` is a 16-byte unique identifier.

    - :py:obj:`~.totalGlobalMem` is the total amount of global memory
      available on the device in bytes;

    - :py:obj:`~.sharedMemPerBlock` is the maximum amount of shared memory
      available to a thread block in bytes;

    - :py:obj:`~.regsPerBlock` is the maximum number of 32-bit registers
      available to a thread block;

    - :py:obj:`~.warpSize` is the warp size in threads;

    - :py:obj:`~.memPitch` is the maximum pitch in bytes allowed by the
      memory copy functions that involve memory regions allocated through
      :py:obj:`~.cudaMallocPitch()`;

    - :py:obj:`~.maxThreadsPerBlock` is the maximum number of threads per
      block;

    - :py:obj:`~.maxThreadsDim[3]` contains the maximum size of each
      dimension of a block;

    - :py:obj:`~.maxGridSize[3]` contains the maximum size of each
      dimension of a grid;

    - :py:obj:`~.clockRate` is the clock frequency in kilohertz;

    - :py:obj:`~.totalConstMem` is the total amount of constant memory
      available on the device in bytes;

    - :py:obj:`~.major`, :py:obj:`~.minor` are the major and minor revision
      numbers defining the device's compute capability;

    - :py:obj:`~.textureAlignment` is the alignment requirement; texture
      base addresses that are aligned to :py:obj:`~.textureAlignment` bytes
      do not need an offset applied to texture fetches;

    - :py:obj:`~.texturePitchAlignment` is the pitch alignment requirement
      for 2D texture references that are bound to pitched memory;

    - :py:obj:`~.deviceOverlap` is 1 if the device can concurrently copy
      memory between host and device while executing a kernel, or 0 if not.
      Deprecated, use instead asyncEngineCount.

    - :py:obj:`~.multiProcessorCount` is the number of multiprocessors on
      the device;

    - :py:obj:`~.kernelExecTimeoutEnabled` is 1 if there is a run time
      limit for kernels executed on the device, or 0 if not.

    - :py:obj:`~.integrated` is 1 if the device is an integrated
      (motherboard) GPU and 0 if it is a discrete (card) component.

    - :py:obj:`~.canMapHostMemory` is 1 if the device can map host memory
      into the CUDA address space for use with
      :py:obj:`~.cudaHostAlloc()`/:py:obj:`~.cudaHostGetDevicePointer()`,
      or 0 if not;

    - :py:obj:`~.computeMode` is the compute mode that the device is
      currently in. Available modes are as follows:

      - cudaComputeModeDefault: Default mode - Device is not restricted and
        multiple threads can use :py:obj:`~.cudaSetDevice()` with this
        device.

      - cudaComputeModeProhibited: Compute-prohibited mode - No threads can
        use :py:obj:`~.cudaSetDevice()` with this device.

      - cudaComputeModeExclusiveProcess: Compute-exclusive-process mode -
        Many threads in one process will be able to use
        :py:obj:`~.cudaSetDevice()` with this device.   When an occupied
        exclusive mode device is chosen with :py:obj:`~.cudaSetDevice`, all
        subsequent non-device management runtime functions will return
        :py:obj:`~.cudaErrorDevicesUnavailable`.

    - :py:obj:`~.maxTexture1D` is the maximum 1D texture size.

    - :py:obj:`~.maxTexture1DMipmap` is the maximum 1D mipmapped texture
      texture size.

    - :py:obj:`~.maxTexture1DLinear` is the maximum 1D texture size for
      textures bound to linear memory.

    - :py:obj:`~.maxTexture2D[2]` contains the maximum 2D texture
      dimensions.

    - :py:obj:`~.maxTexture2DMipmap[2]` contains the maximum 2D mipmapped
      texture dimensions.

    - :py:obj:`~.maxTexture2DLinear[3]` contains the maximum 2D texture
      dimensions for 2D textures bound to pitch linear memory.

    - :py:obj:`~.maxTexture2DGather[2]` contains the maximum 2D texture
      dimensions if texture gather operations have to be performed.

    - :py:obj:`~.maxTexture3D[3]` contains the maximum 3D texture
      dimensions.

    - :py:obj:`~.maxTexture3DAlt[3]` contains the maximum alternate 3D
      texture dimensions.

    - :py:obj:`~.maxTextureCubemap` is the maximum cubemap texture width or
      height.

    - :py:obj:`~.maxTexture1DLayered[2]` contains the maximum 1D layered
      texture dimensions.

    - :py:obj:`~.maxTexture2DLayered[3]` contains the maximum 2D layered
      texture dimensions.

    - :py:obj:`~.maxTextureCubemapLayered[2]` contains the maximum cubemap
      layered texture dimensions.

    - :py:obj:`~.maxSurface1D` is the maximum 1D surface size.

    - :py:obj:`~.maxSurface2D[2]` contains the maximum 2D surface
      dimensions.

    - :py:obj:`~.maxSurface3D[3]` contains the maximum 3D surface
      dimensions.

    - :py:obj:`~.maxSurface1DLayered[2]` contains the maximum 1D layered
      surface dimensions.

    - :py:obj:`~.maxSurface2DLayered[3]` contains the maximum 2D layered
      surface dimensions.

    - :py:obj:`~.maxSurfaceCubemap` is the maximum cubemap surface width or
      height.

    - :py:obj:`~.maxSurfaceCubemapLayered[2]` contains the maximum cubemap
      layered surface dimensions.

    - :py:obj:`~.surfaceAlignment` specifies the alignment requirements for
      surfaces.

    - :py:obj:`~.concurrentKernels` is 1 if the device supports executing
      multiple kernels within the same context simultaneously, or 0 if not.
      It is not guaranteed that multiple kernels will be resident on the
      device concurrently so this feature should not be relied upon for
      correctness;

    - :py:obj:`~.ECCEnabled` is 1 if the device has ECC support turned on,
      or 0 if not.

    - :py:obj:`~.pciBusID` is the PCI bus identifier of the device.

    - :py:obj:`~.pciDeviceID` is the PCI device (sometimes called slot)
      identifier of the device.

    - :py:obj:`~.pciDomainID` is the PCI domain identifier of the device.

    - :py:obj:`~.tccDriver` is 1 if the device is using a TCC driver or 0
      if not.

    - :py:obj:`~.asyncEngineCount` is 1 when the device can concurrently
      copy memory between host and device while executing a kernel. It is 2
      when the device can concurrently copy memory between host and device
      in both directions and execute a kernel at the same time. It is 0 if
      neither of these is supported.

    - :py:obj:`~.unifiedAddressing` is 1 if the device shares a unified
      address space with the host and 0 otherwise.

    - :py:obj:`~.memoryClockRate` is the peak memory clock frequency in
      kilohertz.

    - :py:obj:`~.memoryBusWidth` is the memory bus width   in bits.

    - :py:obj:`~.l2CacheSize` is L2 cache size in bytes.

    - :py:obj:`~.persistingL2CacheMaxSize` is L2 cache's maximum persisting
      lines size in bytes.

    - :py:obj:`~.maxThreadsPerMultiProcessor`   is the number of maximum
      resident threads per multiprocessor.

    - :py:obj:`~.streamPrioritiesSupported` is 1 if the device supports
      stream priorities, or 0 if it is not supported.

    - :py:obj:`~.globalL1CacheSupported` is 1 if the device supports
      caching of globals in L1 cache, or 0 if it is not supported.

    - :py:obj:`~.localL1CacheSupported` is 1 if the device supports caching
      of locals in L1 cache, or 0 if it is not supported.

    - :py:obj:`~.sharedMemPerMultiprocessor` is the maximum amount of
      shared memory available to a multiprocessor in bytes; this amount is
      shared by all thread blocks simultaneously resident on a
      multiprocessor;

    - :py:obj:`~.regsPerMultiprocessor` is the maximum number of 32-bit
      registers available to a multiprocessor; this number is shared by all
      thread blocks simultaneously resident on a multiprocessor;

    - :py:obj:`~.managedMemory` is 1 if the device supports allocating
      managed memory on this system, or 0 if it is not supported.

    - :py:obj:`~.isMultiGpuBoard` is 1 if the device is on a multi-GPU
      board (e.g. Gemini cards), and 0 if not;

    - :py:obj:`~.multiGpuBoardGroupID` is a unique identifier for a group
      of devices associated with the same board. Devices on the same multi-
      GPU board will share the same identifier;

    - :py:obj:`~.singleToDoublePrecisionPerfRatio`   is the ratio of single
      precision performance (in floating-point operations per second) to
      double precision performance.

    - :py:obj:`~.pageableMemoryAccess` is 1 if the device supports
      coherently accessing pageable memory without calling cudaHostRegister
      on it, and 0 otherwise.

    - :py:obj:`~.concurrentManagedAccess` is 1 if the device can coherently
      access managed memory concurrently with the CPU, and 0 otherwise.

    - :py:obj:`~.computePreemptionSupported` is 1 if the device supports
      Compute Preemption, and 0 otherwise.

    - :py:obj:`~.canUseHostPointerForRegisteredMem` is 1 if the device can
      access host registered memory at the same virtual address as the CPU,
      and 0 otherwise.

    - :py:obj:`~.cooperativeLaunch` is 1 if the device supports launching
      cooperative kernels via :py:obj:`~.cudaLaunchCooperativeKernel`, and
      0 otherwise.

    - :py:obj:`~.cooperativeMultiDeviceLaunch` is 1 if the device supports
      launching cooperative kernels via
      :py:obj:`~.cudaLaunchCooperativeKernelMultiDevice`, and 0 otherwise.

    - :py:obj:`~.pageableMemoryAccessUsesHostPageTables` is 1 if the device
      accesses pageable memory via the host's page tables, and 0 otherwise.

    - :py:obj:`~.directManagedMemAccessFromHost` is 1 if the host can
      directly access managed memory on the device without migration, and 0
      otherwise.

    - :py:obj:`~.maxBlocksPerMultiProcessor` is the maximum number of
      thread blocks that can reside on a multiprocessor.

    - :py:obj:`~.accessPolicyMaxWindowSize` is the maximum value of
      :py:obj:`~.cudaAccessPolicyWindow.num_bytes`.

    Parameters
    ----------
    device : int
        Device number to get properties for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`
    prop : :py:obj:`~.cudaDeviceProp`
        Properties for the specified device

    See Also
    --------
    :py:obj:`~.cudaGetDeviceCount`, :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaChooseDevice`, :py:obj:`~.cudaDeviceGetAttribute`, :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetName`
    """
    cdef cudaDeviceProp prop = cudaDeviceProp()
    err = ccudart.cudaGetDeviceProperties(prop._ptr, device)
    return (cudaError_t(err), prop)

@cython.embedsignature(True)
def cudaDeviceGetAttribute(attr not None : cudaDeviceAttr, int device):
    """ Returns information about the device.

    Returns in `*value` the integer value of the attribute `attr` on device
    `device`. The supported attributes are:

    - :py:obj:`~.cudaDevAttrMaxThreadsPerBlock`: Maximum number of threads
      per block

    - :py:obj:`~.cudaDevAttrMaxBlockDimX`: Maximum x-dimension of a block

    - :py:obj:`~.cudaDevAttrMaxBlockDimY`: Maximum y-dimension of a block

    - :py:obj:`~.cudaDevAttrMaxBlockDimZ`: Maximum z-dimension of a block

    - :py:obj:`~.cudaDevAttrMaxGridDimX`: Maximum x-dimension of a grid

    - :py:obj:`~.cudaDevAttrMaxGridDimY`: Maximum y-dimension of a grid

    - :py:obj:`~.cudaDevAttrMaxGridDimZ`: Maximum z-dimension of a grid

    - :py:obj:`~.cudaDevAttrMaxSharedMemoryPerBlock`: Maximum amount of
      shared memory available to a thread block in bytes

    - :py:obj:`~.cudaDevAttrTotalConstantMemory`: Memory available on
      device for constant variables in a CUDA C kernel in bytes

    - :py:obj:`~.cudaDevAttrWarpSize`: Warp size in threads

    - :py:obj:`~.cudaDevAttrMaxPitch`: Maximum pitch in bytes allowed by
      the memory copy functions that involve memory regions allocated
      through :py:obj:`~.cudaMallocPitch()`

    - :py:obj:`~.cudaDevAttrMaxTexture1DWidth`: Maximum 1D texture width

    - :py:obj:`~.cudaDevAttrMaxTexture1DLinearWidth`: Maximum width for a
      1D texture bound to linear memory

    - :py:obj:`~.cudaDevAttrMaxTexture1DMipmappedWidth`: Maximum mipmapped
      1D texture width

    - :py:obj:`~.cudaDevAttrMaxTexture2DWidth`: Maximum 2D texture width

    - :py:obj:`~.cudaDevAttrMaxTexture2DHeight`: Maximum 2D texture height

    - :py:obj:`~.cudaDevAttrMaxTexture2DLinearWidth`: Maximum width for a
      2D texture bound to linear memory

    - :py:obj:`~.cudaDevAttrMaxTexture2DLinearHeight`: Maximum height for a
      2D texture bound to linear memory

    - :py:obj:`~.cudaDevAttrMaxTexture2DLinearPitch`: Maximum pitch in
      bytes for a 2D texture bound to linear memory

    - :py:obj:`~.cudaDevAttrMaxTexture2DMipmappedWidth`: Maximum mipmapped
      2D texture width

    - :py:obj:`~.cudaDevAttrMaxTexture2DMipmappedHeight`: Maximum mipmapped
      2D texture height

    - :py:obj:`~.cudaDevAttrMaxTexture3DWidth`: Maximum 3D texture width

    - :py:obj:`~.cudaDevAttrMaxTexture3DHeight`: Maximum 3D texture height

    - :py:obj:`~.cudaDevAttrMaxTexture3DDepth`: Maximum 3D texture depth

    - :py:obj:`~.cudaDevAttrMaxTexture3DWidthAlt`: Alternate maximum 3D
      texture width, 0 if no alternate maximum 3D texture size is supported

    - :py:obj:`~.cudaDevAttrMaxTexture3DHeightAlt`: Alternate maximum 3D
      texture height, 0 if no alternate maximum 3D texture size is
      supported

    - :py:obj:`~.cudaDevAttrMaxTexture3DDepthAlt`: Alternate maximum 3D
      texture depth, 0 if no alternate maximum 3D texture size is supported

    - :py:obj:`~.cudaDevAttrMaxTextureCubemapWidth`: Maximum cubemap
      texture width or height

    - :py:obj:`~.cudaDevAttrMaxTexture1DLayeredWidth`: Maximum 1D layered
      texture width

    - :py:obj:`~.cudaDevAttrMaxTexture1DLayeredLayers`: Maximum layers in a
      1D layered texture

    - :py:obj:`~.cudaDevAttrMaxTexture2DLayeredWidth`: Maximum 2D layered
      texture width

    - :py:obj:`~.cudaDevAttrMaxTexture2DLayeredHeight`: Maximum 2D layered
      texture height

    - :py:obj:`~.cudaDevAttrMaxTexture2DLayeredLayers`: Maximum layers in a
      2D layered texture

    - :py:obj:`~.cudaDevAttrMaxTextureCubemapLayeredWidth`: Maximum cubemap
      layered texture width or height

    - :py:obj:`~.cudaDevAttrMaxTextureCubemapLayeredLayers`: Maximum layers
      in a cubemap layered texture

    - :py:obj:`~.cudaDevAttrMaxSurface1DWidth`: Maximum 1D surface width

    - :py:obj:`~.cudaDevAttrMaxSurface2DWidth`: Maximum 2D surface width

    - :py:obj:`~.cudaDevAttrMaxSurface2DHeight`: Maximum 2D surface height

    - :py:obj:`~.cudaDevAttrMaxSurface3DWidth`: Maximum 3D surface width

    - :py:obj:`~.cudaDevAttrMaxSurface3DHeight`: Maximum 3D surface height

    - :py:obj:`~.cudaDevAttrMaxSurface3DDepth`: Maximum 3D surface depth

    - :py:obj:`~.cudaDevAttrMaxSurface1DLayeredWidth`: Maximum 1D layered
      surface width

    - :py:obj:`~.cudaDevAttrMaxSurface1DLayeredLayers`: Maximum layers in a
      1D layered surface

    - :py:obj:`~.cudaDevAttrMaxSurface2DLayeredWidth`: Maximum 2D layered
      surface width

    - :py:obj:`~.cudaDevAttrMaxSurface2DLayeredHeight`: Maximum 2D layered
      surface height

    - :py:obj:`~.cudaDevAttrMaxSurface2DLayeredLayers`: Maximum layers in a
      2D layered surface

    - :py:obj:`~.cudaDevAttrMaxSurfaceCubemapWidth`: Maximum cubemap
      surface width

    - :py:obj:`~.cudaDevAttrMaxSurfaceCubemapLayeredWidth`: Maximum cubemap
      layered surface width

    - :py:obj:`~.cudaDevAttrMaxSurfaceCubemapLayeredLayers`: Maximum layers
      in a cubemap layered surface

    - :py:obj:`~.cudaDevAttrMaxRegistersPerBlock`: Maximum number of 32-bit
      registers available to a thread block

    - :py:obj:`~.cudaDevAttrClockRate`: Peak clock frequency in kilohertz

    - :py:obj:`~.cudaDevAttrTextureAlignment`: Alignment requirement;
      texture base addresses aligned to :py:obj:`~.textureAlign` bytes do
      not need an offset applied to texture fetches

    - :py:obj:`~.cudaDevAttrTexturePitchAlignment`: Pitch alignment
      requirement for 2D texture references bound to pitched memory

    - :py:obj:`~.cudaDevAttrGpuOverlap`: 1 if the device can concurrently
      copy memory between host and device while executing a kernel, or 0 if
      not

    - :py:obj:`~.cudaDevAttrMultiProcessorCount`: Number of multiprocessors
      on the device

    - :py:obj:`~.cudaDevAttrKernelExecTimeout`: 1 if there is a run time
      limit for kernels executed on the device, or 0 if not

    - :py:obj:`~.cudaDevAttrIntegrated`: 1 if the device is integrated with
      the memory subsystem, or 0 if not

    - :py:obj:`~.cudaDevAttrCanMapHostMemory`: 1 if the device can map host
      memory into the CUDA address space, or 0 if not

    - :py:obj:`~.cudaDevAttrComputeMode`: Compute mode is the compute mode
      that the device is currently in. Available modes are as follows:

      - :py:obj:`~.cudaComputeModeDefault`: Default mode - Device is not
        restricted and multiple threads can use :py:obj:`~.cudaSetDevice()`
        with this device.

      - :py:obj:`~.cudaComputeModeProhibited`: Compute-prohibited mode - No
        threads can use :py:obj:`~.cudaSetDevice()` with this device.

      - :py:obj:`~.cudaComputeModeExclusiveProcess`: Compute-exclusive-
        process mode - Many threads in one process will be able to use
        :py:obj:`~.cudaSetDevice()` with this device.

    - :py:obj:`~.cudaDevAttrConcurrentKernels`: 1 if the device supports
      executing multiple kernels within the same context simultaneously, or
      0 if not. It is not guaranteed that multiple kernels will be resident
      on the device concurrently so this feature should not be relied upon
      for correctness.

    - :py:obj:`~.cudaDevAttrEccEnabled`: 1 if error correction is enabled
      on the device, 0 if error correction is disabled or not supported by
      the device

    - :py:obj:`~.cudaDevAttrPciBusId`: PCI bus identifier of the device

    - :py:obj:`~.cudaDevAttrPciDeviceId`: PCI device (also known as slot)
      identifier of the device

    - :py:obj:`~.cudaDevAttrTccDriver`: 1 if the device is using a TCC
      driver. TCC is only available on Tesla hardware running Windows Vista
      or later.

    - :py:obj:`~.cudaDevAttrMemoryClockRate`: Peak memory clock frequency
      in kilohertz

    - :py:obj:`~.cudaDevAttrGlobalMemoryBusWidth`: Global memory bus width
      in bits

    - :py:obj:`~.cudaDevAttrL2CacheSize`: Size of L2 cache in bytes. 0 if
      the device doesn't have L2 cache.

    - :py:obj:`~.cudaDevAttrMaxThreadsPerMultiProcessor`: Maximum resident
      threads per multiprocessor

    - :py:obj:`~.cudaDevAttrUnifiedAddressing`: 1 if the device shares a
      unified address space with the host, or 0 if not

    - :py:obj:`~.cudaDevAttrComputeCapabilityMajor`: Major compute
      capability version number

    - :py:obj:`~.cudaDevAttrComputeCapabilityMinor`: Minor compute
      capability version number

    - :py:obj:`~.cudaDevAttrStreamPrioritiesSupported`: 1 if the device
      supports stream priorities, or 0 if not

    - :py:obj:`~.cudaDevAttrGlobalL1CacheSupported`: 1 if device supports
      caching globals in L1 cache, 0 if not

    - :py:obj:`~.cudaDevAttrLocalL1CacheSupported`: 1 if device supports
      caching locals in L1 cache, 0 if not

    - :py:obj:`~.cudaDevAttrMaxSharedMemoryPerMultiprocessor`: Maximum
      amount of shared memory available to a multiprocessor in bytes; this
      amount is shared by all thread blocks simultaneously resident on a
      multiprocessor

    - :py:obj:`~.cudaDevAttrMaxRegistersPerMultiprocessor`: Maximum number
      of 32-bit registers available to a multiprocessor; this number is
      shared by all thread blocks simultaneously resident on a
      multiprocessor

    - :py:obj:`~.cudaDevAttrManagedMemory`: 1 if device supports allocating
      managed memory, 0 if not

    - :py:obj:`~.cudaDevAttrIsMultiGpuBoard`: 1 if device is on a multi-GPU
      board, 0 if not

    - :py:obj:`~.cudaDevAttrMultiGpuBoardGroupID`: Unique identifier for a
      group of devices on the same multi-GPU board

    - :py:obj:`~.cudaDevAttrHostNativeAtomicSupported`: 1 if the link
      between the device and the host supports native atomic operations

    - :py:obj:`~.cudaDevAttrSingleToDoublePrecisionPerfRatio`: Ratio of
      single precision performance (in floating-point operations per
      second) to double precision performance

    - :py:obj:`~.cudaDevAttrPageableMemoryAccess`: 1 if the device supports
      coherently accessing pageable memory without calling cudaHostRegister
      on it, and 0 otherwise

    - :py:obj:`~.cudaDevAttrConcurrentManagedAccess`: 1 if the device can
      coherently access managed memory concurrently with the CPU, and 0
      otherwise

    - :py:obj:`~.cudaDevAttrComputePreemptionSupported`: 1 if the device
      supports Compute Preemption, 0 if not

    - :py:obj:`~.cudaDevAttrCanUseHostPointerForRegisteredMem`: 1 if the
      device can access host registered memory at the same virtual address
      as the CPU, and 0 otherwise

    - :py:obj:`~.cudaDevAttrCooperativeLaunch`: 1 if the device supports
      launching cooperative kernels via
      :py:obj:`~.cudaLaunchCooperativeKernel`, and 0 otherwise

    - :py:obj:`~.cudaDevAttrCooperativeMultiDeviceLaunch`: 1 if the device
      supports launching cooperative kernels via
      :py:obj:`~.cudaLaunchCooperativeKernelMultiDevice`, and 0 otherwise

    - :py:obj:`~.cudaDevAttrCanFlushRemoteWrites`: 1 if the device supports
      flushing of outstanding remote writes, and 0 otherwise

    - :py:obj:`~.cudaDevAttrHostRegisterSupported`: 1 if the device
      supports host memory registration via :py:obj:`~.cudaHostRegister`,
      and 0 otherwise

    - :py:obj:`~.cudaDevAttrPageableMemoryAccessUsesHostPageTables`: 1 if
      the device accesses pageable memory via the host's page tables, and 0
      otherwise

    - :py:obj:`~.cudaDevAttrDirectManagedMemAccessFromHost`: 1 if the host
      can directly access managed memory on the device without migration,
      and 0 otherwise

    - :py:obj:`~.cudaDevAttrMaxSharedMemoryPerBlockOptin`: Maximum per
      block shared memory size on the device. This value can be opted into
      when using :py:obj:`~.cudaFuncSetAttribute`

    - :py:obj:`~.cudaDevAttrMaxBlocksPerMultiprocessor`: Maximum number of
      thread blocks that can reside on a multiprocessor

    - :py:obj:`~.cudaDevAttrMaxPersistingL2CacheSize`: Maximum L2
      persisting lines capacity setting in bytes

    - :py:obj:`~.cudaDevAttrMaxAccessPolicyWindowSize`: Maximum value of
      :py:obj:`~.cudaAccessPolicyWindow.num_bytes`

    - :py:obj:`~.cudaDevAttrReservedSharedMemoryPerBlock`: Shared memory
      reserved by CUDA driver per block in bytes

    - :py:obj:`~.cudaDevAttrSparseCudaArraySupported`: 1 if the device
      supports sparse CUDA arrays and sparse CUDA mipmapped arrays.

    - :py:obj:`~.cudaDevAttrHostRegisterReadOnlySupported`: Device supports
      using the :py:obj:`~.cudaHostRegister` flag cudaHostRegisterReadOnly
      to register memory that must be mapped as read-only to the GPU

    - :py:obj:`~.cudaDevAttrMemoryPoolsSupported`: 1 if the device supports
      using the cudaMallocAsync and cudaMemPool family of APIs, and 0
      otherwise

    - :py:obj:`~.cudaDevAttrGPUDirectRDMASupported`: 1 if the device
      supports GPUDirect RDMA APIs, and 0 otherwise

    - :py:obj:`~.cudaDevAttrGPUDirectRDMAFlushWritesOptions`: bitmask to be
      interpreted according to the
      :py:obj:`~.cudaFlushGPUDirectRDMAWritesOptions` enum

    - :py:obj:`~.cudaDevAttrGPUDirectRDMAWritesOrdering`: see the
      :py:obj:`~.cudaGPUDirectRDMAWritesOrdering` enum for numerical values

    - :py:obj:`~.cudaDevAttrMemoryPoolSupportedHandleTypes`: Bitmask of
      handle types supported with mempool based IPC

    - :py:obj:`~.cudaDevAttrDeferredMappingCudaArraySupported` : 1 if the
      device supports deferred mapping CUDA arrays and CUDA mipmapped
      arrays.

    Parameters
    ----------
    attr : :py:obj:`~.cudaDeviceAttr`
        Device attribute to query
    device : int
        Device number to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`
    value : int
        Returned device attribute value

    See Also
    --------
    :py:obj:`~.cudaGetDeviceCount`, :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaChooseDevice`, :py:obj:`~.cudaGetDeviceProperties`, :py:obj:`~.cuDeviceGetAttribute`
    """
    cdef int value = 0
    cdef ccudart.cudaDeviceAttr cattr = attr.value
    err = ccudart.cudaDeviceGetAttribute(&value, cattr, device)
    return (cudaError_t(err), value)

@cython.embedsignature(True)
def cudaDeviceGetDefaultMemPool(int device):
    """ Returns the default mempool of a device.

    The default mempool of a device contains device memory from that
    device.

    Parameters
    ----------
    device : int
        None

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue` :py:obj:`~.cudaErrorNotSupported`
    memPool : :py:obj:`~.cudaMemPool_t`
        None

    See Also
    --------
    :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaMemPoolTrimTo`, :py:obj:`~.cudaMemPoolGetAttribute`, :py:obj:`~.cudaDeviceSetMemPool`, :py:obj:`~.cudaMemPoolSetAttribute`, :py:obj:`~.cudaMemPoolSetAccess`
    """
    cdef cudaMemPool_t memPool = cudaMemPool_t()
    with nogil:
        err = ccudart.cudaDeviceGetDefaultMemPool(<ccudart.cudaMemPool_t*>memPool._ptr, device)
    return (cudaError_t(err), memPool)

@cython.embedsignature(True)
def cudaDeviceSetMemPool(int device, memPool):
    """ Sets the current memory pool of a device.

    The memory pool must be local to the specified device. Unless a mempool
    is specified in the :py:obj:`~.cudaMallocAsync` call,
    :py:obj:`~.cudaMallocAsync` allocates from the current mempool of the
    provided stream's device. By default, a device's current memory pool is
    its default memory pool.

    Parameters
    ----------
    device : int
        None
    memPool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        None

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue` :py:obj:`~.cudaErrorInvalidDevice` :py:obj:`~.cudaErrorNotSupported`

    See Also
    --------
    :py:obj:`~.cuDeviceSetDefaultMemPool`, :py:obj:`~.cudaDeviceGetMemPool`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaMemPoolCreate`, :py:obj:`~.cudaMemPoolDestroy`, :py:obj:`~.cudaMallocFromPoolAsync`

    Notes
    -----
    Use :py:obj:`~.cudaMallocFromPoolAsync` to specify asynchronous allocations from a device different than the one the stream runs on.
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    with nogil:
        err = ccudart.cudaDeviceSetMemPool(device, cmemPool)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceGetMemPool(int device):
    """ Gets the current mempool for a device.

    Returns the last pool provided to :py:obj:`~.cudaDeviceSetMemPool` for
    this device or the device's default memory pool if
    :py:obj:`~.cudaDeviceSetMemPool` has never been called. By default the
    current mempool is the default mempool for a device, otherwise the
    returned pool must have been set with :py:obj:`~.cuDeviceSetMemPool` or
    :py:obj:`~.cudaDeviceSetMemPool`.

    Parameters
    ----------
    device : int
        None

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue` :py:obj:`~.cudaErrorNotSupported`
    memPool : :py:obj:`~.cudaMemPool_t`
        None

    See Also
    --------
    :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaDeviceSetMemPool`
    """
    cdef cudaMemPool_t memPool = cudaMemPool_t()
    with nogil:
        err = ccudart.cudaDeviceGetMemPool(<ccudart.cudaMemPool_t*>memPool._ptr, device)
    return (cudaError_t(err), memPool)

@cython.embedsignature(True)
def cudaDeviceGetNvSciSyncAttributes(int device, int flags):
    """ Return NvSciSync attributes that this device can support.

    Returns in `nvSciSyncAttrList`, the properties of NvSciSync that this
    CUDA device, `dev` can support. The returned `nvSciSyncAttrList` can be
    used to create an NvSciSync that matches this device's capabilities.

    If NvSciSyncAttrKey_RequiredPerm field in `nvSciSyncAttrList` is
    already set this API will return :py:obj:`~.cudaErrorInvalidValue`.

    The applications should set `nvSciSyncAttrList` to a valid
    NvSciSyncAttrList failing which this API will return
    :py:obj:`~.cudaErrorInvalidHandle`.

    The `flags` controls how applications intends to use the NvSciSync
    created from the `nvSciSyncAttrList`. The valid flags are:

    - :py:obj:`~.cudaNvSciSyncAttrSignal`, specifies that the applications
      intends to signal an NvSciSync on this CUDA device.

    - :py:obj:`~.cudaNvSciSyncAttrWait`, specifies that the applications
      intends to wait on an NvSciSync on this CUDA device.

    At least one of these flags must be set, failing which the API returns
    :py:obj:`~.cudaErrorInvalidValue`. Both the flags are orthogonal to one
    another: a developer may set both these flags that allows to set both
    wait and signal specific attributes in the same `nvSciSyncAttrList`.

    :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorDeviceUninitialized`,
    :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidHandle`,
    :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorNotSupported`,
    :py:obj:`~.cudaErrorMemoryAllocation`

    Parameters
    ----------
    device : int
        Valid Cuda Device to get NvSciSync attributes for.
    flags : int
        flags describing NvSciSync usage.

    Returns
    -------
    cudaError_t

    nvSciSyncAttrList : Any
        Return NvSciSync attributes supported.

    See Also
    --------
    :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaDestroyExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef void_ptr nvSciSyncAttrList = 0
    cdef void* cnvSciSyncAttrList_ptr = <void*>nvSciSyncAttrList
    err = ccudart.cudaDeviceGetNvSciSyncAttributes(cnvSciSyncAttrList_ptr, device, flags)
    return (cudaError_t(err), nvSciSyncAttrList)

@cython.embedsignature(True)
def cudaDeviceGetP2PAttribute(attr not None : cudaDeviceP2PAttr, int srcDevice, int dstDevice):
    """ Queries attributes of the link between two devices.

    Returns in `*value` the value of the requested attribute `attrib` of
    the link between `srcDevice` and `dstDevice`. The supported attributes
    are:

    - :py:obj:`~.cudaDevP2PAttrPerformanceRank`: A relative value
      indicating the performance of the link between two devices. Lower
      value means better performance (0 being the value used for most
      performant link).

    - :py:obj:`~.cudaDevP2PAttrAccessSupported`: 1 if peer access is
      enabled.

    - :py:obj:`~.cudaDevP2PAttrNativeAtomicSupported`: 1 if native atomic
      operations over the link are supported.

    - :py:obj:`~.cudaDevP2PAttrCudaArrayAccessSupported`: 1 if accessing
      CUDA arrays over the link is supported.

    Returns :py:obj:`~.cudaErrorInvalidDevice` if `srcDevice` or
    `dstDevice` are not valid or if they represent the same device.

    Returns :py:obj:`~.cudaErrorInvalidValue` if `attrib` is not valid or
    if `value` is a null pointer.

    Parameters
    ----------
    attrib : :py:obj:`~.cudaDeviceP2PAttr`
        The requested attribute of the link between `srcDevice` and
        `dstDevice`.
    srcDevice : int
        The source device of the target link.
    dstDevice : int
        The destination device of the target link.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`
    value : int
        Returned value of the requested attribute

    See Also
    --------
    :py:obj:`~.cudaCtxEnablePeerAccess`, :py:obj:`~.cudaCtxDisablePeerAccess`, :py:obj:`~.cudaCtxCanAccessPeer`, :py:obj:`~.cuDeviceGetP2PAttribute`
    """
    cdef int value = 0
    cdef ccudart.cudaDeviceP2PAttr cattr = attr.value
    err = ccudart.cudaDeviceGetP2PAttribute(&value, cattr, srcDevice, dstDevice)
    return (cudaError_t(err), value)

@cython.embedsignature(True)
def cudaChooseDevice(prop : cudaDeviceProp):
    """ Select compute-device which best matches criteria.

    Returns in `*device` the device which has properties that best match
    `*prop`.

    Parameters
    ----------
    prop : :py:obj:`~.cudaDeviceProp`
        Desired device properties

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    device : int
        Device with best match

    See Also
    --------
    :py:obj:`~.cudaGetDeviceCount`, :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef int device = 0
    cdef ccudart.cudaDeviceProp* cprop_ptr = prop._ptr if prop != None else NULL
    err = ccudart.cudaChooseDevice(&device, cprop_ptr)
    return (cudaError_t(err), device)

@cython.embedsignature(True)
def cudaSetDevice(int device):
    """ Set device to be used for GPU executions.

    Sets `device` as the current device for the calling host thread. Valid
    device id's are 0 to (:py:obj:`~.cudaGetDeviceCount()` - 1).

    Any device memory subsequently allocated from this host thread using
    :py:obj:`~.cudaMalloc()`, :py:obj:`~.cudaMallocPitch()` or
    :py:obj:`~.cudaMallocArray()` will be physically resident on `device`.
    Any host memory allocated from this host thread using
    :py:obj:`~.cudaMallocHost()` or :py:obj:`~.cudaHostAlloc()` or
    :py:obj:`~.cudaHostRegister()` will have its lifetime associated with
    `device`. Any streams or events created from this host thread will be
    associated with `device`. Any kernels launched from this host thread
    using the <<<>>> operator or :py:obj:`~.cudaLaunchKernel()` will be
    executed on `device`.

    This call may be made from any host thread, to any device, and at any
    time. This function will do no synchronization with the previous or new
    device, and should be considered a very low overhead call. If the
    current context bound to the calling thread is not the primary context,
    this call will bind the primary context to the calling thread and all
    the subsequent memory allocations, stream and event creations, and
    kernel launches will be associated with the primary context. This
    function will not initialize the context until a runtime API requiring
    the context (such as :py:obj:`~.cudaMalloc()`) is used. This function
    will not return an error if the device is in
    :py:obj:`~.cudaComputeModeExclusiveProcess` and is occupied by another
    process or if the device is in :py:obj:`~.cudaComputeModeProhibited`.

    Parameters
    ----------
    device : int
        Device on which the active host thread should execute the device
        code.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaGetDeviceCount`, :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaGetDeviceProperties`, :py:obj:`~.cudaChooseDevice`, :py:obj:`~.cuCtxSetCurrent`
    """
    err = ccudart.cudaSetDevice(device)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGetDevice():
    """ Returns which device is currently being used.

    Returns in `*device` the current device for the calling host thread.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    device : int
        Returns the device on which the active host thread executes the
        device code.

    See Also
    --------
    :py:obj:`~.cudaGetDeviceCount`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaGetDeviceProperties`, :py:obj:`~.cudaChooseDevice`, :py:obj:`~.cuCtxGetCurrent`
    """
    cdef int device = 0
    err = ccudart.cudaGetDevice(&device)
    return (cudaError_t(err), device)

@cython.embedsignature(True)
def cudaSetDeviceFlags(unsigned int flags):
    """ Sets flags to be used for device executions.

    Records `flags` as the flags for the current device. If the current
    device has been set and that device has already been initialized, the
    previous flags are overwritten. If the current device has not been
    initialized, it is initialized with the provided flags. If no device
    has been made current to the calling thread, a default device is
    selected and initialized with the provided flags.

    The two LSBs of the `flags` parameter can be used to control how the
    CPU thread interacts with the OS scheduler when waiting for results
    from the device.

    - :py:obj:`~.cudaDeviceScheduleAuto`: The default value if the `flags`
      parameter is zero, uses a heuristic based on the number of active
      CUDA contexts in the process `C` and the number of logical processors
      in the system `P`. If `C` > `P`, then CUDA will yield to other OS
      threads when waiting for the device, otherwise CUDA will not yield
      while waiting for results and actively spin on the processor.
      Additionally, on Tegra devices, :py:obj:`~.cudaDeviceScheduleAuto`
      uses a heuristic based on the power profile of the platform and may
      choose :py:obj:`~.cudaDeviceScheduleBlockingSync` for low-powered
      devices.

    - :py:obj:`~.cudaDeviceScheduleSpin`: Instruct CUDA to actively spin
      when waiting for results from the device. This can decrease latency
      when waiting for the device, but may lower the performance of CPU
      threads if they are performing work in parallel with the CUDA thread.

    - :py:obj:`~.cudaDeviceScheduleYield`: Instruct CUDA to yield its
      thread when waiting for results from the device. This can increase
      latency when waiting for the device, but can increase the performance
      of CPU threads performing work in parallel with the device.

    - :py:obj:`~.cudaDeviceScheduleBlockingSync`: Instruct CUDA to block
      the CPU thread on a synchronization primitive when waiting for the
      device to finish work.

    - :py:obj:`~.cudaDeviceBlockingSync`: Instruct CUDA to block the CPU
      thread on a synchronization primitive when waiting for the device to
      finish work.   :py:obj:`~.Deprecated:` This flag was deprecated as of
      CUDA 4.0 and replaced with
      :py:obj:`~.cudaDeviceScheduleBlockingSync`.

    - :py:obj:`~.cudaDeviceMapHost`: This flag enables allocating pinned
      host memory that is accessible to the device. It is implicit for the
      runtime but may be absent if a context is created using the driver
      API. If this flag is not set, :py:obj:`~.cudaHostGetDevicePointer()`
      will always return a failure code.

    - :py:obj:`~.cudaDeviceLmemResizeToMax`: Instruct CUDA to not reduce
      local memory after resizing local memory for a kernel. This can
      prevent thrashing by local memory allocations when launching many
      kernels with high local memory usage at the cost of potentially
      increased memory usage.   :py:obj:`~.Deprecated:` This flag is
      deprecated and the behavior enabled   by this flag is now the default
      and cannot be disabled.

    Parameters
    ----------
    flags : unsigned int
        Parameters for device operation

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGetDeviceFlags`, :py:obj:`~.cudaGetDeviceCount`, :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaGetDeviceProperties`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaSetValidDevices`, :py:obj:`~.cudaChooseDevice`, :py:obj:`~.cuDevicePrimaryCtxSetFlags`
    """
    err = ccudart.cudaSetDeviceFlags(flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGetDeviceFlags():
    """ Gets the flags for the current device.

    Returns in `flags` the flags for the current device. If there is a
    current device for the calling thread, the flags for the device are
    returned. If there is no current device, the flags for the first device
    are returned, which may be the default flags. Compare to the behavior
    of :py:obj:`~.cudaSetDeviceFlags`.

    Typically, the flags returned should match the behavior that will be
    seen if the calling thread uses a device after this call, without any
    change to the flags or current device inbetween by this or another
    thread. Note that if the device is not initialized, it is possible for
    another thread to change the flags for the current device before it is
    initialized. Additionally, when using exclusive mode, if this thread
    has not requested a specific device, it may use a device other than the
    first device, contrary to the assumption made by this function.

    If a context has been created via the driver API and is current to the
    calling thread, the flags for that context are always returned.

    Flags returned by this function may specifically include
    :py:obj:`~.cudaDeviceMapHost` even though it is not accepted by
    :py:obj:`~.cudaSetDeviceFlags` because it is implicit in runtime API
    flags. The reason for this is that the current context may have been
    created via the driver API in which case the flag is not implicit and
    may be unset.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`
    flags : unsigned int
        Pointer to store the device flags

    See Also
    --------
    :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaGetDeviceProperties`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaSetDeviceFlags`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuDevicePrimaryCtxGetState`
    """
    cdef unsigned int flags = 0
    err = ccudart.cudaGetDeviceFlags(&flags)
    return (cudaError_t(err), flags)

@cython.embedsignature(True)
def cudaStreamCreate():
    """ Create an asynchronous stream.

    Creates a new asynchronous stream.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pStream : :py:obj:`~.cudaStream_t`
        Pointer to new stream identifier

    See Also
    --------
    :py:obj:`~.cudaStreamCreateWithPriority`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamGetPriority`, :py:obj:`~.cudaStreamGetFlags`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cuStreamCreate`
    """
    cdef cudaStream_t pStream = cudaStream_t()
    err = ccudart.cudaStreamCreate(<ccudart.cudaStream_t*>pStream._ptr)
    return (cudaError_t(err), pStream)

@cython.embedsignature(True)
def cudaStreamCreateWithFlags(unsigned int flags):
    """ Create an asynchronous stream.

    Creates a new asynchronous stream. The `flags` argument determines the
    behaviors of the stream. Valid values for `flags` are

    - :py:obj:`~.cudaStreamDefault`: Default stream creation flag.

    - :py:obj:`~.cudaStreamNonBlocking`: Specifies that work running in the
      created stream may run concurrently with work in stream 0 (the NULL
      stream), and that the created stream should perform no implicit
      synchronization with stream 0.

    Parameters
    ----------
    flags : unsigned int
        Parameters for stream creation

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pStream : :py:obj:`~.cudaStream_t`
        Pointer to new stream identifier

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithPriority`, :py:obj:`~.cudaStreamGetFlags`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cuStreamCreate`
    """
    cdef cudaStream_t pStream = cudaStream_t()
    err = ccudart.cudaStreamCreateWithFlags(<ccudart.cudaStream_t*>pStream._ptr, flags)
    return (cudaError_t(err), pStream)

@cython.embedsignature(True)
def cudaStreamCreateWithPriority(unsigned int flags, int priority):
    """ Create an asynchronous stream with the specified priority.

    Creates a stream with the specified priority and returns a handle in
    `pStream`. This API alters the scheduler priority of work in the
    stream. Work in a higher priority stream may preempt work already
    executing in a low priority stream.

    `priority` follows a convention where lower numbers represent higher
    priorities. '0' represents default priority. The range of meaningful
    numerical priorities can be queried using
    :py:obj:`~.cudaDeviceGetStreamPriorityRange`. If the specified priority
    is outside the numerical range returned by
    :py:obj:`~.cudaDeviceGetStreamPriorityRange`, it will automatically be
    clamped to the lowest or the highest number in the range.

    Parameters
    ----------
    flags : unsigned int
        Flags for stream creation. See
        :py:obj:`~.cudaStreamCreateWithFlags` for a list of valid flags
        that can be passed
    priority : int
        Priority of the stream. Lower numbers represent higher priorities.
        See :py:obj:`~.cudaDeviceGetStreamPriorityRange` for more
        information about the meaningful stream priorities that can be
        passed.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pStream : :py:obj:`~.cudaStream_t`
        Pointer to new stream identifier

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaDeviceGetStreamPriorityRange`, :py:obj:`~.cudaStreamGetPriority`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cuStreamCreateWithPriority`

    Notes
    -----
    Stream priorities are supported only on GPUs with compute capability 3.5 or higher.

    In the current implementation, only compute kernels launched in priority streams are affected by the stream's priority. Stream priorities have no effect on host-to-device and device-to-host memory operations.
    """
    cdef cudaStream_t pStream = cudaStream_t()
    err = ccudart.cudaStreamCreateWithPriority(<ccudart.cudaStream_t*>pStream._ptr, flags, priority)
    return (cudaError_t(err), pStream)

@cython.embedsignature(True)
def cudaStreamGetPriority(hStream):
    """ Query the priority of a stream.

    Query the priority of a stream. The priority is returned in in
    `priority`. Note that if the stream was created with a priority outside
    the meaningful numerical range returned by
    :py:obj:`~.cudaDeviceGetStreamPriorityRange`, this function returns the
    clamped priority. See :py:obj:`~.cudaStreamCreateWithPriority` for
    details about priority clamping.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Handle to the stream to be queried

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    priority : int
        Pointer to a signed integer in which the stream's priority is
        returned

    See Also
    --------
    :py:obj:`~.cudaStreamCreateWithPriority`, :py:obj:`~.cudaDeviceGetStreamPriorityRange`, :py:obj:`~.cudaStreamGetFlags`, :py:obj:`~.cuStreamGetPriority`
    """
    cdef ccudart.cudaStream_t chStream
    if hStream is None:
        chStream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(hStream, (cudaStream_t,cuda.CUstream)):
        phStream = int(hStream)
        chStream = <ccudart.cudaStream_t><void_ptr>phStream
    else:
        phStream = int(cudaStream_t(hStream))
        chStream = <ccudart.cudaStream_t><void_ptr>phStream

    cdef int priority = 0
    err = ccudart.cudaStreamGetPriority(chStream, &priority)
    return (cudaError_t(err), priority)

@cython.embedsignature(True)
def cudaStreamGetFlags(hStream):
    """ Query the flags of a stream.

    Query the flags of a stream. The flags are returned in `flags`. See
    :py:obj:`~.cudaStreamCreateWithFlags` for a list of valid flags.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Handle to the stream to be queried

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    flags : unsigned int
        Pointer to an unsigned integer in which the stream's flags are
        returned

    See Also
    --------
    :py:obj:`~.cudaStreamCreateWithPriority`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamGetPriority`, :py:obj:`~.cuStreamGetFlags`
    """
    cdef ccudart.cudaStream_t chStream
    if hStream is None:
        chStream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(hStream, (cudaStream_t,cuda.CUstream)):
        phStream = int(hStream)
        chStream = <ccudart.cudaStream_t><void_ptr>phStream
    else:
        phStream = int(cudaStream_t(hStream))
        chStream = <ccudart.cudaStream_t><void_ptr>phStream

    cdef unsigned int flags = 0
    err = ccudart.cudaStreamGetFlags(chStream, &flags)
    return (cudaError_t(err), flags)

@cython.embedsignature(True)
def cudaCtxResetPersistingL2Cache():
    """ Resets all persisting lines in cache to normal status.

    Resets all persisting lines in cache to normal status. Takes effect on
    function return.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`,

    See Also
    --------
    :py:obj:`~.cudaAccessPolicyWindow`
    """
    err = ccudart.cudaCtxResetPersistingL2Cache()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamCopyAttributes(dst, src):
    """ Copies attributes from source stream to destination stream.

    Copies attributes from source stream `src` to destination stream `dst`.
    Both streams must have the same context.

    Parameters
    ----------
    dst : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Destination stream
    src : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Source stream For attributes see :py:obj:`~.cudaStreamAttrID`

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorNotSupported`

    See Also
    --------
    :py:obj:`~.cudaAccessPolicyWindow`
    """
    cdef ccudart.cudaStream_t csrc
    if src is None:
        csrc = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(src, (cudaStream_t,cuda.CUstream)):
        psrc = int(src)
        csrc = <ccudart.cudaStream_t><void_ptr>psrc
    else:
        psrc = int(cudaStream_t(src))
        csrc = <ccudart.cudaStream_t><void_ptr>psrc

    cdef ccudart.cudaStream_t cdst
    if dst is None:
        cdst = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(dst, (cudaStream_t,cuda.CUstream)):
        pdst = int(dst)
        cdst = <ccudart.cudaStream_t><void_ptr>pdst
    else:
        pdst = int(cudaStream_t(dst))
        cdst = <ccudart.cudaStream_t><void_ptr>pdst

    err = ccudart.cudaStreamCopyAttributes(cdst, csrc)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamGetAttribute(hStream, attr not None : cudaStreamAttrID):
    """ Queries stream attribute.

    Queries attribute `attr` from `hStream` and stores it in corresponding
    member of `value_out`.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`

    attr : :py:obj:`~.cudaStreamAttrID`


    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    value_out : :py:obj:`~.cudaStreamAttrValue`


    See Also
    --------
    :py:obj:`~.cudaAccessPolicyWindow`
    """
    cdef ccudart.cudaStream_t chStream
    if hStream is None:
        chStream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(hStream, (cudaStream_t,cuda.CUstream)):
        phStream = int(hStream)
        chStream = <ccudart.cudaStream_t><void_ptr>phStream
    else:
        phStream = int(cudaStream_t(hStream))
        chStream = <ccudart.cudaStream_t><void_ptr>phStream

    cdef ccudart.cudaStreamAttrID cattr = attr.value
    cdef cudaStreamAttrValue value_out = cudaStreamAttrValue()
    err = ccudart.cudaStreamGetAttribute(chStream, cattr, value_out._ptr)
    return (cudaError_t(err), value_out)

@cython.embedsignature(True)
def cudaStreamSetAttribute(hStream, attr not None : cudaStreamAttrID, value : cudaStreamAttrValue):
    """ Sets stream attribute.

    Sets attribute `attr` on `hStream` from corresponding attribute of
    `value`. The updated attribute will be applied to subsequent work
    submitted to the stream. It will not affect previously submitted work.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`

    attr : :py:obj:`~.cudaStreamAttrID`

    value : :py:obj:`~.cudaStreamAttrValue`


    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaAccessPolicyWindow`
    """
    cdef ccudart.cudaStream_t chStream
    if hStream is None:
        chStream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(hStream, (cudaStream_t,cuda.CUstream)):
        phStream = int(hStream)
        chStream = <ccudart.cudaStream_t><void_ptr>phStream
    else:
        phStream = int(cudaStream_t(hStream))
        chStream = <ccudart.cudaStream_t><void_ptr>phStream

    cdef ccudart.cudaStreamAttrID cattr = attr.value
    cdef ccudart.cudaStreamAttrValue* cvalue_ptr = value._ptr if value != None else NULL
    err = ccudart.cudaStreamSetAttribute(chStream, cattr, cvalue_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamDestroy(stream):
    """ Destroys and cleans up an asynchronous stream.

    Destroys and cleans up the asynchronous stream specified by `stream`.

    In case the device is still doing work in the stream `stream` when
    :py:obj:`~.cudaStreamDestroy()` is called, the function will return
    immediately and the resources associated with `stream` will be released
    automatically once the device has completed all work in `stream`.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cuStreamDestroy`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    err = ccudart.cudaStreamDestroy(cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamWaitEvent(stream, event, unsigned int flags):
    """ Make a compute stream wait on an event.

    Makes all future work submitted to `stream` wait for all work captured
    in `event`. See :py:obj:`~.cudaEventRecord()` for details on what is
    captured by an event. The synchronization will be performed efficiently
    on the device when applicable. `event` may be from a different device
    than `stream`.

    flags include:

    - :py:obj:`~.cudaEventWaitDefault`: Default event creation flag.

    - :py:obj:`~.cudaEventWaitExternal`: Event is captured in the graph as
      an external event node when performing stream capture.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to wait
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to wait on
    flags : unsigned int
        Parameters for the operation(See above)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cuStreamWaitEvent`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    with nogil:
        err = ccudart.cudaStreamWaitEvent(cstream, cevent, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamAddCallback(stream, callback, userData, unsigned int flags):
    """ Add a callback to a compute stream.

    Adds a callback to be called on the host after all currently enqueued
    items in the stream have completed. For each cudaStreamAddCallback
    call, a callback will be executed exactly once. The callback will block
    later work in the stream until it is finished.

    The callback may be passed :py:obj:`~.cudaSuccess` or an error code. In
    the event of a device error, all subsequently executed callbacks will
    receive an appropriate :py:obj:`~.cudaError_t`.

    Callbacks must not make any CUDA API calls. Attempting to use CUDA APIs
    may result in :py:obj:`~.cudaErrorNotPermitted`. Callbacks must not
    perform any synchronization that may depend on outstanding device work
    or other callbacks that are not mandated to run earlier. Callbacks
    without a mandated order (in independent streams) execute in undefined
    order and may be serialized.

    For the purposes of Unified Memory, callback execution makes a number
    of guarantees:

    - The callback stream is considered idle for the duration of the
      callback. Thus, for example, a callback may always use memory
      attached to the callback stream.

    - The start of execution of a callback has the same effect as
      synchronizing an event recorded in the same stream immediately prior
      to the callback. It thus synchronizes streams which have been
      "joined" prior to the callback.

    - Adding device work to any stream does not have the effect of making
      the stream active until all preceding callbacks have executed. Thus,
      for example, a callback might use global attached memory even if work
      has been added to another stream, if it has been properly ordered
      with an event.

    - Completion of a callback does not cause a stream to become active
      except as described above. The callback stream will remain idle if no
      device work follows the callback, and will remain idle across
      consecutive callbacks without device work in between. Thus, for
      example, stream synchronization can be done by signaling from a
      callback at the end of the stream.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to add callback to
    callback : :py:obj:`~.cudaStreamCallback_t`
        The function to call once preceding stream operations are complete
    userData : Any
        User specified data to be passed to the callback function
    flags : unsigned int
        Reserved for future use, must be 0

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorNotSupported`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cudaMallocManaged`, :py:obj:`~.cudaStreamAttachMemAsync`, :py:obj:`~.cudaLaunchHostFunc`, :py:obj:`~.cuStreamAddCallback`

    Notes
    -----
    This function is slated for eventual deprecation and removal. If you do not require the callback to execute in case of a device error, consider using :py:obj:`~.cudaLaunchHostFunc`. Additionally, this function is not supported with :py:obj:`~.cudaStreamBeginCapture` and :py:obj:`~.cudaStreamEndCapture`, unlike :py:obj:`~.cudaLaunchHostFunc`.
    """
    cdef ccudart.cudaStreamCallback_t ccallback
    if callback is None:
        ccallback = <ccudart.cudaStreamCallback_t><void_ptr>0
    elif isinstance(callback, (cudaStreamCallback_t)):
        pcallback = int(callback)
        ccallback = <ccudart.cudaStreamCallback_t><void_ptr>pcallback
    else:
        pcallback = int(cudaStreamCallback_t(callback))
        ccallback = <ccudart.cudaStreamCallback_t><void_ptr>pcallback

    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cuserData = utils.HelperInputVoidPtr(userData)
    cdef void* cuserData_ptr = <void*><void_ptr>cuserData.cptr
    with nogil:
        err = ccudart.cudaStreamAddCallback(cstream, ccallback, cuserData_ptr, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamSynchronize(stream):
    """ Waits for stream tasks to complete.

    Blocks until `stream` has completed all operations. If the
    :py:obj:`~.cudaDeviceScheduleBlockingSync` flag was set for this
    device, the host thread will block until the stream is finished with
    all of its tasks.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cuStreamSynchronize`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    with nogil:
        err = ccudart.cudaStreamSynchronize(cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamQuery(stream):
    """ Queries an asynchronous stream for completion status.

    Returns :py:obj:`~.cudaSuccess` if all operations in `stream` have
    completed, or :py:obj:`~.cudaErrorNotReady` if not.

    For the purposes of Unified Memory, a return value of
    :py:obj:`~.cudaSuccess` is equivalent to having called
    :py:obj:`~.cudaStreamSynchronize()`.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorNotReady`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cuStreamQuery`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    err = ccudart.cudaStreamQuery(cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamAttachMemAsync(stream, devPtr, size_t length, unsigned int flags):
    """ Attach memory to a stream asynchronously.

    Enqueues an operation in `stream` to specify stream association of
    `length` bytes of memory starting from `devPtr`. This function is a
    stream-ordered operation, meaning that it is dependent on, and will
    only take effect when, previous work in stream has completed. Any
    previous association is automatically replaced.

    `devPtr` must point to an one of the following types of memories:

    - managed memory declared using the managed keyword or allocated with
      :py:obj:`~.cudaMallocManaged`.

    - a valid host-accessible region of system-allocated pageable memory.
      This type of memory may only be specified if the device associated
      with the stream reports a non-zero value for the device attribute
      :py:obj:`~.cudaDevAttrPageableMemoryAccess`.

    For managed allocations, `length` must be either zero or the entire
    allocation's size. Both indicate that the entire allocation's stream
    association is being changed. Currently, it is not possible to change
    stream association for a portion of a managed allocation.

    For pageable allocations, `length` must be non-zero.

    The stream association is specified using `flags` which must be one of
    :py:obj:`~.cudaMemAttachGlobal`, :py:obj:`~.cudaMemAttachHost` or
    :py:obj:`~.cudaMemAttachSingle`. The default value for `flags` is
    :py:obj:`~.cudaMemAttachSingle` If the :py:obj:`~.cudaMemAttachGlobal`
    flag is specified, the memory can be accessed by any stream on any
    device. If the :py:obj:`~.cudaMemAttachHost` flag is specified, the
    program makes a guarantee that it won't access the memory on the device
    from any stream on a device that has a zero value for the device
    attribute :py:obj:`~.cudaDevAttrConcurrentManagedAccess`. If the
    :py:obj:`~.cudaMemAttachSingle` flag is specified and `stream` is
    associated with a device that has a zero value for the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess`, the program makes a
    guarantee that it will only access the memory on the device from
    `stream`. It is illegal to attach singly to the NULL stream, because
    the NULL stream is a virtual global stream and not a specific stream.
    An error will be returned in this case.

    When memory is associated with a single stream, the Unified Memory
    system will allow CPU access to this memory region so long as all
    operations in `stream` have completed, regardless of whether other
    streams are active. In effect, this constrains exclusive ownership of
    the managed memory region by an active GPU to per-stream activity
    instead of whole-GPU activity.

    Accessing memory on the device from streams that are not associated
    with it will produce undefined results. No error checking is performed
    by the Unified Memory system to ensure that kernels launched into other
    streams do not access this region.

    It is a program's responsibility to order calls to
    :py:obj:`~.cudaStreamAttachMemAsync` via events, synchronization or
    other means to ensure legal access to memory at all times. Data
    visibility and coherency will be changed appropriately for all kernels
    which follow a stream-association change.

    If `stream` is destroyed while data is associated with it, the
    association is removed and the association reverts to the default
    visibility of the allocation as specified at
    :py:obj:`~.cudaMallocManaged`. For managed variables, the default
    association is always :py:obj:`~.cudaMemAttachGlobal`. Note that
    destroying a stream is an asynchronous operation, and as a result, the
    change to default association won't happen until all work in the stream
    has completed.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to enqueue the attach operation
    devPtr : Any
        Pointer to memory (must be a pointer to managed memory or to a
        valid host-accessible region of system-allocated memory)
    length : size_t
        Length of memory (defaults to zero)
    flags : unsigned int
        Must be one of :py:obj:`~.cudaMemAttachGlobal`,
        :py:obj:`~.cudaMemAttachHost` or :py:obj:`~.cudaMemAttachSingle`
        (defaults to :py:obj:`~.cudaMemAttachSingle`)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorNotReady`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cudaMallocManaged`, :py:obj:`~.cuStreamAttachMemAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    err = ccudart.cudaStreamAttachMemAsync(cstream, cdevPtr_ptr, length, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaStreamBeginCapture(stream, mode not None : cudaStreamCaptureMode):
    """ Begins graph capture on a stream.

    Begin graph capture on `stream`. When a stream is in capture mode, all
    operations pushed into the stream will not be executed, but will
    instead be captured into a graph, which will be returned via
    :py:obj:`~.cudaStreamEndCapture`. Capture may not be initiated if
    `stream` is :py:obj:`~.cudaStreamLegacy`. Capture must be ended on the
    same stream in which it was initiated, and it may only be initiated if
    the stream is not already in capture mode. The capture mode may be
    queried via :py:obj:`~.cudaStreamIsCapturing`. A unique id representing
    the capture sequence may be queried via
    :py:obj:`~.cudaStreamGetCaptureInfo`.

    If `mode` is not :py:obj:`~.cudaStreamCaptureModeRelaxed`,
    :py:obj:`~.cudaStreamEndCapture` must be called on this stream from the
    same thread.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to initiate capture
    mode : :py:obj:`~.cudaStreamCaptureMode`
        Controls the interaction of this capture sequence with other API
        calls that are potentially unsafe. For more details see
        :py:obj:`~.cudaThreadExchangeStreamCaptureMode`.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamIsCapturing`, :py:obj:`~.cudaStreamEndCapture`, :py:obj:`~.cudaThreadExchangeStreamCaptureMode`

    Notes
    -----
    Kernels captured using this API must not use texture and surface references. Reading or writing through any texture or surface reference is undefined behavior. This restriction does not apply to texture and surface objects.
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaStreamCaptureMode cmode = mode.value
    err = ccudart.cudaStreamBeginCapture(cstream, cmode)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaThreadExchangeStreamCaptureMode(mode not None : cudaStreamCaptureMode):
    """ Swaps the stream capture interaction mode for a thread.

    Sets the calling thread's stream capture interaction mode to the value
    contained in `*mode`, and overwrites `*mode` with the previous mode for
    the thread. To facilitate deterministic behavior across function or
    module boundaries, callers are encouraged to use this API in a push-pop
    fashion:

    **View CUDA Toolkit Documentation for a C++ code example**

    During stream capture (see :py:obj:`~.cudaStreamBeginCapture`), some
    actions, such as a call to :py:obj:`~.cudaMalloc`, may be unsafe. In
    the case of :py:obj:`~.cudaMalloc`, the operation is not enqueued
    asynchronously to a stream, and is not observed by stream capture.
    Therefore, if the sequence of operations captured via
    :py:obj:`~.cudaStreamBeginCapture` depended on the allocation being
    replayed whenever the graph is launched, the captured graph would be
    invalid.

    Therefore, stream capture places restrictions on API calls that can be
    made within or concurrently to a
    :py:obj:`~.cudaStreamBeginCapture`-:py:obj:`~.cudaStreamEndCapture`
    sequence. This behavior can be controlled via this API and flags to
    :py:obj:`~.cudaStreamBeginCapture`.

    A thread's mode is one of the following:

    - `cudaStreamCaptureModeGlobal:` This is the default mode. If the local
      thread has an ongoing capture sequence that was not initiated with
      `cudaStreamCaptureModeRelaxed` at `cuStreamBeginCapture`, or if any
      other thread has a concurrent capture sequence initiated with
      `cudaStreamCaptureModeGlobal`, this thread is prohibited from
      potentially unsafe API calls.

    - `cudaStreamCaptureModeThreadLocal:` If the local thread has an
      ongoing capture sequence not initiated with
      `cudaStreamCaptureModeRelaxed`, it is prohibited from potentially
      unsafe API calls. Concurrent capture sequences in other threads are
      ignored.

    - `cudaStreamCaptureModeRelaxed:` The local thread is not prohibited
      from potentially unsafe API calls. Note that the thread is still
      prohibited from API calls which necessarily conflict with stream
      capture, for example, attempting :py:obj:`~.cudaEventQuery` on an
      event that was last recorded inside a capture sequence.

    Parameters
    ----------
    mode : :py:obj:`~.cudaStreamCaptureMode`
        Pointer to mode value to swap with the current mode

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    mode : :py:obj:`~.cudaStreamCaptureMode`
        Pointer to mode value to swap with the current mode

    See Also
    --------
    :py:obj:`~.cudaStreamBeginCapture`
    """
    cdef ccudart.cudaStreamCaptureMode cmode = mode.value
    err = ccudart.cudaThreadExchangeStreamCaptureMode(&cmode)
    return (cudaError_t(err), cudaStreamCaptureMode(cmode))

@cython.embedsignature(True)
def cudaStreamEndCapture(stream):
    """ Ends capture on a stream, returning the captured graph.

    End capture on `stream`, returning the captured graph via `pGraph`.
    Capture must have been initiated on `stream` via a call to
    :py:obj:`~.cudaStreamBeginCapture`. If capture was invalidated, due to
    a violation of the rules of stream capture, then a NULL graph will be
    returned.

    If the `mode` argument to :py:obj:`~.cudaStreamBeginCapture` was not
    :py:obj:`~.cudaStreamCaptureModeRelaxed`, this call must be from the
    same thread as :py:obj:`~.cudaStreamBeginCapture`.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorStreamCaptureWrongThread`
    pGraph : :py:obj:`~.cudaGraph_t`
        The captured graph

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamBeginCapture`, :py:obj:`~.cudaStreamIsCapturing`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef cudaGraph_t pGraph = cudaGraph_t()
    err = ccudart.cudaStreamEndCapture(cstream, <ccudart.cudaGraph_t*>pGraph._ptr)
    return (cudaError_t(err), pGraph)

@cython.embedsignature(True)
def cudaStreamIsCapturing(stream):
    """ Returns a stream's capture status.

    Return the capture status of `stream` via `pCaptureStatus`. After a
    successful call, `*pCaptureStatus` will contain one of the following:

    - :py:obj:`~.cudaStreamCaptureStatusNone`: The stream is not capturing.

    - :py:obj:`~.cudaStreamCaptureStatusActive`: The stream is capturing.

    - :py:obj:`~.cudaStreamCaptureStatusInvalidated`: The stream was
      capturing but an error has invalidated the capture sequence. The
      capture sequence must be terminated with
      :py:obj:`~.cudaStreamEndCapture` on the stream where it was initiated
      in order to continue using `stream`.

    Note that, if this is called on :py:obj:`~.cudaStreamLegacy` (the "null
    stream") while a blocking stream on the same device is capturing, it
    will return :py:obj:`~.cudaErrorStreamCaptureImplicit` and
    `*pCaptureStatus` is unspecified after the call. The blocking stream
    capture is not invalidated.

    When a blocking stream is capturing, the legacy stream is in an
    unusable state until the blocking stream capture is terminated. The
    legacy stream is not supported for stream capture, but attempted use
    would have an implicit dependency on the capturing stream(s).

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorStreamCaptureImplicit`
    pCaptureStatus : :py:obj:`~.cudaStreamCaptureStatus`
        Returns the stream's capture status

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamBeginCapture`, :py:obj:`~.cudaStreamEndCapture`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaStreamCaptureStatus pCaptureStatus
    err = ccudart.cudaStreamIsCapturing(cstream, &pCaptureStatus)
    return (cudaError_t(err), cudaStreamCaptureStatus(pCaptureStatus))

@cython.embedsignature(True)
def cudaStreamGetCaptureInfo(stream):
    """ Query capture status of a stream.

    Note there is a later version of this API,
    :py:obj:`~.cudaStreamGetCaptureInfo_v2`. It will supplant this version
    in 12.0, which is retained for minor version compatibility.

    Query the capture status of a stream and get a unique id representing
    the capture sequence over the lifetime of the process.

    If called on :py:obj:`~.cudaStreamLegacy` (the "null stream") while a
    stream not created with :py:obj:`~.cudaStreamNonBlocking` is capturing,
    returns :py:obj:`~.cudaErrorStreamCaptureImplicit`.

    A valid id is returned only if both of the following are true:

    - the call returns :py:obj:`~.cudaSuccess`

    - captureStatus is set to :py:obj:`~.cudaStreamCaptureStatusActive`

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorStreamCaptureImplicit`
    pCaptureStatus : :py:obj:`~.cudaStreamCaptureStatus`
        Returns the stream's capture status
    pId : unsigned long long
        Returns the unique id of the capture sequence

    See Also
    --------
    :py:obj:`~.cudaStreamGetCaptureInfo_v2`, :py:obj:`~.cudaStreamBeginCapture`, :py:obj:`~.cudaStreamIsCapturing`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaStreamCaptureStatus pCaptureStatus
    cdef unsigned long long pId = 0
    err = ccudart.cudaStreamGetCaptureInfo(cstream, &pCaptureStatus, &pId)
    return (cudaError_t(err), cudaStreamCaptureStatus(pCaptureStatus), pId)

@cython.embedsignature(True)
def cudaStreamGetCaptureInfo_v2(stream):
    """ Query a stream's capture state (11.3+)

    Query stream state related to stream capture.

    If called on :py:obj:`~.cudaStreamLegacy` (the "null stream") while a
    stream not created with :py:obj:`~.cudaStreamNonBlocking` is capturing,
    returns :py:obj:`~.cudaErrorStreamCaptureImplicit`.

    Valid data (other than capture status) is returned only if both of the
    following are true:

    - the call returns cudaSuccess

    - the returned capture status is
      :py:obj:`~.cudaStreamCaptureStatusActive`

    This version of cudaStreamGetCaptureInfo is introduced in CUDA 11.3 and
    will supplant the previous version :py:obj:`~.cudaStreamGetCaptureInfo`
    in 12.0. Developers requiring compatibility across minor versions to
    CUDA 11.0 (driver version 445) can do one of the following:

    - Use the older version of the API,
      :py:obj:`~.cudaStreamGetCaptureInfo`

    - Pass null for all of `graph_out`, `dependencies_out`, and
      `numDependencies_out`.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorStreamCaptureImplicit`
    captureStatus_out : :py:obj:`~.cudaStreamCaptureStatus`
        Location to return the capture status of the stream; required
    id_out : unsigned long long
        Optional location to return an id for the capture sequence, which
        is unique over the lifetime of the process
    graph_out : :py:obj:`~.cudaGraph_t`
        Optional location to return the graph being captured into. All
        operations other than destroy and node removal are permitted on the
        graph while the capture sequence is in progress. This API does not
        transfer ownership of the graph, which is transferred or destroyed
        at :py:obj:`~.cudaStreamEndCapture`. Note that the graph handle may
        be invalidated before end of capture for certain errors. Nodes that
        are or become unreachable from the original stream at
        :py:obj:`~.cudaStreamEndCapture` due to direct actions on the graph
        do not trigger :py:obj:`~.cudaErrorStreamCaptureUnjoined`.
    dependencies_out : List[:py:obj:`~.cudaGraphNode_t`]
        Optional location to store a pointer to an array of nodes. The next
        node to be captured in the stream will depend on this set of nodes,
        absent operations such as event wait which modify this set. The
        array pointer is valid until the next API call which operates on
        the stream or until end of capture. The node handles may be copied
        out and are valid until they or the graph is destroyed. The driver-
        owned array may also be passed directly to APIs that operate on the
        graph (not the stream) without copying.
    numDependencies_out : int
        Optional location to store the size of the array returned in
        dependencies_out.

    See Also
    --------
    :py:obj:`~.cudaStreamGetCaptureInfo`, :py:obj:`~.cudaStreamBeginCapture`, :py:obj:`~.cudaStreamIsCapturing`, :py:obj:`~.cudaStreamUpdateCaptureDependencies`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaStreamCaptureStatus captureStatus_out
    cdef unsigned long long id_out = 0
    cdef cudaGraph_t graph_out = cudaGraph_t()
    cdef const ccudart.cudaGraphNode_t* cdependencies_out = NULL
    pydependencies_out = []
    cdef size_t numDependencies_out = 0
    err = ccudart.cudaStreamGetCaptureInfo_v2(cstream, &captureStatus_out, &id_out, <ccudart.cudaGraph_t*>graph_out._ptr, &cdependencies_out, &numDependencies_out)
    if cudaError_t(err) == cudaError_t(0):
        pydependencies_out = [cudaGraphNode_t(init_value=<void_ptr>cdependencies_out[idx]) for idx in range(numDependencies_out)]
    return (cudaError_t(err), cudaStreamCaptureStatus(captureStatus_out), id_out, graph_out, pydependencies_out, numDependencies_out)

@cython.embedsignature(True)
def cudaStreamUpdateCaptureDependencies(stream, dependencies : List[cudaGraphNode_t], size_t numDependencies, unsigned int flags):
    """ Update the set of dependencies in a capturing stream (11.3+)

    Modifies the dependency set of a capturing stream. The dependency set
    is the set of nodes that the next captured node in the stream will
    depend on.

    Valid flags are :py:obj:`~.cudaStreamAddCaptureDependencies` and
    :py:obj:`~.cudaStreamSetCaptureDependencies`. These control whether the
    set passed to the API is added to the existing set or replaces it. A
    flags value of 0 defaults to
    :py:obj:`~.cudaStreamAddCaptureDependencies`.

    Nodes that are removed from the dependency set via this API do not
    result in :py:obj:`~.cudaErrorStreamCaptureUnjoined` if they are
    unreachable from the stream at :py:obj:`~.cudaStreamEndCapture`.

    Returns :py:obj:`~.cudaErrorIllegalState` if the stream is not
    capturing.

    This API is new in CUDA 11.3. Developers requiring compatibility across
    minor versions of the CUDA driver to 11.0 should not use this API or
    provide a fallback.

    Parameters
    ----------
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        None
    dependencies : List[:py:obj:`~.cudaGraphNode_t`]
        None
    numDependencies : size_t
        None
    flags : unsigned int
        None

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorIllegalState`

    See Also
    --------
    :py:obj:`~.cudaStreamBeginCapture`, :py:obj:`~.cudaStreamGetCaptureInfo`, :py:obj:`~.cudaStreamGetCaptureInfo_v2`
    """
    dependencies = [] if dependencies is None else dependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in dependencies):
        raise TypeError("Argument 'dependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaGraphNode_t* cdependencies = NULL
    if len(dependencies) > 0:
        cdependencies = <ccudart.cudaGraphNode_t*> calloc(len(dependencies), sizeof(ccudart.cudaGraphNode_t))
        if cdependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(dependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(dependencies)):
                cdependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>dependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(dependencies): raise RuntimeError("List is too small: " + str(len(dependencies)) + " < " + str(numDependencies))
    err = ccudart.cudaStreamUpdateCaptureDependencies(cstream, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>dependencies[0])._ptr if len(dependencies) == 1 else cdependencies, numDependencies, flags)
    if cdependencies is not NULL:
        free(cdependencies)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEventCreate():
    """ Creates an event object.

    Creates an event object for the current device using
    :py:obj:`~.cudaEventDefault`.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorLaunchFailure`, :py:obj:`~.cudaErrorMemoryAllocation`
    event : :py:obj:`~.cudaEvent_t`
        Newly created event

    See Also
    --------
    cudaEventCreate (C++ API), :py:obj:`~.cudaEventCreateWithFlags`, :py:obj:`~.cudaEventRecord`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventElapsedTime`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cuEventCreate`
    """
    cdef cudaEvent_t event = cudaEvent_t()
    err = ccudart.cudaEventCreate(<ccudart.cudaEvent_t*>event._ptr)
    return (cudaError_t(err), event)

@cython.embedsignature(True)
def cudaEventCreateWithFlags(unsigned int flags):
    """ Creates an event object with the specified flags.

    Creates an event object for the current device with the specified
    flags. Valid flags include:

    - :py:obj:`~.cudaEventDefault`: Default event creation flag.

    - :py:obj:`~.cudaEventBlockingSync`: Specifies that event should use
      blocking synchronization. A host thread that uses
      :py:obj:`~.cudaEventSynchronize()` to wait on an event created with
      this flag will block until the event actually completes.

    - :py:obj:`~.cudaEventDisableTiming`: Specifies that the created event
      does not need to record timing data. Events created with this flag
      specified and the :py:obj:`~.cudaEventBlockingSync` flag not
      specified will provide the best performance when used with
      :py:obj:`~.cudaStreamWaitEvent()` and :py:obj:`~.cudaEventQuery()`.

    - :py:obj:`~.cudaEventInterprocess`: Specifies that the created event
      may be used as an interprocess event by
      :py:obj:`~.cudaIpcGetEventHandle()`.
      :py:obj:`~.cudaEventInterprocess` must be specified along with
      :py:obj:`~.cudaEventDisableTiming`.

    Parameters
    ----------
    flags : unsigned int
        Flags for new event

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorLaunchFailure`, :py:obj:`~.cudaErrorMemoryAllocation`
    event : :py:obj:`~.cudaEvent_t`
        Newly created event

    See Also
    --------
    :py:obj:`~.cudaEventCreate (C API)`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventElapsedTime`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cuEventCreate`
    """
    cdef cudaEvent_t event = cudaEvent_t()
    err = ccudart.cudaEventCreateWithFlags(<ccudart.cudaEvent_t*>event._ptr, flags)
    return (cudaError_t(err), event)

@cython.embedsignature(True)
def cudaEventRecord(event, stream):
    """ Records an event.

    Captures in `event` the contents of `stream` at the time of this call.
    `event` and `stream` must be on the same CUDA context. Calls such as
    :py:obj:`~.cudaEventQuery()` or :py:obj:`~.cudaStreamWaitEvent()` will
    then examine or wait for completion of the work that was captured. Uses
    of `stream` after this call do not modify `event`. See note on default
    stream behavior for what is captured in the default case.

    :py:obj:`~.cudaEventRecord()` can be called multiple times on the same
    event and will overwrite the previously captured state. Other APIs such
    as :py:obj:`~.cudaStreamWaitEvent()` use the most recently captured
    state at the time of the API call, and are not affected by later calls
    to :py:obj:`~.cudaEventRecord()`. Before the first call to
    :py:obj:`~.cudaEventRecord()`, an event represents an empty set of
    work, so for example :py:obj:`~.cudaEventQuery()` would return
    :py:obj:`~.cudaSuccess`.

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to record
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to record event

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorLaunchFailure`

    See Also
    --------
    :py:obj:`~.cudaEventCreate (C API)`, :py:obj:`~.cudaEventCreateWithFlags`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventElapsedTime`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cuEventRecord`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    err = ccudart.cudaEventRecord(cevent, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEventRecordWithFlags(event, stream, unsigned int flags):
    """ Records an event.

    Captures in `event` the contents of `stream` at the time of this call.
    `event` and `stream` must be on the same CUDA context. Calls such as
    :py:obj:`~.cudaEventQuery()` or :py:obj:`~.cudaStreamWaitEvent()` will
    then examine or wait for completion of the work that was captured. Uses
    of `stream` after this call do not modify `event`. See note on default
    stream behavior for what is captured in the default case.

    :py:obj:`~.cudaEventRecordWithFlags()` can be called multiple times on
    the same event and will overwrite the previously captured state. Other
    APIs such as :py:obj:`~.cudaStreamWaitEvent()` use the most recently
    captured state at the time of the API call, and are not affected by
    later calls to :py:obj:`~.cudaEventRecordWithFlags()`. Before the first
    call to :py:obj:`~.cudaEventRecordWithFlags()`, an event represents an
    empty set of work, so for example :py:obj:`~.cudaEventQuery()` would
    return :py:obj:`~.cudaSuccess`.

    flags include:

    - :py:obj:`~.cudaEventRecordDefault`: Default event creation flag.

    - :py:obj:`~.cudaEventRecordExternal`: Event is captured in the graph
      as an external event node when performing stream capture.

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to record
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to record event
    flags : unsigned int
        Parameters for the operation(See above)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorLaunchFailure`

    See Also
    --------
    :py:obj:`~.cudaEventCreate (C API)`, :py:obj:`~.cudaEventCreateWithFlags`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventElapsedTime`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaEventRecord`, :py:obj:`~.cuEventRecord`,
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    err = ccudart.cudaEventRecordWithFlags(cevent, cstream, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEventQuery(event):
    """ Queries an event's status.

    Queries the status of all work currently captured by `event`. See
    :py:obj:`~.cudaEventRecord()` for details on what is captured by an
    event.

    Returns :py:obj:`~.cudaSuccess` if all captured work has been
    completed, or :py:obj:`~.cudaErrorNotReady` if any captured work is
    incomplete.

    For the purposes of Unified Memory, a return value of
    :py:obj:`~.cudaSuccess` is equivalent to having called
    :py:obj:`~.cudaEventSynchronize()`.

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorNotReady`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorLaunchFailure`

    See Also
    --------
    :py:obj:`~.cudaEventCreate (C API)`, :py:obj:`~.cudaEventCreateWithFlags`, :py:obj:`~.cudaEventRecord`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventElapsedTime`, :py:obj:`~.cuEventQuery`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    err = ccudart.cudaEventQuery(cevent)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEventSynchronize(event):
    """ Waits for an event to complete.

    Waits until the completion of all work currently captured in `event`.
    See :py:obj:`~.cudaEventRecord()` for details on what is captured by an
    event.

    Waiting for an event that was created with the
    :py:obj:`~.cudaEventBlockingSync` flag will cause the calling CPU
    thread to block until the event has been completed by the device. If
    the :py:obj:`~.cudaEventBlockingSync` flag has not been set, then the
    CPU thread will busy-wait until the event has been completed by the
    device.

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to wait for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorLaunchFailure`

    See Also
    --------
    :py:obj:`~.cudaEventCreate (C API)`, :py:obj:`~.cudaEventCreateWithFlags`, :py:obj:`~.cudaEventRecord`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventElapsedTime`, :py:obj:`~.cuEventSynchronize`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    err = ccudart.cudaEventSynchronize(cevent)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEventDestroy(event):
    """ Destroys an event object.

    Destroys the event specified by `event`.

    An event may be destroyed before it is complete (i.e., while
    :py:obj:`~.cudaEventQuery()` would return
    :py:obj:`~.cudaErrorNotReady`). In this case, the call does not block
    on completion of the event, and any associated resources will
    automatically be released asynchronously at completion.

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to destroy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorLaunchFailure`

    See Also
    --------
    :py:obj:`~.cudaEventCreate (C API)`, :py:obj:`~.cudaEventCreateWithFlags`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventRecord`, :py:obj:`~.cudaEventElapsedTime`, :py:obj:`~.cuEventDestroy`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    err = ccudart.cudaEventDestroy(cevent)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEventElapsedTime(start, end):
    """ Computes the elapsed time between events.

    Computes the elapsed time between two events (in milliseconds with a
    resolution of around 0.5 microseconds).

    If either event was last recorded in a non-NULL stream, the resulting
    time may be greater than expected (even if both used the same stream
    handle). This happens because the :py:obj:`~.cudaEventRecord()`
    operation takes place asynchronously and there is no guarantee that the
    measured latency is actually just between the two events. Any number of
    other different stream operations could execute in between the two
    measured events, thus altering the timing in a significant way.

    If :py:obj:`~.cudaEventRecord()` has not been called on either event,
    then :py:obj:`~.cudaErrorInvalidResourceHandle` is returned. If
    :py:obj:`~.cudaEventRecord()` has been called on both events but one or
    both of them has not yet been completed (that is,
    :py:obj:`~.cudaEventQuery()` would return :py:obj:`~.cudaErrorNotReady`
    on at least one of the events), :py:obj:`~.cudaErrorNotReady` is
    returned. If either event was created with the
    :py:obj:`~.cudaEventDisableTiming` flag, then this function will return
    :py:obj:`~.cudaErrorInvalidResourceHandle`.

    Parameters
    ----------
    start : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Starting event
    end : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Ending event

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorNotReady`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorLaunchFailure`
    ms : float
        Time between `start` and `end` in ms

    See Also
    --------
    :py:obj:`~.cudaEventCreate (C API)`, :py:obj:`~.cudaEventCreateWithFlags`, :py:obj:`~.cudaEventQuery`, :py:obj:`~.cudaEventSynchronize`, :py:obj:`~.cudaEventDestroy`, :py:obj:`~.cudaEventRecord`, :py:obj:`~.cuEventElapsedTime`
    """
    cdef ccudart.cudaEvent_t cend
    if end is None:
        cend = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(end, (cudaEvent_t,cuda.CUevent)):
        pend = int(end)
        cend = <ccudart.cudaEvent_t><void_ptr>pend
    else:
        pend = int(cudaEvent_t(end))
        cend = <ccudart.cudaEvent_t><void_ptr>pend

    cdef ccudart.cudaEvent_t cstart
    if start is None:
        cstart = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(start, (cudaEvent_t,cuda.CUevent)):
        pstart = int(start)
        cstart = <ccudart.cudaEvent_t><void_ptr>pstart
    else:
        pstart = int(cudaEvent_t(start))
        cstart = <ccudart.cudaEvent_t><void_ptr>pstart

    cdef float ms = 0
    err = ccudart.cudaEventElapsedTime(&ms, cstart, cend)
    return (cudaError_t(err), ms)

@cython.embedsignature(True)
def cudaImportExternalMemory(memHandleDesc : cudaExternalMemoryHandleDesc):
    """ Imports an external memory object.

    Imports an externally allocated memory object and returns a handle to
    that in `extMem_out`.

    The properties of the handle being imported must be described in
    `memHandleDesc`. The :py:obj:`~.cudaExternalMemoryHandleDesc` structure
    is defined as follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaExternalMemoryHandleDesc.type` specifies the type
    of handle being imported. :py:obj:`~.cudaExternalMemoryHandleType` is
    defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeOpaqueFd`, then
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::fd must be a valid
    file descriptor referencing a memory object. Ownership of the file
    descriptor is transferred to the CUDA driver when the handle is
    imported successfully. Performing any operations on the file descriptor
    after it is imported results in undefined behavior.

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeOpaqueWin32`, then exactly one
    of :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle and
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name must not
    be NULL. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle is not
    NULL, then it must represent a valid shared NT handle that references a
    memory object. Ownership of this handle is not transferred to CUDA
    after the import operation, so the application must release the handle
    using the appropriate system call. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name is not
    NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a memory object.

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeOpaqueWin32Kmt`, then
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle must be
    non-NULL and
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name must be
    NULL. The handle specified must be a globally shared KMT handle. This
    handle does not hold a reference to the underlying object, and thus
    will be invalid when all references to the memory object are destroyed.

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeD3D12Heap`, then exactly one of
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle and
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name must not
    be NULL. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle is not
    NULL, then it must represent a valid shared NT handle that is returned
    by ID3D12Device::CreateSharedHandle when referring to a ID3D12Heap
    object. This handle holds a reference to the underlying object. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name is not
    NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a ID3D12Heap object.

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeD3D12Resource`, then exactly one
    of :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle and
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name must not
    be NULL. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle is not
    NULL, then it must represent a valid shared NT handle that is returned
    by ID3D12Device::CreateSharedHandle when referring to a ID3D12Resource
    object. This handle holds a reference to the underlying object. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name is not
    NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a ID3D12Resource object.

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeD3D11Resource`,then exactly one
    of :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle and
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name must not
    be NULL. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that is
    returned by IDXGIResource1::CreateSharedHandle when referring to a
    ID3D11Resource object. If
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name is not
    NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a ID3D11Resource object.

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeD3D11ResourceKmt`, then
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::handle must be
    non-NULL and
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::win32::name must be
    NULL. The handle specified must be a valid shared KMT handle that is
    returned by IDXGIResource::GetSharedHandle when referring to a
    ID3D11Resource object.

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is
    :py:obj:`~.cudaExternalMemoryHandleTypeNvSciBuf`, then
    :py:obj:`~.cudaExternalMemoryHandleDesc`::handle::nvSciBufObject must
    be NON-NULL and reference a valid NvSciBuf object. If the NvSciBuf
    object imported into CUDA is also mapped by other drivers, then the
    application must use :py:obj:`~.cudaWaitExternalSemaphoresAsync` or
    :py:obj:`~.cudaSignalExternalSemaphoresAsync` as approprriate barriers
    to maintain coherence between CUDA and the other drivers. See
    :py:obj:`~.cudaExternalSemaphoreWaitSkipNvSciBufMemSync` and
    :py:obj:`~.cudaExternalSemaphoreSignalSkipNvSciBufMemSync` for memory
    synchronization.

    The size of the memory object must be specified in
    :py:obj:`~.cudaExternalMemoryHandleDesc.size`.

    Specifying the flag :py:obj:`~.cudaExternalMemoryDedicated` in
    :py:obj:`~.cudaExternalMemoryHandleDesc.flags` indicates that the
    resource is a dedicated resource. The definition of what a dedicated
    resource is outside the scope of this extension. This flag must be set
    if :py:obj:`~.cudaExternalMemoryHandleDesc.type` is one of the
    following: :py:obj:`~.cudaExternalMemoryHandleTypeD3D12Resource`
    :py:obj:`~.cudaExternalMemoryHandleTypeD3D11Resource`
    :py:obj:`~.cudaExternalMemoryHandleTypeD3D11ResourceKmt`

    Parameters
    ----------
    memHandleDesc : :py:obj:`~.cudaExternalMemoryHandleDesc`
        Memory import handle descriptor

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    extMem_out : :py:obj:`~.cudaExternalMemory_t`
        Returned handle to an external memory object

    See Also
    --------
    :py:obj:`~.cudaDestroyExternalMemory`, :py:obj:`~.cudaExternalMemoryGetMappedBuffer`, :py:obj:`~.cudaExternalMemoryGetMappedMipmappedArray`

    Notes
    -----
    If the Vulkan memory imported into CUDA is mapped on the CPU then the application must use vkInvalidateMappedMemoryRanges/vkFlushMappedMemoryRanges as well as appropriate Vulkan pipeline barriers to maintain coherence between CPU and GPU. For more information on these APIs, please refer to "Synchronization
    and Cache Control" chapter from Vulkan specification.
    """
    cdef cudaExternalMemory_t extMem_out = cudaExternalMemory_t()
    cdef ccudart.cudaExternalMemoryHandleDesc* cmemHandleDesc_ptr = memHandleDesc._ptr if memHandleDesc != None else NULL
    err = ccudart.cudaImportExternalMemory(extMem_out._ptr, cmemHandleDesc_ptr)
    return (cudaError_t(err), extMem_out)

@cython.embedsignature(True)
def cudaExternalMemoryGetMappedBuffer(extMem, bufferDesc : cudaExternalMemoryBufferDesc):
    """ Maps a buffer onto an imported memory object.

    Maps a buffer onto an imported memory object and returns a device
    pointer in `devPtr`.

    The properties of the buffer being mapped must be described in
    `bufferDesc`. The :py:obj:`~.cudaExternalMemoryBufferDesc` structure is
    defined as follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaExternalMemoryBufferDesc.offset` is the offset in
    the memory object where the buffer's base address is.
    :py:obj:`~.cudaExternalMemoryBufferDesc.size` is the size of the
    buffer. :py:obj:`~.cudaExternalMemoryBufferDesc.flags` must be zero.

    The offset and size have to be suitably aligned to match the
    requirements of the external API. Mapping two buffers whose ranges
    overlap may or may not result in the same virtual address being
    returned for the overlapped portion. In such cases, the application
    must ensure that all accesses to that region from the GPU are volatile.
    Otherwise writes made via one address are not guaranteed to be visible
    via the other address, even if they're issued by the same thread. It is
    recommended that applications map the combined range instead of mapping
    separate buffers and then apply the appropriate offsets to the returned
    pointer to derive the individual buffers.

    The returned pointer `devPtr` must be freed using :py:obj:`~.cudaFree`.

    Parameters
    ----------
    extMem : :py:obj:`~.cudaExternalMemory_t`
        Handle to external memory object
    bufferDesc : :py:obj:`~.cudaExternalMemoryBufferDesc`
        Buffer descriptor

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    devPtr : Any
        Returned device pointer to buffer

    See Also
    --------
    :py:obj:`~.cudaImportExternalMemory`, :py:obj:`~.cudaDestroyExternalMemory`, :py:obj:`~.cudaExternalMemoryGetMappedMipmappedArray`
    """
    cdef ccudart.cudaExternalMemory_t cextMem
    if extMem is None:
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>0
    elif isinstance(extMem, (cudaExternalMemory_t)):
        pextMem = int(extMem)
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>pextMem
    else:
        pextMem = int(cudaExternalMemory_t(extMem))
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>pextMem

    cdef void_ptr devPtr = 0
    cdef ccudart.cudaExternalMemoryBufferDesc* cbufferDesc_ptr = bufferDesc._ptr if bufferDesc != None else NULL
    err = ccudart.cudaExternalMemoryGetMappedBuffer(<void**>&devPtr, cextMem, cbufferDesc_ptr)
    return (cudaError_t(err), devPtr)

@cython.embedsignature(True)
def cudaExternalMemoryGetMappedMipmappedArray(extMem, mipmapDesc : cudaExternalMemoryMipmappedArrayDesc):
    """ Maps a CUDA mipmapped array onto an external memory object.

    Maps a CUDA mipmapped array onto an external object and returns a
    handle to it in `mipmap`.

    The properties of the CUDA mipmapped array being mapped must be
    described in `mipmapDesc`. The structure
    :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc` is defined as follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc.offset` is the
    offset in the memory object where the base level of the mipmap chain
    is. :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc.formatDesc`
    describes the format of the data.
    :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc.extent` specifies the
    dimensions of the base level of the mipmap chain.
    :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc.flags` are flags
    associated with CUDA mipmapped arrays. For further details, please
    refer to the documentation for :py:obj:`~.cudaMalloc3DArray`. Note that
    if the mipmapped array is bound as a color target in the graphics API,
    then the flag :py:obj:`~.cudaArrayColorAttachment` must be specified in
    :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc.flags`.
    :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc.numLevels` specifies
    the total number of levels in the mipmap chain.

    The returned CUDA mipmapped array must be freed using
    :py:obj:`~.cudaFreeMipmappedArray`.

    Parameters
    ----------
    extMem : :py:obj:`~.cudaExternalMemory_t`
        Handle to external memory object
    mipmapDesc : :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc`
        CUDA array descriptor

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    mipmap : :py:obj:`~.cudaMipmappedArray_t`
        Returned CUDA mipmapped array

    See Also
    --------
    :py:obj:`~.cudaImportExternalMemory`, :py:obj:`~.cudaDestroyExternalMemory`, :py:obj:`~.cudaExternalMemoryGetMappedBuffer`

    Notes
    -----
    On Tegra devices, this API will always attempt to do a compressed mapping when the :py:obj:`~.cudaExternalMemoryHandleDesc.type` is :py:obj:`~.cudaExternalMemoryHandleTypeOpaqueFd`

    If :py:obj:`~.cudaExternalMemoryHandleDesc.type` is :py:obj:`~.cudaExternalMemoryHandleTypeNvSciBuf`, then :py:obj:`~.cudaExternalMemoryMipmappedArrayDesc.numLevels` must not be greater than 1.
    """
    cdef ccudart.cudaExternalMemory_t cextMem
    if extMem is None:
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>0
    elif isinstance(extMem, (cudaExternalMemory_t)):
        pextMem = int(extMem)
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>pextMem
    else:
        pextMem = int(cudaExternalMemory_t(extMem))
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>pextMem

    cdef cudaMipmappedArray_t mipmap = cudaMipmappedArray_t()
    cdef ccudart.cudaExternalMemoryMipmappedArrayDesc* cmipmapDesc_ptr = mipmapDesc._ptr if mipmapDesc != None else NULL
    err = ccudart.cudaExternalMemoryGetMappedMipmappedArray(mipmap._ptr, cextMem, cmipmapDesc_ptr)
    return (cudaError_t(err), mipmap)

@cython.embedsignature(True)
def cudaDestroyExternalMemory(extMem):
    """ Destroys an external memory object.

    Destroys the specified external memory object. Any existing buffers and
    CUDA mipmapped arrays mapped onto this object must no longer be used
    and must be explicitly freed using :py:obj:`~.cudaFree` and
    :py:obj:`~.cudaFreeMipmappedArray` respectively.

    Parameters
    ----------
    extMem : :py:obj:`~.cudaExternalMemory_t`
        External memory object to be destroyed

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaImportExternalMemory`, :py:obj:`~.cudaExternalMemoryGetMappedBuffer`, :py:obj:`~.cudaExternalMemoryGetMappedMipmappedArray`
    """
    cdef ccudart.cudaExternalMemory_t cextMem
    if extMem is None:
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>0
    elif isinstance(extMem, (cudaExternalMemory_t)):
        pextMem = int(extMem)
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>pextMem
    else:
        pextMem = int(cudaExternalMemory_t(extMem))
        cextMem = <ccudart.cudaExternalMemory_t><void_ptr>pextMem

    err = ccudart.cudaDestroyExternalMemory(cextMem)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaImportExternalSemaphore(semHandleDesc : cudaExternalSemaphoreHandleDesc):
    """ Imports an external semaphore.

    Imports an externally allocated synchronization object and returns a
    handle to that in `extSem_out`.

    The properties of the handle being imported must be described in
    `semHandleDesc`. The :py:obj:`~.cudaExternalSemaphoreHandleDesc` is
    defined as follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` specifies the
    type of handle being imported.
    :py:obj:`~.cudaExternalSemaphoreHandleType` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueFd`, then
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::fd must be a valid
    file descriptor referencing a synchronization object. Ownership of the
    file descriptor is transferred to the CUDA driver when the handle is
    imported successfully. Performing any operations on the file descriptor
    after it is imported results in undefined behavior.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueWin32`, then exactly
    one of
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle and
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name must
    not be NULL. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that
    references a synchronization object. Ownership of this handle is not
    transferred to CUDA after the import operation, so the application must
    release the handle using the appropriate system call. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name is not
    NULL, then it must name a valid synchronization object.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueWin32Kmt`, then
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle must
    be non-NULL and
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name must
    be NULL. The handle specified must be a globally shared KMT handle.
    This handle does not hold a reference to the underlying object, and
    thus will be invalid when all references to the synchronization object
    are destroyed.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeD3D12Fence`, then exactly one
    of :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle
    and :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name
    must not be NULL. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that is
    returned by ID3D12Device::CreateSharedHandle when referring to a
    ID3D12Fence object. This handle holds a reference to the underlying
    object. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name is not
    NULL, then it must name a valid synchronization object that refers to a
    valid ID3D12Fence object.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeD3D11Fence`, then exactly one
    of :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle
    and :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name
    must not be NULL. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that is
    returned by ID3D11Fence::CreateSharedHandle. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name is not
    NULL, then it must name a valid synchronization object that refers to a
    valid ID3D11Fence object.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeNvSciSync`, then
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::nvSciSyncObj
    represents a valid NvSciSyncObj.

    :py:obj:`~.cudaExternalSemaphoreHandleTypeKeyedMutex`, then exactly one
    of :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle
    and :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name
    must not be NULL. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle is
    not NULL, then it represent a valid shared NT handle that is returned
    by IDXGIResource1::CreateSharedHandle when referring to a
    IDXGIKeyedMutex object.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeKeyedMutexKmt`, then
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle must
    be non-NULL and
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name must
    be NULL. The handle specified must represent a valid KMT handle that is
    returned by IDXGIResource::GetSharedHandle when referring to a
    IDXGIKeyedMutex object.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeTimelineSemaphoreFd`, then
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::fd must be a valid
    file descriptor referencing a synchronization object. Ownership of the
    file descriptor is transferred to the CUDA driver when the handle is
    imported successfully. Performing any operations on the file descriptor
    after it is imported results in undefined behavior.

    If :py:obj:`~.cudaExternalSemaphoreHandleDesc.type` is
    :py:obj:`~.cudaExternalSemaphoreHandleTypeTimelineSemaphoreWin32`, then
    exactly one of
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle and
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name must
    not be NULL. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that
    references a synchronization object. Ownership of this handle is not
    transferred to CUDA after the import operation, so the application must
    release the handle using the appropriate system call. If
    :py:obj:`~.cudaExternalSemaphoreHandleDesc`::handle::win32::name is not
    NULL, then it must name a valid synchronization object.

    Parameters
    ----------
    semHandleDesc : :py:obj:`~.cudaExternalSemaphoreHandleDesc`
        Semaphore import handle descriptor

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    extSem_out : :py:obj:`~.cudaExternalSemaphore_t`
        Returned handle to an external semaphore

    See Also
    --------
    :py:obj:`~.cudaDestroyExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef cudaExternalSemaphore_t extSem_out = cudaExternalSemaphore_t()
    cdef ccudart.cudaExternalSemaphoreHandleDesc* csemHandleDesc_ptr = semHandleDesc._ptr if semHandleDesc != None else NULL
    err = ccudart.cudaImportExternalSemaphore(extSem_out._ptr, csemHandleDesc_ptr)
    return (cudaError_t(err), extSem_out)

@cython.embedsignature(True)
def cudaSignalExternalSemaphoresAsync(extSemArray : List[cudaExternalSemaphore_t], paramsArray : List[cudaExternalSemaphoreSignalParams], unsigned int numExtSems, stream):
    """ Signals a set of external semaphore objects.

    Enqueues a signal operation on a set of externally allocated semaphore
    object in the specified stream. The operations will be executed when
    all prior operations in the stream complete.

    The exact semantics of signaling a semaphore depends on the type of the
    object.

    If the semaphore object is any one of the following types:
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueFd`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueWin32`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueWin32Kmt` then
    signaling the semaphore will set it to the signaled state.

    If the semaphore object is any one of the following types:
    :py:obj:`~.cudaExternalSemaphoreHandleTypeD3D12Fence`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeD3D11Fence`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeTimelineSemaphoreFd`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeTimelineSemaphoreWin32` then
    the semaphore will be set to the value specified in
    :py:obj:`~.cudaExternalSemaphoreSignalParams`::params::fence::value.

    If the semaphore object is of the type
    :py:obj:`~.cudaExternalSemaphoreHandleTypeNvSciSync` this API sets
    :py:obj:`~.cudaExternalSemaphoreSignalParams`::params::nvSciSync::fence
    to a value that can be used by subsequent waiters of the same NvSciSync
    object to order operations with those currently submitted in `stream`.
    Such an update will overwrite previous contents of
    :py:obj:`~.cudaExternalSemaphoreSignalParams`::params::nvSciSync::fence.
    By deefault, signaling such an external semaphore object causes
    appropriate memory synchronization operations to be performed over all
    the external memory objects that are imported as
    :py:obj:`~.cudaExternalMemoryHandleTypeNvSciBuf`. This ensures that any
    subsequent accesses made by other importers of the same set of NvSciBuf
    memory object(s) are coherent. These operations can be skipped by
    specifying the flag
    :py:obj:`~.cudaExternalSemaphoreSignalSkipNvSciBufMemSync`, which can
    be used as a performance optimization when data coherency is not
    required. But specifying this flag in scenarios where data coherency is
    required results in undefined behavior. Also, for semaphore object of
    the type :py:obj:`~.cudaExternalSemaphoreHandleTypeNvSciSync`, if the
    NvSciSyncAttrList used to create the NvSciSyncObj had not set the flags
    in :py:obj:`~.cudaDeviceGetNvSciSyncAttributes` to
    cudaNvSciSyncAttrSignal, this API will return cudaErrorNotSupported.

    If the semaphore object is any one of the following types:
    :py:obj:`~.cudaExternalSemaphoreHandleTypeKeyedMutex`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeKeyedMutexKmt`, then the
    keyed mutex will be released with the key specified in
    :py:obj:`~.cudaExternalSemaphoreSignalParams`::params::keyedmutex::key.

    Parameters
    ----------
    extSemArray : List[:py:obj:`~.cudaExternalSemaphore_t`]
        Set of external semaphores to be signaled
    paramsArray : List[:py:obj:`~.cudaExternalSemaphoreSignalParams`]
        Array of semaphore parameters
    numExtSems : unsigned int
        Number of semaphores to signal
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to enqueue the signal operations in

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaDestroyExternalSemaphore`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    paramsArray = [] if paramsArray is None else paramsArray
    if not all(isinstance(_x, (cudaExternalSemaphoreSignalParams)) for _x in paramsArray):
        raise TypeError("Argument 'paramsArray' is not instance of type (expected List[ccudart.cudaExternalSemaphoreSignalParams]")

    extSemArray = [] if extSemArray is None else extSemArray
    if not all(isinstance(_x, (cudaExternalSemaphore_t)) for _x in extSemArray):
        raise TypeError("Argument 'extSemArray' is not instance of type (expected List[ccudart.cudaExternalSemaphore_t]")

    cdef ccudart.cudaExternalSemaphore_t* cextSemArray = NULL
    if len(extSemArray) > 0:
        cextSemArray = <ccudart.cudaExternalSemaphore_t*> calloc(len(extSemArray), sizeof(ccudart.cudaExternalSemaphore_t))
        if cextSemArray is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(extSemArray)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphore_t)))
        else:
            for idx in range(len(extSemArray)):
                cextSemArray[idx] = (<cudaExternalSemaphore_t>extSemArray[idx])._ptr[0]

    cdef ccudart.cudaExternalSemaphoreSignalParams* cparamsArray = NULL
    if len(paramsArray) > 0:
        cparamsArray = <ccudart.cudaExternalSemaphoreSignalParams*> calloc(len(paramsArray), sizeof(ccudart.cudaExternalSemaphoreSignalParams))
        if cparamsArray is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(paramsArray)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphoreSignalParams)))
        for idx in range(len(paramsArray)):
            memcpy(&cparamsArray[idx], (<cudaExternalSemaphoreSignalParams>paramsArray[idx])._ptr, sizeof(ccudart.cudaExternalSemaphoreSignalParams))

    if numExtSems > len(extSemArray): raise RuntimeError("List is too small: " + str(len(extSemArray)) + " < " + str(numExtSems))
    if numExtSems > len(paramsArray): raise RuntimeError("List is too small: " + str(len(paramsArray)) + " < " + str(numExtSems))
    err = ccudart.cudaSignalExternalSemaphoresAsync((<cudaExternalSemaphore_t>extSemArray[0])._ptr if len(extSemArray) == 1 else cextSemArray, (<cudaExternalSemaphoreSignalParams>paramsArray[0])._ptr if len(paramsArray) == 1 else cparamsArray, numExtSems, cstream)
    if cextSemArray is not NULL:
        free(cextSemArray)
    if cparamsArray is not NULL:
        free(cparamsArray)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaWaitExternalSemaphoresAsync(extSemArray : List[cudaExternalSemaphore_t], paramsArray : List[cudaExternalSemaphoreWaitParams], unsigned int numExtSems, stream):
    """ Waits on a set of external semaphore objects.

    Enqueues a wait operation on a set of externally allocated semaphore
    object in the specified stream. The operations will be executed when
    all prior operations in the stream complete.

    The exact semantics of waiting on a semaphore depends on the type of
    the object.

    If the semaphore object is any one of the following types:
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueFd`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueWin32`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeOpaqueWin32Kmt` then waiting
    on the semaphore will wait until the semaphore reaches the signaled
    state. The semaphore will then be reset to the unsignaled state.
    Therefore for every signal operation, there can only be one wait
    operation.

    If the semaphore object is any one of the following types:
    :py:obj:`~.cudaExternalSemaphoreHandleTypeD3D12Fence`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeD3D11Fence`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeTimelineSemaphoreFd`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeTimelineSemaphoreWin32` then
    waiting on the semaphore will wait until the value of the semaphore is
    greater than or equal to
    :py:obj:`~.cudaExternalSemaphoreWaitParams`::params::fence::value.

    If the semaphore object is of the type
    :py:obj:`~.cudaExternalSemaphoreHandleTypeNvSciSync` then, waiting on
    the semaphore will wait until the
    :py:obj:`~.cudaExternalSemaphoreSignalParams`::params::nvSciSync::fence
    is signaled by the signaler of the NvSciSyncObj that was associated
    with this semaphore object. By default, waiting on such an external
    semaphore object causes appropriate memory synchronization operations
    to be performed over all external memory objects that are imported as
    :py:obj:`~.cudaExternalMemoryHandleTypeNvSciBuf`. This ensures that any
    subsequent accesses made by other importers of the same set of NvSciBuf
    memory object(s) are coherent. These operations can be skipped by
    specifying the flag
    :py:obj:`~.cudaExternalSemaphoreWaitSkipNvSciBufMemSync`, which can be
    used as a performance optimization when data coherency is not required.
    But specifying this flag in scenarios where data coherency is required
    results in undefined behavior. Also, for semaphore object of the type
    :py:obj:`~.cudaExternalSemaphoreHandleTypeNvSciSync`, if the
    NvSciSyncAttrList used to create the NvSciSyncObj had not set the flags
    in :py:obj:`~.cudaDeviceGetNvSciSyncAttributes` to
    cudaNvSciSyncAttrWait, this API will return cudaErrorNotSupported.

    If the semaphore object is any one of the following types:
    :py:obj:`~.cudaExternalSemaphoreHandleTypeKeyedMutex`,
    :py:obj:`~.cudaExternalSemaphoreHandleTypeKeyedMutexKmt`, then the
    keyed mutex will be acquired when it is released with the key specified
    in
    :py:obj:`~.cudaExternalSemaphoreSignalParams`::params::keyedmutex::key
    or until the timeout specified by
    :py:obj:`~.cudaExternalSemaphoreSignalParams`::params::keyedmutex::timeoutMs
    has lapsed. The timeout interval can either be a finite value specified
    in milliseconds or an infinite value. In case an infinite value is
    specified the timeout never elapses. The windows INFINITE macro must be
    used to specify infinite timeout

    Parameters
    ----------
    extSemArray : List[:py:obj:`~.cudaExternalSemaphore_t`]
        External semaphores to be waited on
    paramsArray : List[:py:obj:`~.cudaExternalSemaphoreWaitParams`]
        Array of semaphore parameters
    numExtSems : unsigned int
        Number of semaphores to wait on
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to enqueue the wait operations in

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle` :py:obj:`~.cudaErrorTimeout`

    See Also
    --------
    :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaDestroyExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    paramsArray = [] if paramsArray is None else paramsArray
    if not all(isinstance(_x, (cudaExternalSemaphoreWaitParams)) for _x in paramsArray):
        raise TypeError("Argument 'paramsArray' is not instance of type (expected List[ccudart.cudaExternalSemaphoreWaitParams]")

    extSemArray = [] if extSemArray is None else extSemArray
    if not all(isinstance(_x, (cudaExternalSemaphore_t)) for _x in extSemArray):
        raise TypeError("Argument 'extSemArray' is not instance of type (expected List[ccudart.cudaExternalSemaphore_t]")

    cdef ccudart.cudaExternalSemaphore_t* cextSemArray = NULL
    if len(extSemArray) > 0:
        cextSemArray = <ccudart.cudaExternalSemaphore_t*> calloc(len(extSemArray), sizeof(ccudart.cudaExternalSemaphore_t))
        if cextSemArray is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(extSemArray)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphore_t)))
        else:
            for idx in range(len(extSemArray)):
                cextSemArray[idx] = (<cudaExternalSemaphore_t>extSemArray[idx])._ptr[0]

    cdef ccudart.cudaExternalSemaphoreWaitParams* cparamsArray = NULL
    if len(paramsArray) > 0:
        cparamsArray = <ccudart.cudaExternalSemaphoreWaitParams*> calloc(len(paramsArray), sizeof(ccudart.cudaExternalSemaphoreWaitParams))
        if cparamsArray is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(paramsArray)) + 'x' + str(sizeof(ccudart.cudaExternalSemaphoreWaitParams)))
        for idx in range(len(paramsArray)):
            memcpy(&cparamsArray[idx], (<cudaExternalSemaphoreWaitParams>paramsArray[idx])._ptr, sizeof(ccudart.cudaExternalSemaphoreWaitParams))

    if numExtSems > len(extSemArray): raise RuntimeError("List is too small: " + str(len(extSemArray)) + " < " + str(numExtSems))
    if numExtSems > len(paramsArray): raise RuntimeError("List is too small: " + str(len(paramsArray)) + " < " + str(numExtSems))
    err = ccudart.cudaWaitExternalSemaphoresAsync((<cudaExternalSemaphore_t>extSemArray[0])._ptr if len(extSemArray) == 1 else cextSemArray, (<cudaExternalSemaphoreWaitParams>paramsArray[0])._ptr if len(paramsArray) == 1 else cparamsArray, numExtSems, cstream)
    if cextSemArray is not NULL:
        free(cextSemArray)
    if cparamsArray is not NULL:
        free(cparamsArray)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDestroyExternalSemaphore(extSem):
    """ Destroys an external semaphore.

    Destroys an external semaphore object and releases any references to
    the underlying resource. Any outstanding signals or waits must have
    completed before the semaphore is destroyed.

    Parameters
    ----------
    extSem : :py:obj:`~.cudaExternalSemaphore_t`
        External semaphore to be destroyed

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef ccudart.cudaExternalSemaphore_t cextSem
    if extSem is None:
        cextSem = <ccudart.cudaExternalSemaphore_t><void_ptr>0
    elif isinstance(extSem, (cudaExternalSemaphore_t)):
        pextSem = int(extSem)
        cextSem = <ccudart.cudaExternalSemaphore_t><void_ptr>pextSem
    else:
        pextSem = int(cudaExternalSemaphore_t(extSem))
        cextSem = <ccudart.cudaExternalSemaphore_t><void_ptr>pextSem

    err = ccudart.cudaDestroyExternalSemaphore(cextSem)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaFuncSetCacheConfig(func, cacheConfig not None : cudaFuncCache):
    """ Sets the preferred cache configuration for a device function.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this sets through `cacheConfig` the preferred cache
    configuration for the function specified via `func`. This is only a
    preference. The runtime will use the requested configuration if
    possible, but it is free to choose a different configuration if
    required to execute `func`.

    `func` is a device function symbol and must be declared as a `None`
    function. If the specified function does not exist, then
    :py:obj:`~.cudaErrorInvalidDeviceFunction` is returned. For templated
    functions, pass the function symbol as follows:
    func_name<template_arg_0,...,template_arg_N>

    This setting does nothing on devices where the size of the L1 cache and
    shared memory are fixed.

    Launching a kernel with a different preference than the most recent
    preference setting may insert a device-side synchronization point.

    The supported cache configurations are:

    - :py:obj:`~.cudaFuncCachePreferNone`: no preference for shared memory
      or L1 (default)

    - :py:obj:`~.cudaFuncCachePreferShared`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.cudaFuncCachePreferL1`: prefer larger L1 cache and smaller
      shared memory

    - :py:obj:`~.cudaFuncCachePreferEqual`: prefer equal size L1 cache and
      shared memory

    Parameters
    ----------
    func : Any
        Device function symbol
    cacheConfig : :py:obj:`~.cudaFuncCache`
        Requested cache configuration

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDeviceFunction`2

    See Also
    --------
    cudaFuncSetCacheConfig (C++ API), :py:obj:`~.cudaFuncGetAttributes (C API)`, :py:obj:`~.cudaLaunchKernel (C API)`, :py:obj:`~.cudaThreadGetCacheConfig`, :py:obj:`~.cudaThreadSetCacheConfig`, :py:obj:`~.cuFuncSetCacheConfig`
    """
    cfunc = utils.HelperInputVoidPtr(func)
    cdef void* cfunc_ptr = <void*><void_ptr>cfunc.cptr
    cdef ccudart.cudaFuncCache ccacheConfig = cacheConfig.value
    err = ccudart.cudaFuncSetCacheConfig(cfunc_ptr, ccacheConfig)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaFuncSetSharedMemConfig(func, config not None : cudaSharedMemConfig):
    """ Sets the shared memory configuration for a device function.

    On devices with configurable shared memory banks, this function will
    force all subsequent launches of the specified device function to have
    the given shared memory bank size configuration. On any given launch of
    the function, the shared memory configuration of the device will be
    temporarily changed if needed to suit the function's preferred
    configuration. Changes in shared memory configuration between
    subsequent launches of functions, may introduce a device side
    synchronization point.

    Any per-function setting of shared memory bank size set via
    :py:obj:`~.cudaFuncSetSharedMemConfig` will override the device wide
    setting set by :py:obj:`~.cudaDeviceSetSharedMemConfig`.

    Changing the shared memory bank size will not increase shared memory
    usage or affect occupancy of kernels, but may have major effects on
    performance. Larger bank sizes will allow for greater potential
    bandwidth to shared memory, but will change what kinds of accesses to
    shared memory will result in bank conflicts.

    This function will do nothing on devices with fixed shared memory bank
    size.

    For templated functions, pass the function symbol as follows:
    func_name<template_arg_0,...,template_arg_N>

    The supported bank configurations are:

    - :py:obj:`~.cudaSharedMemBankSizeDefault`: use the device's shared
      memory configuration when launching this function.

    - :py:obj:`~.cudaSharedMemBankSizeFourByte`: set shared memory bank
      width to be four bytes natively when launching this function.

    - :py:obj:`~.cudaSharedMemBankSizeEightByte`: set shared memory bank
      width to be eight bytes natively when launching this function.

    Parameters
    ----------
    func : Any
        Device function symbol
    config : :py:obj:`~.cudaSharedMemConfig`
        Requested shared memory configuration

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDeviceFunction`, :py:obj:`~.cudaErrorInvalidValue`,2

    See Also
    --------
    :py:obj:`~.cudaDeviceSetSharedMemConfig`, :py:obj:`~.cudaDeviceGetSharedMemConfig`, :py:obj:`~.cudaDeviceSetCacheConfig`, :py:obj:`~.cudaDeviceGetCacheConfig`, :py:obj:`~.cudaFuncSetCacheConfig`, :py:obj:`~.cuFuncSetSharedMemConfig`
    """
    cfunc = utils.HelperInputVoidPtr(func)
    cdef void* cfunc_ptr = <void*><void_ptr>cfunc.cptr
    cdef ccudart.cudaSharedMemConfig cconfig = config.value
    err = ccudart.cudaFuncSetSharedMemConfig(cfunc_ptr, cconfig)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaFuncGetAttributes(func):
    """ Find out attributes for a given function.

    This function obtains the attributes of a function specified via
    `func`. `func` is a device function symbol and must be declared as a
    `None` function. The fetched attributes are placed in `attr`. If the
    specified function does not exist, then
    :py:obj:`~.cudaErrorInvalidDeviceFunction` is returned. For templated
    functions, pass the function symbol as follows:
    func_name<template_arg_0,...,template_arg_N>

    Note that some function attributes such as
    :py:obj:`~.maxThreadsPerBlock` may vary based on the device that is
    currently being used.

    Parameters
    ----------
    func : Any
        Device function symbol

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDeviceFunction`2
    attr : :py:obj:`~.cudaFuncAttributes`
        Return pointer to function's attributes

    See Also
    --------
    :py:obj:`~.cudaFuncSetCacheConfig (C API)`, cudaFuncGetAttributes (C++ API), :py:obj:`~.cudaLaunchKernel (C API)`, :py:obj:`~.cuFuncGetAttribute`
    """
    cdef cudaFuncAttributes attr = cudaFuncAttributes()
    cfunc = utils.HelperInputVoidPtr(func)
    cdef void* cfunc_ptr = <void*><void_ptr>cfunc.cptr
    err = ccudart.cudaFuncGetAttributes(attr._ptr, cfunc_ptr)
    return (cudaError_t(err), attr)

@cython.embedsignature(True)
def cudaFuncSetAttribute(func, attr not None : cudaFuncAttribute, int value):
    """ Set attributes for a given function.

    This function sets the attributes of a function specified via `func`.
    The parameter `func` must be a pointer to a function that executes on
    the device. The parameter specified by `func` must be declared as a
    `None` function. The enumeration defined by `attr` is set to the value
    defined by `value`. If the specified function does not exist, then
    :py:obj:`~.cudaErrorInvalidDeviceFunction` is returned. If the
    specified attribute cannot be written, or if the value is incorrect,
    then :py:obj:`~.cudaErrorInvalidValue` is returned.

    Valid values for `attr` are:

    - :py:obj:`~.cudaFuncAttributeMaxDynamicSharedMemorySize` - The
      requested maximum size in bytes of dynamically-allocated shared
      memory. The sum of this value and the function attribute
      :py:obj:`~.sharedSizeBytes` cannot exceed the device attribute
      :py:obj:`~.cudaDevAttrMaxSharedMemoryPerBlockOptin`. The maximal size
      of requestable dynamic shared memory may differ by GPU architecture.

    - :py:obj:`~.cudaFuncAttributePreferredSharedMemoryCarveout` - On
      devices where the L1 cache and shared memory use the same hardware
      resources, this sets the shared memory carveout preference, in
      percent of the total shared memory. See
      :py:obj:`~.cudaDevAttrMaxSharedMemoryPerMultiprocessor`. This is only
      a hint, and the driver can choose a different ratio if required to
      execute the function.

    cudaLaunchKernel (C++ API), cudaFuncSetCacheConfig (C++ API),
    :py:obj:`~.cudaFuncGetAttributes (C API)`,

    Parameters
    ----------
    func : Any
        Function to get attributes of
    attr : :py:obj:`~.cudaFuncAttribute`
        Attribute to set
    value : int
        Value to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDeviceFunction`, :py:obj:`~.cudaErrorInvalidValue`
    """
    cfunc = utils.HelperInputVoidPtr(func)
    cdef void* cfunc_ptr = <void*><void_ptr>cfunc.cptr
    cdef ccudart.cudaFuncAttribute cattr = attr.value
    err = ccudart.cudaFuncSetAttribute(cfunc_ptr, cattr, value)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaSetDoubleForDevice(double d):
    """ Converts a double argument to be executed on a device.

    [Deprecated]

    Converts the double value of `d` to an internal float representation if
    the device does not support double arithmetic. If the device does
    natively support doubles, then this function does nothing.

    Parameters
    ----------
    d : double
        Double to convert

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaFuncSetCacheConfig (C API)`, :py:obj:`~.cudaFuncGetAttributes (C API)`, :py:obj:`~.cudaSetDoubleForHost`
    """
    err = ccudart.cudaSetDoubleForDevice(&d)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaSetDoubleForHost(double d):
    """ Converts a double argument after execution on a device.

    [Deprecated]

    Converts the double value of `d` from a potentially internal float
    representation if the device does not support double arithmetic. If the
    device does natively support doubles, then this function does nothing.

    Parameters
    ----------
    d : double
        Double to convert

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaFuncSetCacheConfig (C API)`, :py:obj:`~.cudaFuncGetAttributes (C API)`, :py:obj:`~.cudaSetDoubleForDevice`
    """
    err = ccudart.cudaSetDoubleForHost(&d)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaLaunchHostFunc(stream, fn, userData):
    """ Enqueues a host function call in a stream.

    Enqueues a host function to run in a stream. The function will be
    called after currently enqueued work and will block work added after
    it.

    The host function must not make any CUDA API calls. Attempting to use a
    CUDA API may result in :py:obj:`~.cudaErrorNotPermitted`, but this is
    not required. The host function must not perform any synchronization
    that may depend on outstanding CUDA work not mandated to run earlier.
    Host functions without a mandated order (such as in independent
    streams) execute in undefined order and may be serialized.

    For the purposes of Unified Memory, execution makes a number of
    guarantees:

    - The stream is considered idle for the duration of the function's
      execution. Thus, for example, the function may always use memory
      attached to the stream it was enqueued in.

    - The start of execution of the function has the same effect as
      synchronizing an event recorded in the same stream immediately prior
      to the function. It thus synchronizes streams which have been
      "joined" prior to the function.

    - Adding device work to any stream does not have the effect of making
      the stream active until all preceding host functions and stream
      callbacks have executed. Thus, for example, a function might use
      global attached memory even if work has been added to another stream,
      if the work has been ordered behind the function call with an event.

    - Completion of the function does not cause a stream to become active
      except as described above. The stream will remain idle if no device
      work follows the function, and will remain idle across consecutive
      host functions or stream callbacks without device work in between.
      Thus, for example, stream synchronization can be done by signaling
      from a host function at the end of the stream.

    Note that, in constrast to :py:obj:`~.cuStreamAddCallback`, the
    function will not be called in the event of an error in the CUDA
    context.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to enqueue function call in
    fn : :py:obj:`~.cudaHostFn_t`
        The function to call once preceding stream operations are complete
    userData : Any
        User-specified data to be passed to the function

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorNotSupported`

    See Also
    --------
    :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamQuery`, :py:obj:`~.cudaStreamSynchronize`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaStreamDestroy`, :py:obj:`~.cudaMallocManaged`, :py:obj:`~.cudaStreamAttachMemAsync`, :py:obj:`~.cudaStreamAddCallback`, :py:obj:`~.cuLaunchHostFunc`
    """
    cdef ccudart.cudaHostFn_t cfn
    if fn is None:
        cfn = <ccudart.cudaHostFn_t><void_ptr>0
    elif isinstance(fn, (cudaHostFn_t)):
        pfn = int(fn)
        cfn = <ccudart.cudaHostFn_t><void_ptr>pfn
    else:
        pfn = int(cudaHostFn_t(fn))
        cfn = <ccudart.cudaHostFn_t><void_ptr>pfn

    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cuserData = utils.HelperInputVoidPtr(userData)
    cdef void* cuserData_ptr = <void*><void_ptr>cuserData.cptr
    with nogil:
        err = ccudart.cudaLaunchHostFunc(cstream, cfn, cuserData_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaOccupancyMaxActiveBlocksPerMultiprocessor(func, int blockSize, size_t dynamicSMemSize):
    """ Returns occupancy for a device function.

    Returns in `*numBlocks` the maximum number of active blocks per
    streaming multiprocessor for the device function.

    Parameters
    ----------
    func : Any
        Kernel function for which occupancy is calculated
    blockSize : int
        Block size the kernel is intended to be launched with
    dynamicSMemSize : size_t
        Per-block dynamic shared memory usage intended, in bytes

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidDeviceFunction`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`,
    numBlocks : int
        Returned occupancy

    See Also
    --------
    :py:obj:`~.cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags`, cudaOccupancyMaxPotentialBlockSize (C++ API), cudaOccupancyMaxPotentialBlockSizeWithFlags (C++ API), cudaOccupancyMaxPotentialBlockSizeVariableSMem (C++ API), cudaOccupancyMaxPotentialBlockSizeVariableSMemWithFlags (C++ API), cudaOccupancyAvailableDynamicSMemPerBlock (C++ API), :py:obj:`~.cuOccupancyMaxActiveBlocksPerMultiprocessor`
    """
    cdef int numBlocks = 0
    cfunc = utils.HelperInputVoidPtr(func)
    cdef void* cfunc_ptr = <void*><void_ptr>cfunc.cptr
    err = ccudart.cudaOccupancyMaxActiveBlocksPerMultiprocessor(&numBlocks, cfunc_ptr, blockSize, dynamicSMemSize)
    return (cudaError_t(err), numBlocks)

@cython.embedsignature(True)
def cudaOccupancyAvailableDynamicSMemPerBlock(func, int numBlocks, int blockSize):
    """ Returns dynamic shared memory available per block when launching `numBlocks` blocks on SM.

    Returns in `*dynamicSmemSize` the maximum size of dynamic shared memory
    to allow `numBlocks` blocks per SM.

    Parameters
    ----------
    func : Any
        Kernel function for which occupancy is calculated
    numBlocks : int
        Number of blocks to fit on SM
    blockSize : int
        Size of the block

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidDeviceFunction`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`,
    dynamicSmemSize : int
        Returned maximum dynamic shared memory

    See Also
    --------
    :py:obj:`~.cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags`, cudaOccupancyMaxPotentialBlockSize (C++ API), cudaOccupancyMaxPotentialBlockSizeWithFlags (C++ API), cudaOccupancyMaxPotentialBlockSizeVariableSMem (C++ API), cudaOccupancyMaxPotentialBlockSizeVariableSMemWithFlags (C++ API), :py:obj:`~.cudaOccupancyAvailableDynamicSMemPerBlock`
    """
    cdef size_t dynamicSmemSize = 0
    cfunc = utils.HelperInputVoidPtr(func)
    cdef void* cfunc_ptr = <void*><void_ptr>cfunc.cptr
    err = ccudart.cudaOccupancyAvailableDynamicSMemPerBlock(&dynamicSmemSize, cfunc_ptr, numBlocks, blockSize)
    return (cudaError_t(err), dynamicSmemSize)

@cython.embedsignature(True)
def cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags(func, int blockSize, size_t dynamicSMemSize, unsigned int flags):
    """ Returns occupancy for a device function with the specified flags.

    Returns in `*numBlocks` the maximum number of active blocks per
    streaming multiprocessor for the device function.

    The `flags` parameter controls how special cases are handled. Valid
    flags include:

    - :py:obj:`~.cudaOccupancyDefault`: keeps the default behavior as
      :py:obj:`~.cudaOccupancyMaxActiveBlocksPerMultiprocessor`

    - :py:obj:`~.cudaOccupancyDisableCachingOverride`: This flag suppresses
      the default behavior on platform where global caching affects
      occupancy. On such platforms, if caching is enabled, but per-block SM
      resource usage would result in zero occupancy, the occupancy
      calculator will calculate the occupancy as if caching is disabled.
      Setting this flag makes the occupancy calculator to return 0 in such
      cases. More information can be found about this feature in the
      "Unified L1/Texture Cache" section of the Maxwell tuning guide.

    Parameters
    ----------
    func : Any
        Kernel function for which occupancy is calculated
    blockSize : int
        Block size the kernel is intended to be launched with
    dynamicSMemSize : size_t
        Per-block dynamic shared memory usage intended, in bytes
    flags : unsigned int
        Requested behavior for the occupancy calculator

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidDeviceFunction`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`,
    numBlocks : int
        Returned occupancy

    See Also
    --------
    :py:obj:`~.cudaOccupancyMaxActiveBlocksPerMultiprocessor`, cudaOccupancyMaxPotentialBlockSize (C++ API), cudaOccupancyMaxPotentialBlockSizeWithFlags (C++ API), cudaOccupancyMaxPotentialBlockSizeVariableSMem (C++ API), cudaOccupancyMaxPotentialBlockSizeVariableSMemWithFlags (C++ API), cudaOccupancyAvailableDynamicSMemPerBlock (C++ API), :py:obj:`~.cuOccupancyMaxActiveBlocksPerMultiprocessorWithFlags`
    """
    cdef int numBlocks = 0
    cfunc = utils.HelperInputVoidPtr(func)
    cdef void* cfunc_ptr = <void*><void_ptr>cfunc.cptr
    err = ccudart.cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags(&numBlocks, cfunc_ptr, blockSize, dynamicSMemSize, flags)
    return (cudaError_t(err), numBlocks)

@cython.embedsignature(True)
def cudaMallocManaged(size_t size, unsigned int flags):
    """ Allocates memory that will be automatically managed by the Unified Memory system.

    Allocates `size` bytes of managed memory on the device and returns in
    `*devPtr` a pointer to the allocated memory. If the device doesn't
    support allocating managed memory, :py:obj:`~.cudaErrorNotSupported` is
    returned. Support for managed memory can be queried using the device
    attribute :py:obj:`~.cudaDevAttrManagedMemory`. The allocated memory is
    suitably aligned for any kind of variable. The memory is not cleared.
    If `size` is 0, :py:obj:`~.cudaMallocManaged` returns
    :py:obj:`~.cudaErrorInvalidValue`. The pointer is valid on the CPU and
    on all GPUs in the system that support managed memory. All accesses to
    this pointer must obey the Unified Memory programming model.

    `flags` specifies the default stream association for this allocation.
    `flags` must be one of :py:obj:`~.cudaMemAttachGlobal` or
    :py:obj:`~.cudaMemAttachHost`. The default value for `flags` is
    :py:obj:`~.cudaMemAttachGlobal`. If :py:obj:`~.cudaMemAttachGlobal` is
    specified, then this memory is accessible from any stream on any
    device. If :py:obj:`~.cudaMemAttachHost` is specified, then the
    allocation should not be accessed from devices that have a zero value
    for the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess`; an explicit call to
    :py:obj:`~.cudaStreamAttachMemAsync` will be required to enable access
    on such devices.

    If the association is later changed via
    :py:obj:`~.cudaStreamAttachMemAsync` to a single stream, the default
    association, as specifed during :py:obj:`~.cudaMallocManaged`, is
    restored when that stream is destroyed. For managed variables, the
    default association is always :py:obj:`~.cudaMemAttachGlobal`. Note
    that destroying a stream is an asynchronous operation, and as a result,
    the change to default association won't happen until all work in the
    stream has completed.

    Memory allocated with :py:obj:`~.cudaMallocManaged` should be released
    with :py:obj:`~.cudaFree`.

    Device memory oversubscription is possible for GPUs that have a non-
    zero value for the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess`. Managed memory on such
    GPUs may be evicted from device memory to host memory at any time by
    the Unified Memory driver in order to make room for other allocations.

    In a multi-GPU system where all GPUs have a non-zero value for the
    device attribute :py:obj:`~.cudaDevAttrConcurrentManagedAccess`,
    managed memory may not be populated when this API returns and instead
    may be populated on access. In such systems, managed memory can migrate
    to any processor's memory at any time. The Unified Memory driver will
    employ heuristics to maintain data locality and prevent excessive page
    faults to the extent possible. The application can also guide the
    driver about memory usage patterns via :py:obj:`~.cudaMemAdvise`. The
    application can also explicitly migrate memory to a desired processor's
    memory via :py:obj:`~.cudaMemPrefetchAsync`.

    In a multi-GPU system where all of the GPUs have a zero value for the
    device attribute :py:obj:`~.cudaDevAttrConcurrentManagedAccess` and all
    the GPUs have peer-to-peer support with each other, the physical
    storage for managed memory is created on the GPU which is active at the
    time :py:obj:`~.cudaMallocManaged` is called. All other GPUs will
    reference the data at reduced bandwidth via peer mappings over the PCIe
    bus. The Unified Memory driver does not migrate memory among such GPUs.

    In a multi-GPU system where not all GPUs have peer-to-peer support with
    each other and where the value of the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess` is zero for at least one
    of those GPUs, the location chosen for physical storage of managed
    memory is system-dependent.

    - On Linux, the location chosen will be device memory as long as the
      current set of active contexts are on devices that either have peer-
      to-peer support with each other or have a non-zero value for the
      device attribute :py:obj:`~.cudaDevAttrConcurrentManagedAccess`. If
      there is an active context on a GPU that does not have a non-zero
      value for that device attribute and it does not have peer-to-peer
      support with the other devices that have active contexts on them,
      then the location for physical storage will be 'zero-copy' or host
      memory. Note that this means that managed memory that is located in
      device memory is migrated to host memory if a new context is created
      on a GPU that doesn't have a non-zero value for the device attribute
      and does not support peer-to-peer with at least one of the other
      devices that has an active context. This in turn implies that context
      creation may fail if there is insufficient host memory to migrate all
      managed allocations.

    - On Windows, the physical storage is always created in 'zero-copy' or
      host memory. All GPUs will reference the data at reduced bandwidth
      over the PCIe bus. In these circumstances, use of the environment
      variable CUDA_VISIBLE_DEVICES is recommended to restrict CUDA to only
      use those GPUs that have peer-to-peer support. Alternatively, users
      can also set CUDA_MANAGED_FORCE_DEVICE_ALLOC to a non-zero value to
      force the driver to always use device memory for physical storage.
      When this environment variable is set to a non-zero value, all
      devices used in that process that support managed memory have to be
      peer-to-peer compatible with each other. The error
      :py:obj:`~.cudaErrorInvalidDevice` will be returned if a device that
      supports managed memory is used and it is not peer-to-peer compatible
      with any of the other managed memory supporting devices that were
      previously used in that process, even if :py:obj:`~.cudaDeviceReset`
      has been called on those devices. These environment variables are
      described in the CUDA programming guide under the "CUDA environment
      variables" section.

    Parameters
    ----------
    size : size_t
        Requested allocation size in bytes
    flags : unsigned int
        Must be either :py:obj:`~.cudaMemAttachGlobal` or
        :py:obj:`~.cudaMemAttachHost` (defaults to
        :py:obj:`~.cudaMemAttachGlobal`)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorMemoryAllocation`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`
    devPtr : Any
        Pointer to allocated device memory

    See Also
    --------
    :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cudaDeviceGetAttribute`, :py:obj:`~.cudaStreamAttachMemAsync`, :py:obj:`~.cuMemAllocManaged`
    """
    cdef void_ptr devPtr = 0
    with nogil:
        err = ccudart.cudaMallocManaged(<void**>&devPtr, size, flags)
    return (cudaError_t(err), devPtr)

@cython.embedsignature(True)
def cudaMalloc(size_t size):
    """ Allocate memory on the device.

    Allocates `size` bytes of linear memory on the device and returns in
    `*devPtr` a pointer to the allocated memory. The allocated memory is
    suitably aligned for any kind of variable. The memory is not cleared.
    :py:obj:`~.cudaMalloc()` returns :py:obj:`~.cudaErrorMemoryAllocation`
    in case of failure.

    The device version of :py:obj:`~.cudaFree` cannot be used with a
    `*devPtr` allocated using the host API, and vice versa.

    Parameters
    ----------
    size : size_t
        Requested allocation size in bytes

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    devPtr : Any
        Pointer to allocated device memory

    See Also
    --------
    :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMemAlloc`
    """
    cdef void_ptr devPtr = 0
    with nogil:
        err = ccudart.cudaMalloc(<void**>&devPtr, size)
    return (cudaError_t(err), devPtr)

@cython.embedsignature(True)
def cudaMallocHost(size_t size):
    """ Allocates page-locked memory on the host.

    Allocates `size` bytes of host memory that is page-locked and
    accessible to the device. The driver tracks the virtual memory ranges
    allocated with this function and automatically accelerates calls to
    functions such as :py:obj:`~.cudaMemcpy`*(). Since the memory can be
    accessed directly by the device, it can be read or written with much
    higher bandwidth than pageable memory obtained with functions such as
    :py:obj:`~.malloc()`. Allocating excessive amounts of memory with
    :py:obj:`~.cudaMallocHost()` may degrade system performance, since it
    reduces the amount of memory available to the system for paging. As a
    result, this function is best used sparingly to allocate staging areas
    for data exchange between host and device.

    Parameters
    ----------
    size : size_t
        Requested allocation size in bytes

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    ptr : Any
        Pointer to allocated host memory

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaFreeArray`, cudaMallocHost (C++ API), :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMemAllocHost`
    """
    cdef void_ptr ptr = 0
    err = ccudart.cudaMallocHost(<void**>&ptr, size)
    return (cudaError_t(err), ptr)

@cython.embedsignature(True)
def cudaMallocPitch(size_t width, size_t height):
    """ Allocates pitched memory on the device.

    Allocates at least `width` (in bytes) * `height` bytes of linear memory
    on the device and returns in `*devPtr` a pointer to the allocated
    memory. The function may pad the allocation to ensure that
    corresponding pointers in any given row will continue to meet the
    alignment requirements for coalescing as the address is updated from
    row to row. The pitch returned in `*pitch` by
    :py:obj:`~.cudaMallocPitch()` is the width in bytes of the allocation.
    The intended usage of `pitch` is as a separate parameter of the
    allocation, used to compute addresses within the 2D array. Given the
    row and column of an array element of type `T`, the address is computed
    as:

    **View CUDA Toolkit Documentation for a C++ code example**

    For allocations of 2D arrays, it is recommended that programmers
    consider performing pitch allocations using
    :py:obj:`~.cudaMallocPitch()`. Due to pitch alignment restrictions in
    the hardware, this is especially true if the application will be
    performing 2D memory copies between different regions of device memory
    (whether linear memory or CUDA arrays).

    Parameters
    ----------
    width : size_t
        Requested pitched allocation width (in bytes)
    height : size_t
        Requested pitched allocation height

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    devPtr : Any
        Pointer to allocated pitched device memory
    pitch : int
        Pitch for allocation

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMemAllocPitch`
    """
    cdef void_ptr devPtr = 0
    cdef size_t pitch = 0
    err = ccudart.cudaMallocPitch(<void**>&devPtr, &pitch, width, height)
    return (cudaError_t(err), devPtr, pitch)

@cython.embedsignature(True)
def cudaMallocArray(desc : cudaChannelFormatDesc, size_t width, size_t height, unsigned int flags):
    """ Allocate an array on the device.

    Allocates a CUDA array according to the
    :py:obj:`~.cudaChannelFormatDesc` structure `desc` and returns a handle
    to the new CUDA array in `*array`.

    The :py:obj:`~.cudaChannelFormatDesc` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaChannelFormatKind` is one of
    :py:obj:`~.cudaChannelFormatKindSigned`,
    :py:obj:`~.cudaChannelFormatKindUnsigned`, or
    :py:obj:`~.cudaChannelFormatKindFloat`.

    The `flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.cudaArrayDefault`: This flag's value is defined to be 0
      and provides default array allocation

    - :py:obj:`~.cudaArraySurfaceLoadStore`: Allocates an array that can be
      read from or written to using a surface reference

    - :py:obj:`~.cudaArrayTextureGather`: This flag indicates that texture
      gather operations will be performed on the array.

    - :py:obj:`~.cudaArraySparse`: Allocates a CUDA array without physical
      backing memory. The subregions within this sparse array can later be
      mapped onto a physical memory allocation by calling
      :py:obj:`~.cuMemMapArrayAsync`. The physical backing memory must be
      allocated via :py:obj:`~.cuMemCreate`.

    - :py:obj:`~.cudaArrayDeferredMapping`: Allocates a CUDA array without
      physical backing memory. The entire array can later be mapped onto a
      physical memory allocation by calling :py:obj:`~.cuMemMapArrayAsync`.
      The physical backing memory must be allocated via
      :py:obj:`~.cuMemCreate`.

    `width` and `height` must meet certain size requirements. See
    :py:obj:`~.cudaMalloc3DArray()` for more details.

    Parameters
    ----------
    desc : :py:obj:`~.cudaChannelFormatDesc`
        Requested channel format
    width : size_t
        Requested array allocation width
    height : size_t
        Requested array allocation height
    flags : unsigned int
        Requested properties of allocated array

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    array : :py:obj:`~.cudaArray_t`
        Pointer to allocated array in device memory

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuArrayCreate`
    """
    cdef cudaArray_t array = cudaArray_t()
    cdef ccudart.cudaChannelFormatDesc* cdesc_ptr = desc._ptr if desc != None else NULL
    with nogil:
        err = ccudart.cudaMallocArray(array._ptr, cdesc_ptr, width, height, flags)
    return (cudaError_t(err), array)

@cython.embedsignature(True)
def cudaFree(devPtr):
    """ Frees memory on the device.

    Frees the memory space pointed to by `devPtr`, which must have been
    returned by a previous call to one of the following memory allocation
    APIs - :py:obj:`~.cudaMalloc()`, :py:obj:`~.cudaMallocPitch()`,
    :py:obj:`~.cudaMallocManaged()`, :py:obj:`~.cudaMallocAsync()`,
    :py:obj:`~.cudaMallocFromPoolAsync()`.

    Note - This API will not perform any implicit synchronization when the
    pointer was allocated with :py:obj:`~.cudaMallocAsync` or
    :py:obj:`~.cudaMallocFromPoolAsync`. Callers must ensure that all
    accesses to the pointer have completed before invoking
    :py:obj:`~.cudaFree`. For best performance and memory reuse, users
    should use :py:obj:`~.cudaFreeAsync` to free memory allocated via the
    stream ordered memory allocator.

    If :py:obj:`~.cudaFree`(`devPtr`) has already been called before, an
    error is returned. If `devPtr` is 0, no operation is performed.
    :py:obj:`~.cudaFree()` returns :py:obj:`~.cudaErrorValue` in case of
    failure.

    The device version of :py:obj:`~.cudaFree` cannot be used with a
    `*devPtr` allocated using the host API, and vice versa.

    Parameters
    ----------
    devPtr : Any
        Device pointer to memory to free

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaMallocManaged`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaMallocFromPoolAsync` :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaFreeAsync` :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMemFree`
    """
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    with nogil:
        err = ccudart.cudaFree(cdevPtr_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaFreeHost(ptr):
    """ Frees page-locked memory.

    Frees the memory space pointed to by `hostPtr`, which must have been
    returned by a previous call to :py:obj:`~.cudaMallocHost()` or
    :py:obj:`~.cudaHostAlloc()`.

    Parameters
    ----------
    ptr : Any
        Pointer to memory to free

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMemFreeHost`
    """
    cptr = utils.HelperInputVoidPtr(ptr)
    cdef void* cptr_ptr = <void*><void_ptr>cptr.cptr
    with nogil:
        err = ccudart.cudaFreeHost(cptr_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaFreeArray(array):
    """ Frees an array on the device.

    Frees the CUDA array `array`, which must have been returned by a
    previous call to :py:obj:`~.cudaMallocArray()`. If `devPtr` is 0, no
    operation is performed.

    Parameters
    ----------
    array : :py:obj:`~.cudaArray_t`
        Pointer to array to free

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuArrayDestroy`
    """
    cdef ccudart.cudaArray_t carray
    if array is None:
        carray = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(array, (cudaArray_t)):
        parray = int(array)
        carray = <ccudart.cudaArray_t><void_ptr>parray
    else:
        parray = int(cudaArray_t(array))
        carray = <ccudart.cudaArray_t><void_ptr>parray

    with nogil:
        err = ccudart.cudaFreeArray(carray)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaFreeMipmappedArray(mipmappedArray):
    """ Frees a mipmapped array on the device.

    Frees the CUDA mipmapped array `mipmappedArray`, which must have been
    returned by a previous call to :py:obj:`~.cudaMallocMipmappedArray()`.
    If `devPtr` is 0, no operation is performed.

    Parameters
    ----------
    mipmappedArray : :py:obj:`~.cudaMipmappedArray_t`
        Pointer to mipmapped array to free

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMipmappedArrayDestroy`
    """
    cdef ccudart.cudaMipmappedArray_t cmipmappedArray
    if mipmappedArray is None:
        cmipmappedArray = <ccudart.cudaMipmappedArray_t><void_ptr>0
    elif isinstance(mipmappedArray, (cudaMipmappedArray_t)):
        pmipmappedArray = int(mipmappedArray)
        cmipmappedArray = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmappedArray
    else:
        pmipmappedArray = int(cudaMipmappedArray_t(mipmappedArray))
        cmipmappedArray = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmappedArray

    err = ccudart.cudaFreeMipmappedArray(cmipmappedArray)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaHostAlloc(size_t size, unsigned int flags):
    """ Allocates page-locked memory on the host.

    Allocates `size` bytes of host memory that is page-locked and
    accessible to the device. The driver tracks the virtual memory ranges
    allocated with this function and automatically accelerates calls to
    functions such as :py:obj:`~.cudaMemcpy()`. Since the memory can be
    accessed directly by the device, it can be read or written with much
    higher bandwidth than pageable memory obtained with functions such as
    :py:obj:`~.malloc()`. Allocating excessive amounts of pinned memory may
    degrade system performance, since it reduces the amount of memory
    available to the system for paging. As a result, this function is best
    used sparingly to allocate staging areas for data exchange between host
    and device.

    The `flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.cudaHostAllocDefault`: This flag's value is defined to be
      0 and causes :py:obj:`~.cudaHostAlloc()` to emulate
      :py:obj:`~.cudaMallocHost()`.

    - :py:obj:`~.cudaHostAllocPortable`: The memory returned by this call
      will be considered as pinned memory by all CUDA contexts, not just
      the one that performed the allocation.

    - :py:obj:`~.cudaHostAllocMapped`: Maps the allocation into the CUDA
      address space. The device pointer to the memory may be obtained by
      calling :py:obj:`~.cudaHostGetDevicePointer()`.

    - :py:obj:`~.cudaHostAllocWriteCombined`: Allocates the memory as
      write-combined (WC). WC memory can be transferred across the PCI
      Express bus more quickly on some system configurations, but cannot be
      read efficiently by most CPUs. WC memory is a good option for buffers
      that will be written by the CPU and read by the device via mapped
      pinned memory or host->device transfers.

    All of these flags are orthogonal to one another: a developer may
    allocate memory that is portable, mapped and/or write-combined with no
    restrictions.

    In order for the :py:obj:`~.cudaHostAllocMapped` flag to have any
    effect, the CUDA context must support the :py:obj:`~.cudaDeviceMapHost`
    flag, which can be checked via :py:obj:`~.cudaGetDeviceFlags()`. The
    :py:obj:`~.cudaDeviceMapHost` flag is implicitly set for contexts
    created via the runtime API.

    The :py:obj:`~.cudaHostAllocMapped` flag may be specified on CUDA
    contexts for devices that do not support mapped pinned memory. The
    failure is deferred to :py:obj:`~.cudaHostGetDevicePointer()` because
    the memory may be mapped into other CUDA contexts via the
    :py:obj:`~.cudaHostAllocPortable` flag.

    Memory allocated by this function must be freed with
    :py:obj:`~.cudaFreeHost()`.

    Parameters
    ----------
    size : size_t
        Requested allocation size in bytes
    flags : unsigned int
        Requested properties of allocated memory

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    pHost : Any
        Device pointer to allocated memory

    See Also
    --------
    :py:obj:`~.cudaSetDeviceFlags`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaGetDeviceFlags`, :py:obj:`~.cuMemHostAlloc`
    """
    cdef void_ptr pHost = 0
    with nogil:
        err = ccudart.cudaHostAlloc(<void**>&pHost, size, flags)
    return (cudaError_t(err), pHost)

@cython.embedsignature(True)
def cudaHostRegister(ptr, size_t size, unsigned int flags):
    """ Registers an existing host memory range for use by CUDA.

    Page-locks the memory range specified by `ptr` and `size` and maps it
    for the device(s) as specified by `flags`. This memory range also is
    added to the same tracking mechanism as :py:obj:`~.cudaHostAlloc()` to
    automatically accelerate calls to functions such as
    :py:obj:`~.cudaMemcpy()`. Since the memory can be accessed directly by
    the device, it can be read or written with much higher bandwidth than
    pageable memory that has not been registered. Page-locking excessive
    amounts of memory may degrade system performance, since it reduces the
    amount of memory available to the system for paging. As a result, this
    function is best used sparingly to register staging areas for data
    exchange between host and device.

    :py:obj:`~.cudaHostRegister` is supported only on I/O coherent devices
    that have a non-zero value for the device attribute
    :py:obj:`~.cudaDevAttrHostRegisterSupported`.

    The `flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.cudaHostRegisterDefault`: On a system with unified virtual
      addressing, the memory will be both mapped and portable. On a system
      with no unified virtual addressing, the memory will be neither mapped
      nor portable.

    - :py:obj:`~.cudaHostRegisterPortable`: The memory returned by this
      call will be considered as pinned memory by all CUDA contexts, not
      just the one that performed the allocation.

    - :py:obj:`~.cudaHostRegisterMapped`: Maps the allocation into the CUDA
      address space. The device pointer to the memory may be obtained by
      calling :py:obj:`~.cudaHostGetDevicePointer()`.

    - :py:obj:`~.cudaHostRegisterIoMemory`: The passed memory pointer is
      treated as pointing to some memory-mapped I/O space, e.g. belonging
      to a third-party PCIe device, and it will marked as non cache-
      coherent and contiguous.

    - :py:obj:`~.cudaHostRegisterReadOnly`: The passed memory pointer is
      treated as pointing to memory that is considered read-only by the
      device. On platforms without
      :py:obj:`~.cudaDevAttrPageableMemoryAccessUsesHostPageTables`, this
      flag is required in order to register memory mapped to the CPU as
      read-only. Support for the use of this flag can be queried from the
      device attribute cudaDeviceAttrReadOnlyHostRegisterSupported. Using
      this flag with a current context associated with a device that does
      not have this attribute set will cause :py:obj:`~.cudaHostRegister`
      to error with cudaErrorNotSupported.

    All of these flags are orthogonal to one another: a developer may page-
    lock memory that is portable or mapped with no restrictions.

    The CUDA context must have been created with the
    :py:obj:`~.cudaMapHost` flag in order for the
    :py:obj:`~.cudaHostRegisterMapped` flag to have any effect.

    The :py:obj:`~.cudaHostRegisterMapped` flag may be specified on CUDA
    contexts for devices that do not support mapped pinned memory. The
    failure is deferred to :py:obj:`~.cudaHostGetDevicePointer()` because
    the memory may be mapped into other CUDA contexts via the
    :py:obj:`~.cudaHostRegisterPortable` flag.

    For devices that have a non-zero value for the device attribute
    :py:obj:`~.cudaDevAttrCanUseHostPointerForRegisteredMem`, the memory
    can also be accessed from the device using the host pointer `ptr`. The
    device pointer returned by :py:obj:`~.cudaHostGetDevicePointer()` may
    or may not match the original host pointer `ptr` and depends on the
    devices visible to the application. If all devices visible to the
    application have a non-zero value for the device attribute, the device
    pointer returned by :py:obj:`~.cudaHostGetDevicePointer()` will match
    the original pointer `ptr`. If any device visible to the application
    has a zero value for the device attribute, the device pointer returned
    by :py:obj:`~.cudaHostGetDevicePointer()` will not match the original
    host pointer `ptr`, but it will be suitable for use on all devices
    provided Unified Virtual Addressing is enabled. In such systems, it is
    valid to access the memory using either pointer on devices that have a
    non-zero value for the device attribute. Note however that such devices
    should access the memory using only of the two pointers and not both.

    The memory page-locked by this function must be unregistered with
    :py:obj:`~.cudaHostUnregister()`.

    Parameters
    ----------
    ptr : Any
        Host pointer to memory to page-lock
    size : size_t
        Size in bytes of the address range to page-lock in bytes
    flags : unsigned int
        Flags for allocation request

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`, :py:obj:`~.cudaErrorHostMemoryAlreadyRegistered`, :py:obj:`~.cudaErrorNotSupported`

    See Also
    --------
    :py:obj:`~.cudaHostUnregister`, :py:obj:`~.cudaHostGetFlags`, :py:obj:`~.cudaHostGetDevicePointer`, :py:obj:`~.cuMemHostRegister`
    """
    cptr = utils.HelperInputVoidPtr(ptr)
    cdef void* cptr_ptr = <void*><void_ptr>cptr.cptr
    with nogil:
        err = ccudart.cudaHostRegister(cptr_ptr, size, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaHostUnregister(ptr):
    """ Unregisters a memory range that was registered with cudaHostRegister.

    Unmaps the memory range whose base address is specified by `ptr`, and
    makes it pageable again.

    The base address must be the same one specified to
    :py:obj:`~.cudaHostRegister()`.

    Parameters
    ----------
    ptr : Any
        Host pointer to memory to unregister

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorHostMemoryNotRegistered`

    See Also
    --------
    :py:obj:`~.cudaHostUnregister`, :py:obj:`~.cuMemHostUnregister`
    """
    cptr = utils.HelperInputVoidPtr(ptr)
    cdef void* cptr_ptr = <void*><void_ptr>cptr.cptr
    with nogil:
        err = ccudart.cudaHostUnregister(cptr_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaHostGetDevicePointer(pHost, unsigned int flags):
    """ Passes back device pointer of mapped host memory allocated by cudaHostAlloc or registered by cudaHostRegister.

    Passes back the device pointer corresponding to the mapped, pinned host
    buffer allocated by :py:obj:`~.cudaHostAlloc()` or registered by
    :py:obj:`~.cudaHostRegister()`.

    :py:obj:`~.cudaHostGetDevicePointer()` will fail if the
    :py:obj:`~.cudaDeviceMapHost` flag was not specified before deferred
    context creation occurred, or if called on a device that does not
    support mapped, pinned memory.

    For devices that have a non-zero value for the device attribute
    :py:obj:`~.cudaDevAttrCanUseHostPointerForRegisteredMem`, the memory
    can also be accessed from the device using the host pointer `pHost`.
    The device pointer returned by :py:obj:`~.cudaHostGetDevicePointer()`
    may or may not match the original host pointer `pHost` and depends on
    the devices visible to the application. If all devices visible to the
    application have a non-zero value for the device attribute, the device
    pointer returned by :py:obj:`~.cudaHostGetDevicePointer()` will match
    the original pointer `pHost`. If any device visible to the application
    has a zero value for the device attribute, the device pointer returned
    by :py:obj:`~.cudaHostGetDevicePointer()` will not match the original
    host pointer `pHost`, but it will be suitable for use on all devices
    provided Unified Virtual Addressing is enabled. In such systems, it is
    valid to access the memory using either pointer on devices that have a
    non-zero value for the device attribute. Note however that such devices
    should access the memory using only of the two pointers and not both.

    `flags` provides for future releases. For now, it must be set to 0.

    Parameters
    ----------
    pHost : Any
        Requested host pointer mapping
    flags : unsigned int
        Flags for extensions (must be 0 for now)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    pDevice : Any
        Returned device pointer for mapped memory

    See Also
    --------
    :py:obj:`~.cudaSetDeviceFlags`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`
    """
    cdef void_ptr pDevice = 0
    cpHost = utils.HelperInputVoidPtr(pHost)
    cdef void* cpHost_ptr = <void*><void_ptr>cpHost.cptr
    err = ccudart.cudaHostGetDevicePointer(<void**>&pDevice, cpHost_ptr, flags)
    return (cudaError_t(err), pDevice)

@cython.embedsignature(True)
def cudaHostGetFlags(pHost):
    """ Passes back flags used to allocate pinned host memory allocated by cudaHostAlloc.

    :py:obj:`~.cudaHostGetFlags()` will fail if the input pointer does not
    reside in an address range allocated by :py:obj:`~.cudaHostAlloc()`.

    Parameters
    ----------
    pHost : Any
        Host pointer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pFlags : unsigned int
        Returned flags word

    See Also
    --------
    :py:obj:`~.cudaHostAlloc`, :py:obj:`~.cuMemHostGetFlags`
    """
    cdef unsigned int pFlags = 0
    cpHost = utils.HelperInputVoidPtr(pHost)
    cdef void* cpHost_ptr = <void*><void_ptr>cpHost.cptr
    err = ccudart.cudaHostGetFlags(&pFlags, cpHost_ptr)
    return (cudaError_t(err), pFlags)

@cython.embedsignature(True)
def cudaMalloc3D(extent not None : cudaExtent):
    """ Allocates logical 1D, 2D, or 3D memory objects on the device.

    Allocates at least `width` * `height` * `depth` bytes of linear memory
    on the device and returns a :py:obj:`~.cudaPitchedPtr` in which `ptr`
    is a pointer to the allocated memory. The function may pad the
    allocation to ensure hardware alignment requirements are met. The pitch
    returned in the `pitch` field of `pitchedDevPtr` is the width in bytes
    of the allocation.

    The returned :py:obj:`~.cudaPitchedPtr` contains additional fields
    `xsize` and `ysize`, the logical width and height of the allocation,
    which are equivalent to the `width` and `height` `extent` parameters
    provided by the programmer during allocation.

    For allocations of 2D and 3D objects, it is highly recommended that
    programmers perform allocations using :py:obj:`~.cudaMalloc3D()` or
    :py:obj:`~.cudaMallocPitch()`. Due to alignment restrictions in the
    hardware, this is especially true if the application will be performing
    memory copies involving 2D or 3D objects (whether linear memory or CUDA
    arrays).

    Parameters
    ----------
    extent : :py:obj:`~.cudaExtent`
        Requested allocation size (`width` field in bytes)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    pitchedDevPtr : :py:obj:`~.cudaPitchedPtr`
        Pointer to allocated pitched device memory

    See Also
    --------
    :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaMemcpy3D`, :py:obj:`~.cudaMemset3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaMallocArray`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.make_cudaPitchedPtr`, :py:obj:`~.make_cudaExtent`, :py:obj:`~.cuMemAllocPitch`
    """
    cdef cudaPitchedPtr pitchedDevPtr = cudaPitchedPtr()
    err = ccudart.cudaMalloc3D(pitchedDevPtr._ptr, extent._ptr[0])
    return (cudaError_t(err), pitchedDevPtr)

@cython.embedsignature(True)
def cudaMalloc3DArray(desc : cudaChannelFormatDesc, extent not None : cudaExtent, unsigned int flags):
    """ Allocate an array on the device.

    Allocates a CUDA array according to the
    :py:obj:`~.cudaChannelFormatDesc` structure `desc` and returns a handle
    to the new CUDA array in `*array`.

    The :py:obj:`~.cudaChannelFormatDesc` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaChannelFormatKind` is one of
    :py:obj:`~.cudaChannelFormatKindSigned`,
    :py:obj:`~.cudaChannelFormatKindUnsigned`, or
    :py:obj:`~.cudaChannelFormatKindFloat`.

    :py:obj:`~.cudaMalloc3DArray()` can allocate the following:

    - A 1D array is allocated if the height and depth extents are both
      zero.

    - A 2D array is allocated if only the depth extent is zero.

    - A 3D array is allocated if all three extents are non-zero.

    - A 1D layered CUDA array is allocated if only the height extent is
      zero and the cudaArrayLayered flag is set. Each layer is a 1D array.
      The number of layers is determined by the depth extent.

    - A 2D layered CUDA array is allocated if all three extents are non-
      zero and the cudaArrayLayered flag is set. Each layer is a 2D array.
      The number of layers is determined by the depth extent.

    - A cubemap CUDA array is allocated if all three extents are non-zero
      and the cudaArrayCubemap flag is set. Width must be equal to height,
      and depth must be six. A cubemap is a special type of 2D layered CUDA
      array, where the six layers represent the six faces of a cube. The
      order of the six layers in memory is the same as that listed in
      :py:obj:`~.cudaGraphicsCubeFace`.

    - A cubemap layered CUDA array is allocated if all three extents are
      non-zero, and both, cudaArrayCubemap and cudaArrayLayered flags are
      set. Width must be equal to height, and depth must be a multiple of
      six. A cubemap layered CUDA array is a special type of 2D layered
      CUDA array that consists of a collection of cubemaps. The first six
      layers represent the first cubemap, the next six layers form the
      second cubemap, and so on.

    The `flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.cudaArrayDefault`: This flag's value is defined to be 0
      and provides default array allocation

    - :py:obj:`~.cudaArrayLayered`: Allocates a layered CUDA array, with
      the depth extent indicating the number of layers

    - :py:obj:`~.cudaArrayCubemap`: Allocates a cubemap CUDA array. Width
      must be equal to height, and depth must be six. If the
      cudaArrayLayered flag is also set, depth must be a multiple of six.

    - :py:obj:`~.cudaArraySurfaceLoadStore`: Allocates a CUDA array that
      could be read from or written to using a surface reference.

    - :py:obj:`~.cudaArrayTextureGather`: This flag indicates that texture
      gather operations will be performed on the CUDA array. Texture gather
      can only be performed on 2D CUDA arrays.

    - :py:obj:`~.cudaArraySparse`: Allocates a CUDA array without physical
      backing memory. The subregions within this sparse array can later be
      mapped onto a physical memory allocation by calling
      :py:obj:`~.cuMemMapArrayAsync`. This flag can only be used for
      creating 2D, 3D or 2D layered sparse CUDA arrays. The physical
      backing memory must be allocated via :py:obj:`~.cuMemCreate`.

    - :py:obj:`~.cudaArrayDeferredMapping`: Allocates a CUDA array without
      physical backing memory. The entire array can later be mapped onto a
      physical memory allocation by calling :py:obj:`~.cuMemMapArrayAsync`.
      The physical backing memory must be allocated via
      :py:obj:`~.cuMemCreate`.

    The width, height and depth extents must meet certain size requirements
    as listed in the following table. All values are specified in elements.

    Note that 2D CUDA arrays have different size requirements if the
    :py:obj:`~.cudaArrayTextureGather` flag is set. In that case, the valid
    range for (width, height, depth) is ((1,maxTexture2DGather[0]),
    (1,maxTexture2DGather[1]), 0).

    **View CUDA Toolkit Documentation for a table example**

    Parameters
    ----------
    desc : :py:obj:`~.cudaChannelFormatDesc`
        Requested channel format
    extent : :py:obj:`~.cudaExtent`
        Requested allocation size (`width` field in elements)
    flags : unsigned int
        Flags for extensions

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    array : :py:obj:`~.cudaArray_t`
        Pointer to allocated array in device memory

    See Also
    --------
    :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.make_cudaExtent`, :py:obj:`~.cuArray3DCreate`
    """
    cdef cudaArray_t array = cudaArray_t()
    cdef ccudart.cudaChannelFormatDesc* cdesc_ptr = desc._ptr if desc != None else NULL
    with nogil:
        err = ccudart.cudaMalloc3DArray(array._ptr, cdesc_ptr, extent._ptr[0], flags)
    return (cudaError_t(err), array)

@cython.embedsignature(True)
def cudaMallocMipmappedArray(desc : cudaChannelFormatDesc, extent not None : cudaExtent, unsigned int numLevels, unsigned int flags):
    """ Allocate a mipmapped array on the device.

    Allocates a CUDA mipmapped array according to the
    :py:obj:`~.cudaChannelFormatDesc` structure `desc` and returns a handle
    to the new CUDA mipmapped array in `*mipmappedArray`. `numLevels`
    specifies the number of mipmap levels to be allocated. This value is
    clamped to the range [1, 1 + floor(log2(max(width, height, depth)))].

    The :py:obj:`~.cudaChannelFormatDesc` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaChannelFormatKind` is one of
    :py:obj:`~.cudaChannelFormatKindSigned`,
    :py:obj:`~.cudaChannelFormatKindUnsigned`, or
    :py:obj:`~.cudaChannelFormatKindFloat`.

    :py:obj:`~.cudaMallocMipmappedArray()` can allocate the following:

    - A 1D mipmapped array is allocated if the height and depth extents are
      both zero.

    - A 2D mipmapped array is allocated if only the depth extent is zero.

    - A 3D mipmapped array is allocated if all three extents are non-zero.

    - A 1D layered CUDA mipmapped array is allocated if only the height
      extent is zero and the cudaArrayLayered flag is set. Each layer is a
      1D mipmapped array. The number of layers is determined by the depth
      extent.

    - A 2D layered CUDA mipmapped array is allocated if all three extents
      are non-zero and the cudaArrayLayered flag is set. Each layer is a 2D
      mipmapped array. The number of layers is determined by the depth
      extent.

    - A cubemap CUDA mipmapped array is allocated if all three extents are
      non-zero and the cudaArrayCubemap flag is set. Width must be equal to
      height, and depth must be six. The order of the six layers in memory
      is the same as that listed in :py:obj:`~.cudaGraphicsCubeFace`.

    - A cubemap layered CUDA mipmapped array is allocated if all three
      extents are non-zero, and both, cudaArrayCubemap and cudaArrayLayered
      flags are set. Width must be equal to height, and depth must be a
      multiple of six. A cubemap layered CUDA mipmapped array is a special
      type of 2D layered CUDA mipmapped array that consists of a collection
      of cubemap mipmapped arrays. The first six layers represent the first
      cubemap mipmapped array, the next six layers form the second cubemap
      mipmapped array, and so on.

    The `flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.cudaArrayDefault`: This flag's value is defined to be 0
      and provides default mipmapped array allocation

    - :py:obj:`~.cudaArrayLayered`: Allocates a layered CUDA mipmapped
      array, with the depth extent indicating the number of layers

    - :py:obj:`~.cudaArrayCubemap`: Allocates a cubemap CUDA mipmapped
      array. Width must be equal to height, and depth must be six. If the
      cudaArrayLayered flag is also set, depth must be a multiple of six.

    - :py:obj:`~.cudaArraySurfaceLoadStore`: This flag indicates that
      individual mipmap levels of the CUDA mipmapped array will be read
      from or written to using a surface reference.

    - :py:obj:`~.cudaArrayTextureGather`: This flag indicates that texture
      gather operations will be performed on the CUDA array. Texture gather
      can only be performed on 2D CUDA mipmapped arrays, and the gather
      operations are performed only on the most detailed mipmap level.

    - :py:obj:`~.cudaArraySparse`: Allocates a CUDA mipmapped array without
      physical backing memory. The subregions within this sparse array can
      later be mapped onto a physical memory allocation by calling
      :py:obj:`~.cuMemMapArrayAsync`. This flag can only be used for
      creating 2D, 3D or 2D layered sparse CUDA mipmapped arrays. The
      physical backing memory must be allocated via
      :py:obj:`~.cuMemCreate`.

    - :py:obj:`~.cudaArrayDeferredMapping`: Allocates a CUDA mipmapped
      array without physical backing memory. The entire array can later be
      mapped onto a physical memory allocation by calling
      :py:obj:`~.cuMemMapArrayAsync`. The physical backing memory must be
      allocated via :py:obj:`~.cuMemCreate`.

    The width, height and depth extents must meet certain size requirements
    as listed in the following table. All values are specified in elements.

    **View CUDA Toolkit Documentation for a table example**

    Parameters
    ----------
    desc : :py:obj:`~.cudaChannelFormatDesc`
        Requested channel format
    extent : :py:obj:`~.cudaExtent`
        Requested allocation size (`width` field in elements)
    numLevels : unsigned int
        Number of mipmap levels to allocate
    flags : unsigned int
        Flags for extensions

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    mipmappedArray : :py:obj:`~.cudaMipmappedArray_t`
        Pointer to allocated mipmapped array in device memory

    See Also
    --------
    :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.make_cudaExtent`, :py:obj:`~.cuMipmappedArrayCreate`
    """
    cdef cudaMipmappedArray_t mipmappedArray = cudaMipmappedArray_t()
    cdef ccudart.cudaChannelFormatDesc* cdesc_ptr = desc._ptr if desc != None else NULL
    err = ccudart.cudaMallocMipmappedArray(mipmappedArray._ptr, cdesc_ptr, extent._ptr[0], numLevels, flags)
    return (cudaError_t(err), mipmappedArray)

@cython.embedsignature(True)
def cudaGetMipmappedArrayLevel(mipmappedArray, unsigned int level):
    """ Gets a mipmap level of a CUDA mipmapped array.

    Returns in `*levelArray` a CUDA array that represents a single mipmap
    level of the CUDA mipmapped array `mipmappedArray`.

    If `level` is greater than the maximum number of levels in this
    mipmapped array, :py:obj:`~.cudaErrorInvalidValue` is returned.

    If `mipmappedArray` is NULL, :py:obj:`~.cudaErrorInvalidResourceHandle`
    is returned.

    Parameters
    ----------
    mipmappedArray : :py:obj:`~.cudaMipmappedArray_const_t`
        CUDA mipmapped array
    level : unsigned int
        Mipmap level

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue` :py:obj:`~.cudaErrorInvalidResourceHandle`
    levelArray : :py:obj:`~.cudaArray_t`
        Returned mipmap level CUDA array

    See Also
    --------
    :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc`, :py:obj:`~.cudaMallocPitch`, :py:obj:`~.cudaFree`, :py:obj:`~.cudaFreeArray`, :py:obj:`~.cudaMallocHost (C API)`, :py:obj:`~.cudaFreeHost`, :py:obj:`~.cudaHostAlloc`, :py:obj:`~.make_cudaExtent`, :py:obj:`~.cuMipmappedArrayGetLevel`
    """
    cdef ccudart.cudaMipmappedArray_const_t cmipmappedArray
    if mipmappedArray is None:
        cmipmappedArray = <ccudart.cudaMipmappedArray_const_t><void_ptr>0
    elif isinstance(mipmappedArray, (cudaMipmappedArray_const_t)):
        pmipmappedArray = int(mipmappedArray)
        cmipmappedArray = <ccudart.cudaMipmappedArray_const_t><void_ptr>pmipmappedArray
    else:
        pmipmappedArray = int(cudaMipmappedArray_const_t(mipmappedArray))
        cmipmappedArray = <ccudart.cudaMipmappedArray_const_t><void_ptr>pmipmappedArray

    cdef cudaArray_t levelArray = cudaArray_t()
    err = ccudart.cudaGetMipmappedArrayLevel(levelArray._ptr, cmipmappedArray, level)
    return (cudaError_t(err), levelArray)

@cython.embedsignature(True)
def cudaMemcpy3D(p : cudaMemcpy3DParms):
    """ Copies data between 3D objects.

    **View CUDA Toolkit Documentation for a C++ code example**

    :py:obj:`~.cudaMemcpy3D()` copies data betwen two 3D objects. The
    source and destination objects may be in either host memory, device
    memory, or a CUDA array. The source, destination, extent, and kind of
    copy performed is specified by the :py:obj:`~.cudaMemcpy3DParms` struct
    which should be initialized to zero before use:

    **View CUDA Toolkit Documentation for a C++ code example**

    The struct passed to :py:obj:`~.cudaMemcpy3D()` must specify one of
    `srcArray` or `srcPtr` and one of `dstArray` or `dstPtr`. Passing more
    than one non-zero source or destination will cause
    :py:obj:`~.cudaMemcpy3D()` to return an error.

    The `srcPos` and `dstPos` fields are optional offsets into the source
    and destination objects and are defined in units of each object's
    elements. The element for a host or device pointer is assumed to be
    unsigned char.

    The `extent` field defines the dimensions of the transferred area in
    elements. If a CUDA array is participating in the copy, the extent is
    defined in terms of that array's elements. If no CUDA array is
    participating in the copy then the extents are defined in elements of
    unsigned char.

    The `kind` field defines the direction of the copy. It must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. For :py:obj:`~.cudaMemcpyHostToHost` or
    :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` passed as kind and cudaArray type
    passed as source or destination, if the kind implies cudaArray type to
    be present on the host, :py:obj:`~.cudaMemcpy3D()` will disregard that
    implication and silently correct the kind based on the fact that
    cudaArray type can only be present on the device.

    If the source and destination are both arrays,
    :py:obj:`~.cudaMemcpy3D()` will return an error if they do not have the
    same element size.

    The source and destination object may not overlap. If overlapping
    source and destination objects are specified, undefined behavior will
    result.

    The source object must entirely contain the region defined by `srcPos`
    and `extent`. The destination object must entirely contain the region
    defined by `dstPos` and `extent`.

    :py:obj:`~.cudaMemcpy3D()` returns an error if the pitch of `srcPtr` or
    `dstPtr` exceeds the maximum allowed. The pitch of a
    :py:obj:`~.cudaPitchedPtr` allocated with :py:obj:`~.cudaMalloc3D()`
    will always be valid.

    Parameters
    ----------
    p : :py:obj:`~.cudaMemcpy3DParms`
        3D memory copy parameters

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaMemset3D`, :py:obj:`~.cudaMemcpy3DAsync`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.make_cudaExtent`, :py:obj:`~.make_cudaPos`, :py:obj:`~.cuMemcpy3D`
    """
    cdef ccudart.cudaMemcpy3DParms* cp_ptr = p._ptr if p != None else NULL
    with nogil:
        err = ccudart.cudaMemcpy3D(cp_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy3DPeer(p : cudaMemcpy3DPeerParms):
    """ Copies memory between devices.

    Perform a 3D memory copy according to the parameters specified in `p`.
    See the definition of the :py:obj:`~.cudaMemcpy3DPeerParms` structure
    for documentation of its parameters.

    Note that this function is synchronous with respect to the host only if
    the source or destination of the transfer is host memory. Note also
    that this copy is serialized with respect to all pending and future
    asynchronous work in to the current device, the copy's source device,
    and the copy's destination device (use
    :py:obj:`~.cudaMemcpy3DPeerAsync` to avoid this synchronization).

    Parameters
    ----------
    p : :py:obj:`~.cudaMemcpy3DPeerParms`
        Parameters for the memory copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyPeer`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyPeerAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`, :py:obj:`~.cuMemcpy3DPeer`
    """
    cdef ccudart.cudaMemcpy3DPeerParms* cp_ptr = p._ptr if p != None else NULL
    err = ccudart.cudaMemcpy3DPeer(cp_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy3DAsync(p : cudaMemcpy3DParms, stream):
    """ Copies data between 3D objects.

    **View CUDA Toolkit Documentation for a C++ code example**

    :py:obj:`~.cudaMemcpy3DAsync()` copies data betwen two 3D objects. The
    source and destination objects may be in either host memory, device
    memory, or a CUDA array. The source, destination, extent, and kind of
    copy performed is specified by the :py:obj:`~.cudaMemcpy3DParms` struct
    which should be initialized to zero before use:

    **View CUDA Toolkit Documentation for a C++ code example**

    The struct passed to :py:obj:`~.cudaMemcpy3DAsync()` must specify one
    of `srcArray` or `srcPtr` and one of `dstArray` or `dstPtr`. Passing
    more than one non-zero source or destination will cause
    :py:obj:`~.cudaMemcpy3DAsync()` to return an error.

    The `srcPos` and `dstPos` fields are optional offsets into the source
    and destination objects and are defined in units of each object's
    elements. The element for a host or device pointer is assumed to be
    unsigned char. For CUDA arrays, positions must be in the range [0,
    2048) for any dimension.

    The `extent` field defines the dimensions of the transferred area in
    elements. If a CUDA array is participating in the copy, the extent is
    defined in terms of that array's elements. If no CUDA array is
    participating in the copy then the extents are defined in elements of
    unsigned char.

    The `kind` field defines the direction of the copy. It must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. For :py:obj:`~.cudaMemcpyHostToHost` or
    :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` passed as kind and cudaArray type
    passed as source or destination, if the kind implies cudaArray type to
    be present on the host, :py:obj:`~.cudaMemcpy3DAsync()` will disregard
    that implication and silently correct the kind based on the fact that
    cudaArray type can only be present on the device.

    If the source and destination are both arrays,
    :py:obj:`~.cudaMemcpy3DAsync()` will return an error if they do not
    have the same element size.

    The source and destination object may not overlap. If overlapping
    source and destination objects are specified, undefined behavior will
    result.

    The source object must lie entirely within the region defined by
    `srcPos` and `extent`. The destination object must lie entirely within
    the region defined by `dstPos` and `extent`.

    :py:obj:`~.cudaMemcpy3DAsync()` returns an error if the pitch of
    `srcPtr` or `dstPtr` exceeds the maximum allowed. The pitch of a
    :py:obj:`~.cudaPitchedPtr` allocated with :py:obj:`~.cudaMalloc3D()`
    will always be valid.

    :py:obj:`~.cudaMemcpy3DAsync()` is asynchronous with respect to the
    host, so the call may return before the copy is complete. The copy can
    optionally be associated to a stream by passing a non-zero `stream`
    argument. If `kind` is :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` and `stream` is non-zero, the copy
    may overlap with operations in other streams.

    The device version of this function only handles device to device
    copies and cannot be given local or shared pointers.

    Parameters
    ----------
    p : :py:obj:`~.cudaMemcpy3DParms`
        3D memory copy parameters
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMalloc3D`, :py:obj:`~.cudaMalloc3DArray`, :py:obj:`~.cudaMemset3D`, :py:obj:`~.cudaMemcpy3D`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, ::::py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.make_cudaExtent`, :py:obj:`~.make_cudaPos`, :py:obj:`~.cuMemcpy3DAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaMemcpy3DParms* cp_ptr = p._ptr if p != None else NULL
    with nogil:
        err = ccudart.cudaMemcpy3DAsync(cp_ptr, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy3DPeerAsync(p : cudaMemcpy3DPeerParms, stream):
    """ Copies memory between devices asynchronously.

    Perform a 3D memory copy according to the parameters specified in `p`.
    See the definition of the :py:obj:`~.cudaMemcpy3DPeerParms` structure
    for documentation of its parameters.

    Parameters
    ----------
    p : :py:obj:`~.cudaMemcpy3DPeerParms`
        Parameters for the memory copy
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyPeer`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyPeerAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaMemcpy3DPeerParms* cp_ptr = p._ptr if p != None else NULL
    err = ccudart.cudaMemcpy3DPeerAsync(cp_ptr, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemGetInfo():
    """ Gets free and total device memory.

    Returns in `*total` the total amount of memory available to the the
    current context. Returns in `*free` the amount of memory on the device
    that is free according to the OS. CUDA is not guaranteed to be able to
    allocate all of the memory that the OS reports as free. In a multi-
    tenet situation, free estimate returned is prone to race condition
    where a new allocation/free done by a different process or a different
    thread in the same process between the time when free memory was
    estimated and reported, will result in deviation in free value reported
    and actual free memory.

    The integrated GPU on Tegra shares memory with CPU and other component
    of the SoC. The free and total values returned by the API excludes the
    SWAP memory space maintained by the OS on some platforms. The OS may
    move some of the memory pages into swap area as the GPU or CPU allocate
    or access memory. See Tegra app note on how to calculate total and free
    memory on Tegra.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorLaunchFailure`
    free : int
        Returned free memory in bytes
    total : int
        Returned total memory in bytes

    See Also
    --------
    :py:obj:`~.cuMemGetInfo`
    """
    cdef size_t free = 0
    cdef size_t total = 0
    err = ccudart.cudaMemGetInfo(&free, &total)
    return (cudaError_t(err), free, total)

@cython.embedsignature(True)
def cudaArrayGetInfo(array):
    """ Gets info about the specified cudaArray.

    Returns in `*desc`, `*extent` and `*flags` respectively, the type,
    shape and flags of `array`.

    Any of `*desc`, `*extent` and `*flags` may be specified as NULL.

    Parameters
    ----------
    array : :py:obj:`~.cudaArray_t`
        The :py:obj:`~.cudaArray` to get info for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    desc : :py:obj:`~.cudaChannelFormatDesc`
        Returned array type
    extent : :py:obj:`~.cudaExtent`
        Returned array shape. 2D arrays will have depth of zero
    flags : unsigned int
        Returned array flags

    See Also
    --------
    :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuArray3DGetDescriptor`
    """
    cdef ccudart.cudaArray_t carray
    if array is None:
        carray = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(array, (cudaArray_t)):
        parray = int(array)
        carray = <ccudart.cudaArray_t><void_ptr>parray
    else:
        parray = int(cudaArray_t(array))
        carray = <ccudart.cudaArray_t><void_ptr>parray

    cdef cudaChannelFormatDesc desc = cudaChannelFormatDesc()
    cdef cudaExtent extent = cudaExtent()
    cdef unsigned int flags = 0
    err = ccudart.cudaArrayGetInfo(desc._ptr, extent._ptr, &flags, carray)
    return (cudaError_t(err), desc, extent, flags)

@cython.embedsignature(True)
def cudaArrayGetPlane(hArray, unsigned int planeIdx):
    """ Gets a CUDA array plane from a CUDA array.

    Returns in `pPlaneArray` a CUDA array that represents a single format
    plane of the CUDA array `hArray`.

    If `planeIdx` is greater than the maximum number of planes in this
    array or if the array does not have a multi-planar format e.g:
    :py:obj:`~.cudaChannelFormatKindNV12`, then
    :py:obj:`~.cudaErrorInvalidValue` is returned.

    Note that if the `hArray` has format
    :py:obj:`~.cudaChannelFormatKindNV12`, then passing in 0 for `planeIdx`
    returns a CUDA array of the same size as `hArray` but with one 8-bit
    channel and :py:obj:`~.cudaChannelFormatKindUnsigned` as its format
    kind. If 1 is passed for `planeIdx`, then the returned CUDA array has
    half the height and width of `hArray` with two 8-bit channels and
    :py:obj:`~.cudaChannelFormatKindUnsigned` as its format kind.

    Parameters
    ----------
    hArray : :py:obj:`~.cudaArray_t`
        CUDA array
    planeIdx : unsigned int
        Plane index

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue` :py:obj:`~.cudaErrorInvalidResourceHandle`
    pPlaneArray : :py:obj:`~.cudaArray_t`
        Returned CUDA array referenced by the `planeIdx`

    See Also
    --------
    :py:obj:`~.cuArrayGetPlane`
    """
    cdef ccudart.cudaArray_t chArray
    if hArray is None:
        chArray = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(hArray, (cudaArray_t)):
        phArray = int(hArray)
        chArray = <ccudart.cudaArray_t><void_ptr>phArray
    else:
        phArray = int(cudaArray_t(hArray))
        chArray = <ccudart.cudaArray_t><void_ptr>phArray

    cdef cudaArray_t pPlaneArray = cudaArray_t()
    err = ccudart.cudaArrayGetPlane(pPlaneArray._ptr, chArray, planeIdx)
    return (cudaError_t(err), pPlaneArray)

@cython.embedsignature(True)
def cudaArrayGetMemoryRequirements(array, int device):
    """ Returns the memory requirements of a CUDA array.

    Returns the memory requirements of a CUDA array in `memoryRequirements`
    If the CUDA array is not allocated with flag
    :py:obj:`~.cudaArrayDeferredMapping` :py:obj:`~.cudaErrorInvalidValue`
    will be returned.

    The returned value in :py:obj:`~.cudaArrayMemoryRequirements.size`
    represents the total size of the CUDA array. The returned value in
    :py:obj:`~.cudaArrayMemoryRequirements.alignment` represents the
    alignment necessary for mapping the CUDA array.

    Parameters
    ----------
    array : :py:obj:`~.cudaArray_t`
        CUDA array to get the memory requirements of
    device : int
        Device to get the memory requirements for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess` :py:obj:`~.cudaErrorInvalidValue`
    memoryRequirements : :py:obj:`~.cudaArrayMemoryRequirements`
        Pointer to :py:obj:`~.cudaArrayMemoryRequirements`

    See Also
    --------
    :py:obj:`~.cudaMipmappedArrayGetMemoryRequirements`
    """
    cdef ccudart.cudaArray_t carray
    if array is None:
        carray = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(array, (cudaArray_t)):
        parray = int(array)
        carray = <ccudart.cudaArray_t><void_ptr>parray
    else:
        parray = int(cudaArray_t(array))
        carray = <ccudart.cudaArray_t><void_ptr>parray

    cdef cudaArrayMemoryRequirements memoryRequirements = cudaArrayMemoryRequirements()
    err = ccudart.cudaArrayGetMemoryRequirements(memoryRequirements._ptr, carray, device)
    return (cudaError_t(err), memoryRequirements)

@cython.embedsignature(True)
def cudaMipmappedArrayGetMemoryRequirements(mipmap, int device):
    """ Returns the memory requirements of a CUDA mipmapped array.

    Returns the memory requirements of a CUDA mipmapped array in
    `memoryRequirements` If the CUDA mipmapped array is not allocated with
    flag :py:obj:`~.cudaArrayDeferredMapping`
    :py:obj:`~.cudaErrorInvalidValue` will be returned.

    The returned value in :py:obj:`~.cudaArrayMemoryRequirements.size`
    represents the total size of the CUDA mipmapped array. The returned
    value in :py:obj:`~.cudaArrayMemoryRequirements.alignment` represents
    the alignment necessary for mapping the CUDA mipmapped array.

    Parameters
    ----------
    mipmap : :py:obj:`~.cudaMipmappedArray_t`
        CUDA mipmapped array to get the memory requirements of
    device : int
        Device to get the memory requirements for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess` :py:obj:`~.cudaErrorInvalidValue`
    memoryRequirements : :py:obj:`~.cudaArrayMemoryRequirements`
        Pointer to :py:obj:`~.cudaArrayMemoryRequirements`

    See Also
    --------
    :py:obj:`~.cudaArrayGetMemoryRequirements`
    """
    cdef ccudart.cudaMipmappedArray_t cmipmap
    if mipmap is None:
        cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>0
    elif isinstance(mipmap, (cudaMipmappedArray_t)):
        pmipmap = int(mipmap)
        cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmap
    else:
        pmipmap = int(cudaMipmappedArray_t(mipmap))
        cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmap

    cdef cudaArrayMemoryRequirements memoryRequirements = cudaArrayMemoryRequirements()
    err = ccudart.cudaMipmappedArrayGetMemoryRequirements(memoryRequirements._ptr, cmipmap, device)
    return (cudaError_t(err), memoryRequirements)

@cython.embedsignature(True)
def cudaArrayGetSparseProperties(array):
    """ Returns the layout properties of a sparse CUDA array.

    Returns the layout properties of a sparse CUDA array in
    `sparseProperties`. If the CUDA array is not allocated with flag
    :py:obj:`~.cudaArraySparse` :py:obj:`~.cudaErrorInvalidValue` will be
    returned.

    If the returned value in :py:obj:`~.cudaArraySparseProperties.flags`
    contains :py:obj:`~.cudaArraySparsePropertiesSingleMipTail`, then
    :py:obj:`~.cudaArraySparseProperties.miptailSize` represents the total
    size of the array. Otherwise, it will be zero. Also, the returned value
    in :py:obj:`~.cudaArraySparseProperties.miptailFirstLevel` is always
    zero. Note that the `array` must have been allocated using
    :py:obj:`~.cudaMallocArray` or :py:obj:`~.cudaMalloc3DArray`. For CUDA
    arrays obtained using :py:obj:`~.cudaMipmappedArrayGetLevel`,
    :py:obj:`~.cudaErrorInvalidValue` will be returned. Instead,
    :py:obj:`~.cudaMipmappedArrayGetSparseProperties` must be used to
    obtain the sparse properties of the entire CUDA mipmapped array to
    which `array` belongs to.

    Parameters
    ----------
    array : :py:obj:`~.cudaArray_t`
        The CUDA array to get the sparse properties of

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess` :py:obj:`~.cudaErrorInvalidValue`
    sparseProperties : :py:obj:`~.cudaArraySparseProperties`
        Pointer to return the :py:obj:`~.cudaArraySparseProperties`

    See Also
    --------
    :py:obj:`~.cudaMipmappedArrayGetSparseProperties`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef ccudart.cudaArray_t carray
    if array is None:
        carray = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(array, (cudaArray_t)):
        parray = int(array)
        carray = <ccudart.cudaArray_t><void_ptr>parray
    else:
        parray = int(cudaArray_t(array))
        carray = <ccudart.cudaArray_t><void_ptr>parray

    cdef cudaArraySparseProperties sparseProperties = cudaArraySparseProperties()
    err = ccudart.cudaArrayGetSparseProperties(sparseProperties._ptr, carray)
    return (cudaError_t(err), sparseProperties)

@cython.embedsignature(True)
def cudaMipmappedArrayGetSparseProperties(mipmap):
    """ Returns the layout properties of a sparse CUDA mipmapped array.

    Returns the sparse array layout properties in `sparseProperties`. If
    the CUDA mipmapped array is not allocated with flag
    :py:obj:`~.cudaArraySparse` :py:obj:`~.cudaErrorInvalidValue` will be
    returned.

    For non-layered CUDA mipmapped arrays,
    :py:obj:`~.cudaArraySparseProperties.miptailSize` returns the size of
    the mip tail region. The mip tail region includes all mip levels whose
    width, height or depth is less than that of the tile. For layered CUDA
    mipmapped arrays, if :py:obj:`~.cudaArraySparseProperties.flags`
    contains :py:obj:`~.cudaArraySparsePropertiesSingleMipTail`, then
    :py:obj:`~.cudaArraySparseProperties.miptailSize` specifies the size of
    the mip tail of all layers combined. Otherwise,
    :py:obj:`~.cudaArraySparseProperties.miptailSize` specifies mip tail
    size per layer. The returned value of
    :py:obj:`~.cudaArraySparseProperties.miptailFirstLevel` is valid only
    if :py:obj:`~.cudaArraySparseProperties.miptailSize` is non-zero.

    Parameters
    ----------
    mipmap : :py:obj:`~.cudaMipmappedArray_t`
        The CUDA mipmapped array to get the sparse properties of

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess` :py:obj:`~.cudaErrorInvalidValue`
    sparseProperties : :py:obj:`~.cudaArraySparseProperties`
        Pointer to return :py:obj:`~.cudaArraySparseProperties`

    See Also
    --------
    :py:obj:`~.cudaArrayGetSparseProperties`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef ccudart.cudaMipmappedArray_t cmipmap
    if mipmap is None:
        cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>0
    elif isinstance(mipmap, (cudaMipmappedArray_t)):
        pmipmap = int(mipmap)
        cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmap
    else:
        pmipmap = int(cudaMipmappedArray_t(mipmap))
        cmipmap = <ccudart.cudaMipmappedArray_t><void_ptr>pmipmap

    cdef cudaArraySparseProperties sparseProperties = cudaArraySparseProperties()
    err = ccudart.cudaMipmappedArrayGetSparseProperties(sparseProperties._ptr, cmipmap)
    return (cudaError_t(err), sparseProperties)

@cython.embedsignature(True)
def cudaMemcpy(dst, src, size_t count, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    Copies `count` bytes from the memory area pointed to by `src` to the
    memory area pointed to by `dst`, where `kind` specifies the direction
    of the copy, and must be one of :py:obj:`~.cudaMemcpyHostToHost`,
    :py:obj:`~.cudaMemcpyHostToDevice`, :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. Calling :py:obj:`~.cudaMemcpy()` with dst
    and src pointers that do not match the direction of the copy results in
    an undefined behavior.

    \note_sync

    Parameters
    ----------
    dst : Any
        Destination memory address
    src : Any
        Source memory address
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpy`
    """
    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpy(cdst_ptr, csrc_ptr, count, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpyPeer(dst, int dstDevice, src, int srcDevice, size_t count):
    """ Copies memory between two devices.

    Copies memory from one device to memory on another device. `dst` is the
    base device pointer of the destination memory and `dstDevice` is the
    destination device. `src` is the base device pointer of the source
    memory and `srcDevice` is the source device. `count` specifies the
    number of bytes to copy.

    Note that this function is asynchronous with respect to the host, but
    serialized with respect all pending and future asynchronous work in to
    the current device, `srcDevice`, and `dstDevice` (use
    :py:obj:`~.cudaMemcpyPeerAsync` to avoid this synchronization).

    Parameters
    ----------
    dst : Any
        Destination device pointer
    dstDevice : int
        Destination device
    src : Any
        Source device pointer
    srcDevice : int
        Source device
    count : size_t
        Size of memory copy in bytes

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyPeerAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`, :py:obj:`~.cuMemcpyPeer`
    """
    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    with nogil:
        err = ccudart.cudaMemcpyPeer(cdst_ptr, dstDevice, csrc_ptr, srcDevice, count)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy2D(dst, size_t dpitch, src, size_t spitch, size_t width, size_t height, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    Copies a matrix (`height` rows of `width` bytes each) from the memory
    area pointed to by `src` to the memory area pointed to by `dst`, where
    `kind` specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. `dpitch` and `spitch` are the widths in
    memory in bytes of the 2D arrays pointed to by `dst` and `src`,
    including any padding added to the end of each row. The memory areas
    may not overlap. `width` must not exceed either `dpitch` or `spitch`.
    Calling :py:obj:`~.cudaMemcpy2D()` with `dst` and `src` pointers that
    do not match the direction of the copy results in an undefined
    behavior. :py:obj:`~.cudaMemcpy2D()` returns an error if `dpitch` or
    `spitch` exceeds the maximum allowed.

    Parameters
    ----------
    dst : Any
        Destination memory address
    dpitch : size_t
        Pitch of destination memory
    src : Any
        Source memory address
    spitch : size_t
        Pitch of source memory
    width : size_t
        Width of matrix transfer (columns in bytes)
    height : size_t
        Height of matrix transfer (rows)
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DUnaligned`
    """
    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpy2D(cdst_ptr, dpitch, csrc_ptr, spitch, width, height, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy2DToArray(dst, size_t wOffset, size_t hOffset, src, size_t spitch, size_t width, size_t height, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    Copies a matrix (`height` rows of `width` bytes each) from the memory
    area pointed to by `src` to the CUDA array `dst` starting at `hOffset`
    rows and `wOffset` bytes from the upper left corner, where `kind`
    specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. `spitch` is the width in memory in bytes of
    the 2D array pointed to by `src`, including any padding added to the
    end of each row. `wOffset` + `width` must not exceed the width of the
    CUDA array `dst`. `width` must not exceed `spitch`.
    :py:obj:`~.cudaMemcpy2DToArray()` returns an error if `spitch` exceeds
    the maximum allowed.

    Parameters
    ----------
    dst : :py:obj:`~.cudaArray_t`
        Destination memory address
    wOffset : size_t
        Destination starting X offset (columns in bytes)
    hOffset : size_t
        Destination starting Y offset (rows)
    src : Any
        Source memory address
    spitch : size_t
        Pitch of source memory
    width : size_t
        Width of matrix transfer (columns in bytes)
    height : size_t
        Height of matrix transfer (rows)
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DUnaligned`
    """
    cdef ccudart.cudaArray_t cdst
    if dst is None:
        cdst = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(dst, (cudaArray_t)):
        pdst = int(dst)
        cdst = <ccudart.cudaArray_t><void_ptr>pdst
    else:
        pdst = int(cudaArray_t(dst))
        cdst = <ccudart.cudaArray_t><void_ptr>pdst

    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpy2DToArray(cdst, wOffset, hOffset, csrc_ptr, spitch, width, height, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy2DFromArray(dst, size_t dpitch, src, size_t wOffset, size_t hOffset, size_t width, size_t height, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    Copies a matrix (`height` rows of `width` bytes each) from the CUDA
    array `src` starting at `hOffset` rows and `wOffset` bytes from the
    upper left corner to the memory area pointed to by `dst`, where `kind`
    specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. `dpitch` is the width in memory in bytes of
    the 2D array pointed to by `dst`, including any padding added to the
    end of each row. `wOffset` + `width` must not exceed the width of the
    CUDA array `src`. `width` must not exceed `dpitch`.
    :py:obj:`~.cudaMemcpy2DFromArray()` returns an error if `dpitch`
    exceeds the maximum allowed.

    Parameters
    ----------
    dst : Any
        Destination memory address
    dpitch : size_t
        Pitch of destination memory
    src : :py:obj:`~.cudaArray_const_t`
        Source memory address
    wOffset : size_t
        Source starting X offset (columns in bytes)
    hOffset : size_t
        Source starting Y offset (rows)
    width : size_t
        Width of matrix transfer (columns in bytes)
    height : size_t
        Height of matrix transfer (rows)
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DUnaligned`
    """
    cdef ccudart.cudaArray_const_t csrc
    if src is None:
        csrc = <ccudart.cudaArray_const_t><void_ptr>0
    elif isinstance(src, (cudaArray_const_t)):
        psrc = int(src)
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc
    else:
        psrc = int(cudaArray_const_t(src))
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpy2DFromArray(cdst_ptr, dpitch, csrc, wOffset, hOffset, width, height, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy2DArrayToArray(dst, size_t wOffsetDst, size_t hOffsetDst, src, size_t wOffsetSrc, size_t hOffsetSrc, size_t width, size_t height, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    Copies a matrix (`height` rows of `width` bytes each) from the CUDA
    array `src` starting at `hOffsetSrc` rows and `wOffsetSrc` bytes from
    the upper left corner to the CUDA array `dst` starting at `hOffsetDst`
    rows and `wOffsetDst` bytes from the upper left corner, where `kind`
    specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. `wOffsetDst` + `width` must not exceed the
    width of the CUDA array `dst`. `wOffsetSrc` + `width` must not exceed
    the width of the CUDA array `src`.

    Parameters
    ----------
    dst : :py:obj:`~.cudaArray_t`
        Destination memory address
    wOffsetDst : size_t
        Destination starting X offset (columns in bytes)
    hOffsetDst : size_t
        Destination starting Y offset (rows)
    src : :py:obj:`~.cudaArray_const_t`
        Source memory address
    wOffsetSrc : size_t
        Source starting X offset (columns in bytes)
    hOffsetSrc : size_t
        Source starting Y offset (rows)
    width : size_t
        Width of matrix transfer (columns in bytes)
    height : size_t
        Height of matrix transfer (rows)
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DUnaligned`
    """
    cdef ccudart.cudaArray_const_t csrc
    if src is None:
        csrc = <ccudart.cudaArray_const_t><void_ptr>0
    elif isinstance(src, (cudaArray_const_t)):
        psrc = int(src)
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc
    else:
        psrc = int(cudaArray_const_t(src))
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc

    cdef ccudart.cudaArray_t cdst
    if dst is None:
        cdst = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(dst, (cudaArray_t)):
        pdst = int(dst)
        cdst = <ccudart.cudaArray_t><void_ptr>pdst
    else:
        pdst = int(cudaArray_t(dst))
        cdst = <ccudart.cudaArray_t><void_ptr>pdst

    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaMemcpy2DArrayToArray(cdst, wOffsetDst, hOffsetDst, csrc, wOffsetSrc, hOffsetSrc, width, height, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpyAsync(dst, src, size_t count, kind not None : cudaMemcpyKind, stream):
    """ Copies data between host and device.

    Copies `count` bytes from the memory area pointed to by `src` to the
    memory area pointed to by `dst`, where `kind` specifies the direction
    of the copy, and must be one of :py:obj:`~.cudaMemcpyHostToHost`,
    :py:obj:`~.cudaMemcpyHostToDevice`, :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing.

    The memory areas may not overlap. Calling :py:obj:`~.cudaMemcpyAsync()`
    with `dst` and `src` pointers that do not match the direction of the
    copy results in an undefined behavior.

    :py:obj:`~.cudaMemcpyAsync()` is asynchronous with respect to the host,
    so the call may return before the copy is complete. The copy can
    optionally be associated to a stream by passing a non-zero `stream`
    argument. If `kind` is :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` and the `stream` is non-zero, the
    copy may overlap with operations in other streams.

    The device version of this function only handles device to device
    copies and cannot be given local or shared pointers.

    Parameters
    ----------
    dst : Any
        Destination memory address
    src : Any
        Source memory address
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpyAsync`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemcpyDtoDAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpyAsync(cdst_ptr, csrc_ptr, count, ckind, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpyPeerAsync(dst, int dstDevice, src, int srcDevice, size_t count, stream):
    """ Copies memory between two devices asynchronously.

    Copies memory from one device to memory on another device. `dst` is the
    base device pointer of the destination memory and `dstDevice` is the
    destination device. `src` is the base device pointer of the source
    memory and `srcDevice` is the source device. `count` specifies the
    number of bytes to copy.

    Note that this function is asynchronous with respect to the host and
    all work on other devices.

    Parameters
    ----------
    dst : Any
        Destination device pointer
    dstDevice : int
        Destination device
    src : Any
        Source device pointer
    srcDevice : int
        Source device
    count : size_t
        Size of memory copy in bytes
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyPeer`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`, :py:obj:`~.cuMemcpyPeerAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    with nogil:
        err = ccudart.cudaMemcpyPeerAsync(cdst_ptr, dstDevice, csrc_ptr, srcDevice, count, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy2DAsync(dst, size_t dpitch, src, size_t spitch, size_t width, size_t height, kind not None : cudaMemcpyKind, stream):
    """ Copies data between host and device.

    Copies a matrix (`height` rows of `width` bytes each) from the memory
    area pointed to by `src` to the memory area pointed to by `dst`, where
    `kind` specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. `dpitch` and `spitch` are the widths in
    memory in bytes of the 2D arrays pointed to by `dst` and `src`,
    including any padding added to the end of each row. The memory areas
    may not overlap. `width` must not exceed either `dpitch` or `spitch`.

    Calling :py:obj:`~.cudaMemcpy2DAsync()` with `dst` and `src` pointers
    that do not match the direction of the copy results in an undefined
    behavior. :py:obj:`~.cudaMemcpy2DAsync()` returns an error if `dpitch`
    or `spitch` is greater than the maximum allowed.

    :py:obj:`~.cudaMemcpy2DAsync()` is asynchronous with respect to the
    host, so the call may return before the copy is complete. The copy can
    optionally be associated to a stream by passing a non-zero `stream`
    argument. If `kind` is :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` and `stream` is non-zero, the copy
    may overlap with operations in other streams.

    The device version of this function only handles device to device
    copies and cannot be given local or shared pointers.

    Parameters
    ----------
    dst : Any
        Destination memory address
    dpitch : size_t
        Pitch of destination memory
    src : Any
        Source memory address
    spitch : size_t
        Pitch of source memory
    width : size_t
        Width of matrix transfer (columns in bytes)
    height : size_t
        Height of matrix transfer (rows)
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpy2DAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpy2DAsync(cdst_ptr, dpitch, csrc_ptr, spitch, width, height, ckind, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy2DToArrayAsync(dst, size_t wOffset, size_t hOffset, src, size_t spitch, size_t width, size_t height, kind not None : cudaMemcpyKind, stream):
    """ Copies data between host and device.

    Copies a matrix (`height` rows of `width` bytes each) from the memory
    area pointed to by `src` to the CUDA array `dst` starting at `hOffset`
    rows and `wOffset` bytes from the upper left corner, where `kind`
    specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. `spitch` is the width in memory in bytes of
    the 2D array pointed to by `src`, including any padding added to the
    end of each row. `wOffset` + `width` must not exceed the width of the
    CUDA array `dst`. `width` must not exceed `spitch`.
    :py:obj:`~.cudaMemcpy2DToArrayAsync()` returns an error if `spitch`
    exceeds the maximum allowed.

    :py:obj:`~.cudaMemcpy2DToArrayAsync()` is asynchronous with respect to
    the host, so the call may return before the copy is complete. The copy
    can optionally be associated to a stream by passing a non-zero `stream`
    argument. If `kind` is :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` and `stream` is non-zero, the copy
    may overlap with operations in other streams.

    :py:obj:`~.cudaMemcpy2DFromArrayAsync`,
    :py:obj:`~.cudaMemcpyToSymbolAsync`,
    :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpy2DAsync`

    Parameters
    ----------
    dst : :py:obj:`~.cudaArray_t`
        Destination memory address
    wOffset : size_t
        Destination starting X offset (columns in bytes)
    hOffset : size_t
        Destination starting Y offset (rows)
    src : Any
        Source memory address
    spitch : size_t
        Pitch of source memory
    width : size_t
        Width of matrix transfer (columns in bytes)
    height : size_t
        Height of matrix transfer (rows)
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`,
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaArray_t cdst
    if dst is None:
        cdst = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(dst, (cudaArray_t)):
        pdst = int(dst)
        cdst = <ccudart.cudaArray_t><void_ptr>pdst
    else:
        pdst = int(cudaArray_t(dst))
        cdst = <ccudart.cudaArray_t><void_ptr>pdst

    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpy2DToArrayAsync(cdst, wOffset, hOffset, csrc_ptr, spitch, width, height, ckind, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpy2DFromArrayAsync(dst, size_t dpitch, src, size_t wOffset, size_t hOffset, size_t width, size_t height, kind not None : cudaMemcpyKind, stream):
    """ Copies data between host and device.

    Copies a matrix (`height` rows of `width` bytes each) from the CUDA
    array `src` starting at `hOffset` rows and `wOffset` bytes from the
    upper left corner to the memory area pointed to by `dst`, where `kind`
    specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. `dpitch` is the width in memory in bytes of
    the 2D array pointed to by `dst`, including any padding added to the
    end of each row. `wOffset` + `width` must not exceed the width of the
    CUDA array `src`. `width` must not exceed `dpitch`.
    :py:obj:`~.cudaMemcpy2DFromArrayAsync()` returns an error if `dpitch`
    exceeds the maximum allowed.

    :py:obj:`~.cudaMemcpy2DFromArrayAsync()` is asynchronous with respect
    to the host, so the call may return before the copy is complete. The
    copy can optionally be associated to a stream by passing a non-zero
    `stream` argument. If `kind` is :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` and `stream` is non-zero, the copy
    may overlap with operations in other streams.

    :py:obj:`~.cudaMemcpyToSymbolAsync`,
    :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpy2DAsync`

    Parameters
    ----------
    dst : Any
        Destination memory address
    dpitch : size_t
        Pitch of destination memory
    src : :py:obj:`~.cudaArray_const_t`
        Source memory address
    wOffset : size_t
        Source starting X offset (columns in bytes)
    hOffset : size_t
        Source starting Y offset (rows)
    width : size_t
        Width of matrix transfer (columns in bytes)
    height : size_t
        Height of matrix transfer (rows)
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidPitchValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`,
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaArray_const_t csrc
    if src is None:
        csrc = <ccudart.cudaArray_const_t><void_ptr>0
    elif isinstance(src, (cudaArray_const_t)):
        psrc = int(src)
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc
    else:
        psrc = int(cudaArray_const_t(src))
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    with nogil:
        err = ccudart.cudaMemcpy2DFromArrayAsync(cdst_ptr, dpitch, csrc, wOffset, hOffset, width, height, ckind, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemset(devPtr, int value, size_t count):
    """ Initializes or sets device memory to a value.

    Fills the first `count` bytes of the memory area pointed to by `devPtr`
    with the constant byte value `value`.

    Note that this function is asynchronous with respect to the host unless
    `devPtr` refers to pinned host memory.

    Parameters
    ----------
    devPtr : Any
        Pointer to device memory
    value : int
        Value to set for each byte of specified memory
    count : size_t
        Size in bytes to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`
    """
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    with nogil:
        err = ccudart.cudaMemset(cdevPtr_ptr, value, count)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemset2D(devPtr, size_t pitch, int value, size_t width, size_t height):
    """ Initializes or sets device memory to a value.

    Sets to the specified value `value` a matrix (`height` rows of `width`
    bytes each) pointed to by `dstPtr`. `pitch` is the width in bytes of
    the 2D array pointed to by `dstPtr`, including any padding added to the
    end of each row. This function performs fastest when the pitch is one
    that has been passed back by :py:obj:`~.cudaMallocPitch()`.

    Note that this function is asynchronous with respect to the host unless
    `devPtr` refers to pinned host memory.

    Parameters
    ----------
    devPtr : Any
        Pointer to 2D device memory
    pitch : size_t
        Pitch in bytes of 2D device memory(Unused if `height` is 1)
    value : int
        Value to set for each byte of specified memory
    width : size_t
        Width of matrix set (columns in bytes)
    height : size_t
        Height of matrix set (rows)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaMemset`, :py:obj:`~.cudaMemset3D`, :py:obj:`~.cudaMemsetAsync`, :py:obj:`~.cudaMemset2DAsync`, :py:obj:`~.cudaMemset3DAsync`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`
    """
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    err = ccudart.cudaMemset2D(cdevPtr_ptr, pitch, value, width, height)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemset3D(pitchedDevPtr not None : cudaPitchedPtr, int value, extent not None : cudaExtent):
    """ Initializes or sets device memory to a value.

    Initializes each element of a 3D array to the specified value `value`.
    The object to initialize is defined by `pitchedDevPtr`. The `pitch`
    field of `pitchedDevPtr` is the width in memory in bytes of the 3D
    array pointed to by `pitchedDevPtr`, including any padding added to the
    end of each row. The `xsize` field specifies the logical width of each
    row in bytes, while the `ysize` field specifies the height of each 2D
    slice in rows. The `pitch` field of `pitchedDevPtr` is ignored when
    `height` and `depth` are both equal to 1.

    The extents of the initialized region are specified as a `width` in
    bytes, a `height` in rows, and a `depth` in slices.

    Extents with `width` greater than or equal to the `xsize` of
    `pitchedDevPtr` may perform significantly faster than extents narrower
    than the `xsize`. Secondarily, extents with `height` equal to the
    `ysize` of `pitchedDevPtr` will perform faster than when the `height`
    is shorter than the `ysize`.

    This function performs fastest when the `pitchedDevPtr` has been
    allocated by :py:obj:`~.cudaMalloc3D()`.

    Note that this function is asynchronous with respect to the host unless
    `pitchedDevPtr` refers to pinned host memory.

    Parameters
    ----------
    pitchedDevPtr : :py:obj:`~.cudaPitchedPtr`
        Pointer to pitched device memory
    value : int
        Value to set for each byte of specified memory
    extent : :py:obj:`~.cudaExtent`
        Size parameters for where to set device memory (`width` field in
        bytes)

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaMemset`, :py:obj:`~.cudaMemset2D`, :py:obj:`~.cudaMemsetAsync`, :py:obj:`~.cudaMemset2DAsync`, :py:obj:`~.cudaMemset3DAsync`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.make_cudaPitchedPtr`, :py:obj:`~.make_cudaExtent`
    """
    err = ccudart.cudaMemset3D(pitchedDevPtr._ptr[0], value, extent._ptr[0])
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemsetAsync(devPtr, int value, size_t count, stream):
    """ Initializes or sets device memory to a value.

    Fills the first `count` bytes of the memory area pointed to by `devPtr`
    with the constant byte value `value`.

    :py:obj:`~.cudaMemsetAsync()` is asynchronous with respect to the host,
    so the call may return before the memset is complete. The operation can
    optionally be associated to a stream by passing a non-zero `stream`
    argument. If `stream` is non-zero, the operation may overlap with
    operations in other streams.

    The device version of this function only handles device to device
    copies and cannot be given local or shared pointers.

    Parameters
    ----------
    devPtr : Any
        Pointer to device memory
    value : int
        Value to set for each byte of specified memory
    count : size_t
        Size in bytes to set
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaMemset`, :py:obj:`~.cudaMemset2D`, :py:obj:`~.cudaMemset3D`, :py:obj:`~.cudaMemset2DAsync`, :py:obj:`~.cudaMemset3DAsync`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32Async`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    with nogil:
        err = ccudart.cudaMemsetAsync(cdevPtr_ptr, value, count, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemset2DAsync(devPtr, size_t pitch, int value, size_t width, size_t height, stream):
    """ Initializes or sets device memory to a value.

    Sets to the specified value `value` a matrix (`height` rows of `width`
    bytes each) pointed to by `dstPtr`. `pitch` is the width in bytes of
    the 2D array pointed to by `dstPtr`, including any padding added to the
    end of each row. This function performs fastest when the pitch is one
    that has been passed back by :py:obj:`~.cudaMallocPitch()`.

    :py:obj:`~.cudaMemset2DAsync()` is asynchronous with respect to the
    host, so the call may return before the memset is complete. The
    operation can optionally be associated to a stream by passing a non-
    zero `stream` argument. If `stream` is non-zero, the operation may
    overlap with operations in other streams.

    The device version of this function only handles device to device
    copies and cannot be given local or shared pointers.

    Parameters
    ----------
    devPtr : Any
        Pointer to 2D device memory
    pitch : size_t
        Pitch in bytes of 2D device memory(Unused if `height` is 1)
    value : int
        Value to set for each byte of specified memory
    width : size_t
        Width of matrix set (columns in bytes)
    height : size_t
        Height of matrix set (rows)
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaMemset`, :py:obj:`~.cudaMemset2D`, :py:obj:`~.cudaMemset3D`, :py:obj:`~.cudaMemsetAsync`, :py:obj:`~.cudaMemset3DAsync`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32Async`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    err = ccudart.cudaMemset2DAsync(cdevPtr_ptr, pitch, value, width, height, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemset3DAsync(pitchedDevPtr not None : cudaPitchedPtr, int value, extent not None : cudaExtent, stream):
    """ Initializes or sets device memory to a value.

    Initializes each element of a 3D array to the specified value `value`.
    The object to initialize is defined by `pitchedDevPtr`. The `pitch`
    field of `pitchedDevPtr` is the width in memory in bytes of the 3D
    array pointed to by `pitchedDevPtr`, including any padding added to the
    end of each row. The `xsize` field specifies the logical width of each
    row in bytes, while the `ysize` field specifies the height of each 2D
    slice in rows. The `pitch` field of `pitchedDevPtr` is ignored when
    `height` and `depth` are both equal to 1.

    The extents of the initialized region are specified as a `width` in
    bytes, a `height` in rows, and a `depth` in slices.

    Extents with `width` greater than or equal to the `xsize` of
    `pitchedDevPtr` may perform significantly faster than extents narrower
    than the `xsize`. Secondarily, extents with `height` equal to the
    `ysize` of `pitchedDevPtr` will perform faster than when the `height`
    is shorter than the `ysize`.

    This function performs fastest when the `pitchedDevPtr` has been
    allocated by :py:obj:`~.cudaMalloc3D()`.

    :py:obj:`~.cudaMemset3DAsync()` is asynchronous with respect to the
    host, so the call may return before the memset is complete. The
    operation can optionally be associated to a stream by passing a non-
    zero `stream` argument. If `stream` is non-zero, the operation may
    overlap with operations in other streams.

    The device version of this function only handles device to device
    copies and cannot be given local or shared pointers.

    Parameters
    ----------
    pitchedDevPtr : :py:obj:`~.cudaPitchedPtr`
        Pointer to pitched device memory
    value : int
        Value to set for each byte of specified memory
    extent : :py:obj:`~.cudaExtent`
        Size parameters for where to set device memory (`width` field in
        bytes)
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaMemset`, :py:obj:`~.cudaMemset2D`, :py:obj:`~.cudaMemset3D`, :py:obj:`~.cudaMemsetAsync`, :py:obj:`~.cudaMemset2DAsync`, :py:obj:`~.cudaMalloc3D`, :py:obj:`~.make_cudaPitchedPtr`, :py:obj:`~.make_cudaExtent`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    err = ccudart.cudaMemset3DAsync(pitchedDevPtr._ptr[0], value, extent._ptr[0], cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemPrefetchAsync(devPtr, size_t count, int dstDevice, stream):
    """ Prefetches memory to the specified destination device.

    Prefetches memory to the specified destination device. `devPtr` is the
    base device pointer of the memory to be prefetched and `dstDevice` is
    the destination device. `count` specifies the number of bytes to copy.
    `stream` is the stream in which the operation is enqueued. The memory
    range must refer to managed memory allocated via
    :py:obj:`~.cudaMallocManaged` or declared via managed variables.

    Passing in cudaCpuDeviceId for `dstDevice` will prefetch the data to
    host memory. If `dstDevice` is a GPU, then the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess` must be non-zero.
    Additionally, `stream` must be associated with a device that has a non-
    zero value for the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess`.

    The start address and end address of the memory range will be rounded
    down and rounded up respectively to be aligned to CPU page size before
    the prefetch operation is enqueued in the stream.

    If no physical memory has been allocated for this region, then this
    memory region will be populated and mapped on the destination device.
    If there's insufficient memory to prefetch the desired region, the
    Unified Memory driver may evict pages from other
    :py:obj:`~.cudaMallocManaged` allocations to host memory in order to
    make room. Device memory allocated using :py:obj:`~.cudaMalloc` or
    :py:obj:`~.cudaMallocArray` will not be evicted.

    By default, any mappings to the previous location of the migrated pages
    are removed and mappings for the new location are only setup on
    `dstDevice`. The exact behavior however also depends on the settings
    applied to this memory range via :py:obj:`~.cudaMemAdvise` as described
    below:

    If :py:obj:`~.cudaMemAdviseSetReadMostly` was set on any subset of this
    memory range, then that subset will create a read-only copy of the
    pages on `dstDevice`.

    If :py:obj:`~.cudaMemAdviseSetPreferredLocation` was called on any
    subset of this memory range, then the pages will be migrated to
    `dstDevice` even if `dstDevice` is not the preferred location of any
    pages in the memory range.

    If :py:obj:`~.cudaMemAdviseSetAccessedBy` was called on any subset of
    this memory range, then mappings to those pages from all the
    appropriate processors are updated to refer to the new location if
    establishing such a mapping is possible. Otherwise, those mappings are
    cleared.

    Note that this API is not required for functionality and only serves to
    improve performance by allowing the application to migrate data to a
    suitable location before it is accessed. Memory accesses to this range
    are always coherent and are allowed even when the data is actively
    being migrated.

    Note that this function is asynchronous with respect to the host and
    all work on other devices.

    Parameters
    ----------
    devPtr : Any
        Pointer to be prefetched
    count : size_t
        Size in bytes
    dstDevice : int
        Destination device to prefetch to
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to enqueue prefetch operation

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyPeer`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`, :py:obj:`~.cudaMemAdvise`, :py:obj:`~.cuMemPrefetchAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    with nogil:
        err = ccudart.cudaMemPrefetchAsync(cdevPtr_ptr, count, dstDevice, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemAdvise(devPtr, size_t count, advice not None : cudaMemoryAdvise, int device):
    """ Advise about the usage of a given memory range.

    Advise the Unified Memory subsystem about the usage pattern for the
    memory range starting at `devPtr` with a size of `count` bytes. The
    start address and end address of the memory range will be rounded down
    and rounded up respectively to be aligned to CPU page size before the
    advice is applied. The memory range must refer to managed memory
    allocated via :py:obj:`~.cudaMallocManaged` or declared via managed
    variables. The memory range could also refer to system-allocated
    pageable memory provided it represents a valid, host-accessible region
    of memory and all additional constraints imposed by `advice` as
    outlined below are also satisfied. Specifying an invalid system-
    allocated pageable memory range results in an error being returned.

    The `advice` parameter can take the following values:

    - :py:obj:`~.cudaMemAdviseSetReadMostly`: This implies that the data is
      mostly going to be read from and only occasionally written to. Any
      read accesses from any processor to this region will create a read-
      only copy of at least the accessed pages in that processor's memory.
      Additionally, if :py:obj:`~.cudaMemPrefetchAsync` is called on this
      region, it will create a read-only copy of the data on the
      destination processor. If any processor writes to this region, all
      copies of the corresponding page will be invalidated except for the
      one where the write occurred. The `device` argument is ignored for
      this advice. Note that for a page to be read-duplicated, the
      accessing processor must either be the CPU or a GPU that has a non-
      zero value for the device attribute
      :py:obj:`~.cudaDevAttrConcurrentManagedAccess`. Also, if a context is
      created on a device that does not have the device attribute
      :py:obj:`~.cudaDevAttrConcurrentManagedAccess` set, then read-
      duplication will not occur until all such contexts are destroyed. If
      the memory region refers to valid system-allocated pageable memory,
      then the accessing device must have a non-zero value for the device
      attribute :py:obj:`~.cudaDevAttrPageableMemoryAccess` for a read-only
      copy to be created on that device. Note however that if the accessing
      device also has a non-zero value for the device attribute
      :py:obj:`~.cudaDevAttrPageableMemoryAccessUsesHostPageTables`, then
      setting this advice will not create a read-only copy when that device
      accesses this memory region.

    - :py:obj:`~.cudaMemAdviceUnsetReadMostly`: Undoes the effect of
      :py:obj:`~.cudaMemAdviceReadMostly` and also prevents the Unified
      Memory driver from attempting heuristic read-duplication on the
      memory range. Any read-duplicated copies of the data will be
      collapsed into a single copy. The location for the collapsed copy
      will be the preferred location if the page has a preferred location
      and one of the read-duplicated copies was resident at that location.
      Otherwise, the location chosen is arbitrary.

    - :py:obj:`~.cudaMemAdviseSetPreferredLocation`: This advice sets the
      preferred location for the data to be the memory belonging to
      `device`. Passing in cudaCpuDeviceId for `device` sets the preferred
      location as host memory. If `device` is a GPU, then it must have a
      non-zero value for the device attribute
      :py:obj:`~.cudaDevAttrConcurrentManagedAccess`. Setting the preferred
      location does not cause data to migrate to that location immediately.
      Instead, it guides the migration policy when a fault occurs on that
      memory region. If the data is already in its preferred location and
      the faulting processor can establish a mapping without requiring the
      data to be migrated, then data migration will be avoided. On the
      other hand, if the data is not in its preferred location or if a
      direct mapping cannot be established, then it will be migrated to the
      processor accessing it. It is important to note that setting the
      preferred location does not prevent data prefetching done using
      :py:obj:`~.cudaMemPrefetchAsync`. Having a preferred location can
      override the page thrash detection and resolution logic in the
      Unified Memory driver. Normally, if a page is detected to be
      constantly thrashing between for example host and device memory, the
      page may eventually be pinned to host memory by the Unified Memory
      driver. But if the preferred location is set as device memory, then
      the page will continue to thrash indefinitely. If
      :py:obj:`~.cudaMemAdviseSetReadMostly` is also set on this memory
      region or any subset of it, then the policies associated with that
      advice will override the policies of this advice, unless read
      accesses from `device` will not result in a read-only copy being
      created on that device as outlined in description for the advice
      :py:obj:`~.cudaMemAdviseSetReadMostly`. If the memory region refers
      to valid system-allocated pageable memory, then `device` must have a
      non-zero value for the device attribute
      :py:obj:`~.cudaDevAttrPageableMemoryAccess`. Additionally, if
      `device` has a non-zero value for the device attribute
      :py:obj:`~.cudaDevAttrPageableMemoryAccessUsesHostPageTables`, then
      this call has no effect. Note however that this behavior may change
      in the future.

    - :py:obj:`~.cudaMemAdviseUnsetPreferredLocation`: Undoes the effect of
      :py:obj:`~.cudaMemAdviseSetPreferredLocation` and changes the
      preferred location to none.

    - :py:obj:`~.cudaMemAdviseSetAccessedBy`: This advice implies that the
      data will be accessed by `device`. Passing in
      :py:obj:`~.cudaCpuDeviceId` for `device` will set the advice for the
      CPU. If `device` is a GPU, then the device attribute
      :py:obj:`~.cudaDevAttrConcurrentManagedAccess` must be non-zero. This
      advice does not cause data migration and has no impact on the
      location of the data per se. Instead, it causes the data to always be
      mapped in the specified processor's page tables, as long as the
      location of the data permits a mapping to be established. If the data
      gets migrated for any reason, the mappings are updated accordingly.
      This advice is recommended in scenarios where data locality is not
      important, but avoiding faults is. Consider for example a system
      containing multiple GPUs with peer-to-peer access enabled, where the
      data located on one GPU is occasionally accessed by peer GPUs. In
      such scenarios, migrating data over to the other GPUs is not as
      important because the accesses are infrequent and the overhead of
      migration may be too high. But preventing faults can still help
      improve performance, and so having a mapping set up in advance is
      useful. Note that on CPU access of this data, the data may be
      migrated to host memory because the CPU typically cannot access
      device memory directly. Any GPU that had the
      :py:obj:`~.cudaMemAdviceSetAccessedBy` flag set for this data will
      now have its mapping updated to point to the page in host memory. If
      :py:obj:`~.cudaMemAdviseSetReadMostly` is also set on this memory
      region or any subset of it, then the policies associated with that
      advice will override the policies of this advice. Additionally, if
      the preferred location of this memory region or any subset of it is
      also `device`, then the policies associated with
      :py:obj:`~.cudaMemAdviseSetPreferredLocation` will override the
      policies of this advice. If the memory region refers to valid system-
      allocated pageable memory, then `device` must have a non-zero value
      for the device attribute :py:obj:`~.cudaDevAttrPageableMemoryAccess`.
      Additionally, if `device` has a non-zero value for the device
      attribute
      :py:obj:`~.cudaDevAttrPageableMemoryAccessUsesHostPageTables`, then
      this call has no effect.

    - :py:obj:`~.cudaMemAdviseUnsetAccessedBy`: Undoes the effect of
      :py:obj:`~.cudaMemAdviseSetAccessedBy`. Any mappings to the data from
      `device` may be removed at any time causing accesses to result in
      non-fatal page faults. If the memory region refers to valid system-
      allocated pageable memory, then `device` must have a non-zero value
      for the device attribute :py:obj:`~.cudaDevAttrPageableMemoryAccess`.
      Additionally, if `device` has a non-zero value for the device
      attribute
      :py:obj:`~.cudaDevAttrPageableMemoryAccessUsesHostPageTables`, then
      this call has no effect.

    Parameters
    ----------
    devPtr : Any
        Pointer to memory to set the advice for
    count : size_t
        Size in bytes of the memory range
    advice : :py:obj:`~.cudaMemoryAdvise`
        Advice to be applied for the specified memory range
    device : int
        Device to apply the advice for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyPeer`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`, :py:obj:`~.cudaMemPrefetchAsync`, :py:obj:`~.cuMemAdvise`
    """
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    cdef ccudart.cudaMemoryAdvise cadvice = advice.value
    with nogil:
        err = ccudart.cudaMemAdvise(cdevPtr_ptr, count, cadvice, device)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemRangeGetAttribute(size_t dataSize, attribute not None : cudaMemRangeAttribute, devPtr, size_t count):
    """ Query an attribute of a given memory range.

    Query an attribute about the memory range starting at `devPtr` with a
    size of `count` bytes. The memory range must refer to managed memory
    allocated via :py:obj:`~.cudaMallocManaged` or declared via managed
    variables.

    The `attribute` parameter can take the following values:

    - :py:obj:`~.cudaMemRangeAttributeReadMostly`: If this attribute is
      specified, `data` will be interpreted as a 32-bit integer, and
      `dataSize` must be 4. The result returned will be 1 if all pages in
      the given memory range have read-duplication enabled, or 0 otherwise.

    - :py:obj:`~.cudaMemRangeAttributePreferredLocation`: If this attribute
      is specified, `data` will be interpreted as a 32-bit integer, and
      `dataSize` must be 4. The result returned will be a GPU device id if
      all pages in the memory range have that GPU as their preferred
      location, or it will be cudaCpuDeviceId if all pages in the memory
      range have the CPU as their preferred location, or it will be
      cudaInvalidDeviceId if either all the pages don't have the same
      preferred location or some of the pages don't have a preferred
      location at all. Note that the actual location of the pages in the
      memory range at the time of the query may be different from the
      preferred location.

    - :py:obj:`~.cudaMemRangeAttributeAccessedBy`: If this attribute is
      specified, `data` will be interpreted as an array of 32-bit integers,
      and `dataSize` must be a non-zero multiple of 4. The result returned
      will be a list of device ids that had
      :py:obj:`~.cudaMemAdviceSetAccessedBy` set for that entire memory
      range. If any device does not have that advice set for the entire
      memory range, that device will not be included. If `data` is larger
      than the number of devices that have that advice set for that memory
      range, cudaInvalidDeviceId will be returned in all the extra space
      provided. For ex., if `dataSize` is 12 (i.e. `data` has 3 elements)
      and only device 0 has the advice set, then the result returned will
      be { 0, cudaInvalidDeviceId, cudaInvalidDeviceId }. If `data` is
      smaller than the number of devices that have that advice set, then
      only as many devices will be returned as can fit in the array. There
      is no guarantee on which specific devices will be returned, however.

    - :py:obj:`~.cudaMemRangeAttributeLastPrefetchLocation`: If this
      attribute is specified, `data` will be interpreted as a 32-bit
      integer, and `dataSize` must be 4. The result returned will be the
      last location to which all pages in the memory range were prefetched
      explicitly via :py:obj:`~.cudaMemPrefetchAsync`. This will either be
      a GPU id or cudaCpuDeviceId depending on whether the last location
      for prefetch was a GPU or the CPU respectively. If any page in the
      memory range was never explicitly prefetched or if all pages were not
      prefetched to the same location, cudaInvalidDeviceId will be
      returned. Note that this simply returns the last location that the
      applicaton requested to prefetch the memory range to. It gives no
      indication as to whether the prefetch operation to that location has
      completed or even begun.

    Parameters
    ----------
    dataSize : size_t
        Array containing the size of data
    attribute : :py:obj:`~.cudaMemRangeAttribute`
        The attribute to query
    devPtr : Any
        Start of the range to query
    count : size_t
        Size of the range to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    data : Any
        A pointers to a memory location where the result of each attribute
        query will be written to.

    See Also
    --------
    :py:obj:`~.cudaMemRangeGetAttributes`, :py:obj:`~.cudaMemPrefetchAsync`, :py:obj:`~.cudaMemAdvise`, :py:obj:`~.cuMemRangeGetAttribute`
    """
    cdef utils.HelperCUmem_range_attribute cdata = utils.HelperCUmem_range_attribute(attribute, dataSize)
    cdef void* cdata_ptr = <void*><void_ptr>cdata.cptr
    cdef ccudart.cudaMemRangeAttribute cattribute = attribute.value
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    err = ccudart.cudaMemRangeGetAttribute(cdata_ptr, dataSize, cattribute, cdevPtr_ptr, count)
    return (cudaError_t(err), cdata.pyObj())

@cython.embedsignature(True)
def cudaMemRangeGetAttributes(dataSizes : List[int], attributes : List[cudaMemRangeAttribute], size_t numAttributes, devPtr, size_t count):
    """ Query attributes of a given memory range.

    Query attributes of the memory range starting at `devPtr` with a size
    of `count` bytes. The memory range must refer to managed memory
    allocated via :py:obj:`~.cudaMallocManaged` or declared via managed
    variables. The `attributes` array will be interpreted to have
    `numAttributes` entries. The `dataSizes` array will also be interpreted
    to have `numAttributes` entries. The results of the query will be
    stored in `data`.

    The list of supported attributes are given below. Please refer to
    :py:obj:`~.cudaMemRangeGetAttribute` for attribute descriptions and
    restrictions.

    - :py:obj:`~.cudaMemRangeAttributeReadMostly`

    - :py:obj:`~.cudaMemRangeAttributePreferredLocation`

    - :py:obj:`~.cudaMemRangeAttributeAccessedBy`

    - :py:obj:`~.cudaMemRangeAttributeLastPrefetchLocation`

    Parameters
    ----------
    dataSizes : List[int]
        Array containing the sizes of each result
    attributes : List[:py:obj:`~.cudaMemRangeAttribute`]
        An array of attributes to query (numAttributes and the number of
        attributes in this array should match)
    numAttributes : size_t
        Number of attributes to query
    devPtr : Any
        Start of the range to query
    count : size_t
        Size of the range to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    data : List[Any]
        A two-dimensional array containing pointers to memory locations
        where the result of each attribute query will be written to.

    See Also
    --------
    :py:obj:`~.cudaMemRangeGetAttribute`, :py:obj:`~.cudaMemAdvise`, :py:obj:`~.cudaMemPrefetchAsync`, :py:obj:`~.cuMemRangeGetAttributes`
    """
    attributes = [] if attributes is None else attributes
    if not all(isinstance(_x, (cudaMemRangeAttribute)) for _x in attributes):
        raise TypeError("Argument 'attributes' is not instance of type (expected List[ccudart.cudaMemRangeAttribute]")

    if not all(isinstance(_x, (int)) for _x in dataSizes):
        raise TypeError("Argument 'dataSizes' is not instance of type (expected List[int]")

    pylist = [utils.HelperCUmem_range_attribute(pyattributes, pydataSizes) for (pyattributes, pydataSizes) in zip(attributes, dataSizes)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelper = utils.InputVoidPtrPtrHelper(pylist)
    cdef void** cvoidStarHelper_ptr = <void**><void_ptr>voidStarHelper.cptr
    cdef vector[size_t] cdataSizes = dataSizes
    cdef vector[ccudart.cudaMemRangeAttribute] cattributes = [pyattributes.value for pyattributes in (attributes)]
    if numAttributes > <size_t>len(dataSizes): raise RuntimeError("List is too small: " + str(len(dataSizes)) + " < " + str(numAttributes))
    if numAttributes > <size_t>len(attributes): raise RuntimeError("List is too small: " + str(len(attributes)) + " < " + str(numAttributes))
    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    err = ccudart.cudaMemRangeGetAttributes(cvoidStarHelper_ptr, cdataSizes.data(), cattributes.data(), numAttributes, cdevPtr_ptr, count)
    return (cudaError_t(err), [obj.pyObj() for obj in pylist])

@cython.embedsignature(True)
def cudaMemcpyToArray(dst, size_t wOffset, size_t hOffset, src, size_t count, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    [Deprecated]

    Copies `count` bytes from the memory area pointed to by `src` to the
    CUDA array `dst` starting at `hOffset` rows and `wOffset` bytes from
    the upper left corner, where `kind` specifies the direction of the
    copy, and must be one of :py:obj:`~.cudaMemcpyHostToHost`,
    :py:obj:`~.cudaMemcpyHostToDevice`, :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing.

    Parameters
    ----------
    dst : :py:obj:`~.cudaArray_t`
        Destination memory address
    wOffset : size_t
        Destination starting X offset (columns in bytes)
    hOffset : size_t
        Destination starting Y offset (rows)
    src : Any
        Source memory address
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpyFromArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpyArrayToArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpyToArrayAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpyFromArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyDtoA`
    """
    cdef ccudart.cudaArray_t cdst
    if dst is None:
        cdst = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(dst, (cudaArray_t)):
        pdst = int(dst)
        cdst = <ccudart.cudaArray_t><void_ptr>pdst
    else:
        pdst = int(cudaArray_t(dst))
        cdst = <ccudart.cudaArray_t><void_ptr>pdst

    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaMemcpyToArray(cdst, wOffset, hOffset, csrc_ptr, count, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpyFromArray(dst, src, size_t wOffset, size_t hOffset, size_t count, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    [Deprecated]

    Copies `count` bytes from the CUDA array `src` starting at `hOffset`
    rows and `wOffset` bytes from the upper left corner to the memory area
    pointed to by `dst`, where `kind` specifies the direction of the copy,
    and must be one of :py:obj:`~.cudaMemcpyHostToHost`,
    :py:obj:`~.cudaMemcpyHostToDevice`, :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing.

    Parameters
    ----------
    dst : Any
        Destination memory address
    src : :py:obj:`~.cudaArray_const_t`
        Source memory address
    wOffset : size_t
        Source starting X offset (columns in bytes)
    hOffset : size_t
        Source starting Y offset (rows)
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpyToArray`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpyArrayToArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpyToArrayAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpyFromArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoD`
    """
    cdef ccudart.cudaArray_const_t csrc
    if src is None:
        csrc = <ccudart.cudaArray_const_t><void_ptr>0
    elif isinstance(src, (cudaArray_const_t)):
        psrc = int(src)
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc
    else:
        psrc = int(cudaArray_const_t(src))
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaMemcpyFromArray(cdst_ptr, csrc, wOffset, hOffset, count, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpyArrayToArray(dst, size_t wOffsetDst, size_t hOffsetDst, src, size_t wOffsetSrc, size_t hOffsetSrc, size_t count, kind not None : cudaMemcpyKind):
    """ Copies data between host and device.

    [Deprecated]

    Copies `count` bytes from the CUDA array `src` starting at `hOffsetSrc`
    rows and `wOffsetSrc` bytes from the upper left corner to the CUDA
    array `dst` starting at `hOffsetDst` rows and `wOffsetDst` bytes from
    the upper left corner, where `kind` specifies the direction of the
    copy, and must be one of :py:obj:`~.cudaMemcpyHostToHost`,
    :py:obj:`~.cudaMemcpyHostToDevice`, :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing.

    Parameters
    ----------
    dst : :py:obj:`~.cudaArray_t`
        Destination memory address
    wOffsetDst : size_t
        Destination starting X offset (columns in bytes)
    hOffsetDst : size_t
        Destination starting Y offset (rows)
    src : :py:obj:`~.cudaArray_const_t`
        Source memory address
    wOffsetSrc : size_t
        Source starting X offset (columns in bytes)
    hOffsetSrc : size_t
        Source starting Y offset (rows)
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpyToArray`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpyFromArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpyToArrayAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpyFromArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpyAtoA`
    """
    cdef ccudart.cudaArray_const_t csrc
    if src is None:
        csrc = <ccudart.cudaArray_const_t><void_ptr>0
    elif isinstance(src, (cudaArray_const_t)):
        psrc = int(src)
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc
    else:
        psrc = int(cudaArray_const_t(src))
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc

    cdef ccudart.cudaArray_t cdst
    if dst is None:
        cdst = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(dst, (cudaArray_t)):
        pdst = int(dst)
        cdst = <ccudart.cudaArray_t><void_ptr>pdst
    else:
        pdst = int(cudaArray_t(dst))
        cdst = <ccudart.cudaArray_t><void_ptr>pdst

    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaMemcpyArrayToArray(cdst, wOffsetDst, hOffsetDst, csrc, wOffsetSrc, hOffsetSrc, count, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpyToArrayAsync(dst, size_t wOffset, size_t hOffset, src, size_t count, kind not None : cudaMemcpyKind, stream):
    """ Copies data between host and device.

    [Deprecated]

    Copies `count` bytes from the memory area pointed to by `src` to the
    CUDA array `dst` starting at `hOffset` rows and `wOffset` bytes from
    the upper left corner, where `kind` specifies the direction of the
    copy, and must be one of :py:obj:`~.cudaMemcpyHostToHost`,
    :py:obj:`~.cudaMemcpyHostToDevice`, :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing.

    :py:obj:`~.cudaMemcpyToArrayAsync()` is asynchronous with respect to
    the host, so the call may return before the copy is complete. The copy
    can optionally be associated to a stream by passing a non-zero `stream`
    argument. If `kind` is :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` and `stream` is non-zero, the copy
    may overlap with operations in other streams.

    Parameters
    ----------
    dst : :py:obj:`~.cudaArray_t`
        Destination memory address
    wOffset : size_t
        Destination starting X offset (columns in bytes)
    hOffset : size_t
        Destination starting Y offset (rows)
    src : Any
        Source memory address
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpyToArray`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpyFromArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpyArrayToArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpyFromArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpy2DAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaArray_t cdst
    if dst is None:
        cdst = <ccudart.cudaArray_t><void_ptr>0
    elif isinstance(dst, (cudaArray_t)):
        pdst = int(dst)
        cdst = <ccudart.cudaArray_t><void_ptr>pdst
    else:
        pdst = int(cudaArray_t(dst))
        cdst = <ccudart.cudaArray_t><void_ptr>pdst

    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaMemcpyToArrayAsync(cdst, wOffset, hOffset, csrc_ptr, count, ckind, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemcpyFromArrayAsync(dst, src, size_t wOffset, size_t hOffset, size_t count, kind not None : cudaMemcpyKind, stream):
    """ Copies data between host and device.

    [Deprecated]

    Copies `count` bytes from the CUDA array `src` starting at `hOffset`
    rows and `wOffset` bytes from the upper left corner to the memory area
    pointed to by `dst`, where `kind` specifies the direction of the copy,
    and must be one of :py:obj:`~.cudaMemcpyHostToHost`,
    :py:obj:`~.cudaMemcpyHostToDevice`, :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing.

    :py:obj:`~.cudaMemcpyFromArrayAsync()` is asynchronous with respect to
    the host, so the call may return before the copy is complete. The copy
    can optionally be associated to a stream by passing a non-zero `stream`
    argument. If `kind` is :py:obj:`~.cudaMemcpyHostToDevice` or
    :py:obj:`~.cudaMemcpyDeviceToHost` and `stream` is non-zero, the copy
    may overlap with operations in other streams.

    Parameters
    ----------
    dst : Any
        Destination memory address
    src : :py:obj:`~.cudaArray_const_t`
        Source memory address
    wOffset : size_t
        Source starting X offset (columns in bytes)
    hOffset : size_t
        Source starting Y offset (rows)
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidMemcpyDirection`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpyToArray`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpyFromArray`, :py:obj:`~.cudaMemcpy2DFromArray`, :py:obj:`~.cudaMemcpyArrayToArray`, :py:obj:`~.cudaMemcpy2DArrayToArray`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpyToArrayAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpy2DAsync`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaArray_const_t csrc
    if src is None:
        csrc = <ccudart.cudaArray_const_t><void_ptr>0
    elif isinstance(src, (cudaArray_const_t)):
        psrc = int(src)
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc
    else:
        psrc = int(cudaArray_const_t(src))
        csrc = <ccudart.cudaArray_const_t><void_ptr>psrc

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaMemcpyFromArrayAsync(cdst_ptr, csrc, wOffset, hOffset, count, ckind, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMallocAsync(size_t size, hStream):
    """ Allocates memory with stream ordered semantics.

    Inserts an allocation operation into `hStream`. A pointer to the
    allocated memory is returned immediately in *dptr. The allocation must
    not be accessed until the the allocation operation completes. The
    allocation comes from the memory pool associated with the stream's
    device.

    Parameters
    ----------
    size : size_t
        Number of bytes to allocate
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream establishing the stream ordering contract and the memory
        pool to allocate from

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorOutOfMemory`,
    devPtr : Any
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, cudaMallocAsync (C++ API), :py:obj:`~.cudaMallocFromPoolAsync`, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaDeviceSetMemPool`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaDeviceGetMemPool`, :py:obj:`~.cudaMemPoolSetAccess`, :py:obj:`~.cudaMemPoolSetAttribute`, :py:obj:`~.cudaMemPoolGetAttribute`

    Notes
    -----
    The default memory pool of a device contains device memory from that device.

    Basic stream ordering allows future work submitted into the same stream to use the allocation. Stream query, stream synchronize, and CUDA events can be used to guarantee that the allocation operation completes before work submitted in a separate stream runs.

    During stream capture, this function results in the creation of an allocation node. In this case, the allocation is owned by the graph instead of the memory pool. The memory pool's properties are used to set the node's creation parameters.
    """
    cdef ccudart.cudaStream_t chStream
    if hStream is None:
        chStream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(hStream, (cudaStream_t,cuda.CUstream)):
        phStream = int(hStream)
        chStream = <ccudart.cudaStream_t><void_ptr>phStream
    else:
        phStream = int(cudaStream_t(hStream))
        chStream = <ccudart.cudaStream_t><void_ptr>phStream

    cdef void_ptr devPtr = 0
    with nogil:
        err = ccudart.cudaMallocAsync(<void**>&devPtr, size, chStream)
    return (cudaError_t(err), devPtr)

@cython.embedsignature(True)
def cudaFreeAsync(devPtr, hStream):
    """ Frees memory with stream ordered semantics.

    Inserts a free operation into `hStream`. The allocation must not be
    accessed after stream execution reaches the free. After this API
    returns, accessing the memory from any subsequent work launched on the
    GPU or querying its pointer attributes results in undefined behavior.

    Parameters
    ----------
    dptr : Any
        memory to free
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream establishing the stream ordering promise

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorNotSupported`

    See Also
    --------
    :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cudaMallocAsync`

    Notes
    -----
    During stream capture, this function results in the creation of a free node and must therefore be passed the address of a graph allocation.
    """
    cdef ccudart.cudaStream_t chStream
    if hStream is None:
        chStream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(hStream, (cudaStream_t,cuda.CUstream)):
        phStream = int(hStream)
        chStream = <ccudart.cudaStream_t><void_ptr>phStream
    else:
        phStream = int(cudaStream_t(hStream))
        chStream = <ccudart.cudaStream_t><void_ptr>phStream

    cdevPtr = utils.HelperInputVoidPtr(devPtr)
    cdef void* cdevPtr_ptr = <void*><void_ptr>cdevPtr.cptr
    with nogil:
        err = ccudart.cudaFreeAsync(cdevPtr_ptr, chStream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemPoolTrimTo(memPool, size_t minBytesToKeep):
    """ Tries to release memory back to the OS.

    Releases memory back to the OS until the pool contains fewer than
    minBytesToKeep reserved bytes, or there is no more memory that the
    allocator can safely release. The allocator cannot release OS
    allocations that back outstanding asynchronous allocations. The OS
    allocations may happen at different granularity from the user
    allocations.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The memory pool to trim
    minBytesToKeep : size_t
        If the pool has less than minBytesToKeep reserved, the TrimTo
        operation is a no-op. Otherwise the pool will be guaranteed to have
        at least minBytesToKeep bytes reserved after the operation.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cuMemPoolTrimTo`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaDeviceGetMemPool`, :py:obj:`~.cudaMemPoolCreate`

    Notes
    -----
    : Allocations that have not been freed count as outstanding.

    : Allocations that have been asynchronously freed but whose completion has not been observed on the host (eg. by a synchronize) can count as outstanding.
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    with nogil:
        err = ccudart.cudaMemPoolTrimTo(cmemPool, minBytesToKeep)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemPoolSetAttribute(memPool, attr not None : cudaMemPoolAttr, value):
    """ Sets attributes of a memory pool.

    Supported attributes are:

    - :py:obj:`~.cudaMemPoolAttrReleaseThreshold`: (value type =
      cuuint64_t) Amount of reserved memory in bytes to hold onto before
      trying to release memory back to the OS. When more than the release
      threshold bytes of memory are held by the memory pool, the allocator
      will try to release memory back to the OS on the next call to stream,
      event or context synchronize. (default 0)

    - :py:obj:`~.cudaMemPoolReuseFollowEventDependencies`: (value type =
      int) Allow :py:obj:`~.cudaMallocAsync` to use memory asynchronously
      freed in another stream as long as a stream ordering dependency of
      the allocating stream on the free action exists. Cuda events and null
      stream interactions can create the required stream ordered
      dependencies. (default enabled)

    - :py:obj:`~.cudaMemPoolReuseAllowOpportunistic`: (value type = int)
      Allow reuse of already completed frees when there is no dependency
      between the free and allocation. (default enabled)

    - :py:obj:`~.cudaMemPoolReuseAllowInternalDependencies`: (value type =
      int) Allow :py:obj:`~.cudaMallocAsync` to insert new stream
      dependencies in order to establish the stream ordering required to
      reuse a piece of memory released by :py:obj:`~.cudaFreeAsync`
      (default enabled).

    - :py:obj:`~.cudaMemPoolAttrReservedMemHigh`: (value type = cuuint64_t)
      Reset the high watermark that tracks the amount of backing memory
      that was allocated for the memory pool. It is illegal to set this
      attribute to a non-zero value.

    - :py:obj:`~.cudaMemPoolAttrUsedMemHigh`: (value type = cuuint64_t)
      Reset the high watermark that tracks the amount of used memory that
      was allocated for the memory pool. It is illegal to set this
      attribute to a non-zero value.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The memory pool to modify
    attr : :py:obj:`~.cudaMemPoolAttr`
        The attribute to modify
    value : Any
        Pointer to the value to assign

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cuMemPoolSetAttribute`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaDeviceGetMemPool`, :py:obj:`~.cudaMemPoolCreate`
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    cdef ccudart.cudaMemPoolAttr cattr = attr.value
    cdef utils.HelperCUmemPool_attribute cvalue = utils.HelperCUmemPool_attribute(attr, value, is_getter=False)
    cdef void* cvalue_ptr = <void*><void_ptr>cvalue.cptr
    with nogil:
        err = ccudart.cudaMemPoolSetAttribute(cmemPool, cattr, cvalue_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemPoolGetAttribute(memPool, attr not None : cudaMemPoolAttr):
    """ Gets attributes of a memory pool.

    Supported attributes are:

    - :py:obj:`~.cudaMemPoolAttrReleaseThreshold`: (value type =
      cuuint64_t) Amount of reserved memory in bytes to hold onto before
      trying to release memory back to the OS. When more than the release
      threshold bytes of memory are held by the memory pool, the allocator
      will try to release memory back to the OS on the next call to stream,
      event or context synchronize. (default 0)

    - :py:obj:`~.cudaMemPoolReuseFollowEventDependencies`: (value type =
      int) Allow :py:obj:`~.cudaMallocAsync` to use memory asynchronously
      freed in another stream as long as a stream ordering dependency of
      the allocating stream on the free action exists. Cuda events and null
      stream interactions can create the required stream ordered
      dependencies. (default enabled)

    - :py:obj:`~.cudaMemPoolReuseAllowOpportunistic`: (value type = int)
      Allow reuse of already completed frees when there is no dependency
      between the free and allocation. (default enabled)

    - :py:obj:`~.cudaMemPoolReuseAllowInternalDependencies`: (value type =
      int) Allow :py:obj:`~.cudaMallocAsync` to insert new stream
      dependencies in order to establish the stream ordering required to
      reuse a piece of memory released by :py:obj:`~.cudaFreeAsync`
      (default enabled).

    - :py:obj:`~.cudaMemPoolAttrReservedMemCurrent`: (value type =
      cuuint64_t) Amount of backing memory currently allocated for the
      mempool.

    - :py:obj:`~.cudaMemPoolAttrReservedMemHigh`: (value type = cuuint64_t)
      High watermark of backing memory allocated for the mempool since the
      last time it was reset.

    - :py:obj:`~.cudaMemPoolAttrUsedMemCurrent`: (value type = cuuint64_t)
      Amount of memory from the pool that is currently in use by the
      application.

    - :py:obj:`~.cudaMemPoolAttrUsedMemHigh`: (value type = cuuint64_t)
      High watermark of the amount of memory from the pool that was in use
      by the application since the last time it was reset.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The memory pool to get attributes of
    attr : :py:obj:`~.cudaMemPoolAttr`
        The attribute to get

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    value : Any
        Retrieved value

    See Also
    --------
    :py:obj:`~.cuMemPoolGetAttribute`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaDeviceGetMemPool`, :py:obj:`~.cudaMemPoolCreate`
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    cdef ccudart.cudaMemPoolAttr cattr = attr.value
    cdef utils.HelperCUmemPool_attribute cvalue = utils.HelperCUmemPool_attribute(attr, 0, is_getter=True)
    cdef void* cvalue_ptr = <void*><void_ptr>cvalue.cptr
    with nogil:
        err = ccudart.cudaMemPoolGetAttribute(cmemPool, cattr, cvalue_ptr)
    return (cudaError_t(err), cvalue.pyObj())

@cython.embedsignature(True)
def cudaMemPoolSetAccess(memPool, descList : List[cudaMemAccessDesc], size_t count):
    """ Controls visibility of pools between devices.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The pool being modified
    map : List[:py:obj:`~.cudaMemAccessDesc`]
        Array of access descriptors. Each descriptor instructs the access
        to enable for a single gpu
    count : size_t
        Number of descriptors in the map array.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cuMemPoolSetAccess`, :py:obj:`~.cudaMemPoolGetAccess`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`
    """
    descList = [] if descList is None else descList
    if not all(isinstance(_x, (cudaMemAccessDesc)) for _x in descList):
        raise TypeError("Argument 'descList' is not instance of type (expected List[ccudart.cudaMemAccessDesc]")

    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    cdef ccudart.cudaMemAccessDesc* cdescList = NULL
    if len(descList) > 0:
        cdescList = <ccudart.cudaMemAccessDesc*> calloc(len(descList), sizeof(ccudart.cudaMemAccessDesc))
        if cdescList is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(descList)) + 'x' + str(sizeof(ccudart.cudaMemAccessDesc)))
        for idx in range(len(descList)):
            memcpy(&cdescList[idx], (<cudaMemAccessDesc>descList[idx])._ptr, sizeof(ccudart.cudaMemAccessDesc))

    if count > <size_t>len(descList): raise RuntimeError("List is too small: " + str(len(descList)) + " < " + str(count))
    err = ccudart.cudaMemPoolSetAccess(cmemPool, (<cudaMemAccessDesc>descList[0])._ptr if len(descList) == 1 else cdescList, count)
    if cdescList is not NULL:
        free(cdescList)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMemPoolGetAccess(memPool, location : cudaMemLocation):
    """ Returns the accessibility of a pool from a device.

    Returns the accessibility of the pool's memory from the specified
    location.

    Parameters
    ----------
    memPool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        the pool being queried
    location : :py:obj:`~.cudaMemLocation`
        the location accessing the pool

    Returns
    -------
    cudaError_t

    flags : :py:obj:`~.cudaMemAccessFlags`
        the accessibility of the pool from the specified location

    See Also
    --------
    :py:obj:`~.cuMemPoolGetAccess`, :py:obj:`~.cudaMemPoolSetAccess`
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    cdef ccudart.cudaMemAccessFlags flags
    cdef ccudart.cudaMemLocation* clocation_ptr = location._ptr if location != None else NULL
    err = ccudart.cudaMemPoolGetAccess(&flags, cmemPool, clocation_ptr)
    return (cudaError_t(err), cudaMemAccessFlags(flags))

@cython.embedsignature(True)
def cudaMemPoolCreate(poolProps : cudaMemPoolProps):
    """ Creates a memory pool.

    Creates a CUDA memory pool and returns the handle in `pool`. The
    `poolProps` determines the properties of the pool such as the backing
    device and IPC capabilities.

    By default, the pool's memory will be accessible from the device it is
    allocated on.

    Parameters
    ----------
    poolProps : :py:obj:`~.cudaMemPoolProps`
        None

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorNotSupported`
    memPool : :py:obj:`~.cudaMemPool_t`
        None

    See Also
    --------
    :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cudaDeviceSetMemPool`, :py:obj:`~.cudaMallocFromPoolAsync`, :py:obj:`~.cudaMemPoolExportToShareableHandle`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaDeviceGetMemPool`

    Notes
    -----
    Specifying cudaMemHandleTypeNone creates a memory pool that will not support IPC.
    """
    cdef cudaMemPool_t memPool = cudaMemPool_t()
    cdef ccudart.cudaMemPoolProps* cpoolProps_ptr = poolProps._ptr if poolProps != None else NULL
    err = ccudart.cudaMemPoolCreate(<ccudart.cudaMemPool_t*>memPool._ptr, cpoolProps_ptr)
    return (cudaError_t(err), memPool)

@cython.embedsignature(True)
def cudaMemPoolDestroy(memPool):
    """ Destroys the specified memory pool.

    If any pointers obtained from this pool haven't been freed or the pool
    has free operations that haven't completed when
    :py:obj:`~.cudaMemPoolDestroy` is invoked, the function will return
    immediately and the resources associated with the pool will be released
    automatically once there are no more outstanding allocations.

    Destroying the current mempool of a device sets the default mempool of
    that device as the current mempool for that device.

    Parameters
    ----------
    memPool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        None

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    cuMemPoolDestroy, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaDeviceSetMemPool`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaDeviceGetMemPool`, :py:obj:`~.cudaMemPoolCreate`

    Notes
    -----
    A device's default memory pool cannot be destroyed.
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    err = ccudart.cudaMemPoolDestroy(cmemPool)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaMallocFromPoolAsync(size_t size, memPool, stream):
    """ Allocates memory from a specified pool with stream ordered semantics.

    Inserts an allocation operation into `hStream`. A pointer to the
    allocated memory is returned immediately in *dptr. The allocation must
    not be accessed until the the allocation operation completes. The
    allocation comes from the specified memory pool.

    Parameters
    ----------
    bytesize : size_t
        Number of bytes to allocate
    memPool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The pool to allocate from
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream establishing the stream ordering semantic

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorOutOfMemory`
    ptr : Any
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuMemAllocFromPoolAsync`, cudaMallocAsync (C++ API), :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaDeviceGetDefaultMemPool`, :py:obj:`~.cudaMemPoolCreate`, :py:obj:`~.cudaMemPoolSetAccess`, :py:obj:`~.cudaMemPoolSetAttribute`

    Notes
    -----
    During stream capture, this function results in the creation of an allocation node. In this case, the allocation is owned by the graph instead of the memory pool. The memory pool's properties are used to set the node's creation parameters.
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    cdef void_ptr ptr = 0
    err = ccudart.cudaMallocFromPoolAsync(<void**>&ptr, size, cmemPool, cstream)
    return (cudaError_t(err), ptr)

@cython.embedsignature(True)
def cudaMemPoolExportToShareableHandle(memPool, handleType not None : cudaMemAllocationHandleType, unsigned int flags):
    """ Exports a memory pool to the requested handle type.

    Given an IPC capable mempool, create an OS handle to share the pool
    with another process. A recipient process can convert the shareable
    handle into a mempool with
    :py:obj:`~.cudaMemPoolImportFromShareableHandle`. Individual pointers
    can then be shared with the :py:obj:`~.cudaMemPoolExportPointer` and
    :py:obj:`~.cudaMemPoolImportPointer` APIs. The implementation of what
    the shareable handle is and how it can be transferred is defined by the
    requested handle type.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        pool to export
    handleType : :py:obj:`~.cudaMemAllocationHandleType`
        the type of handle to create
    flags : unsigned int
        must be 0

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorOutOfMemory`
    handle_out : Any
        pointer to the location in which to store the requested handle

    See Also
    --------
    :py:obj:`~.cuMemPoolExportToShareableHandle`, :py:obj:`~.cudaMemPoolImportFromShareableHandle`, :py:obj:`~.cudaMemPoolExportPointer`, :py:obj:`~.cudaMemPoolImportPointer`

    Notes
    -----
    : To create an IPC capable mempool, create a mempool with a CUmemAllocationHandleType other than cudaMemHandleTypeNone.
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    cdef void_ptr shareableHandle = 0
    cdef void* cshareableHandle_ptr = <void*>shareableHandle
    cdef ccudart.cudaMemAllocationHandleType chandleType = handleType.value
    err = ccudart.cudaMemPoolExportToShareableHandle(cshareableHandle_ptr, cmemPool, chandleType, flags)
    return (cudaError_t(err), shareableHandle)

@cython.embedsignature(True)
def cudaMemPoolImportFromShareableHandle(shareableHandle, handleType not None : cudaMemAllocationHandleType, unsigned int flags):
    """ imports a memory pool from a shared handle.

    Specific allocations can be imported from the imported pool with
    :py:obj:`~.cudaMemPoolImportPointer`.

    Parameters
    ----------
    handle : Any
        OS handle of the pool to open
    handleType : :py:obj:`~.cudaMemAllocationHandleType`
        The type of handle being imported
    flags : unsigned int
        must be 0

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorOutOfMemory`
    pool_out : :py:obj:`~.cudaMemPool_t`
        Returned memory pool

    See Also
    --------
    :py:obj:`~.cuMemPoolImportFromShareableHandle`, :py:obj:`~.cudaMemPoolExportToShareableHandle`, :py:obj:`~.cudaMemPoolExportPointer`, :py:obj:`~.cudaMemPoolImportPointer`

    Notes
    -----
    Imported memory pools do not support creating new allocations. As such imported memory pools may not be used in :py:obj:`~.cudaDeviceSetMemPool` or :py:obj:`~.cudaMallocFromPoolAsync` calls.
    """
    cdef cudaMemPool_t memPool = cudaMemPool_t()
    cshareableHandle = utils.HelperInputVoidPtr(shareableHandle)
    cdef void* cshareableHandle_ptr = <void*><void_ptr>cshareableHandle.cptr
    cdef ccudart.cudaMemAllocationHandleType chandleType = handleType.value
    err = ccudart.cudaMemPoolImportFromShareableHandle(<ccudart.cudaMemPool_t*>memPool._ptr, cshareableHandle_ptr, chandleType, flags)
    return (cudaError_t(err), memPool)

@cython.embedsignature(True)
def cudaMemPoolExportPointer(ptr):
    """ Export data to share a memory pool allocation between processes.

    Constructs `shareData_out` for sharing a specific allocation from an
    already shared memory pool. The recipient process can import the
    allocation with the :py:obj:`~.cudaMemPoolImportPointer` api. The data
    is not a handle and may be shared through any IPC mechanism.

    Parameters
    ----------
    ptr : Any
        pointer to memory being exported

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorOutOfMemory`
    shareData_out : :py:obj:`~.cudaMemPoolPtrExportData`
        Returned export data

    See Also
    --------
    :py:obj:`~.cuMemPoolExportPointer`, :py:obj:`~.cudaMemPoolExportToShareableHandle`, :py:obj:`~.cudaMemPoolImportFromShareableHandle`, :py:obj:`~.cudaMemPoolImportPointer`
    """
    cdef cudaMemPoolPtrExportData exportData = cudaMemPoolPtrExportData()
    cptr = utils.HelperInputVoidPtr(ptr)
    cdef void* cptr_ptr = <void*><void_ptr>cptr.cptr
    err = ccudart.cudaMemPoolExportPointer(exportData._ptr, cptr_ptr)
    return (cudaError_t(err), exportData)

@cython.embedsignature(True)
def cudaMemPoolImportPointer(memPool, exportData : cudaMemPoolPtrExportData):
    """ Import a memory pool allocation from another process.

    Returns in `ptr_out` a pointer to the imported memory. The imported
    memory must not be accessed before the allocation operation completes
    in the exporting process. The imported memory must be freed from all
    importing processes before being freed in the exporting process. The
    pointer may be freed with cudaFree or cudaFreeAsync. If
    :py:obj:`~.cudaFreeAsync` is used, the free must be completed on the
    importing process before the free operation on the exporting process.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        pool from which to import
    shareData : :py:obj:`~.cudaMemPoolPtrExportData`
        data specifying the memory to import

    Returns
    -------
    cudaError_t
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    ptr_out : Any
        pointer to imported memory

    See Also
    --------
    :py:obj:`~.cuMemPoolImportPointer`, :py:obj:`~.cudaMemPoolExportToShareableHandle`, :py:obj:`~.cudaMemPoolImportFromShareableHandle`, :py:obj:`~.cudaMemPoolExportPointer`

    Notes
    -----
    The :py:obj:`~.cudaFreeAsync` api may be used in the exporting process before the :py:obj:`~.cudaFreeAsync` operation completes in its stream as long as the :py:obj:`~.cudaFreeAsync` in the exporting process specifies a stream with a stream dependency on the importing process's :py:obj:`~.cudaFreeAsync`.
    """
    cdef ccudart.cudaMemPool_t cmemPool
    if memPool is None:
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>0
    elif isinstance(memPool, (cudaMemPool_t,cuda.CUmemoryPool)):
        pmemPool = int(memPool)
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool
    else:
        pmemPool = int(cudaMemPool_t(memPool))
        cmemPool = <ccudart.cudaMemPool_t><void_ptr>pmemPool

    cdef void_ptr ptr = 0
    cdef ccudart.cudaMemPoolPtrExportData* cexportData_ptr = exportData._ptr if exportData != None else NULL
    err = ccudart.cudaMemPoolImportPointer(<void**>&ptr, cmemPool, cexportData_ptr)
    return (cudaError_t(err), ptr)

@cython.embedsignature(True)
def cudaPointerGetAttributes(ptr):
    """ Returns attributes about a specified pointer.

    Returns in `*attributes` the attributes of the pointer `ptr`. If
    pointer was not allocated in, mapped by or registered with context
    supporting unified addressing :py:obj:`~.cudaErrorInvalidValue` is
    returned.

    The :py:obj:`~.cudaPointerAttributes` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    In this structure, the individual fields mean

    - :py:obj:`~.cudaPointerAttributes.type` identifies type of memory. It
      can be :py:obj:`~.cudaMemoryTypeUnregistered` for unregistered host
      memory, :py:obj:`~.cudaMemoryTypeHost` for registered host memory,
      :py:obj:`~.cudaMemoryTypeDevice` for device memory or
      :py:obj:`~.cudaMemoryTypeManaged` for managed memory.

    - :py:obj:`~.device` is the device against which `ptr` was allocated.
      If `ptr` has memory type :py:obj:`~.cudaMemoryTypeDevice` then this
      identifies the device on which the memory referred to by `ptr`
      physically resides. If `ptr` has memory type
      :py:obj:`~.cudaMemoryTypeHost` then this identifies the device which
      was current when the allocation was made (and if that device is
      deinitialized then this allocation will vanish with that device's
      state).

    - :py:obj:`~.devicePointer` is the device pointer alias through which
      the memory referred to by `ptr` may be accessed on the current
      device. If the memory referred to by `ptr` cannot be accessed
      directly by the current device then this is NULL.

    - :py:obj:`~.hostPointer` is the host pointer alias through which the
      memory referred to by `ptr` may be accessed on the host. If the
      memory referred to by `ptr` cannot be accessed directly by the host
      then this is NULL.

    Parameters
    ----------
    ptr : Any
        Pointer to get attributes for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorInvalidValue`
    attributes : :py:obj:`~.cudaPointerAttributes`
        Attributes for the specified pointer

    See Also
    --------
    :py:obj:`~.cudaGetDeviceCount`, :py:obj:`~.cudaGetDevice`, :py:obj:`~.cudaSetDevice`, :py:obj:`~.cudaChooseDevice`, :py:obj:`~.cuPointerGetAttributes`

    Notes
    -----
    In CUDA 11.0 forward passing host pointer will return :py:obj:`~.cudaMemoryTypeUnregistered` in :py:obj:`~.cudaPointerAttributes.type` and call will return :py:obj:`~.cudaSuccess`.
    """
    cdef cudaPointerAttributes attributes = cudaPointerAttributes()
    cptr = utils.HelperInputVoidPtr(ptr)
    cdef void* cptr_ptr = <void*><void_ptr>cptr.cptr
    err = ccudart.cudaPointerGetAttributes(attributes._ptr, cptr_ptr)
    return (cudaError_t(err), attributes)

@cython.embedsignature(True)
def cudaDeviceCanAccessPeer(int device, int peerDevice):
    """ Queries if a device may directly access a peer device's memory.

    Returns in `*canAccessPeer` a value of 1 if device `device` is capable
    of directly accessing memory from `peerDevice` and 0 otherwise. If
    direct access of `peerDevice` from `device` is possible, then access
    may be enabled by calling :py:obj:`~.cudaDeviceEnablePeerAccess()`.

    Parameters
    ----------
    device : int
        Device from which allocations on `peerDevice` are to be directly
        accessed.
    peerDevice : int
        Device on which the allocations to be directly accessed by `device`
        reside.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`
    canAccessPeer : int
        Returned access capability

    See Also
    --------
    :py:obj:`~.cudaDeviceEnablePeerAccess`, :py:obj:`~.cudaDeviceDisablePeerAccess`, :py:obj:`~.cuDeviceCanAccessPeer`
    """
    cdef int canAccessPeer = 0
    err = ccudart.cudaDeviceCanAccessPeer(&canAccessPeer, device, peerDevice)
    return (cudaError_t(err), canAccessPeer)

@cython.embedsignature(True)
def cudaDeviceEnablePeerAccess(int peerDevice, unsigned int flags):
    """ Enables direct access to memory allocations on a peer device.

    On success, all allocations from `peerDevice` will immediately be
    accessible by the current device. They will remain accessible until
    access is explicitly disabled using
    :py:obj:`~.cudaDeviceDisablePeerAccess()` or either device is reset
    using :py:obj:`~.cudaDeviceReset()`.

    Note that access granted by this call is unidirectional and that in
    order to access memory on the current device from `peerDevice`, a
    separate symmetric call to :py:obj:`~.cudaDeviceEnablePeerAccess()` is
    required.

    Note that there are both device-wide and system-wide limitations per
    system configuration, as noted in the CUDA Programming Guide under the
    section "Peer-to-Peer Memory Access".

    Returns :py:obj:`~.cudaErrorInvalidDevice` if
    :py:obj:`~.cudaDeviceCanAccessPeer()` indicates that the current device
    cannot directly access memory from `peerDevice`.

    Returns :py:obj:`~.cudaErrorPeerAccessAlreadyEnabled` if direct access
    of `peerDevice` from the current device has already been enabled.

    Returns :py:obj:`~.cudaErrorInvalidValue` if `flags` is not 0.

    Parameters
    ----------
    peerDevice : int
        Peer device to enable direct access to from the current device
    flags : unsigned int
        Reserved for future use and must be set to 0

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`, :py:obj:`~.cudaErrorPeerAccessAlreadyEnabled`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaDeviceCanAccessPeer`, :py:obj:`~.cudaDeviceDisablePeerAccess`, :py:obj:`~.cuCtxEnablePeerAccess`
    """
    err = ccudart.cudaDeviceEnablePeerAccess(peerDevice, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceDisablePeerAccess(int peerDevice):
    """ Disables direct access to memory allocations on a peer device.

    Returns :py:obj:`~.cudaErrorPeerAccessNotEnabled` if direct access to
    memory on `peerDevice` has not yet been enabled from the current
    device.

    Parameters
    ----------
    peerDevice : int
        Peer device to disable direct access to

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorPeerAccessNotEnabled`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaDeviceCanAccessPeer`, :py:obj:`~.cudaDeviceEnablePeerAccess`, :py:obj:`~.cuCtxDisablePeerAccess`
    """
    err = ccudart.cudaDeviceDisablePeerAccess(peerDevice)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsUnregisterResource(resource):
    """ Unregisters a graphics resource for access by CUDA.

    Unregisters the graphics resource `resource` so it is not accessible by
    CUDA unless registered again.

    If `resource` is invalid then
    :py:obj:`~.cudaErrorInvalidResourceHandle` is returned.

    Parameters
    ----------
    resource : :py:obj:`~.cudaGraphicsResource_t`
        Resource to unregister

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaGraphicsD3D9RegisterResource`, :py:obj:`~.cudaGraphicsD3D10RegisterResource`, :py:obj:`~.cudaGraphicsD3D11RegisterResource`, :py:obj:`~.cudaGraphicsGLRegisterBuffer`, :py:obj:`~.cudaGraphicsGLRegisterImage`, :py:obj:`~.cuGraphicsUnregisterResource`
    """
    cdef ccudart.cudaGraphicsResource_t cresource
    if resource is None:
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>0
    elif isinstance(resource, (cudaGraphicsResource_t)):
        presource = int(resource)
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource
    else:
        presource = int(cudaGraphicsResource_t(resource))
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource

    err = ccudart.cudaGraphicsUnregisterResource(cresource)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsResourceSetMapFlags(resource, unsigned int flags):
    """ Set usage flags for mapping a graphics resource.

    Set `flags` for mapping the graphics resource `resource`.

    Changes to `flags` will take effect the next time `resource` is mapped.
    The `flags` argument may be any of the following:

    - :py:obj:`~.cudaGraphicsMapFlagsNone`: Specifies no hints about how
      `resource` will be used. It is therefore assumed that CUDA may read
      from or write to `resource`.

    - :py:obj:`~.cudaGraphicsMapFlagsReadOnly`: Specifies that CUDA will
      not write to `resource`.

    - :py:obj:`~.cudaGraphicsMapFlagsWriteDiscard`: Specifies CUDA will not
      read from `resource` and will write over the entire contents of
      `resource`, so none of the data previously stored in `resource` will
      be preserved.

    If `resource` is presently mapped for access by CUDA then
    :py:obj:`~.cudaErrorUnknown` is returned. If `flags` is not one of the
    above values then :py:obj:`~.cudaErrorInvalidValue` is returned.

    Parameters
    ----------
    resource : :py:obj:`~.cudaGraphicsResource_t`
        Registered resource to set flags for
    flags : unsigned int
        Parameters for resource mapping

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`,

    See Also
    --------
    :py:obj:`~.cudaGraphicsMapResources`, :py:obj:`~.cuGraphicsResourceSetMapFlags`
    """
    cdef ccudart.cudaGraphicsResource_t cresource
    if resource is None:
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>0
    elif isinstance(resource, (cudaGraphicsResource_t)):
        presource = int(resource)
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource
    else:
        presource = int(cudaGraphicsResource_t(resource))
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource

    err = ccudart.cudaGraphicsResourceSetMapFlags(cresource, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsMapResources(int count, resources, stream):
    """ Map graphics resources for access by CUDA.

    Maps the `count` graphics resources in `resources` for access by CUDA.

    The resources in `resources` may be accessed by CUDA until they are
    unmapped. The graphics API from which `resources` were registered
    should not access any resources while they are mapped by CUDA. If an
    application does so, the results are undefined.

    This function provides the synchronization guarantee that any graphics
    calls issued before :py:obj:`~.cudaGraphicsMapResources()` will
    complete before any subsequent CUDA work issued in `stream` begins.

    If `resources` contains any duplicate entries then
    :py:obj:`~.cudaErrorInvalidResourceHandle` is returned. If any of
    `resources` are presently mapped for access by CUDA then
    :py:obj:`~.cudaErrorUnknown` is returned.

    Parameters
    ----------
    count : int
        Number of resources to map
    resources : :py:obj:`~.cudaGraphicsResource_t`
        Resources to map for CUDA
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream for synchronization

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaGraphicsResourceGetMappedPointer`, :py:obj:`~.cudaGraphicsSubResourceGetMappedArray`, :py:obj:`~.cudaGraphicsUnmapResources`, :py:obj:`~.cuGraphicsMapResources`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaGraphicsResource_t *cresources
    if resources is None:
        cresources = <ccudart.cudaGraphicsResource_t*><void_ptr>NULL
    elif isinstance(resources, (cudaGraphicsResource_t)):
        presources = resources.getPtr()
        cresources = <ccudart.cudaGraphicsResource_t*><void_ptr>presources
    elif isinstance(resources, (int)):
        cresources = <ccudart.cudaGraphicsResource_t*><void_ptr>resources
    else:
        raise TypeError("Argument 'resources' is not instance of type (expected <class 'int, cudart.cudaGraphicsResource_t'>, found " + str(type(resources)))

    err = ccudart.cudaGraphicsMapResources(count, cresources, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsUnmapResources(int count, resources, stream):
    """ Unmap graphics resources.

    Unmaps the `count` graphics resources in `resources`.

    Once unmapped, the resources in `resources` may not be accessed by CUDA
    until they are mapped again.

    This function provides the synchronization guarantee that any CUDA work
    issued in `stream` before :py:obj:`~.cudaGraphicsUnmapResources()` will
    complete before any subsequently issued graphics work begins.

    If `resources` contains any duplicate entries then
    :py:obj:`~.cudaErrorInvalidResourceHandle` is returned. If any of
    `resources` are not presently mapped for access by CUDA then
    :py:obj:`~.cudaErrorUnknown` is returned.

    Parameters
    ----------
    count : int
        Number of resources to unmap
    resources : :py:obj:`~.cudaGraphicsResource_t`
        Resources to unmap
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream for synchronization

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaGraphicsMapResources`, :py:obj:`~.cuGraphicsUnmapResources`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaGraphicsResource_t *cresources
    if resources is None:
        cresources = <ccudart.cudaGraphicsResource_t*><void_ptr>NULL
    elif isinstance(resources, (cudaGraphicsResource_t)):
        presources = resources.getPtr()
        cresources = <ccudart.cudaGraphicsResource_t*><void_ptr>presources
    elif isinstance(resources, (int)):
        cresources = <ccudart.cudaGraphicsResource_t*><void_ptr>resources
    else:
        raise TypeError("Argument 'resources' is not instance of type (expected <class 'int, cudart.cudaGraphicsResource_t'>, found " + str(type(resources)))

    err = ccudart.cudaGraphicsUnmapResources(count, cresources, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsResourceGetMappedPointer(resource):
    """ Get an device pointer through which to access a mapped graphics resource.

    Returns in `*devPtr` a pointer through which the mapped graphics
    resource `resource` may be accessed. Returns in `*size` the size of the
    memory in bytes which may be accessed from that pointer. The value set
    in `devPtr` may change every time that `resource` is mapped.

    If `resource` is not a buffer then it cannot be accessed via a pointer
    and :py:obj:`~.cudaErrorUnknown` is returned. If `resource` is not
    mapped then :py:obj:`~.cudaErrorUnknown` is returned.

    Parameters
    ----------
    resource : :py:obj:`~.cudaGraphicsResource_t`
        None

    Returns
    -------
    cudaError_t

    devPtr : Any
        None
    size : int
        None
    """
    cdef ccudart.cudaGraphicsResource_t cresource
    if resource is None:
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>0
    elif isinstance(resource, (cudaGraphicsResource_t)):
        presource = int(resource)
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource
    else:
        presource = int(cudaGraphicsResource_t(resource))
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource

    cdef void_ptr devPtr = 0
    cdef size_t size = 0
    err = ccudart.cudaGraphicsResourceGetMappedPointer(<void**>&devPtr, &size, cresource)
    return (cudaError_t(err), devPtr, size)

@cython.embedsignature(True)
def cudaGraphicsSubResourceGetMappedArray(resource, unsigned int arrayIndex, unsigned int mipLevel):
    """ Get an array through which to access a subresource of a mapped graphics resource.

    Returns in `*array` an array through which the subresource of the
    mapped graphics resource `resource` which corresponds to array index
    `arrayIndex` and mipmap level `mipLevel` may be accessed. The value set
    in `array` may change every time that `resource` is mapped.

    If `resource` is not a texture then it cannot be accessed via an array
    and :py:obj:`~.cudaErrorUnknown` is returned. If `arrayIndex` is not a
    valid array index for `resource` then :py:obj:`~.cudaErrorInvalidValue`
    is returned. If `mipLevel` is not a valid mipmap level for `resource`
    then :py:obj:`~.cudaErrorInvalidValue` is returned. If `resource` is
    not mapped then :py:obj:`~.cudaErrorUnknown` is returned.

    Parameters
    ----------
    resource : :py:obj:`~.cudaGraphicsResource_t`
        Mapped resource to access
    arrayIndex : unsigned int
        Array index for array textures or cubemap face index as defined by
        :py:obj:`~.cudaGraphicsCubeFace` for cubemap textures for the
        subresource to access
    mipLevel : unsigned int
        Mipmap level for the subresource to access

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`
    array : :py:obj:`~.cudaArray_t`
        Returned array through which a subresource of `resource` may be
        accessed

    See Also
    --------
    :py:obj:`~.cudaGraphicsResourceGetMappedPointer`, :py:obj:`~.cuGraphicsSubResourceGetMappedArray`
    """
    cdef ccudart.cudaGraphicsResource_t cresource
    if resource is None:
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>0
    elif isinstance(resource, (cudaGraphicsResource_t)):
        presource = int(resource)
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource
    else:
        presource = int(cudaGraphicsResource_t(resource))
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource

    cdef cudaArray_t array = cudaArray_t()
    err = ccudart.cudaGraphicsSubResourceGetMappedArray(array._ptr, cresource, arrayIndex, mipLevel)
    return (cudaError_t(err), array)

@cython.embedsignature(True)
def cudaGraphicsResourceGetMappedMipmappedArray(resource):
    """ Get a mipmapped array through which to access a mapped graphics resource.

    Returns in `*mipmappedArray` a mipmapped array through which the mapped
    graphics resource `resource` may be accessed. The value set in
    `mipmappedArray` may change every time that `resource` is mapped.

    If `resource` is not a texture then it cannot be accessed via an array
    and :py:obj:`~.cudaErrorUnknown` is returned. If `resource` is not
    mapped then :py:obj:`~.cudaErrorUnknown` is returned.

    Parameters
    ----------
    resource : :py:obj:`~.cudaGraphicsResource_t`
        Mapped resource to access

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorUnknown`
    mipmappedArray : :py:obj:`~.cudaMipmappedArray_t`
        Returned mipmapped array through which `resource` may be accessed

    See Also
    --------
    :py:obj:`~.cudaGraphicsResourceGetMappedPointer`, :py:obj:`~.cuGraphicsResourceGetMappedMipmappedArray`
    """
    cdef ccudart.cudaGraphicsResource_t cresource
    if resource is None:
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>0
    elif isinstance(resource, (cudaGraphicsResource_t)):
        presource = int(resource)
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource
    else:
        presource = int(cudaGraphicsResource_t(resource))
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource

    cdef cudaMipmappedArray_t mipmappedArray = cudaMipmappedArray_t()
    err = ccudart.cudaGraphicsResourceGetMappedMipmappedArray(mipmappedArray._ptr, cresource)
    return (cudaError_t(err), mipmappedArray)

@cython.embedsignature(True)
def cudaGetChannelDesc(array):
    """ Get the channel descriptor of an array.

    Returns in `*desc` the channel descriptor of the CUDA array `array`.

    Parameters
    ----------
    array : :py:obj:`~.cudaArray_const_t`
        Memory array on device

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    desc : :py:obj:`~.cudaChannelFormatDesc`
        Channel format

    See Also
    --------
    :py:obj:`~.cudaCreateChannelDesc (C API)`, :py:obj:`~.cudaCreateTextureObject`, :py:obj:`~.cudaCreateSurfaceObject`
    """
    cdef ccudart.cudaArray_const_t carray
    if array is None:
        carray = <ccudart.cudaArray_const_t><void_ptr>0
    elif isinstance(array, (cudaArray_const_t)):
        parray = int(array)
        carray = <ccudart.cudaArray_const_t><void_ptr>parray
    else:
        parray = int(cudaArray_const_t(array))
        carray = <ccudart.cudaArray_const_t><void_ptr>parray

    cdef cudaChannelFormatDesc desc = cudaChannelFormatDesc()
    with nogil:
        err = ccudart.cudaGetChannelDesc(desc._ptr, carray)
    return (cudaError_t(err), desc)

@cython.embedsignature(True)
def cudaCreateChannelDesc(int x, int y, int z, int w, f not None : cudaChannelFormatKind):
    """ Returns a channel descriptor using the specified format.

    Returns a channel descriptor with format `f` and number of bits of each
    component `x`, `y`, `z`, and `w`. The :py:obj:`~.cudaChannelFormatDesc`
    is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.cudaChannelFormatKind` is one of
    :py:obj:`~.cudaChannelFormatKindSigned`,
    :py:obj:`~.cudaChannelFormatKindUnsigned`, or
    :py:obj:`~.cudaChannelFormatKindFloat`.

    Parameters
    ----------
    x : int
        X component
    y : int
        Y component
    z : int
        Z component
    w : int
        W component
    f : :py:obj:`~.cudaChannelFormatKind`
        Channel format

    Returns
    -------
    cudaError_t.cudaSuccess
        cudaError_t.cudaSuccess
    :py:obj:`~.cudaChannelFormatDesc`
        Channel descriptor with format `f`

    See Also
    --------
    cudaCreateChannelDesc (C++ API), :py:obj:`~.cudaGetChannelDesc`, :py:obj:`~.cudaCreateTextureObject`, :py:obj:`~.cudaCreateSurfaceObject`
    """
    cdef ccudart.cudaChannelFormatKind cf = f.value
    cdef ccudart.cudaChannelFormatDesc err
    err = ccudart.cudaCreateChannelDesc(x, y, z, w, cf)
    cdef cudaChannelFormatDesc wrapper = cudaChannelFormatDesc()
    wrapper._ptr[0] = err
    return (cudaError_t.cudaSuccess, wrapper)

@cython.embedsignature(True)
def cudaCreateTextureObject(pResDesc : cudaResourceDesc, pTexDesc : cudaTextureDesc, pResViewDesc : cudaResourceViewDesc):
    """ Creates a texture object.

    Creates a texture object and returns it in `pTexObject`. `pResDesc`
    describes the data to texture from. `pTexDesc` describes how the data
    should be sampled. `pResViewDesc` is an optional argument that
    specifies an alternate format for the data described by `pResDesc`, and
    also describes the subresource region to restrict access to when
    texturing. `pResViewDesc` can only be specified if the type of resource
    is a CUDA array or a CUDA mipmapped array.

    Texture objects are only supported on devices of compute capability 3.0
    or higher. Additionally, a texture object is an opaque value, and, as
    such, should only be accessed through CUDA API calls.

    The :py:obj:`~.cudaResourceDesc` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.cudaResourceDesc.resType` specifies the type of resource
      to texture from. CUresourceType is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.cudaResourceDesc.resType` is set to
    :py:obj:`~.cudaResourceTypeArray`,
    :py:obj:`~.cudaResourceDesc`::res::array::array must be set to a valid
    CUDA array handle.

    If :py:obj:`~.cudaResourceDesc.resType` is set to
    :py:obj:`~.cudaResourceTypeMipmappedArray`,
    :py:obj:`~.cudaResourceDesc`::res::mipmap::mipmap must be set to a
    valid CUDA mipmapped array handle and
    :py:obj:`~.cudaTextureDesc.normalizedCoords` must be set to true.

    If :py:obj:`~.cudaResourceDesc.resType` is set to
    :py:obj:`~.cudaResourceTypeLinear`,
    :py:obj:`~.cudaResourceDesc`::res::linear::devPtr must be set to a
    valid device pointer, that is aligned to
    :py:obj:`~.cudaDeviceProp.textureAlignment`.
    :py:obj:`~.cudaResourceDesc`::res::linear::desc describes the format
    and the number of components per array element.
    :py:obj:`~.cudaResourceDesc`::res::linear::sizeInBytes specifies the
    size of the array in bytes. The total number of elements in the linear
    address range cannot exceed
    :py:obj:`~.cudaDeviceProp.maxTexture1DLinear`. The number of elements
    is computed as (sizeInBytes / sizeof(desc)).

    If :py:obj:`~.cudaResourceDesc.resType` is set to
    :py:obj:`~.cudaResourceTypePitch2D`,
    :py:obj:`~.cudaResourceDesc`::res::pitch2D::devPtr must be set to a
    valid device pointer, that is aligned to
    :py:obj:`~.cudaDeviceProp.textureAlignment`.
    :py:obj:`~.cudaResourceDesc`::res::pitch2D::desc describes the format
    and the number of components per array element.
    :py:obj:`~.cudaResourceDesc`::res::pitch2D::width and
    :py:obj:`~.cudaResourceDesc`::res::pitch2D::height specify the width
    and height of the array in elements, and cannot exceed
    :py:obj:`~.cudaDeviceProp.maxTexture2DLinear`[0] and
    :py:obj:`~.cudaDeviceProp.maxTexture2DLinear`[1] respectively.
    :py:obj:`~.cudaResourceDesc`::res::pitch2D::pitchInBytes specifies the
    pitch between two rows in bytes and has to be aligned to
    :py:obj:`~.cudaDeviceProp.texturePitchAlignment`. Pitch cannot exceed
    :py:obj:`~.cudaDeviceProp.maxTexture2DLinear`[2].

    The :py:obj:`~.cudaTextureDesc` struct is defined as

    **View CUDA Toolkit Documentation for a C++ code example**

    where

    - :py:obj:`~.cudaTextureDesc.addressMode` specifies the addressing mode
      for each dimension of the texture data.
      :py:obj:`~.cudaTextureAddressMode` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - This is ignored if :py:obj:`~.cudaResourceDesc.resType` is
      :py:obj:`~.cudaResourceTypeLinear`. Also, if
      :py:obj:`~.cudaTextureDesc.normalizedCoords` is set to zero,
      :py:obj:`~.cudaAddressModeWrap` and :py:obj:`~.cudaAddressModeMirror`
      won't be supported and will be switched to
      :py:obj:`~.cudaAddressModeClamp`.

    - :py:obj:`~.cudaTextureDesc.filterMode` specifies the filtering mode
      to be used when fetching from the texture.
      :py:obj:`~.cudaTextureFilterMode` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - This is ignored if :py:obj:`~.cudaResourceDesc.resType` is
      :py:obj:`~.cudaResourceTypeLinear`.

    - :py:obj:`~.cudaTextureDesc.readMode` specifies whether integer data
      should be converted to floating point or not.
      :py:obj:`~.cudaTextureReadMode` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - Note that this applies only to 8-bit and 16-bit integer formats.
      32-bit integer format would not be promoted, regardless of whether or
      not this :py:obj:`~.cudaTextureDesc.readMode` is set
      :py:obj:`~.cudaReadModeNormalizedFloat` is specified.

    - :py:obj:`~.cudaTextureDesc.sRGB` specifies whether sRGB to linear
      conversion should be performed during texture fetch.

    - :py:obj:`~.cudaTextureDesc.borderColor` specifies the float values of
      color. where: :py:obj:`~.cudaTextureDesc.borderColor`[0] contains
      value of 'R', :py:obj:`~.cudaTextureDesc.borderColor`[1] contains
      value of 'G', :py:obj:`~.cudaTextureDesc.borderColor`[2] contains
      value of 'B', :py:obj:`~.cudaTextureDesc.borderColor`[3] contains
      value of 'A' Note that application using integer border color values
      will need to <reinterpret_cast> these values to float. The values are
      set only when the addressing mode specified by
      :py:obj:`~.cudaTextureDesc.addressMode` is cudaAddressModeBorder.

    - :py:obj:`~.cudaTextureDesc.normalizedCoords` specifies whether the
      texture coordinates will be normalized or not.

    - :py:obj:`~.cudaTextureDesc.maxAnisotropy` specifies the maximum
      anistropy ratio to be used when doing anisotropic filtering. This
      value will be clamped to the range [1,16].

    - :py:obj:`~.cudaTextureDesc.mipmapFilterMode` specifies the filter
      mode when the calculated mipmap level lies between two defined mipmap
      levels.

    - :py:obj:`~.cudaTextureDesc.mipmapLevelBias` specifies the offset to
      be applied to the calculated mipmap level.

    - :py:obj:`~.cudaTextureDesc.minMipmapLevelClamp` specifies the lower
      end of the mipmap level range to clamp access to.

    - :py:obj:`~.cudaTextureDesc.maxMipmapLevelClamp` specifies the upper
      end of the mipmap level range to clamp access to.

    - :py:obj:`~.cudaTextureDesc.disableTrilinearOptimization` specifies
      whether the trilinear filtering optimizations will be disabled.

    - :py:obj:`~.cudaTextureDesc.seamlessCubemap` specifies whether
      seamless cube map filtering is enabled. This flag can only be
      specified if the underlying resource is a CUDA array or a CUDA
      mipmapped array that was created with the flag
      :py:obj:`~.cudaArrayCubemap`. When seamless cube map filtering is
      enabled, texture address modes specified by
      :py:obj:`~.cudaTextureDesc.addressMode` are ignored. Instead, if the
      :py:obj:`~.cudaTextureDesc.filterMode` is set to
      :py:obj:`~.cudaFilterModePoint` the address mode
      :py:obj:`~.cudaAddressModeClamp` will be applied for all dimensions.
      If the :py:obj:`~.cudaTextureDesc.filterMode` is set to
      :py:obj:`~.cudaFilterModeLinear` seamless cube map filtering will be
      performed when sampling along the cube face borders.

    The :py:obj:`~.cudaResourceViewDesc` struct is defined as

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.cudaResourceViewDesc.format` specifies how the data
      contained in the CUDA array or CUDA mipmapped array should be
      interpreted. Note that this can incur a change in size of the texture
      data. If the resource view format is a block compressed format, then
      the underlying CUDA array or CUDA mipmapped array has to have a
      32-bit unsigned integer format with 2 or 4 channels, depending on the
      block compressed format. For ex., BC1 and BC4 require the underlying
      CUDA array to have a 32-bit unsigned int with 2 channels. The other
      BC formats require the underlying resource to have the same 32-bit
      unsigned int format but with 4 channels.

    - :py:obj:`~.cudaResourceViewDesc.width` specifies the new width of the
      texture data. If the resource view format is a block compressed
      format, this value has to be 4 times the original width of the
      resource. For non block compressed formats, this value has to be
      equal to that of the original resource.

    - :py:obj:`~.cudaResourceViewDesc.height` specifies the new height of
      the texture data. If the resource view format is a block compressed
      format, this value has to be 4 times the original height of the
      resource. For non block compressed formats, this value has to be
      equal to that of the original resource.

    - :py:obj:`~.cudaResourceViewDesc.depth` specifies the new depth of the
      texture data. This value has to be equal to that of the original
      resource.

    - :py:obj:`~.cudaResourceViewDesc.firstMipmapLevel` specifies the most
      detailed mipmap level. This will be the new mipmap level zero. For
      non-mipmapped resources, this value has to be
      zero.:py:obj:`~.cudaTextureDesc.minMipmapLevelClamp` and
      :py:obj:`~.cudaTextureDesc.maxMipmapLevelClamp` will be relative to
      this value. For ex., if the firstMipmapLevel is set to 2, and a
      minMipmapLevelClamp of 1.2 is specified, then the actual minimum
      mipmap level clamp will be 3.2.

    - :py:obj:`~.cudaResourceViewDesc.lastMipmapLevel` specifies the least
      detailed mipmap level. For non-mipmapped resources, this value has to
      be zero.

    - :py:obj:`~.cudaResourceViewDesc.firstLayer` specifies the first layer
      index for layered textures. This will be the new layer zero. For non-
      layered resources, this value has to be zero.

    - :py:obj:`~.cudaResourceViewDesc.lastLayer` specifies the last layer
      index for layered textures. For non-layered resources, this value has
      to be zero.

    Parameters
    ----------
    pResDesc : :py:obj:`~.cudaResourceDesc`
        Resource descriptor
    pTexDesc : :py:obj:`~.cudaTextureDesc`
        Texture descriptor
    pResViewDesc : :py:obj:`~.cudaResourceViewDesc`
        Resource view descriptor

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pTexObject : :py:obj:`~.cudaTextureObject_t`
        Texture object to create

    See Also
    --------
    :py:obj:`~.cudaDestroyTextureObject`, :py:obj:`~.cuTexObjectCreate`
    """
    cdef cudaTextureObject_t pTexObject = cudaTextureObject_t()
    cdef ccudart.cudaResourceDesc* cpResDesc_ptr = pResDesc._ptr if pResDesc != None else NULL
    cdef ccudart.cudaTextureDesc* cpTexDesc_ptr = pTexDesc._ptr if pTexDesc != None else NULL
    cdef ccudart.cudaResourceViewDesc* cpResViewDesc_ptr = pResViewDesc._ptr if pResViewDesc != None else NULL
    err = ccudart.cudaCreateTextureObject(pTexObject._ptr, cpResDesc_ptr, cpTexDesc_ptr, cpResViewDesc_ptr)
    return (cudaError_t(err), pTexObject)

@cython.embedsignature(True)
def cudaDestroyTextureObject(texObject):
    """ Destroys a texture object.

    Destroys the texture object specified by `texObject`.

    Parameters
    ----------
    texObject : :py:obj:`~.cudaTextureObject_t`
        Texture object to destroy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaCreateTextureObject`, :py:obj:`~.cuTexObjectDestroy`
    """
    cdef ccudart.cudaTextureObject_t ctexObject
    if texObject is None:
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>0
    elif isinstance(texObject, (cudaTextureObject_t)):
        ptexObject = int(texObject)
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject
    else:
        ptexObject = int(cudaTextureObject_t(texObject))
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject

    with nogil:
        err = ccudart.cudaDestroyTextureObject(ctexObject)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGetTextureObjectResourceDesc(texObject):
    """ Returns a texture object's resource descriptor.

    Returns the resource descriptor for the texture object specified by
    `texObject`.

    Parameters
    ----------
    texObject : :py:obj:`~.cudaTextureObject_t`
        Texture object

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pResDesc : :py:obj:`~.cudaResourceDesc`
        Resource descriptor

    See Also
    --------
    :py:obj:`~.cudaCreateTextureObject`, :py:obj:`~.cuTexObjectGetResourceDesc`
    """
    cdef ccudart.cudaTextureObject_t ctexObject
    if texObject is None:
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>0
    elif isinstance(texObject, (cudaTextureObject_t)):
        ptexObject = int(texObject)
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject
    else:
        ptexObject = int(cudaTextureObject_t(texObject))
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject

    cdef cudaResourceDesc pResDesc = cudaResourceDesc()
    with nogil:
        err = ccudart.cudaGetTextureObjectResourceDesc(pResDesc._ptr, ctexObject)
    return (cudaError_t(err), pResDesc)

@cython.embedsignature(True)
def cudaGetTextureObjectTextureDesc(texObject):
    """ Returns a texture object's texture descriptor.

    Returns the texture descriptor for the texture object specified by
    `texObject`.

    Parameters
    ----------
    texObject : :py:obj:`~.cudaTextureObject_t`
        Texture object

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pTexDesc : :py:obj:`~.cudaTextureDesc`
        Texture descriptor

    See Also
    --------
    :py:obj:`~.cudaCreateTextureObject`, :py:obj:`~.cuTexObjectGetTextureDesc`
    """
    cdef ccudart.cudaTextureObject_t ctexObject
    if texObject is None:
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>0
    elif isinstance(texObject, (cudaTextureObject_t)):
        ptexObject = int(texObject)
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject
    else:
        ptexObject = int(cudaTextureObject_t(texObject))
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject

    cdef cudaTextureDesc pTexDesc = cudaTextureDesc()
    with nogil:
        err = ccudart.cudaGetTextureObjectTextureDesc(pTexDesc._ptr, ctexObject)
    return (cudaError_t(err), pTexDesc)

@cython.embedsignature(True)
def cudaGetTextureObjectResourceViewDesc(texObject):
    """ Returns a texture object's resource view descriptor.

    Returns the resource view descriptor for the texture object specified
    by `texObject`. If no resource view was specified,
    :py:obj:`~.cudaErrorInvalidValue` is returned.

    Parameters
    ----------
    texObject : :py:obj:`~.cudaTextureObject_t`
        Texture object

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pResViewDesc : :py:obj:`~.cudaResourceViewDesc`
        Resource view descriptor

    See Also
    --------
    :py:obj:`~.cudaCreateTextureObject`, :py:obj:`~.cuTexObjectGetResourceViewDesc`
    """
    cdef ccudart.cudaTextureObject_t ctexObject
    if texObject is None:
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>0
    elif isinstance(texObject, (cudaTextureObject_t)):
        ptexObject = int(texObject)
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject
    else:
        ptexObject = int(cudaTextureObject_t(texObject))
        ctexObject = <ccudart.cudaTextureObject_t><void_ptr>ptexObject

    cdef cudaResourceViewDesc pResViewDesc = cudaResourceViewDesc()
    err = ccudart.cudaGetTextureObjectResourceViewDesc(pResViewDesc._ptr, ctexObject)
    return (cudaError_t(err), pResViewDesc)

@cython.embedsignature(True)
def cudaCreateSurfaceObject(pResDesc : cudaResourceDesc):
    """ Creates a surface object.

    Creates a surface object and returns it in `pSurfObject`. `pResDesc`
    describes the data to perform surface load/stores on.
    :py:obj:`~.cudaResourceDesc.resType` must be
    :py:obj:`~.cudaResourceTypeArray` and
    :py:obj:`~.cudaResourceDesc`::res::array::array must be set to a valid
    CUDA array handle.

    Surface objects are only supported on devices of compute capability 3.0
    or higher. Additionally, a surface object is an opaque value, and, as
    such, should only be accessed through CUDA API calls.

    Parameters
    ----------
    pResDesc : :py:obj:`~.cudaResourceDesc`
        Resource descriptor

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidChannelDescriptor`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    pSurfObject : :py:obj:`~.cudaSurfaceObject_t`
        Surface object to create

    See Also
    --------
    :py:obj:`~.cudaDestroySurfaceObject`, :py:obj:`~.cuSurfObjectCreate`
    """
    cdef cudaSurfaceObject_t pSurfObject = cudaSurfaceObject_t()
    cdef ccudart.cudaResourceDesc* cpResDesc_ptr = pResDesc._ptr if pResDesc != None else NULL
    with nogil:
        err = ccudart.cudaCreateSurfaceObject(pSurfObject._ptr, cpResDesc_ptr)
    return (cudaError_t(err), pSurfObject)

@cython.embedsignature(True)
def cudaDestroySurfaceObject(surfObject):
    """ Destroys a surface object.

    Destroys the surface object specified by `surfObject`.

    Parameters
    ----------
    surfObject : :py:obj:`~.cudaSurfaceObject_t`
        Surface object to destroy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaCreateSurfaceObject`, :py:obj:`~.cuSurfObjectDestroy`
    """
    cdef ccudart.cudaSurfaceObject_t csurfObject
    if surfObject is None:
        csurfObject = <ccudart.cudaSurfaceObject_t><void_ptr>0
    elif isinstance(surfObject, (cudaSurfaceObject_t)):
        psurfObject = int(surfObject)
        csurfObject = <ccudart.cudaSurfaceObject_t><void_ptr>psurfObject
    else:
        psurfObject = int(cudaSurfaceObject_t(surfObject))
        csurfObject = <ccudart.cudaSurfaceObject_t><void_ptr>psurfObject

    with nogil:
        err = ccudart.cudaDestroySurfaceObject(csurfObject)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGetSurfaceObjectResourceDesc(surfObject):
    """ Returns a surface object's resource descriptor Returns the resource descriptor for the surface object specified by `surfObject`.

    Parameters
    ----------
    surfObject : :py:obj:`~.cudaSurfaceObject_t`
        Surface object

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pResDesc : :py:obj:`~.cudaResourceDesc`
        Resource descriptor

    See Also
    --------
    :py:obj:`~.cudaCreateSurfaceObject`, :py:obj:`~.cuSurfObjectGetResourceDesc`
    """
    cdef ccudart.cudaSurfaceObject_t csurfObject
    if surfObject is None:
        csurfObject = <ccudart.cudaSurfaceObject_t><void_ptr>0
    elif isinstance(surfObject, (cudaSurfaceObject_t)):
        psurfObject = int(surfObject)
        csurfObject = <ccudart.cudaSurfaceObject_t><void_ptr>psurfObject
    else:
        psurfObject = int(cudaSurfaceObject_t(surfObject))
        csurfObject = <ccudart.cudaSurfaceObject_t><void_ptr>psurfObject

    cdef cudaResourceDesc pResDesc = cudaResourceDesc()
    err = ccudart.cudaGetSurfaceObjectResourceDesc(pResDesc._ptr, csurfObject)
    return (cudaError_t(err), pResDesc)

@cython.embedsignature(True)
def cudaDriverGetVersion():
    """ Returns the latest version of CUDA supported by the driver.

    Returns in `*driverVersion` the latest version of CUDA supported by the
    driver. The version is returned as (1000 * major + 10 * minor). For
    example, CUDA 9.2 would be represented by 9020. If no driver is
    installed, then 0 is returned as the driver version.

    This function automatically returns :py:obj:`~.cudaErrorInvalidValue`
    if `driverVersion` is NULL.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    driverVersion : int
        Returns the CUDA driver version.

    See Also
    --------
    :py:obj:`~.cudaRuntimeGetVersion`, :py:obj:`~.cuDriverGetVersion`
    """
    cdef int driverVersion = 0
    err = ccudart.cudaDriverGetVersion(&driverVersion)
    return (cudaError_t(err), driverVersion)

@cython.embedsignature(True)
def cudaRuntimeGetVersion():
    """ Returns the CUDA Runtime version.

    Returns in `*runtimeVersion` the version number of the current CUDA
    Runtime instance. The version is returned as (1000 * major + 10 *
    minor). For example, CUDA 9.2 would be represented by 9020.

    This function automatically returns :py:obj:`~.cudaErrorInvalidValue`
    if the `runtimeVersion` argument is NULL.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    runtimeVersion : int
        Returns the CUDA Runtime version.

    See Also
    --------
    :py:obj:`~.cudaDriverGetVersion`, :py:obj:`~.cuDriverGetVersion`
    """
    cdef int runtimeVersion = 0
    err = ccudart.cudaRuntimeGetVersion(&runtimeVersion)
    return (cudaError_t(err), runtimeVersion)

@cython.embedsignature(True)
def cudaGraphCreate(unsigned int flags):
    """ Creates a graph.

    Creates an empty graph, which is returned via `pGraph`.

    Parameters
    ----------
    flags : unsigned int
        Graph creation flags, must be 0

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    pGraph : :py:obj:`~.cudaGraph_t`
        Returns newly created graph

    See Also
    --------
    :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`, :py:obj:`~.cudaGraphInstantiate`, :py:obj:`~.cudaGraphDestroy`, :py:obj:`~.cudaGraphGetNodes`, :py:obj:`~.cudaGraphGetRootNodes`, :py:obj:`~.cudaGraphGetEdges`, :py:obj:`~.cudaGraphClone`
    """
    cdef cudaGraph_t pGraph = cudaGraph_t()
    err = ccudart.cudaGraphCreate(<ccudart.cudaGraph_t*>pGraph._ptr, flags)
    return (cudaError_t(err), pGraph)

@cython.embedsignature(True)
def cudaGraphAddKernelNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, pNodeParams : cudaKernelNodeParams):
    """ Creates a kernel execution node and adds it to a graph.

    Creates a new kernel execution node and adds it to `graph` with
    `numDependencies` dependencies specified via `pDependencies` and
    arguments specified in `pNodeParams`. It is possible for
    `numDependencies` to be 0, in which case the node will be placed at the
    root of the graph. `pDependencies` may not have any duplicate entries.
    A handle to the new node will be returned in `pGraphNode`.

    The :py:obj:`~.cudaKernelNodeParams` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    When the graph is launched, the node will invoke kernel `func` on a
    (`gridDim.x` x `gridDim.y` x `gridDim.z`) grid of blocks. Each block
    contains (`blockDim.x` x `blockDim.y` x `blockDim.z`) threads.

    `sharedMem` sets the amount of dynamic shared memory that will be
    available to each thread block.

    Kernel parameters to `func` can be specified in one of two ways:

    1) Kernel parameters can be specified via `kernelParams`. If the kernel
    has N parameters, then `kernelParams` needs to be an array of N
    pointers. Each pointer, from `kernelParams`[0] to `kernelParams`[N-1],
    points to the region of memory from which the actual parameter will be
    copied. The number of kernel parameters and their offsets and sizes do
    not need to be specified as that information is retrieved directly from
    the kernel's image.

    2) Kernel parameters can also be packaged by the application into a
    single buffer that is passed in via `extra`. This places the burden on
    the application of knowing each kernel parameter's size and
    alignment/padding within the buffer. The `extra` parameter exists to
    allow this function to take additional less commonly used arguments.
    `extra` specifies a list of names of extra settings and their
    corresponding values. Each extra setting name is immediately followed
    by the corresponding value. The list must be terminated with either
    NULL or CU_LAUNCH_PARAM_END.

    - :py:obj:`~.CU_LAUNCH_PARAM_END`, which indicates the end of the
      `extra` array;

    - :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER`, which specifies that the
      next value in `extra` will be a pointer to a buffer containing all
      the kernel parameters for launching kernel `func`;

    - :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_SIZE`, which specifies that the
      next value in `extra` will be a pointer to a size_t containing the
      size of the buffer specified with
      :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER`;

    The error :py:obj:`~.cudaErrorInvalidValue` will be returned if kernel
    parameters are specified with both `kernelParams` and `extra` (i.e.
    both `kernelParams` and `extra` are non-NULL).

    The `kernelParams` or `extra` array, as well as the argument values it
    points to, are copied during this call.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    pNodeParams : :py:obj:`~.cudaKernelNodeParams`
        Parameters for the GPU execution node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDeviceFunction`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaLaunchKernel`, :py:obj:`~.cudaGraphKernelNodeGetParams`, :py:obj:`~.cudaGraphKernelNodeSetParams`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`

    Notes
    -----
    Kernels launched using graphs must not use texture and surface references. Reading or writing through any texture or surface reference is undefined behavior. This restriction does not apply to texture and surface objects.
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdef ccudart.cudaKernelNodeParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphAddKernelNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cpNodeParams_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphKernelNodeGetParams(node):
    """ Returns a kernel node's parameters.

    Returns the parameters of kernel node `node` in `pNodeParams`. The
    `kernelParams` or `extra` array returned in `pNodeParams`, as well as
    the argument values it points to, are owned by the node. This memory
    remains valid until the node is destroyed or its parameters are
    modified, and should not be modified directly. Use
    :py:obj:`~.cudaGraphKernelNodeSetParams` to update the parameters of
    this node.

    The params will contain either `kernelParams` or `extra`, according to
    which of these was most recently set on the node.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDeviceFunction`
    pNodeParams : :py:obj:`~.cudaKernelNodeParams`
        Pointer to return the parameters

    See Also
    --------
    :py:obj:`~.cudaLaunchKernel`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphKernelNodeSetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaKernelNodeParams pNodeParams = cudaKernelNodeParams()
    err = ccudart.cudaGraphKernelNodeGetParams(cnode, pNodeParams._ptr)
    return (cudaError_t(err), pNodeParams)

@cython.embedsignature(True)
def cudaGraphKernelNodeSetParams(node, pNodeParams : cudaKernelNodeParams):
    """ Sets a kernel node's parameters.

    Sets the parameters of kernel node `node` to `pNodeParams`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the parameters for
    pNodeParams : :py:obj:`~.cudaKernelNodeParams`
        Parameters to copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorMemoryAllocation`

    See Also
    --------
    :py:obj:`~.cudaLaunchKernel`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphKernelNodeGetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaKernelNodeParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphKernelNodeSetParams(cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphKernelNodeCopyAttributes(hSrc, hDst):
    """ Copies attributes from source node to destination node.

    Copies attributes from source node `src` to destination node `dst`.
    Both node must have the same context.

    Parameters
    ----------
    dst : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Destination node
    src : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Source node For list of attributes see
        :py:obj:`~.cudaKernelNodeAttrID`

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidContext`

    See Also
    --------
    :py:obj:`~.cudaAccessPolicyWindow`
    """
    cdef ccudart.cudaGraphNode_t chDst
    if hDst is None:
        chDst = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hDst, (cudaGraphNode_t,cuda.CUgraphNode)):
        phDst = int(hDst)
        chDst = <ccudart.cudaGraphNode_t><void_ptr>phDst
    else:
        phDst = int(cudaGraphNode_t(hDst))
        chDst = <ccudart.cudaGraphNode_t><void_ptr>phDst

    cdef ccudart.cudaGraphNode_t chSrc
    if hSrc is None:
        chSrc = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hSrc, (cudaGraphNode_t,cuda.CUgraphNode)):
        phSrc = int(hSrc)
        chSrc = <ccudart.cudaGraphNode_t><void_ptr>phSrc
    else:
        phSrc = int(cudaGraphNode_t(hSrc))
        chSrc = <ccudart.cudaGraphNode_t><void_ptr>phSrc

    err = ccudart.cudaGraphKernelNodeCopyAttributes(chSrc, chDst)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphKernelNodeGetAttribute(hNode, attr not None : cudaKernelNodeAttrID):
    """ Queries node attribute.

    Queries attribute `attr` from node `hNode` and stores it in
    corresponding member of `value_out`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`

    attr : :py:obj:`~.cudaKernelNodeAttrID`


    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`
    value_out : :py:obj:`~.cudaKernelNodeAttrValue`


    See Also
    --------
    :py:obj:`~.cudaAccessPolicyWindow`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaKernelNodeAttrID cattr = attr.value
    cdef cudaKernelNodeAttrValue value_out = cudaKernelNodeAttrValue()
    err = ccudart.cudaGraphKernelNodeGetAttribute(chNode, cattr, value_out._ptr)
    return (cudaError_t(err), value_out)

@cython.embedsignature(True)
def cudaGraphKernelNodeSetAttribute(hNode, attr not None : cudaKernelNodeAttrID, value : cudaKernelNodeAttrValue):
    """ Sets node attribute.

    Sets attribute `attr` on node `hNode` from corresponding attribute of
    `value`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`

    attr : :py:obj:`~.cudaKernelNodeAttrID`

    value : :py:obj:`~.cudaKernelNodeAttrValue`


    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidResourceHandle`

    See Also
    --------
    :py:obj:`~.cudaAccessPolicyWindow`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaKernelNodeAttrID cattr = attr.value
    cdef ccudart.cudaKernelNodeAttrValue* cvalue_ptr = value._ptr if value != None else NULL
    err = ccudart.cudaGraphKernelNodeSetAttribute(chNode, cattr, cvalue_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddMemcpyNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, pCopyParams : cudaMemcpy3DParms):
    """ Creates a memcpy node and adds it to a graph.

    Creates a new memcpy node and adds it to `graph` with `numDependencies`
    dependencies specified via `pDependencies`. It is possible for
    `numDependencies` to be 0, in which case the node will be placed at the
    root of the graph. `pDependencies` may not have any duplicate entries.
    A handle to the new node will be returned in `pGraphNode`.

    When the graph is launched, the node will perform the memcpy described
    by `pCopyParams`. See :py:obj:`~.cudaMemcpy3D()` for a description of
    the structure and its restrictions.

    Memcpy nodes have some additional restrictions with regards to managed
    memory, if the system contains at least one device which has a zero
    value for the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess`.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    pCopyParams : :py:obj:`~.cudaMemcpy3DParms`
        Parameters for the memory copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaMemcpy3D`, :py:obj:`~.cudaGraphAddMemcpyNodeToSymbol`, :py:obj:`~.cudaGraphAddMemcpyNodeFromSymbol`, :py:obj:`~.cudaGraphAddMemcpyNode1D`, :py:obj:`~.cudaGraphMemcpyNodeGetParams`, :py:obj:`~.cudaGraphMemcpyNodeSetParams`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemsetNode`
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdef ccudart.cudaMemcpy3DParms* cpCopyParams_ptr = pCopyParams._ptr if pCopyParams != None else NULL
    err = ccudart.cudaGraphAddMemcpyNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cpCopyParams_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphAddMemcpyNode1D(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, dst, src, size_t count, kind not None : cudaMemcpyKind):
    """ Creates a 1D memcpy node and adds it to a graph.

    Creates a new 1D memcpy node and adds it to `graph` with
    `numDependencies` dependencies specified via `pDependencies`. It is
    possible for `numDependencies` to be 0, in which case the node will be
    placed at the root of the graph. `pDependencies` may not have any
    duplicate entries. A handle to the new node will be returned in
    `pGraphNode`.

    When the graph is launched, the node will copy `count` bytes from the
    memory area pointed to by `src` to the memory area pointed to by `dst`,
    where `kind` specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. Launching a memcpy node with dst and src
    pointers that do not match the direction of the copy results in an
    undefined behavior.

    Memcpy nodes have some additional restrictions with regards to managed
    memory, if the system contains at least one device which has a zero
    value for the device attribute
    :py:obj:`~.cudaDevAttrConcurrentManagedAccess`.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    dst : Any
        Destination memory address
    src : Any
        Source memory address
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphMemcpyNodeGetParams`, :py:obj:`~.cudaGraphMemcpyNodeSetParams`, :py:obj:`~.cudaGraphMemcpyNodeSetParams1D`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemsetNode`
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaGraphAddMemcpyNode1D(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cdst_ptr, csrc_ptr, count, ckind)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphMemcpyNodeGetParams(node):
    """ Returns a memcpy node's parameters.

    Returns the parameters of memcpy node `node` in `pNodeParams`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pNodeParams : :py:obj:`~.cudaMemcpy3DParms`
        Pointer to return the parameters

    See Also
    --------
    :py:obj:`~.cudaMemcpy3D`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphMemcpyNodeSetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaMemcpy3DParms pNodeParams = cudaMemcpy3DParms()
    err = ccudart.cudaGraphMemcpyNodeGetParams(cnode, pNodeParams._ptr)
    return (cudaError_t(err), pNodeParams)

@cython.embedsignature(True)
def cudaGraphMemcpyNodeSetParams(node, pNodeParams : cudaMemcpy3DParms):
    """ Sets a memcpy node's parameters.

    Sets the parameters of memcpy node `node` to `pNodeParams`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the parameters for
    pNodeParams : :py:obj:`~.cudaMemcpy3DParms`
        Parameters to copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaMemcpy3D`, :py:obj:`~.cudaGraphMemcpyNodeSetParamsToSymbol`, :py:obj:`~.cudaGraphMemcpyNodeSetParamsFromSymbol`, :py:obj:`~.cudaGraphMemcpyNodeSetParams1D`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphMemcpyNodeGetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaMemcpy3DParms* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphMemcpyNodeSetParams(cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphMemcpyNodeSetParams1D(node, dst, src, size_t count, kind not None : cudaMemcpyKind):
    """ Sets a memcpy node's parameters to perform a 1-dimensional copy.

    Sets the parameters of memcpy node `node` to the copy described by the
    provided parameters.

    When the graph is launched, the node will copy `count` bytes from the
    memory area pointed to by `src` to the memory area pointed to by `dst`,
    where `kind` specifies the direction of the copy, and must be one of
    :py:obj:`~.cudaMemcpyHostToHost`, :py:obj:`~.cudaMemcpyHostToDevice`,
    :py:obj:`~.cudaMemcpyDeviceToHost`,
    :py:obj:`~.cudaMemcpyDeviceToDevice`, or :py:obj:`~.cudaMemcpyDefault`.
    Passing :py:obj:`~.cudaMemcpyDefault` is recommended, in which case the
    type of transfer is inferred from the pointer values. However,
    :py:obj:`~.cudaMemcpyDefault` is only allowed on systems that support
    unified virtual addressing. Launching a memcpy node with dst and src
    pointers that do not match the direction of the copy results in an
    undefined behavior.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the parameters for
    dst : Any
        Destination memory address
    src : Any
        Source memory address
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaGraphMemcpyNodeSetParams`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphMemcpyNodeGetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaGraphMemcpyNodeSetParams1D(cnode, cdst_ptr, csrc_ptr, count, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddMemsetNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, pMemsetParams : cudaMemsetParams):
    """ Creates a memset node and adds it to a graph.

    Creates a new memset node and adds it to `graph` with `numDependencies`
    dependencies specified via `pDependencies`. It is possible for
    `numDependencies` to be 0, in which case the node will be placed at the
    root of the graph. `pDependencies` may not have any duplicate entries.
    A handle to the new node will be returned in `pGraphNode`.

    The element size must be 1, 2, or 4 bytes. When the graph is launched,
    the node will perform the memset described by `pMemsetParams`.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    pMemsetParams : :py:obj:`~.cudaMemsetParams`
        Parameters for the memory set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorInvalidDevice`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaMemset2D`, :py:obj:`~.cudaGraphMemsetNodeGetParams`, :py:obj:`~.cudaGraphMemsetNodeSetParams`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemcpyNode`
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdef ccudart.cudaMemsetParams* cpMemsetParams_ptr = pMemsetParams._ptr if pMemsetParams != None else NULL
    err = ccudart.cudaGraphAddMemsetNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cpMemsetParams_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphMemsetNodeGetParams(node):
    """ Returns a memset node's parameters.

    Returns the parameters of memset node `node` in `pNodeParams`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pNodeParams : :py:obj:`~.cudaMemsetParams`
        Pointer to return the parameters

    See Also
    --------
    :py:obj:`~.cudaMemset2D`, :py:obj:`~.cudaGraphAddMemsetNode`, :py:obj:`~.cudaGraphMemsetNodeSetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaMemsetParams pNodeParams = cudaMemsetParams()
    err = ccudart.cudaGraphMemsetNodeGetParams(cnode, pNodeParams._ptr)
    return (cudaError_t(err), pNodeParams)

@cython.embedsignature(True)
def cudaGraphMemsetNodeSetParams(node, pNodeParams : cudaMemsetParams):
    """ Sets a memset node's parameters.

    Sets the parameters of memset node `node` to `pNodeParams`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the parameters for
    pNodeParams : :py:obj:`~.cudaMemsetParams`
        Parameters to copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaMemset2D`, :py:obj:`~.cudaGraphAddMemsetNode`, :py:obj:`~.cudaGraphMemsetNodeGetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaMemsetParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphMemsetNodeSetParams(cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddHostNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, pNodeParams : cudaHostNodeParams):
    """ Creates a host execution node and adds it to a graph.

    Creates a new CPU execution node and adds it to `graph` with
    `numDependencies` dependencies specified via `pDependencies` and
    arguments specified in `pNodeParams`. It is possible for
    `numDependencies` to be 0, in which case the node will be placed at the
    root of the graph. `pDependencies` may not have any duplicate entries.
    A handle to the new node will be returned in `pGraphNode`.

    When the graph is launched, the node will invoke the specified CPU
    function. Host nodes are not supported under MPS with pre-Volta GPUs.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    pNodeParams : :py:obj:`~.cudaHostNodeParams`
        Parameters for the host node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaLaunchHostFunc`, :py:obj:`~.cudaGraphHostNodeGetParams`, :py:obj:`~.cudaGraphHostNodeSetParams`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdef ccudart.cudaHostNodeParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphAddHostNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cpNodeParams_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphHostNodeGetParams(node):
    """ Returns a host node's parameters.

    Returns the parameters of host node `node` in `pNodeParams`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pNodeParams : :py:obj:`~.cudaHostNodeParams`
        Pointer to return the parameters

    See Also
    --------
    :py:obj:`~.cudaLaunchHostFunc`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphHostNodeSetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaHostNodeParams pNodeParams = cudaHostNodeParams()
    err = ccudart.cudaGraphHostNodeGetParams(cnode, pNodeParams._ptr)
    return (cudaError_t(err), pNodeParams)

@cython.embedsignature(True)
def cudaGraphHostNodeSetParams(node, pNodeParams : cudaHostNodeParams):
    """ Sets a host node's parameters.

    Sets the parameters of host node `node` to `nodeParams`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the parameters for
    pNodeParams : :py:obj:`~.cudaHostNodeParams`
        Parameters to copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaLaunchHostFunc`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphHostNodeGetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaHostNodeParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphHostNodeSetParams(cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddChildGraphNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, childGraph):
    """ Creates a child graph node and adds it to a graph.

    Creates a new node which executes an embedded graph, and adds it to
    `graph` with `numDependencies` dependencies specified via
    `pDependencies`. It is possible for `numDependencies` to be 0, in which
    case the node will be placed at the root of the graph. `pDependencies`
    may not have any duplicate entries. A handle to the new node will be
    returned in `pGraphNode`.

    If `hGraph` contains allocation or free nodes, this call will return an
    error.

    The node executes an embedded child graph. The child graph is cloned in
    this call.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    childGraph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        The graph to clone into this node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphChildGraphNodeGetGraph`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`, :py:obj:`~.cudaGraphClone`
    """
    cdef ccudart.cudaGraph_t cchildGraph
    if childGraph is None:
        cchildGraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(childGraph, (cudaGraph_t,cuda.CUgraph)):
        pchildGraph = int(childGraph)
        cchildGraph = <ccudart.cudaGraph_t><void_ptr>pchildGraph
    else:
        pchildGraph = int(cudaGraph_t(childGraph))
        cchildGraph = <ccudart.cudaGraph_t><void_ptr>pchildGraph

    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    err = ccudart.cudaGraphAddChildGraphNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cchildGraph)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphChildGraphNodeGetGraph(node):
    """ Gets a handle to the embedded graph of a child graph node.

    Gets a handle to the embedded graph in a child graph node. This call
    does not clone the graph. Changes to the graph will be reflected in the
    node, and the node retains ownership of the graph.

    Allocation and free nodes cannot be added to the returned graph.
    Attempting to do so will return an error.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the embedded graph for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraph : :py:obj:`~.cudaGraph_t`
        Location to store a handle to the graph

    See Also
    --------
    :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphNodeFindInClone`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaGraph_t pGraph = cudaGraph_t()
    err = ccudart.cudaGraphChildGraphNodeGetGraph(cnode, <ccudart.cudaGraph_t*>pGraph._ptr)
    return (cudaError_t(err), pGraph)

@cython.embedsignature(True)
def cudaGraphAddEmptyNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies):
    """ Creates an empty node and adds it to a graph.

    Creates a new node which performs no operation, and adds it to `graph`
    with `numDependencies` dependencies specified via `pDependencies`. It
    is possible for `numDependencies` to be 0, in which case the node will
    be placed at the root of the graph. `pDependencies` may not have any
    duplicate entries. A handle to the new node will be returned in
    `pGraphNode`.

    An empty node performs no operation during execution, but can be used
    for transitive ordering. For example, a phased execution graph with 2
    groups of n nodes with a barrier between them can be represented using
    an empty node and 2*n dependency edges, rather than no empty node and
    n^2 dependency edges.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    err = ccudart.cudaGraphAddEmptyNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphAddEventRecordNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, event):
    """ Creates an event record node and adds it to a graph.

    Creates a new event record node and adds it to `hGraph` with
    `numDependencies` dependencies specified via `dependencies` and event
    specified in `event`. It is possible for `numDependencies` to be 0, in
    which case the node will be placed at the root of the graph.
    `dependencies` may not have any duplicate entries. A handle to the new
    node will be returned in `phGraphNode`.

    Each launch of the graph will record `event` to capture execution of
    the node's dependencies.

    These nodes may not be used in loops or conditionals.

    Parameters
    ----------
    hGraph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    dependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event for the node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    phGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`,
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    err = ccudart.cudaGraphAddEventRecordNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cevent)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphEventRecordNodeGetEvent(node):
    """ Returns the event associated with an event record node.

    Returns the event of event record node `hNode` in `event_out`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the event for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    event_out : :py:obj:`~.cudaEvent_t`
        Pointer to return the event

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaGraphEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphEventWaitNodeGetEvent`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaEvent_t event_out = cudaEvent_t()
    err = ccudart.cudaGraphEventRecordNodeGetEvent(cnode, <ccudart.cudaEvent_t*>event_out._ptr)
    return (cudaError_t(err), event_out)

@cython.embedsignature(True)
def cudaGraphEventRecordNodeSetEvent(node, event):
    """ Sets an event record node's event.

    Sets the event of event record node `hNode` to `event`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the event for
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to use

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaGraphEventRecordNodeGetEvent`, :py:obj:`~.cudaGraphEventWaitNodeSetEvent`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    err = ccudart.cudaGraphEventRecordNodeSetEvent(cnode, cevent)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddEventWaitNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, event):
    """ Creates an event wait node and adds it to a graph.

    Creates a new event wait node and adds it to `hGraph` with
    `numDependencies` dependencies specified via `dependencies` and event
    specified in `event`. It is possible for `numDependencies` to be 0, in
    which case the node will be placed at the root of the graph.
    `dependencies` may not have any duplicate entries. A handle to the new
    node will be returned in `phGraphNode`.

    The graph node will wait for all work captured in `event`. See
    :py:obj:`~.cuEventRecord()` for details on what is captured by an
    event. The synchronization will be performed efficiently on the device
    when applicable. `event` may be from a different context or device than
    the launch stream.

    These nodes may not be used in loops or conditionals.

    Parameters
    ----------
    hGraph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    dependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event for the node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    phGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`,
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    err = ccudart.cudaGraphAddEventWaitNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cevent)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphEventWaitNodeGetEvent(node):
    """ Returns the event associated with an event wait node.

    Returns the event of event wait node `hNode` in `event_out`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the event for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    event_out : :py:obj:`~.cudaEvent_t`
        Pointer to return the event

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaGraphEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphEventRecordNodeGetEvent`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaEvent_t event_out = cudaEvent_t()
    err = ccudart.cudaGraphEventWaitNodeGetEvent(cnode, <ccudart.cudaEvent_t*>event_out._ptr)
    return (cudaError_t(err), event_out)

@cython.embedsignature(True)
def cudaGraphEventWaitNodeSetEvent(node, event):
    """ Sets an event wait node's event.

    Sets the event of event wait node `hNode` to `event`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the event for
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to use

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaGraphEventWaitNodeGetEvent`, :py:obj:`~.cudaGraphEventRecordNodeSetEvent`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    err = ccudart.cudaGraphEventWaitNodeSetEvent(cnode, cevent)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddExternalSemaphoresSignalNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, nodeParams : cudaExternalSemaphoreSignalNodeParams):
    """ Creates an external semaphore signal node and adds it to a graph.

    Creates a new external semaphore signal node and adds it to `graph`
    with `numDependencies` dependencies specified via `dependencies` and
    arguments specified in `nodeParams`. It is possible for
    `numDependencies` to be 0, in which case the node will be placed at the
    root of the graph. `dependencies` may not have any duplicate entries. A
    handle to the new node will be returned in `pGraphNode`.

    Performs a signal operation on a set of externally allocated semaphore
    objects when the node is launched. The operation(s) will occur after
    all of the node's dependencies have completed.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    nodeParams : :py:obj:`~.cudaExternalSemaphoreSignalNodeParams`
        Parameters for the node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphExternalSemaphoresSignalNodeGetParams`, :py:obj:`~.cudaGraphExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`,
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdef ccudart.cudaExternalSemaphoreSignalNodeParams* cnodeParams_ptr = nodeParams._ptr if nodeParams != None else NULL
    err = ccudart.cudaGraphAddExternalSemaphoresSignalNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cnodeParams_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphExternalSemaphoresSignalNodeGetParams(hNode):
    """ Returns an external semaphore signal node's parameters.

    Returns the parameters of an external semaphore signal node `hNode` in
    `params_out`. The `extSemArray` and `paramsArray` returned in
    `params_out`, are owned by the node. This memory remains valid until
    the node is destroyed or its parameters are modified, and should not be
    modified directly. Use
    :py:obj:`~.cudaGraphExternalSemaphoresSignalNodeSetParams` to update
    the parameters of this node.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    params_out : :py:obj:`~.cudaExternalSemaphoreSignalNodeParams`
        Pointer to return the parameters

    See Also
    --------
    :py:obj:`~.cudaLaunchKernel`, :py:obj:`~.cudaGraphAddExternalSemaphoresSignalNode`, :py:obj:`~.cudaGraphExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef cudaExternalSemaphoreSignalNodeParams params_out = cudaExternalSemaphoreSignalNodeParams()
    err = ccudart.cudaGraphExternalSemaphoresSignalNodeGetParams(chNode, params_out._ptr)
    return (cudaError_t(err), params_out)

@cython.embedsignature(True)
def cudaGraphExternalSemaphoresSignalNodeSetParams(hNode, nodeParams : cudaExternalSemaphoreSignalNodeParams):
    """ Sets an external semaphore signal node's parameters.

    Sets the parameters of an external semaphore signal node `hNode` to
    `nodeParams`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the parameters for
    nodeParams : :py:obj:`~.cudaExternalSemaphoreSignalNodeParams`
        Parameters to copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddExternalSemaphoresSignalNode`, :py:obj:`~.cudaGraphExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaExternalSemaphoreSignalNodeParams* cnodeParams_ptr = nodeParams._ptr if nodeParams != None else NULL
    err = ccudart.cudaGraphExternalSemaphoresSignalNodeSetParams(chNode, cnodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddExternalSemaphoresWaitNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, nodeParams : cudaExternalSemaphoreWaitNodeParams):
    """ Creates an external semaphore wait node and adds it to a graph.

    Creates a new external semaphore wait node and adds it to `graph` with
    `numDependencies` dependencies specified via `dependencies` and
    arguments specified in `nodeParams`. It is possible for
    `numDependencies` to be 0, in which case the node will be placed at the
    root of the graph. `dependencies` may not have any duplicate entries. A
    handle to the new node will be returned in `pGraphNode`.

    Performs a wait operation on a set of externally allocated semaphore
    objects when the node is launched. The node's dependencies will not be
    launched until the wait operation has completed.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    nodeParams : :py:obj:`~.cudaExternalSemaphoreWaitNodeParams`
        Parameters for the node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphExternalSemaphoresWaitNodeGetParams`, :py:obj:`~.cudaGraphExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphAddExternalSemaphoresSignalNode`, :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`,
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdef ccudart.cudaExternalSemaphoreWaitNodeParams* cnodeParams_ptr = nodeParams._ptr if nodeParams != None else NULL
    err = ccudart.cudaGraphAddExternalSemaphoresWaitNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cnodeParams_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphExternalSemaphoresWaitNodeGetParams(hNode):
    """ Returns an external semaphore wait node's parameters.

    Returns the parameters of an external semaphore wait node `hNode` in
    `params_out`. The `extSemArray` and `paramsArray` returned in
    `params_out`, are owned by the node. This memory remains valid until
    the node is destroyed or its parameters are modified, and should not be
    modified directly. Use
    :py:obj:`~.cudaGraphExternalSemaphoresSignalNodeSetParams` to update
    the parameters of this node.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    params_out : :py:obj:`~.cudaExternalSemaphoreWaitNodeParams`
        Pointer to return the parameters

    See Also
    --------
    :py:obj:`~.cudaLaunchKernel`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaGraphExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef cudaExternalSemaphoreWaitNodeParams params_out = cudaExternalSemaphoreWaitNodeParams()
    err = ccudart.cudaGraphExternalSemaphoresWaitNodeGetParams(chNode, params_out._ptr)
    return (cudaError_t(err), params_out)

@cython.embedsignature(True)
def cudaGraphExternalSemaphoresWaitNodeSetParams(hNode, nodeParams : cudaExternalSemaphoreWaitNodeParams):
    """ Sets an external semaphore wait node's parameters.

    Sets the parameters of an external semaphore wait node `hNode` to
    `nodeParams`.

    Parameters
    ----------
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to set the parameters for
    nodeParams : :py:obj:`~.cudaExternalSemaphoreWaitNodeParams`
        Parameters to copy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaGraphExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaExternalSemaphoreWaitNodeParams* cnodeParams_ptr = nodeParams._ptr if nodeParams != None else NULL
    err = ccudart.cudaGraphExternalSemaphoresWaitNodeSetParams(chNode, cnodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphAddMemAllocNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, nodeParams : cudaMemAllocNodeParams):
    """ Creates an allocation node and adds it to a graph.

    Creates a new allocation node and adds it to `graph` with
    `numDependencies` dependencies specified via `pDependencies` and
    arguments specified in `nodeParams`. It is possible for
    `numDependencies` to be 0, in which case the node will be placed at the
    root of the graph. `pDependencies` may not have any duplicate entries.
    A handle to the new node will be returned in `pGraphNode`.

    When :py:obj:`~.cudaGraphAddMemAllocNode` creates an allocation node,
    it returns the address of the allocation in `nodeParams.dptr`. The
    allocation's address remains fixed across instantiations and launches.

    If the allocation is freed in the same graph, by creating a free node
    using :py:obj:`~.cudaGraphAddMemFreeNode`, the allocation can be
    accessed by nodes ordered after the allocation node but before the free
    node. These allocations cannot be freed outside the owning graph, and
    they can only be freed once in the owning graph.

    If the allocation is not freed in the same graph, then it can be
    accessed not only by nodes in the graph which are ordered after the
    allocation node, but also by stream operations ordered after the
    graph's execution but before the allocation is freed.

    Allocations which are not freed in the same graph can be freed by:

    - passing the allocation to :py:obj:`~.cudaMemFreeAsync` or
      :py:obj:`~.cudaMemFree`;

    - launching a graph with a free node for that allocation; or

    - specifying :py:obj:`~.cudaGraphInstantiateFlagAutoFreeOnLaunch`
      during instantiation, which makes each launch behave as though it
      called :py:obj:`~.cudaMemFreeAsync` for every unfreed allocation.

    It is not possible to free an allocation in both the owning graph and
    another graph. If the allocation is freed in the same graph, a free
    node cannot be added to another graph. If the allocation is freed in
    another graph, a free node can no longer be added to the owning graph.

    The following restrictions apply to graphs which contain allocation
    and/or memory free nodes:

    - Nodes and edges of the graph cannot be deleted.

    - The graph cannot be used in a child node.

    - Only one instantiation of the graph may exist at any point in time.

    - The graph cannot be cloned.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    nodeParams : :py:obj:`~.cudaMemAllocNodeParams`
        Parameters for the node

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorCudartUnloading`, :py:obj:`~.cudaErrorInitializationError`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorOutOfMemory`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemFreeNode`, :py:obj:`~.cudaGraphMemAllocNodeGetParams`, :py:obj:`~.cudaDeviceGraphMemTrim`, :py:obj:`~.cudaDeviceGetGraphMemAttribute`, :py:obj:`~.cudaDeviceSetGraphMemAttribute`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaGraphAddExternalSemaphoresSignalNode`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdef ccudart.cudaMemAllocNodeParams* cnodeParams_ptr = nodeParams._ptr if nodeParams != None else NULL
    err = ccudart.cudaGraphAddMemAllocNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cnodeParams_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphMemAllocNodeGetParams(node):
    """ Returns a memory alloc node's parameters.

    Returns the parameters of a memory alloc node `hNode` in `params_out`.
    The `poolProps` and `accessDescs` returned in `params_out`, are owned
    by the node. This memory remains valid until the node is destroyed. The
    returned parameters must not be modified.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    params_out : :py:obj:`~.cudaMemAllocNodeParams`
        Pointer to return the parameters

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemAllocNode`, :py:obj:`~.cudaGraphMemFreeNodeGetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef cudaMemAllocNodeParams params_out = cudaMemAllocNodeParams()
    err = ccudart.cudaGraphMemAllocNodeGetParams(cnode, params_out._ptr)
    return (cudaError_t(err), params_out)

@cython.embedsignature(True)
def cudaGraphAddMemFreeNode(graph, pDependencies : List[cudaGraphNode_t], size_t numDependencies, dptr):
    """ Creates a memory free node and adds it to a graph.

    Creates a new memory free node and adds it to `graph` with
    `numDependencies` dependencies specified via `pDependencies` and
    address specified in `dptr`. It is possible for `numDependencies` to be
    0, in which case the node will be placed at the root of the graph.
    `pDependencies` may not have any duplicate entries. A handle to the new
    node will be returned in `pGraphNode`.

    :py:obj:`~.cudaGraphAddMemFreeNode` will return
    :py:obj:`~.cudaErrorInvalidValue` if the user attempts to free:

    - an allocation twice in the same graph.

    - an address that was not returned by an allocation node.

    - an invalid address.

    The following restrictions apply to graphs which contain allocation
    and/or memory free nodes:

    - Nodes and edges of the graph cannot be deleted.

    - The graph cannot be used in a child node.

    - Only one instantiation of the graph may exist at any point in time.

    - The graph cannot be cloned.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which to add the node
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Dependencies of the node
    numDependencies : size_t
        Number of dependencies
    dptr : Any
        Address of memory to free

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorCudartUnloading`, :py:obj:`~.cudaErrorInitializationError`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorOutOfMemory`
    pGraphNode : :py:obj:`~.cudaGraphNode_t`
        Returns newly created node

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemAllocNode`, :py:obj:`~.cudaGraphMemFreeNodeGetParams`, :py:obj:`~.cudaDeviceGraphMemTrim`, :py:obj:`~.cudaDeviceGetGraphMemAttribute`, :py:obj:`~.cudaDeviceSetGraphMemAttribute`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphDestroyNode`, :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaGraphAddExternalSemaphoresSignalNode`, :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`
    """
    pDependencies = [] if pDependencies is None else pDependencies
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in pDependencies):
        raise TypeError("Argument 'pDependencies' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphNode_t pGraphNode = cudaGraphNode_t()
    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    if len(pDependencies) > 0:
        cpDependencies = <ccudart.cudaGraphNode_t*> calloc(len(pDependencies), sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(pDependencies)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(pDependencies)):
                cpDependencies[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>pDependencies[idx])._ptr[0]

    if numDependencies > <size_t>len(pDependencies): raise RuntimeError("List is too small: " + str(len(pDependencies)) + " < " + str(numDependencies))
    cdptr = utils.HelperInputVoidPtr(dptr)
    cdef void* cdptr_ptr = <void*><void_ptr>cdptr.cptr
    err = ccudart.cudaGraphAddMemFreeNode(<ccudart.cudaGraphNode_t*>pGraphNode._ptr, cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>pDependencies[0])._ptr if len(pDependencies) == 1 else cpDependencies, numDependencies, cdptr_ptr)
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pGraphNode)

@cython.embedsignature(True)
def cudaGraphMemFreeNodeGetParams(node):
    """ Returns a memory free node's parameters.

    Returns the address of a memory free node `hNode` in `dptr_out`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to get the parameters for

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    dptr_out : Any
        Pointer to return the device address

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemFreeNode`, :py:obj:`~.cudaGraphMemFreeNodeGetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef void_ptr dptr_out = 0
    cdef void* cdptr_out_ptr = <void*>dptr_out
    err = ccudart.cudaGraphMemFreeNodeGetParams(cnode, cdptr_out_ptr)
    return (cudaError_t(err), dptr_out)

@cython.embedsignature(True)
def cudaDeviceGraphMemTrim(int device):
    """ Free unused memory that was cached on the specified device for use with graphs back to the OS.

    Blocks which are not in use by a graph that is either currently
    executing or scheduled to execute are freed back to the operating
    system.

    Parameters
    ----------
    device : int
        The device for which cached memory should be freed.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemAllocNode`, :py:obj:`~.cudaGraphAddMemFreeNode`, :py:obj:`~.cudaDeviceGetGraphMemAttribute`, :py:obj:`~.cudaDeviceSetGraphMemAttribute`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`,
    """
    err = ccudart.cudaDeviceGraphMemTrim(device)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaDeviceGetGraphMemAttribute(int device, attr not None : cudaGraphMemAttributeType):
    """ Query asynchronous allocation attributes related to graphs.

    Valid attributes are:

    - :py:obj:`~.cudaGraphMemAttrUsedMemCurrent`: Amount of memory, in
      bytes, currently associated with graphs

    - :py:obj:`~.cudaGraphMemAttrUsedMemHigh`: High watermark of memory, in
      bytes, associated with graphs since the last time it was reset. High
      watermark can only be reset to zero.

    - :py:obj:`~.cudaGraphMemAttrReservedMemCurrent`: Amount of memory, in
      bytes, currently allocated for use by the CUDA graphs asynchronous
      allocator.

    - :py:obj:`~.cudaGraphMemAttrReservedMemHigh`: High watermark of
      memory, in bytes, currently allocated for use by the CUDA graphs
      asynchronous allocator.

    Parameters
    ----------
    device : int
        Specifies the scope of the query
    attr : :py:obj:`~.cudaGraphMemAttributeType`
        attribute to get

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`
    value : Any
        retrieved value

    See Also
    --------
    :py:obj:`~.cudaDeviceSetGraphMemAttribute`, :py:obj:`~.cudaGraphAddMemAllocNode`, :py:obj:`~.cudaGraphAddMemFreeNode`, :py:obj:`~.cudaDeviceGraphMemTrim`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`,
    """
    cdef ccudart.cudaGraphMemAttributeType cattr = attr.value
    cdef utils.HelperCUgraphMem_attribute cvalue = utils.HelperCUgraphMem_attribute(attr, 0, is_getter=True)
    cdef void* cvalue_ptr = <void*><void_ptr>cvalue.cptr
    err = ccudart.cudaDeviceGetGraphMemAttribute(device, cattr, cvalue_ptr)
    return (cudaError_t(err), cvalue.pyObj())

@cython.embedsignature(True)
def cudaDeviceSetGraphMemAttribute(int device, attr not None : cudaGraphMemAttributeType, value):
    """ Set asynchronous allocation attributes related to graphs.

    Valid attributes are:

    - :py:obj:`~.cudaGraphMemAttrUsedMemHigh`: High watermark of memory, in
      bytes, associated with graphs since the last time it was reset. High
      watermark can only be reset to zero.

    - :py:obj:`~.cudaGraphMemAttrReservedMemHigh`: High watermark of
      memory, in bytes, currently allocated for use by the CUDA graphs
      asynchronous allocator.

    Parameters
    ----------
    device : int
        Specifies the scope of the query
    attr : :py:obj:`~.cudaGraphMemAttributeType`
        attribute to get
    value : Any
        pointer to value to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidDevice`

    See Also
    --------
    :py:obj:`~.cudaDeviceGetGraphMemAttribute`, :py:obj:`~.cudaGraphAddMemAllocNode`, :py:obj:`~.cudaGraphAddMemFreeNode`, :py:obj:`~.cudaDeviceGraphMemTrim`, :py:obj:`~.cudaMallocAsync`, :py:obj:`~.cudaFreeAsync`,
    """
    cdef ccudart.cudaGraphMemAttributeType cattr = attr.value
    cdef utils.HelperCUgraphMem_attribute cvalue = utils.HelperCUgraphMem_attribute(attr, value, is_getter=False)
    cdef void* cvalue_ptr = <void*><void_ptr>cvalue.cptr
    err = ccudart.cudaDeviceSetGraphMemAttribute(device, cattr, cvalue_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphClone(originalGraph):
    """ Clones a graph.

    This function creates a copy of `originalGraph` and returns it in
    `pGraphClone`. All parameters are copied into the cloned graph. The
    original graph may be modified after this call without affecting the
    clone.

    Child graph nodes in the original graph are recursively copied into the
    clone.

    Parameters
    ----------
    originalGraph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to clone

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorMemoryAllocation`
    pGraphClone : :py:obj:`~.cudaGraph_t`
        Returns newly created cloned graph

    See Also
    --------
    :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphNodeFindInClone`
    """
    cdef ccudart.cudaGraph_t coriginalGraph
    if originalGraph is None:
        coriginalGraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(originalGraph, (cudaGraph_t,cuda.CUgraph)):
        poriginalGraph = int(originalGraph)
        coriginalGraph = <ccudart.cudaGraph_t><void_ptr>poriginalGraph
    else:
        poriginalGraph = int(cudaGraph_t(originalGraph))
        coriginalGraph = <ccudart.cudaGraph_t><void_ptr>poriginalGraph

    cdef cudaGraph_t pGraphClone = cudaGraph_t()
    err = ccudart.cudaGraphClone(<ccudart.cudaGraph_t*>pGraphClone._ptr, coriginalGraph)
    return (cudaError_t(err), pGraphClone)

@cython.embedsignature(True)
def cudaGraphNodeFindInClone(originalNode, clonedGraph):
    """ Finds a cloned version of a node.

    This function returns the node in `clonedGraph` corresponding to
    `originalNode` in the original graph.

    `clonedGraph` must have been cloned from `originalGraph` via
    :py:obj:`~.cudaGraphClone`. `originalNode` must have been in
    `originalGraph` at the time of the call to :py:obj:`~.cudaGraphClone`,
    and the corresponding cloned node in `clonedGraph` must not have been
    removed. The cloned node is then returned via `pClonedNode`.

    Parameters
    ----------
    originalNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Handle to the original node
    clonedGraph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Cloned graph to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pNode : :py:obj:`~.cudaGraphNode_t`
        Returns handle to the cloned node

    See Also
    --------
    :py:obj:`~.cudaGraphClone`
    """
    cdef ccudart.cudaGraph_t cclonedGraph
    if clonedGraph is None:
        cclonedGraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(clonedGraph, (cudaGraph_t,cuda.CUgraph)):
        pclonedGraph = int(clonedGraph)
        cclonedGraph = <ccudart.cudaGraph_t><void_ptr>pclonedGraph
    else:
        pclonedGraph = int(cudaGraph_t(clonedGraph))
        cclonedGraph = <ccudart.cudaGraph_t><void_ptr>pclonedGraph

    cdef ccudart.cudaGraphNode_t coriginalNode
    if originalNode is None:
        coriginalNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(originalNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        poriginalNode = int(originalNode)
        coriginalNode = <ccudart.cudaGraphNode_t><void_ptr>poriginalNode
    else:
        poriginalNode = int(cudaGraphNode_t(originalNode))
        coriginalNode = <ccudart.cudaGraphNode_t><void_ptr>poriginalNode

    cdef cudaGraphNode_t pNode = cudaGraphNode_t()
    err = ccudart.cudaGraphNodeFindInClone(<ccudart.cudaGraphNode_t*>pNode._ptr, coriginalNode, cclonedGraph)
    return (cudaError_t(err), pNode)

@cython.embedsignature(True)
def cudaGraphNodeGetType(node):
    """ Returns a node's type.

    Returns the node type of `node` in `pType`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to query

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pType : :py:obj:`~.cudaGraphNodeType`
        Pointer to return the node type

    See Also
    --------
    :py:obj:`~.cudaGraphGetNodes`, :py:obj:`~.cudaGraphGetRootNodes`, :py:obj:`~.cudaGraphChildGraphNodeGetGraph`, :py:obj:`~.cudaGraphKernelNodeGetParams`, :py:obj:`~.cudaGraphKernelNodeSetParams`, :py:obj:`~.cudaGraphHostNodeGetParams`, :py:obj:`~.cudaGraphHostNodeSetParams`, :py:obj:`~.cudaGraphMemcpyNodeGetParams`, :py:obj:`~.cudaGraphMemcpyNodeSetParams`, :py:obj:`~.cudaGraphMemsetNodeGetParams`, :py:obj:`~.cudaGraphMemsetNodeSetParams`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphNodeType pType
    err = ccudart.cudaGraphNodeGetType(cnode, &pType)
    return (cudaError_t(err), cudaGraphNodeType(pType))

@cython.embedsignature(True)
def cudaGraphGetNodes(graph, size_t numNodes = 0):
    """ Returns a graph's nodes.

    Returns a list of `graph's` nodes. `nodes` may be NULL, in which case
    this function will return the number of nodes in `numNodes`. Otherwise,
    `numNodes` entries will be filled in. If `numNodes` is higher than the
    actual number of nodes, the remaining entries in `nodes` will be set to
    NULL, and the number of nodes actually obtained will be returned in
    `numNodes`.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to query
    numNodes : int
        See description

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    nodes : List[:py:obj:`~.cudaGraphNode_t`]
        Pointer to return the nodes
    numNodes : int
        See description

    See Also
    --------
    :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphGetRootNodes`, :py:obj:`~.cudaGraphGetEdges`, :py:obj:`~.cudaGraphNodeGetType`, :py:obj:`~.cudaGraphNodeGetDependencies`, :py:obj:`~.cudaGraphNodeGetDependentNodes`
    """
    cdef size_t _graph_length = numNodes
    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef ccudart.cudaGraphNode_t* cnodes = NULL
    pynodes = []
    if _graph_length != 0:
        cnodes = <ccudart.cudaGraphNode_t*>calloc(_graph_length, sizeof(ccudart.cudaGraphNode_t))
        if cnodes is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(_graph_length) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
    err = ccudart.cudaGraphGetNodes(cgraph, cnodes, &numNodes)
    if cudaError_t(err) == cudaError_t(0):
        pynodes = [cudaGraphNode_t(init_value=<void_ptr>cnodes[idx]) for idx in range(_graph_length)]
    if cnodes is not NULL:
        free(cnodes)
    return (cudaError_t(err), pynodes, numNodes)

@cython.embedsignature(True)
def cudaGraphGetRootNodes(graph, size_t pNumRootNodes = 0):
    """ Returns a graph's root nodes.

    Returns a list of `graph's` root nodes. `pRootNodes` may be NULL, in
    which case this function will return the number of root nodes in
    `pNumRootNodes`. Otherwise, `pNumRootNodes` entries will be filled in.
    If `pNumRootNodes` is higher than the actual number of root nodes, the
    remaining entries in `pRootNodes` will be set to NULL, and the number
    of nodes actually obtained will be returned in `pNumRootNodes`.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to query
    pNumRootNodes : int
        See description

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pRootNodes : List[:py:obj:`~.cudaGraphNode_t`]
        Pointer to return the root nodes
    pNumRootNodes : int
        See description

    See Also
    --------
    :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphGetNodes`, :py:obj:`~.cudaGraphGetEdges`, :py:obj:`~.cudaGraphNodeGetType`, :py:obj:`~.cudaGraphNodeGetDependencies`, :py:obj:`~.cudaGraphNodeGetDependentNodes`
    """
    cdef size_t _graph_length = pNumRootNodes
    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef ccudart.cudaGraphNode_t* cpRootNodes = NULL
    pypRootNodes = []
    if _graph_length != 0:
        cpRootNodes = <ccudart.cudaGraphNode_t*>calloc(_graph_length, sizeof(ccudart.cudaGraphNode_t))
        if cpRootNodes is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(_graph_length) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
    err = ccudart.cudaGraphGetRootNodes(cgraph, cpRootNodes, &pNumRootNodes)
    if cudaError_t(err) == cudaError_t(0):
        pypRootNodes = [cudaGraphNode_t(init_value=<void_ptr>cpRootNodes[idx]) for idx in range(_graph_length)]
    if cpRootNodes is not NULL:
        free(cpRootNodes)
    return (cudaError_t(err), pypRootNodes, pNumRootNodes)

@cython.embedsignature(True)
def cudaGraphGetEdges(graph, size_t numEdges = 0):
    """ Returns a graph's dependency edges.

    Returns a list of `graph's` dependency edges. Edges are returned via
    corresponding indices in `from` and `to`; that is, the node in `to`[i]
    has a dependency on the node in `from`[i]. `from` and `to` may both be
    NULL, in which case this function only returns the number of edges in
    `numEdges`. Otherwise, `numEdges` entries will be filled in. If
    `numEdges` is higher than the actual number of edges, the remaining
    entries in `from` and `to` will be set to NULL, and the number of edges
    actually returned will be written to `numEdges`.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to get the edges from
    numEdges : int
        See description

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    from : List[:py:obj:`~.cudaGraphNode_t`]
        Location to return edge endpoints
    to : List[:py:obj:`~.cudaGraphNode_t`]
        Location to return edge endpoints
    numEdges : int
        See description

    See Also
    --------
    :py:obj:`~.cudaGraphGetNodes`, :py:obj:`~.cudaGraphGetRootNodes`, :py:obj:`~.cudaGraphAddDependencies`, :py:obj:`~.cudaGraphRemoveDependencies`, :py:obj:`~.cudaGraphNodeGetDependencies`, :py:obj:`~.cudaGraphNodeGetDependentNodes`
    """
    cdef size_t _graph_length = numEdges
    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef ccudart.cudaGraphNode_t* cfrom_ = NULL
    pyfrom_ = []
    if _graph_length != 0:
        cfrom_ = <ccudart.cudaGraphNode_t*>calloc(_graph_length, sizeof(ccudart.cudaGraphNode_t))
        if cfrom_ is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(_graph_length) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
    cdef ccudart.cudaGraphNode_t* cto = NULL
    pyto = []
    if _graph_length != 0:
        cto = <ccudart.cudaGraphNode_t*>calloc(_graph_length, sizeof(ccudart.cudaGraphNode_t))
        if cto is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(_graph_length) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
    err = ccudart.cudaGraphGetEdges(cgraph, cfrom_, cto, &numEdges)
    if cudaError_t(err) == cudaError_t(0):
        pyfrom_ = [cudaGraphNode_t(init_value=<void_ptr>cfrom_[idx]) for idx in range(_graph_length)]
    if cfrom_ is not NULL:
        free(cfrom_)
    if cudaError_t(err) == cudaError_t(0):
        pyto = [cudaGraphNode_t(init_value=<void_ptr>cto[idx]) for idx in range(_graph_length)]
    if cto is not NULL:
        free(cto)
    return (cudaError_t(err), pyfrom_, pyto, numEdges)

@cython.embedsignature(True)
def cudaGraphNodeGetDependencies(node, size_t pNumDependencies = 0):
    """ Returns a node's dependencies.

    Returns a list of `node's` dependencies. `pDependencies` may be NULL,
    in which case this function will return the number of dependencies in
    `pNumDependencies`. Otherwise, `pNumDependencies` entries will be
    filled in. If `pNumDependencies` is higher than the actual number of
    dependencies, the remaining entries in `pDependencies` will be set to
    NULL, and the number of nodes actually obtained will be returned in
    `pNumDependencies`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to query
    pNumDependencies : int
        See description

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pDependencies : List[:py:obj:`~.cudaGraphNode_t`]
        Pointer to return the dependencies
    pNumDependencies : int
        See description

    See Also
    --------
    :py:obj:`~.cudaGraphNodeGetDependentNodes`, :py:obj:`~.cudaGraphGetNodes`, :py:obj:`~.cudaGraphGetRootNodes`, :py:obj:`~.cudaGraphGetEdges`, :py:obj:`~.cudaGraphAddDependencies`, :py:obj:`~.cudaGraphRemoveDependencies`
    """
    cdef size_t _graph_length = pNumDependencies
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphNode_t* cpDependencies = NULL
    pypDependencies = []
    if _graph_length != 0:
        cpDependencies = <ccudart.cudaGraphNode_t*>calloc(_graph_length, sizeof(ccudart.cudaGraphNode_t))
        if cpDependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(_graph_length) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
    err = ccudart.cudaGraphNodeGetDependencies(cnode, cpDependencies, &pNumDependencies)
    if cudaError_t(err) == cudaError_t(0):
        pypDependencies = [cudaGraphNode_t(init_value=<void_ptr>cpDependencies[idx]) for idx in range(_graph_length)]
    if cpDependencies is not NULL:
        free(cpDependencies)
    return (cudaError_t(err), pypDependencies, pNumDependencies)

@cython.embedsignature(True)
def cudaGraphNodeGetDependentNodes(node, size_t pNumDependentNodes = 0):
    """ Returns a node's dependent nodes.

    Returns a list of `node's` dependent nodes. `pDependentNodes` may be
    NULL, in which case this function will return the number of dependent
    nodes in `pNumDependentNodes`. Otherwise, `pNumDependentNodes` entries
    will be filled in. If `pNumDependentNodes` is higher than the actual
    number of dependent nodes, the remaining entries in `pDependentNodes`
    will be set to NULL, and the number of nodes actually obtained will be
    returned in `pNumDependentNodes`.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to query
    pNumDependentNodes : int
        See description

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pDependentNodes : List[:py:obj:`~.cudaGraphNode_t`]
        Pointer to return the dependent nodes
    pNumDependentNodes : int
        See description

    See Also
    --------
    :py:obj:`~.cudaGraphNodeGetDependencies`, :py:obj:`~.cudaGraphGetNodes`, :py:obj:`~.cudaGraphGetRootNodes`, :py:obj:`~.cudaGraphGetEdges`, :py:obj:`~.cudaGraphAddDependencies`, :py:obj:`~.cudaGraphRemoveDependencies`
    """
    cdef size_t _graph_length = pNumDependentNodes
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphNode_t* cpDependentNodes = NULL
    pypDependentNodes = []
    if _graph_length != 0:
        cpDependentNodes = <ccudart.cudaGraphNode_t*>calloc(_graph_length, sizeof(ccudart.cudaGraphNode_t))
        if cpDependentNodes is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(_graph_length) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
    err = ccudart.cudaGraphNodeGetDependentNodes(cnode, cpDependentNodes, &pNumDependentNodes)
    if cudaError_t(err) == cudaError_t(0):
        pypDependentNodes = [cudaGraphNode_t(init_value=<void_ptr>cpDependentNodes[idx]) for idx in range(_graph_length)]
    if cpDependentNodes is not NULL:
        free(cpDependentNodes)
    return (cudaError_t(err), pypDependentNodes, pNumDependentNodes)

@cython.embedsignature(True)
def cudaGraphAddDependencies(graph, from_ : List[cudaGraphNode_t], to : List[cudaGraphNode_t], size_t numDependencies):
    """ Adds dependency edges to a graph.

    The number of dependencies to be added is defined by `numDependencies`
    Elements in `pFrom` and `pTo` at corresponding indices define a
    dependency. Each node in `pFrom` and `pTo` must belong to `graph`.

    If `numDependencies` is 0, elements in `pFrom` and `pTo` will be
    ignored. Specifying an existing dependency will return an error.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to which dependencies are added
    from : List[:py:obj:`~.cudaGraphNode_t`]
        Array of nodes that provide the dependencies
    to : List[:py:obj:`~.cudaGraphNode_t`]
        Array of dependent nodes
    numDependencies : size_t
        Number of dependencies to be added

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphRemoveDependencies`, :py:obj:`~.cudaGraphGetEdges`, :py:obj:`~.cudaGraphNodeGetDependencies`, :py:obj:`~.cudaGraphNodeGetDependentNodes`
    """
    to = [] if to is None else to
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in to):
        raise TypeError("Argument 'to' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    from_ = [] if from_ is None else from_
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in from_):
        raise TypeError("Argument 'from_' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef ccudart.cudaGraphNode_t* cfrom_ = NULL
    if len(from_) > 0:
        cfrom_ = <ccudart.cudaGraphNode_t*> calloc(len(from_), sizeof(ccudart.cudaGraphNode_t))
        if cfrom_ is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(from_)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(from_)):
                cfrom_[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>from_[idx])._ptr[0]

    cdef ccudart.cudaGraphNode_t* cto = NULL
    if len(to) > 0:
        cto = <ccudart.cudaGraphNode_t*> calloc(len(to), sizeof(ccudart.cudaGraphNode_t))
        if cto is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(to)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(to)):
                cto[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>to[idx])._ptr[0]

    if numDependencies > <size_t>len(from_): raise RuntimeError("List is too small: " + str(len(from_)) + " < " + str(numDependencies))
    if numDependencies > <size_t>len(to): raise RuntimeError("List is too small: " + str(len(to)) + " < " + str(numDependencies))
    err = ccudart.cudaGraphAddDependencies(cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>from_[0])._ptr if len(from_) == 1 else cfrom_, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>to[0])._ptr if len(to) == 1 else cto, numDependencies)
    if cfrom_ is not NULL:
        free(cfrom_)
    if cto is not NULL:
        free(cto)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphRemoveDependencies(graph, from_ : List[cudaGraphNode_t], to : List[cudaGraphNode_t], size_t numDependencies):
    """ Removes dependency edges from a graph.

    The number of `pDependencies` to be removed is defined by
    `numDependencies`. Elements in `pFrom` and `pTo` at corresponding
    indices define a dependency. Each node in `pFrom` and `pTo` must belong
    to `graph`.

    If `numDependencies` is 0, elements in `pFrom` and `pTo` will be
    ignored. Specifying a non-existing dependency will return an error.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph from which to remove dependencies
    from : List[:py:obj:`~.cudaGraphNode_t`]
        Array of nodes that provide the dependencies
    to : List[:py:obj:`~.cudaGraphNode_t`]
        Array of dependent nodes
    numDependencies : size_t
        Number of dependencies to be removed

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddDependencies`, :py:obj:`~.cudaGraphGetEdges`, :py:obj:`~.cudaGraphNodeGetDependencies`, :py:obj:`~.cudaGraphNodeGetDependentNodes`
    """
    to = [] if to is None else to
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in to):
        raise TypeError("Argument 'to' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    from_ = [] if from_ is None else from_
    if not all(isinstance(_x, (cudaGraphNode_t, cuda.CUgraphNode)) for _x in from_):
        raise TypeError("Argument 'from_' is not instance of type (expected List[ccudart.cudaGraphNode_t, cuda.CUgraphNode]")

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef ccudart.cudaGraphNode_t* cfrom_ = NULL
    if len(from_) > 0:
        cfrom_ = <ccudart.cudaGraphNode_t*> calloc(len(from_), sizeof(ccudart.cudaGraphNode_t))
        if cfrom_ is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(from_)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(from_)):
                cfrom_[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>from_[idx])._ptr[0]

    cdef ccudart.cudaGraphNode_t* cto = NULL
    if len(to) > 0:
        cto = <ccudart.cudaGraphNode_t*> calloc(len(to), sizeof(ccudart.cudaGraphNode_t))
        if cto is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(to)) + 'x' + str(sizeof(ccudart.cudaGraphNode_t)))
        else:
            for idx in range(len(to)):
                cto[idx] = <ccudart.cudaGraphNode_t>(<cudaGraphNode_t>to[idx])._ptr[0]

    if numDependencies > <size_t>len(from_): raise RuntimeError("List is too small: " + str(len(from_)) + " < " + str(numDependencies))
    if numDependencies > <size_t>len(to): raise RuntimeError("List is too small: " + str(len(to)) + " < " + str(numDependencies))
    err = ccudart.cudaGraphRemoveDependencies(cgraph, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>from_[0])._ptr if len(from_) == 1 else cfrom_, <ccudart.cudaGraphNode_t*>(<cudaGraphNode_t>to[0])._ptr if len(to) == 1 else cto, numDependencies)
    if cfrom_ is not NULL:
        free(cfrom_)
    if cto is not NULL:
        free(cto)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphDestroyNode(node):
    """ Remove a node from the graph.

    Removes `node` from its graph. This operation also severs any
    dependencies of other nodes on `node` and vice versa.

    Dependencies cannot be removed from graphs which contain allocation or
    free nodes. Any attempt to do so will return an error.

    Parameters
    ----------
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node to remove

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphAddEmptyNode`, :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemsetNode`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    err = ccudart.cudaGraphDestroyNode(cnode)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphInstantiate(graph, char* pLogBuffer, size_t bufferSize):
    """ Creates an executable graph from a graph.

    Instantiates `graph` as an executable graph. The graph is validated for
    any structural constraints or intra-node constraints which were not
    previously validated. If instantiation is successful, a handle to the
    instantiated graph is returned in `pGraphExec`.

    If there are any errors, diagnostic information may be returned in
    `pErrorNode` and `pLogBuffer`. This is the primary way to inspect
    instantiation errors. The output will be null terminated unless the
    diagnostics overflow the buffer. In this case, they will be truncated,
    and the last byte can be inspected to determine if truncation occurred.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to instantiate
    pLogBuffer : bytes
        A character buffer to store diagnostic messages
    bufferSize : size_t
        Size of the log buffer in bytes

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphExec : :py:obj:`~.cudaGraphExec_t`
        Returns instantiated graph
    pErrorNode : :py:obj:`~.cudaGraphNode_t`
        In case of an instantiation error, this may be modified to indicate
        a node contributing to the error

    See Also
    --------
    :py:obj:`~.cudaGraphInstantiateWithFlags`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphUpload`, :py:obj:`~.cudaGraphLaunch`, :py:obj:`~.cudaGraphExecDestroy`
    """
    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphExec_t pGraphExec = cudaGraphExec_t()
    cdef cudaGraphNode_t pErrorNode = cudaGraphNode_t()
    err = ccudart.cudaGraphInstantiate(<ccudart.cudaGraphExec_t*>pGraphExec._ptr, cgraph, <ccudart.cudaGraphNode_t*>pErrorNode._ptr, pLogBuffer, bufferSize)
    return (cudaError_t(err), pGraphExec, pErrorNode)

@cython.embedsignature(True)
def cudaGraphInstantiateWithFlags(graph, unsigned long long flags):
    """ Creates an executable graph from a graph.

    Instantiates `graph` as an executable graph. The graph is validated for
    any structural constraints or intra-node constraints which were not
    previously validated. If instantiation is successful, a handle to the
    instantiated graph is returned in `pGraphExec`.

    The `flags` parameter controls the behavior of instantiation and
    subsequent graph launches. Valid flags are:

    - :py:obj:`~.cudaGraphInstantiateFlagAutoFreeOnLaunch`, which
      configures a graph containing memory allocation nodes to
      automatically free any unfreed memory allocations before the graph is
      relaunched.

    If `graph` contains any allocation or free nodes, there can be at most
    one executable graph in existence for that graph at a time.

    An attempt to instantiate a second executable graph before destroying
    the first with :py:obj:`~.cudaGraphExecDestroy` will result in an
    error.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to instantiate
    flags : unsigned long long
        Flags to control instantiation. See
        :py:obj:`~.CUgraphInstantiate_flags`.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    pGraphExec : :py:obj:`~.cudaGraphExec_t`
        Returns instantiated graph

    See Also
    --------
    :py:obj:`~.cudaGraphInstantiate`, :py:obj:`~.cudaGraphCreate`, :py:obj:`~.cudaGraphUpload`, :py:obj:`~.cudaGraphLaunch`, :py:obj:`~.cudaGraphExecDestroy`
    """
    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    cdef cudaGraphExec_t pGraphExec = cudaGraphExec_t()
    err = ccudart.cudaGraphInstantiateWithFlags(<ccudart.cudaGraphExec_t*>pGraphExec._ptr, cgraph, flags)
    return (cudaError_t(err), pGraphExec)

@cython.embedsignature(True)
def cudaGraphExecKernelNodeSetParams(hGraphExec, node, pNodeParams : cudaKernelNodeParams):
    """ Sets the parameters for a kernel node in the given graphExec.

    Sets the parameters of a kernel node in an executable graph
    `hGraphExec`. The node is identified by the corresponding node `node`
    in the non-executable graph, from which the executable graph was
    instantiated.

    `node` must not have been removed from the original graph. The `func`
    field of `nodeParams` cannot be modified and must match the original
    value. All other values can be modified.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `node` is also not modified by this call.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        kernel node from the graph from which graphExec was instantiated
    pNodeParams : :py:obj:`~.cudaKernelNodeParams`
        Updated Parameters to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddKernelNode`, :py:obj:`~.cudaGraphKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef ccudart.cudaKernelNodeParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphExecKernelNodeSetParams(chGraphExec, cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecMemcpyNodeSetParams(hGraphExec, node, pNodeParams : cudaMemcpy3DParms):
    """ Sets the parameters for a memcpy node in the given graphExec.

    Updates the work represented by `node` in `hGraphExec` as though `node`
    had contained `pNodeParams` at instantiation. `node` must remain in the
    graph which was used to instantiate `hGraphExec`. Changed edges to and
    from `node` are ignored.

    The source and destination memory in `pNodeParams` must be allocated
    from the same contexts as the original source and destination memory.
    Both the instantiation-time memory operands and the memory operands in
    `pNodeParams` must be 1-dimensional. Zero-length operations are not
    supported.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `node` is also not modified by this call.

    Returns :py:obj:`~.cudaErrorInvalidValue` if the memory operands'
    mappings changed or either the original or new memory operands are
    multidimensional.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Memcpy node from the graph which was used to instantiate graphExec
    pNodeParams : :py:obj:`~.cudaMemcpy3DParms`
        Updated Parameters to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParamsToSymbol`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParamsFromSymbol`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams1D`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef ccudart.cudaMemcpy3DParms* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphExecMemcpyNodeSetParams(chGraphExec, cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecMemcpyNodeSetParams1D(hGraphExec, node, dst, src, size_t count, kind not None : cudaMemcpyKind):
    """ Sets the parameters for a memcpy node in the given graphExec to perform a 1-dimensional copy.

    Updates the work represented by `node` in `hGraphExec` as though `node`
    had contained the given params at instantiation. `node` must remain in
    the graph which was used to instantiate `hGraphExec`. Changed edges to
    and from `node` are ignored.

    `src` and `dst` must be allocated from the same contexts as the
    original source and destination memory. The instantiation-time memory
    operands must be 1-dimensional. Zero-length operations are not
    supported.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `node` is also not modified by this call.

    Returns :py:obj:`~.cudaErrorInvalidValue` if the memory operands'
    mappings changed or the original memory operands are multidimensional.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Memcpy node from the graph which was used to instantiate graphExec
    dst : Any
        Destination memory address
    src : Any
        Source memory address
    count : size_t
        Size in bytes to copy
    kind : :py:obj:`~.cudaMemcpyKind`
        Type of transfer

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemcpyNode`, :py:obj:`~.cudaGraphAddMemcpyNode1D`, :py:obj:`~.cudaGraphMemcpyNodeSetParams`, :py:obj:`~.cudaGraphMemcpyNodeSetParams1D`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdst = utils.HelperInputVoidPtr(dst)
    cdef void* cdst_ptr = <void*><void_ptr>cdst.cptr
    csrc = utils.HelperInputVoidPtr(src)
    cdef void* csrc_ptr = <void*><void_ptr>csrc.cptr
    cdef ccudart.cudaMemcpyKind ckind = kind.value
    err = ccudart.cudaGraphExecMemcpyNodeSetParams1D(chGraphExec, cnode, cdst_ptr, csrc_ptr, count, ckind)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecMemsetNodeSetParams(hGraphExec, node, pNodeParams : cudaMemsetParams):
    """ Sets the parameters for a memset node in the given graphExec.

    Updates the work represented by `node` in `hGraphExec` as though `node`
    had contained `pNodeParams` at instantiation. `node` must remain in the
    graph which was used to instantiate `hGraphExec`. Changed edges to and
    from `node` are ignored.

    The destination memory in `pNodeParams` must be allocated from the same
    context as the original destination memory. Both the instantiation-time
    memory operand and the memory operand in `pNodeParams` must be
    1-dimensional. Zero-length operations are not supported.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `node` is also not modified by this call.

    Returns cudaErrorInvalidValue if the memory operand's mappings changed
    or either the original or new memory operand are multidimensional.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Memset node from the graph which was used to instantiate graphExec
    pNodeParams : :py:obj:`~.cudaMemsetParams`
        Updated Parameters to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddMemsetNode`, :py:obj:`~.cudaGraphMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef ccudart.cudaMemsetParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphExecMemsetNodeSetParams(chGraphExec, cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecHostNodeSetParams(hGraphExec, node, pNodeParams : cudaHostNodeParams):
    """ Sets the parameters for a host node in the given graphExec.

    Updates the work represented by `node` in `hGraphExec` as though `node`
    had contained `pNodeParams` at instantiation. `node` must remain in the
    graph which was used to instantiate `hGraphExec`. Changed edges to and
    from `node` are ignored.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `node` is also not modified by this call.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Host node from the graph which was used to instantiate graphExec
    pNodeParams : :py:obj:`~.cudaHostNodeParams`
        Updated Parameters to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddHostNode`, :py:obj:`~.cudaGraphHostNodeSetParams`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef ccudart.cudaHostNodeParams* cpNodeParams_ptr = pNodeParams._ptr if pNodeParams != None else NULL
    err = ccudart.cudaGraphExecHostNodeSetParams(chGraphExec, cnode, cpNodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecChildGraphNodeSetParams(hGraphExec, node, childGraph):
    """ Updates node parameters in the child graph node in the given graphExec.

    Updates the work represented by `node` in `hGraphExec` as though the
    nodes contained in `node's` graph had the parameters contained in
    `childGraph's` nodes at instantiation. `node` must remain in the graph
    which was used to instantiate `hGraphExec`. Changed edges to and from
    `node` are ignored.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `node` is also not modified by this call.

    The topology of `childGraph`, as well as the node insertion order, must
    match that of the graph contained in `node`. See
    :py:obj:`~.cudaGraphExecUpdate()` for a list of restrictions on what
    can be updated in an instantiated graph. The update is recursive, so
    child graph nodes contained within the top level child graph will also
    be updated.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    node : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Host node from the graph which was used to instantiate graphExec
    childGraph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        The graph supplying the updated parameters

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddChildGraphNode`, :py:obj:`~.cudaGraphChildGraphNodeGetGraph`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraph_t cchildGraph
    if childGraph is None:
        cchildGraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(childGraph, (cudaGraph_t,cuda.CUgraph)):
        pchildGraph = int(childGraph)
        cchildGraph = <ccudart.cudaGraph_t><void_ptr>pchildGraph
    else:
        pchildGraph = int(cudaGraph_t(childGraph))
        cchildGraph = <ccudart.cudaGraph_t><void_ptr>pchildGraph

    cdef ccudart.cudaGraphNode_t cnode
    if node is None:
        cnode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(node, (cudaGraphNode_t,cuda.CUgraphNode)):
        pnode = int(node)
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode
    else:
        pnode = int(cudaGraphNode_t(node))
        cnode = <ccudart.cudaGraphNode_t><void_ptr>pnode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    err = ccudart.cudaGraphExecChildGraphNodeSetParams(chGraphExec, cnode, cchildGraph)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecEventRecordNodeSetEvent(hGraphExec, hNode, event):
    """ Sets the event for an event record node in the given graphExec.

    Sets the event of an event record node in an executable graph
    `hGraphExec`. The node is identified by the corresponding node `hNode`
    in the non-executable graph, from which the executable graph was
    instantiated.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `hNode` is also not modified by this call.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Event record node from the graph from which graphExec was
        instantiated
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Updated event to use

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventRecordNode`, :py:obj:`~.cudaGraphEventRecordNodeGetEvent`, :py:obj:`~.cudaGraphEventWaitNodeSetEvent`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    err = ccudart.cudaGraphExecEventRecordNodeSetEvent(chGraphExec, chNode, cevent)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecEventWaitNodeSetEvent(hGraphExec, hNode, event):
    """ Sets the event for an event wait node in the given graphExec.

    Sets the event of an event wait node in an executable graph
    `hGraphExec`. The node is identified by the corresponding node `hNode`
    in the non-executable graph, from which the executable graph was
    instantiated.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `hNode` is also not modified by this call.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Event wait node from the graph from which graphExec was
        instantiated
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Updated event to use

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddEventWaitNode`, :py:obj:`~.cudaGraphEventWaitNodeGetEvent`, :py:obj:`~.cudaGraphEventRecordNodeSetEvent`, :py:obj:`~.cudaEventRecordWithFlags`, :py:obj:`~.cudaStreamWaitEvent`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaEvent_t cevent
    if event is None:
        cevent = <ccudart.cudaEvent_t><void_ptr>0
    elif isinstance(event, (cudaEvent_t,cuda.CUevent)):
        pevent = int(event)
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent
    else:
        pevent = int(cudaEvent_t(event))
        cevent = <ccudart.cudaEvent_t><void_ptr>pevent

    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    err = ccudart.cudaGraphExecEventWaitNodeSetEvent(chGraphExec, chNode, cevent)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecExternalSemaphoresSignalNodeSetParams(hGraphExec, hNode, nodeParams : cudaExternalSemaphoreSignalNodeParams):
    """ Sets the parameters for an external semaphore signal node in the given graphExec.

    Sets the parameters of an external semaphore signal node in an
    executable graph `hGraphExec`. The node is identified by the
    corresponding node `hNode` in the non-executable graph, from which the
    executable graph was instantiated.

    `hNode` must not have been removed from the original graph.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `hNode` is also not modified by this call.

    Changing `nodeParams->numExtSems` is not supported.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        semaphore signal node from the graph from which graphExec was
        instantiated
    nodeParams : :py:obj:`~.cudaExternalSemaphoreSignalNodeParams`
        Updated Parameters to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddExternalSemaphoresSignalNode`, :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresWaitNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef ccudart.cudaExternalSemaphoreSignalNodeParams* cnodeParams_ptr = nodeParams._ptr if nodeParams != None else NULL
    err = ccudart.cudaGraphExecExternalSemaphoresSignalNodeSetParams(chGraphExec, chNode, cnodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecExternalSemaphoresWaitNodeSetParams(hGraphExec, hNode, nodeParams : cudaExternalSemaphoreWaitNodeParams):
    """ Sets the parameters for an external semaphore wait node in the given graphExec.

    Sets the parameters of an external semaphore wait node in an executable
    graph `hGraphExec`. The node is identified by the corresponding node
    `hNode` in the non-executable graph, from which the executable graph
    was instantiated.

    `hNode` must not have been removed from the original graph.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `hNode` is also not modified by this call.

    Changing `nodeParams->numExtSems` is not supported.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        semaphore wait node from the graph from which graphExec was
        instantiated
    nodeParams : :py:obj:`~.cudaExternalSemaphoreWaitNodeParams`
        Updated Parameters to set

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphAddExternalSemaphoresWaitNode`, :py:obj:`~.cudaImportExternalSemaphore`, :py:obj:`~.cudaSignalExternalSemaphoresAsync`, :py:obj:`~.cudaWaitExternalSemaphoresAsync`, :py:obj:`~.cudaGraphExecKernelNodeSetParams`, :py:obj:`~.cudaGraphExecMemcpyNodeSetParams`, :py:obj:`~.cudaGraphExecMemsetNodeSetParams`, :py:obj:`~.cudaGraphExecHostNodeSetParams`, :py:obj:`~.cudaGraphExecChildGraphNodeSetParams`, :py:obj:`~.cudaGraphExecEventRecordNodeSetEvent`, :py:obj:`~.cudaGraphExecEventWaitNodeSetEvent`, :py:obj:`~.cudaGraphExecExternalSemaphoresSignalNodeSetParams`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate`
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef ccudart.cudaExternalSemaphoreWaitNodeParams* cnodeParams_ptr = nodeParams._ptr if nodeParams != None else NULL
    err = ccudart.cudaGraphExecExternalSemaphoresWaitNodeSetParams(chGraphExec, chNode, cnodeParams_ptr)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphNodeSetEnabled(hGraphExec, hNode, unsigned int isEnabled):
    """ Enables or disables the specified node in the given graphExec.

    Sets `hNode` to be either enabled or disabled. Disabled nodes are
    functionally equivalent to empty nodes until they are reenabled.
    Existing node parameters are not affected by disabling/enabling the
    node.

    The node is identified by the corresponding node `hNode` in the non-
    executable graph, from which the executable graph was instantiated.

    `hNode` must not have been removed from the original graph.

    The modifications only affect future launches of `hGraphExec`. Already
    enqueued or running launches of `hGraphExec` are not affected by this
    call. `hNode` is also not modified by this call.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node from the graph from which graphExec was instantiated
    isEnabled : unsigned int
        Node is enabled if != 0, otherwise the node is disabled

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphNodeGetEnabled`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate` :py:obj:`~.cudaGraphLaunch`

    Notes
    -----
    Currently only kernel, memset and memcpy nodes are supported.
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    err = ccudart.cudaGraphNodeSetEnabled(chGraphExec, chNode, isEnabled)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphNodeGetEnabled(hGraphExec, hNode):
    """ Query whether a node in the given graphExec is enabled.

    Sets isEnabled to 1 if `hNode` is enabled, or 0 if `hNode` is disabled.

    The node is identified by the corresponding node `hNode` in the non-
    executable graph, from which the executable graph was instantiated.

    `hNode` must not have been removed from the original graph.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The executable graph in which to set the specified node
    hNode : :py:obj:`~.CUgraphNode` or :py:obj:`~.cudaGraphNode_t`
        Node from the graph from which graphExec was instantiated

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,
    isEnabled : unsigned int
        Location to return the enabled status of the node

    See Also
    --------
    :py:obj:`~.cudaGraphNodeSetEnabled`, :py:obj:`~.cudaGraphExecUpdate`, :py:obj:`~.cudaGraphInstantiate` :py:obj:`~.cudaGraphLaunch`

    Notes
    -----
    Currently only kernel, memset and memcpy nodes are supported.
    """
    cdef ccudart.cudaGraphNode_t chNode
    if hNode is None:
        chNode = <ccudart.cudaGraphNode_t><void_ptr>0
    elif isinstance(hNode, (cudaGraphNode_t,cuda.CUgraphNode)):
        phNode = int(hNode)
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode
    else:
        phNode = int(cudaGraphNode_t(hNode))
        chNode = <ccudart.cudaGraphNode_t><void_ptr>phNode

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef unsigned int isEnabled = 0
    err = ccudart.cudaGraphNodeGetEnabled(chGraphExec, chNode, &isEnabled)
    return (cudaError_t(err), isEnabled)

@cython.embedsignature(True)
def cudaGraphExecUpdate(hGraphExec, hGraph):
    """ Check whether an executable graph can be updated with a graph and perform the update if possible.

    Updates the node parameters in the instantiated graph specified by
    `hGraphExec` with the node parameters in a topologically identical
    graph specified by `hGraph`.

    Limitations:

    - Kernel nodes:

      - The owning context of the function cannot change.

      - A node whose function originally did not use CUDA dynamic
        parallelism cannot be updated to a function which uses CDP.

      - A cooperative node cannot be updated to a non-cooperative node, and
        vice-versa.

    - Memset and memcpy nodes:

      - The CUDA device(s) to which the operand(s) was allocated/mapped
        cannot change.

      - The source/destination memory must be allocated from the same
        contexts as the original source/destination memory.

      - Only 1D memsets can be changed.

    - Additional memcpy node restrictions:

      - Changing either the source or destination memory type(i.e.
        CU_MEMORYTYPE_DEVICE, CU_MEMORYTYPE_ARRAY, etc.) is not supported.

    Note: The API may add further restrictions in future releases. The
    return code should always be checked.

    cudaGraphExecUpdate sets `updateResult_out` to
    cudaGraphExecUpdateErrorTopologyChanged under the following conditions:

    - The count of nodes directly in `hGraphExec` and `hGraph` differ, in
      which case `hErrorNode_out` is NULL.

    - A node is deleted in `hGraph` but not not its pair from `hGraphExec`,
      in which case `hErrorNode_out` is NULL.

    - A node is deleted in `hGraphExec` but not its pair from `hGraph`, in
      which case `hErrorNode_out` is the pairless node from `hGraph`.

    - The dependent nodes of a pair differ, in which case `hErrorNode_out`
      is the node from `hGraph`.

    cudaGraphExecUpdate sets `updateResult_out` to:

    - cudaGraphExecUpdateError if passed an invalid value.

    - cudaGraphExecUpdateErrorTopologyChanged if the graph topology changed

    - cudaGraphExecUpdateErrorNodeTypeChanged if the type of a node
      changed, in which case `hErrorNode_out` is set to the node from
      `hGraph`.

    - cudaGraphExecUpdateErrorFunctionChanged if the function of a kernel
      node changed (CUDA driver < 11.2)

    - cudaGraphExecUpdateErrorUnsupportedFunctionChange if the func field
      of a kernel changed in an unsupported way(see note above), in which
      case `hErrorNode_out` is set to the node from `hGraph`

    - cudaGraphExecUpdateErrorParametersChanged if any parameters to a node
      changed in a way that is not supported, in which case
      `hErrorNode_out` is set to the node from `hGraph`

    - cudaGraphExecUpdateErrorAttributesChanged if any attributes of a node
      changed in a way that is not supported, in which case
      `hErrorNode_out` is set to the node from `hGraph`

    - cudaGraphExecUpdateErrorNotSupported if something about a node is
      unsupported, like the node's type or configuration, in which case
      `hErrorNode_out` is set to the node from `hGraph`

    If `updateResult_out` isn't set in one of the situations described
    above, the update check passes and cudaGraphExecUpdate updates
    `hGraphExec` to match the contents of `hGraph`. If an error happens
    during the update, `updateResult_out` will be set to
    cudaGraphExecUpdateError; otherwise, `updateResult_out` is set to
    cudaGraphExecUpdateSuccess.

    cudaGraphExecUpdate returns cudaSuccess when the updated was performed
    successfully. It returns cudaErrorGraphExecUpdateFailure if the graph
    update was not performed because it included changes which violated
    constraints specific to instantiated graph update.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        The instantiated graph to be updated
    hGraph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        The graph containing the updated parameters

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorGraphExecUpdateFailure`,
    hErrorNode_out : :py:obj:`~.cudaGraphNode_t`
        The node which caused the permissibility check to forbid the
        update, if any
    updateResult_out : :py:obj:`~.cudaGraphExecUpdateResult`
        Whether the graph update was permitted. If was forbidden, the
        reason why

    See Also
    --------
    :py:obj:`~.cudaGraphInstantiate`,
    """
    cdef ccudart.cudaGraph_t chGraph
    if hGraph is None:
        chGraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(hGraph, (cudaGraph_t,cuda.CUgraph)):
        phGraph = int(hGraph)
        chGraph = <ccudart.cudaGraph_t><void_ptr>phGraph
    else:
        phGraph = int(cudaGraph_t(hGraph))
        chGraph = <ccudart.cudaGraph_t><void_ptr>phGraph

    cdef ccudart.cudaGraphExec_t chGraphExec
    if hGraphExec is None:
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(hGraphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        phGraphExec = int(hGraphExec)
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec
    else:
        phGraphExec = int(cudaGraphExec_t(hGraphExec))
        chGraphExec = <ccudart.cudaGraphExec_t><void_ptr>phGraphExec

    cdef cudaGraphNode_t hErrorNode_out = cudaGraphNode_t()
    cdef ccudart.cudaGraphExecUpdateResult updateResult_out
    err = ccudart.cudaGraphExecUpdate(chGraphExec, chGraph, <ccudart.cudaGraphNode_t*>hErrorNode_out._ptr, &updateResult_out)
    return (cudaError_t(err), hErrorNode_out, cudaGraphExecUpdateResult(updateResult_out))

@cython.embedsignature(True)
def cudaGraphUpload(graphExec, stream):
    """ Uploads an executable graph in a stream.

    Uploads `hGraphExec` to the device in `hStream` without executing it.
    Uploads of the same `hGraphExec` will be serialized. Each upload is
    ordered behind both any previous work in `hStream` and any previous
    launches of `hGraphExec`. Uses memory cached by `stream` to back the
    allocations owned by `graphExec`.

    Parameters
    ----------
    hGraphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        Executable graph to upload
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to upload the graph

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`,

    See Also
    --------
    :py:obj:`~.cudaGraphInstantiate`, :py:obj:`~.cudaGraphLaunch`, :py:obj:`~.cudaGraphExecDestroy`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaGraphExec_t cgraphExec
    if graphExec is None:
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(graphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        pgraphExec = int(graphExec)
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>pgraphExec
    else:
        pgraphExec = int(cudaGraphExec_t(graphExec))
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>pgraphExec

    err = ccudart.cudaGraphUpload(cgraphExec, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphLaunch(graphExec, stream):
    """ Launches an executable graph in a stream.

    Executes `graphExec` in `stream`. Only one instance of `graphExec` may
    be executing at a time. Each launch is ordered behind both any previous
    work in `stream` and any previous launches of `graphExec`. To execute a
    graph concurrently, it must be instantiated multiple times into
    multiple executable graphs.

    If any allocations created by `graphExec` remain unfreed (from a
    previous launch) and `graphExec` was not instantiated with
    :py:obj:`~.cudaGraphInstantiateFlagAutoFreeOnLaunch`, the launch will
    fail with :py:obj:`~.cudaErrorInvalidValue`.

    Parameters
    ----------
    graphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        Executable graph to launch
    stream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to launch the graph

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphInstantiate`, :py:obj:`~.cudaGraphUpload`, :py:obj:`~.cudaGraphExecDestroy`
    """
    cdef ccudart.cudaStream_t cstream
    if stream is None:
        cstream = <ccudart.cudaStream_t><void_ptr>0
    elif isinstance(stream, (cudaStream_t,cuda.CUstream)):
        pstream = int(stream)
        cstream = <ccudart.cudaStream_t><void_ptr>pstream
    else:
        pstream = int(cudaStream_t(stream))
        cstream = <ccudart.cudaStream_t><void_ptr>pstream

    cdef ccudart.cudaGraphExec_t cgraphExec
    if graphExec is None:
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(graphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        pgraphExec = int(graphExec)
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>pgraphExec
    else:
        pgraphExec = int(cudaGraphExec_t(graphExec))
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>pgraphExec

    err = ccudart.cudaGraphLaunch(cgraphExec, cstream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphExecDestroy(graphExec):
    """ Destroys an executable graph.

    Destroys the executable graph specified by `graphExec`.

    Parameters
    ----------
    graphExec : :py:obj:`~.CUgraphExec` or :py:obj:`~.cudaGraphExec_t`
        Executable graph to destroy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphInstantiate`, :py:obj:`~.cudaGraphUpload`, :py:obj:`~.cudaGraphLaunch`
    """
    cdef ccudart.cudaGraphExec_t cgraphExec
    if graphExec is None:
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>0
    elif isinstance(graphExec, (cudaGraphExec_t,cuda.CUgraphExec)):
        pgraphExec = int(graphExec)
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>pgraphExec
    else:
        pgraphExec = int(cudaGraphExec_t(graphExec))
        cgraphExec = <ccudart.cudaGraphExec_t><void_ptr>pgraphExec

    err = ccudart.cudaGraphExecDestroy(cgraphExec)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphDestroy(graph):
    """ Destroys a graph.

    Destroys the graph specified by `graph`, as well as all of its nodes.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        Graph to destroy

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaGraphCreate`
    """
    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    err = ccudart.cudaGraphDestroy(cgraph)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphDebugDotPrint(graph, char* path, unsigned int flags):
    """ Write a DOT file describing graph structure.

    Using the provided `graph`, write to `path` a DOT formatted description
    of the graph. By default this includes the graph topology, node types,
    node id, kernel names and memcpy direction. `flags` can be specified to
    write more detailed information about each node type such as parameter
    values, kernel attributes, node and function handles.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        The graph to create a DOT file from
    path : bytes
        The path to write the DOT file to
    flags : unsigned int
        Flags from cudaGraphDebugDotFlags for specifying which additional
        node information to write

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorOperatingSystem`
    """
    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    err = ccudart.cudaGraphDebugDotPrint(cgraph, path, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaUserObjectCreate(ptr, destroy, unsigned int initialRefcount, unsigned int flags):
    """ Create a user object.

    Create a user object with the specified destructor callback and initial
    reference count. The initial references are owned by the caller.

    Destructor callbacks cannot make CUDA API calls and should avoid
    blocking behavior, as they are executed by a shared internal thread.
    Another thread may be signaled to perform such actions, if it does not
    block forward progress of tasks scheduled through CUDA.

    See CUDA User Objects in the CUDA C++ Programming Guide for more
    information on user objects.

    Parameters
    ----------
    ptr : Any
        The pointer to pass to the destroy function
    destroy : :py:obj:`~.cudaHostFn_t`
        Callback to free the user object when it is no longer in use
    initialRefcount : unsigned int
        The initial refcount to create the object with, typically 1. The
        initial references are owned by the calling thread.
    flags : unsigned int
        Currently it is required to pass
        :py:obj:`~.cudaUserObjectNoDestructorSync`, which is the only
        defined flag. This indicates that the destroy callback cannot be
        waited on by any CUDA API. Users requiring synchronization of the
        callback should signal its completion manually.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`
    object_out : :py:obj:`~.cudaUserObject_t`
        Location to return the user object handle

    See Also
    --------
    :py:obj:`~.cudaUserObjectRetain`, :py:obj:`~.cudaUserObjectRelease`, :py:obj:`~.cudaGraphRetainUserObject`, :py:obj:`~.cudaGraphReleaseUserObject`, :py:obj:`~.cudaGraphCreate`
    """
    cdef ccudart.cudaHostFn_t cdestroy
    if destroy is None:
        cdestroy = <ccudart.cudaHostFn_t><void_ptr>0
    elif isinstance(destroy, (cudaHostFn_t)):
        pdestroy = int(destroy)
        cdestroy = <ccudart.cudaHostFn_t><void_ptr>pdestroy
    else:
        pdestroy = int(cudaHostFn_t(destroy))
        cdestroy = <ccudart.cudaHostFn_t><void_ptr>pdestroy

    cdef cudaUserObject_t object_out = cudaUserObject_t()
    cptr = utils.HelperInputVoidPtr(ptr)
    cdef void* cptr_ptr = <void*><void_ptr>cptr.cptr
    err = ccudart.cudaUserObjectCreate(<ccudart.cudaUserObject_t*>object_out._ptr, cptr_ptr, cdestroy, initialRefcount, flags)
    return (cudaError_t(err), object_out)

@cython.embedsignature(True)
def cudaUserObjectRetain(object, unsigned int count):
    """ Retain a reference to a user object.

    Retains new references to a user object. The new references are owned
    by the caller.

    See CUDA User Objects in the CUDA C++ Programming Guide for more
    information on user objects.

    Parameters
    ----------
    object : :py:obj:`~.cudaUserObject_t`
        The object to retain
    count : unsigned int
        The number of references to retain, typically 1. Must be nonzero
        and not larger than INT_MAX.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaUserObjectCreate`, :py:obj:`~.cudaUserObjectRelease`, :py:obj:`~.cudaGraphRetainUserObject`, :py:obj:`~.cudaGraphReleaseUserObject`, :py:obj:`~.cudaGraphCreate`
    """
    cdef ccudart.cudaUserObject_t cobject
    if object is None:
        cobject = <ccudart.cudaUserObject_t><void_ptr>0
    elif isinstance(object, (cudaUserObject_t,cuda.CUuserObject)):
        pobject = int(object)
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject
    else:
        pobject = int(cudaUserObject_t(object))
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject

    err = ccudart.cudaUserObjectRetain(cobject, count)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaUserObjectRelease(object, unsigned int count):
    """ Release a reference to a user object.

    Releases user object references owned by the caller. The object's
    destructor is invoked if the reference count reaches zero.

    It is undefined behavior to release references not owned by the caller,
    or to use a user object handle after all references are released.

    See CUDA User Objects in the CUDA C++ Programming Guide for more
    information on user objects.

    Parameters
    ----------
    object : :py:obj:`~.cudaUserObject_t`
        The object to release
    count : unsigned int
        The number of references to release, typically 1. Must be nonzero
        and not larger than INT_MAX.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaUserObjectCreate`, :py:obj:`~.cudaUserObjectRetain`, :py:obj:`~.cudaGraphRetainUserObject`, :py:obj:`~.cudaGraphReleaseUserObject`, :py:obj:`~.cudaGraphCreate`
    """
    cdef ccudart.cudaUserObject_t cobject
    if object is None:
        cobject = <ccudart.cudaUserObject_t><void_ptr>0
    elif isinstance(object, (cudaUserObject_t,cuda.CUuserObject)):
        pobject = int(object)
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject
    else:
        pobject = int(cudaUserObject_t(object))
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject

    err = ccudart.cudaUserObjectRelease(cobject, count)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphRetainUserObject(graph, object, unsigned int count, unsigned int flags):
    """ Retain a reference to a user object from a graph.

    Creates or moves user object references that will be owned by a CUDA
    graph.

    See CUDA User Objects in the CUDA C++ Programming Guide for more
    information on user objects.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        The graph to associate the reference with
    object : :py:obj:`~.cudaUserObject_t`
        The user object to retain a reference for
    count : unsigned int
        The number of references to add to the graph, typically 1. Must be
        nonzero and not larger than INT_MAX.
    flags : unsigned int
        The optional flag :py:obj:`~.cudaGraphUserObjectMove` transfers
        references from the calling thread, rather than create new
        references. Pass 0 to create new references.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaUserObjectCreate` :py:obj:`~.cudaUserObjectRetain`, :py:obj:`~.cudaUserObjectRelease`, :py:obj:`~.cudaGraphReleaseUserObject`, :py:obj:`~.cudaGraphCreate`
    """
    cdef ccudart.cudaUserObject_t cobject
    if object is None:
        cobject = <ccudart.cudaUserObject_t><void_ptr>0
    elif isinstance(object, (cudaUserObject_t,cuda.CUuserObject)):
        pobject = int(object)
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject
    else:
        pobject = int(cudaUserObject_t(object))
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    err = ccudart.cudaGraphRetainUserObject(cgraph, cobject, count, flags)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphReleaseUserObject(graph, object, unsigned int count):
    """ Release a user object reference from a graph.

    Releases user object references owned by a graph.

    See CUDA User Objects in the CUDA C++ Programming Guide for more
    information on user objects.

    Parameters
    ----------
    graph : :py:obj:`~.CUgraph` or :py:obj:`~.cudaGraph_t`
        The graph that will release the reference
    object : :py:obj:`~.cudaUserObject_t`
        The user object to release a reference for
    count : unsigned int
        The number of references to release, typically 1. Must be nonzero
        and not larger than INT_MAX.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`

    See Also
    --------
    :py:obj:`~.cudaUserObjectCreate` :py:obj:`~.cudaUserObjectRetain`, :py:obj:`~.cudaUserObjectRelease`, :py:obj:`~.cudaGraphRetainUserObject`, :py:obj:`~.cudaGraphCreate`
    """
    cdef ccudart.cudaUserObject_t cobject
    if object is None:
        cobject = <ccudart.cudaUserObject_t><void_ptr>0
    elif isinstance(object, (cudaUserObject_t,cuda.CUuserObject)):
        pobject = int(object)
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject
    else:
        pobject = int(cudaUserObject_t(object))
        cobject = <ccudart.cudaUserObject_t><void_ptr>pobject

    cdef ccudart.cudaGraph_t cgraph
    if graph is None:
        cgraph = <ccudart.cudaGraph_t><void_ptr>0
    elif isinstance(graph, (cudaGraph_t,cuda.CUgraph)):
        pgraph = int(graph)
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph
    else:
        pgraph = int(cudaGraph_t(graph))
        cgraph = <ccudart.cudaGraph_t><void_ptr>pgraph

    err = ccudart.cudaGraphReleaseUserObject(cgraph, cobject, count)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGetDriverEntryPoint(char* symbol, unsigned long long flags):
    """ Returns the requested driver API function pointer.

    Returns in `**funcPtr` the address of the CUDA driver function for the
    requested flags.

    For a requested driver symbol, if the CUDA version in which the driver
    symbol was introduced is less than or equal to the CUDA runtime
    version, the API will return the function pointer to the corresponding
    versioned driver function.

    The pointer returned by the API should be cast to a function pointer
    matching the requested driver function's definition in the API header
    file. The function pointer typedef can be picked up from the
    corresponding typedefs header file. For example, cudaTypedefs.h
    consists of function pointer typedefs for driver APIs defined in
    cuda.h.

    The API will return :py:obj:`~.cudaErrorSymbolNotFound` if the
    requested driver function is not supported on the platform, no ABI
    compatible driver function exists for the CUDA runtime version or if
    the driver symbol is invalid.

    The requested flags can be:

    - :py:obj:`~.cudaEnableDefault`: This is the default mode. This is
      equivalent to :py:obj:`~.cudaEnablePerThreadDefaultStream` if the
      code is compiled with --default-stream per-thread compilation flag or
      the macro CUDA_API_PER_THREAD_DEFAULT_STREAM is defined;
      :py:obj:`~.cudaEnableLegacyStream` otherwise.

    - :py:obj:`~.cudaEnableLegacyStream`: This will enable the search for
      all driver symbols that match the requested driver symbol name except
      the corresponding per-thread versions.

    - :py:obj:`~.cudaEnablePerThreadDefaultStream`: This will enable the
      search for all driver symbols that match the requested driver symbol
      name including the per-thread versions. If a per-thread version is
      not found, the API will return the legacy version of the driver
      function.

    Parameters
    ----------
    symbol : bytes
        The base name of the driver API function to look for. As an
        example, for the driver API :py:obj:`~.cuMemAlloc_v2`, `symbol`
        would be cuMemAlloc. Note that the API will use the CUDA runtime
        version to return the address to the most recent ABI compatible
        driver symbol, :py:obj:`~.cuMemAlloc` or :py:obj:`~.cuMemAlloc_v2`.
    flags : unsigned long long
        Flags to specify search options.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorNotSupported`, :py:obj:`~.cudaErrorSymbolNotFound`
    funcPtr : Any
        Location to return the function pointer to the requested driver
        function

    See Also
    --------
    :py:obj:`~.cuGetProcAddress`
    """
    cdef void_ptr funcPtr = 0
    err = ccudart.cudaGetDriverEntryPoint(symbol, <void**>&funcPtr, flags)
    return (cudaError_t(err), funcPtr)

@cython.embedsignature(True)
def cudaGetExportTable(pExportTableId : cudaUUID_t):
    """"""
    cdef void_ptr ppExportTable = 0
    cdef ccudart.cudaUUID_t* cpExportTableId_ptr = pExportTableId._ptr if pExportTableId != None else NULL
    err = ccudart.cudaGetExportTable(<const void**>&ppExportTable, cpExportTableId_ptr)
    return (cudaError_t(err), ppExportTable)

@cython.embedsignature(True)
def make_cudaPitchedPtr(d, size_t p, size_t xsz, size_t ysz):
    """ Returns a :py:obj:`~.cudaPitchedPtr` based on input parameters.

    Returns a :py:obj:`~.cudaPitchedPtr` based on the specified input
    parameters `d`, `p`, `xsz`, and `ysz`.

    Parameters
    ----------
    d : Any
        Pointer to allocated memory
    p : size_t
        Pitch of allocated memory in bytes
    xsz : size_t
        Logical width of allocation in elements
    ysz : size_t
        Logical height of allocation in elements

    Returns
    -------
    cudaError_t.cudaSuccess
        cudaError_t.cudaSuccess
    :py:obj:`~.cudaPitchedPtr`
        :py:obj:`~.cudaPitchedPtr` specified by `d`, `p`, `xsz`, and `ysz`

    See Also
    --------
    make_cudaExtent, make_cudaPos
    """
    cd = utils.HelperInputVoidPtr(d)
    cdef void* cd_ptr = <void*><void_ptr>cd.cptr
    cdef ccudart.cudaPitchedPtr err
    err = ccudart.make_cudaPitchedPtr(cd_ptr, p, xsz, ysz)
    cdef cudaPitchedPtr wrapper = cudaPitchedPtr()
    wrapper._ptr[0] = err
    return wrapper

@cython.embedsignature(True)
def make_cudaPos(size_t x, size_t y, size_t z):
    """ Returns a :py:obj:`~.cudaPos` based on input parameters.

    Returns a :py:obj:`~.cudaPos` based on the specified input parameters
    `x`, `y`, and `z`.

    Parameters
    ----------
    x : size_t
        X position
    y : size_t
        Y position
    z : size_t
        Z position

    Returns
    -------
    cudaError_t.cudaSuccess
        cudaError_t.cudaSuccess
    :py:obj:`~.cudaPos`
        :py:obj:`~.cudaPos` specified by `x`, `y`, and `z`

    See Also
    --------
    make_cudaExtent, make_cudaPitchedPtr
    """
    cdef ccudart.cudaPos err
    err = ccudart.make_cudaPos(x, y, z)
    cdef cudaPos wrapper = cudaPos()
    wrapper._ptr[0] = err
    return wrapper

@cython.embedsignature(True)
def make_cudaExtent(size_t w, size_t h, size_t d):
    """ Returns a :py:obj:`~.cudaExtent` based on input parameters.

    Returns a :py:obj:`~.cudaExtent` based on the specified input
    parameters `w`, `h`, and `d`.

    Parameters
    ----------
    w : size_t
        Width in elements when referring to array memory, in bytes when
        referring to linear memory
    h : size_t
        Height in elements
    d : size_t
        Depth in elements

    Returns
    -------
    cudaError_t.cudaSuccess
        cudaError_t.cudaSuccess
    :py:obj:`~.cudaExtent`
        :py:obj:`~.cudaExtent` specified by `w`, `h`, and `d`

    See Also
    --------
    make_cudaPitchedPtr, make_cudaPos
    """
    cdef ccudart.cudaExtent err
    err = ccudart.make_cudaExtent(w, h, d)
    cdef cudaExtent wrapper = cudaExtent()
    wrapper._ptr[0] = err
    return wrapper

@cython.embedsignature(True)
def cudaProfilerInitialize(char* configFile, char* outputFile, outputMode not None : cudaOutputMode_t):
    """ Initialize the CUDA profiler.

    [Deprecated]

    Using this API user can initialize the CUDA profiler by specifying the
    configuration file, output file and output file format. This API is
    generally used to profile different set of counters by looping the
    kernel launch. The `configFile` parameter can be used to select
    profiling options including profiler counters. Refer to the "Compute
    Command Line Profiler User Guide" for supported profiler options and
    counters.

    Limitation: The CUDA profiler cannot be initialized with this API if
    another profiling tool is already active, as indicated by the
    :py:obj:`~.cudaErrorProfilerDisabled` return code.

    Typical usage of the profiling APIs is as follows:

    for each set of counters/options {
    :py:obj:`~.cudaProfilerInitialize()`; //Initialize profiling,set the
    counters/options in the config file   ...
    :py:obj:`~.cudaProfilerStart()`;   // code to be profiled
    :py:obj:`~.cudaProfilerStop()`;  ...  :py:obj:`~.cudaProfilerStart()`;
    // code to be profiled   :py:obj:`~.cudaProfilerStop()`;  ... }

    Parameters
    ----------
    configFile : bytes
        Name of the config file that lists the counters/options for
        profiling.
    outputFile : bytes
        Name of the outputFile where the profiling results will be stored.
    outputMode : :py:obj:`~.cudaOutputMode_t`
        outputMode, can be :py:obj:`~.cudaKeyValuePair` OR
        :py:obj:`~.cudaCSV`.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorProfilerDisabled`

    See Also
    --------
    :py:obj:`~.cudaProfilerStart`, :py:obj:`~.cudaProfilerStop`, :py:obj:`~.cuProfilerInitialize`
    """
    cdef ccudart.cudaOutputMode_t coutputMode = outputMode.value
    err = ccudart.cudaProfilerInitialize(configFile, outputFile, coutputMode)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaProfilerStart():
    """ Enable profiling.

    Enables profile collection by the active profiling tool for the current
    context. If profiling is already enabled, then
    :py:obj:`~.cudaProfilerStart()` has no effect.

    cudaProfilerStart and cudaProfilerStop APIs are used to
    programmatically control the profiling granularity by allowing
    profiling to be done only on selective pieces of code.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaProfilerInitialize`, :py:obj:`~.cudaProfilerStop`, :py:obj:`~.cuProfilerStart`
    """
    err = ccudart.cudaProfilerStart()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaProfilerStop():
    """ Disable profiling.

    Disables profile collection by the active profiling tool for the
    current context. If profiling is already disabled, then
    :py:obj:`~.cudaProfilerStop()` has no effect.

    cudaProfilerStart and cudaProfilerStop APIs are used to
    programmatically control the profiling granularity by allowing
    profiling to be done only on selective pieces of code.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`

    See Also
    --------
    :py:obj:`~.cudaProfilerInitialize`, :py:obj:`~.cudaProfilerStart`, :py:obj:`~.cuProfilerStop`
    """
    err = ccudart.cudaProfilerStop()
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsEGLRegisterImage(image, unsigned int flags):
    """ Registers an EGL image.

    Registers the EGLImageKHR specified by `image` for access by CUDA. A
    handle to the registered object is returned as `pCudaResource`.
    Additional Mapping/Unmapping is not required for the registered
    resource and :py:obj:`~.cudaGraphicsResourceGetMappedEglFrame` can be
    directly called on the `pCudaResource`.

    The application will be responsible for synchronizing access to shared
    objects. The application must ensure that any pending operation which
    access the objects have completed before passing control to CUDA. This
    may be accomplished by issuing and waiting for glFinish command on all
    GLcontexts (for OpenGL and likewise for other APIs). The application
    will be also responsible for ensuring that any pending operation on the
    registered CUDA resource has completed prior to executing subsequent
    commands in other APIs accesing the same memory objects. This can be
    accomplished by calling cuCtxSynchronize or cuEventSynchronize
    (preferably).

    The surface's intended usage is specified using `flags`, as follows:

    - :py:obj:`~.cudaGraphicsRegisterFlagsNone`: Specifies no hints about
      how this resource will be used. It is therefore assumed that this
      resource will be read from and written to by CUDA. This is the
      default value.

    - :py:obj:`~.cudaGraphicsRegisterFlagsReadOnly`: Specifies that CUDA
      will not write to this resource.

    - :py:obj:`~.cudaGraphicsRegisterFlagsWriteDiscard`: Specifies that
      CUDA will not read from this resource and will write over the entire
      contents of the resource, so none of the data previously stored in
      the resource will be preserved.

    The EGLImageKHR is an object which can be used to create EGLImage
    target resource. It is defined as a void pointer. typedef void*
    EGLImageKHR

    Parameters
    ----------
    image : :py:obj:`~.EGLImageKHR`
        An EGLImageKHR image which can be used to create target resource.
    flags : unsigned int
        Map flags

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidResourceHandle`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`
    pCudaResource : :py:obj:`~.cudaGraphicsResource_t`
        Pointer to the returned object handle

    See Also
    --------
    :py:obj:`~.cudaGraphicsUnregisterResource`, :py:obj:`~.cudaGraphicsResourceGetMappedEglFrame`, :py:obj:`~.cuGraphicsEGLRegisterImage`
    """
    cdef ccudart.EGLImageKHR cimage
    if image is None:
        cimage = <ccudart.EGLImageKHR><void_ptr>0
    elif isinstance(image, (EGLImageKHR)):
        pimage = int(image)
        cimage = <ccudart.EGLImageKHR><void_ptr>pimage
    else:
        pimage = int(EGLImageKHR(image))
        cimage = <ccudart.EGLImageKHR><void_ptr>pimage

    cdef cudaGraphicsResource_t pCudaResource = cudaGraphicsResource_t()
    err = ccudart.cudaGraphicsEGLRegisterImage(pCudaResource._ptr, cimage, flags)
    return (cudaError_t(err), pCudaResource)

@cython.embedsignature(True)
def cudaEGLStreamConsumerConnect(eglStream):
    """ Connect CUDA to EGLStream as a consumer.

    Connect CUDA as a consumer to EGLStreamKHR specified by `eglStream`.

    The EGLStreamKHR is an EGL object that transfers a sequence of image
    frames from one API to another.

    Parameters
    ----------
    eglStream : :py:obj:`~.EGLStreamKHR`
        EGLStreamKHR handle

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`
    conn : :py:obj:`~.cudaEglStreamConnection`
        Pointer to the returned connection handle

    See Also
    --------
    :py:obj:`~.cudaEGLStreamConsumerDisconnect`, :py:obj:`~.cudaEGLStreamConsumerAcquireFrame`, :py:obj:`~.cudaEGLStreamConsumerReleaseFrame`, :py:obj:`~.cuEGLStreamConsumerConnect`
    """
    cdef ccudart.EGLStreamKHR ceglStream
    if eglStream is None:
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>0
    elif isinstance(eglStream, (EGLStreamKHR)):
        peglStream = int(eglStream)
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>peglStream
    else:
        peglStream = int(EGLStreamKHR(eglStream))
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>peglStream

    cdef cudaEglStreamConnection conn = cudaEglStreamConnection()
    err = ccudart.cudaEGLStreamConsumerConnect(<ccudart.cudaEglStreamConnection*>conn._ptr, ceglStream)
    return (cudaError_t(err), conn)

@cython.embedsignature(True)
def cudaEGLStreamConsumerConnectWithFlags(eglStream, unsigned int flags):
    """ Connect CUDA to EGLStream as a consumer with given flags.

    Connect CUDA as a consumer to EGLStreamKHR specified by `stream` with
    specified `flags` defined by :py:obj:`~.cudaEglResourceLocationFlags`.

    The flags specify whether the consumer wants to access frames from
    system memory or video memory. Default is
    :py:obj:`~.cudaEglResourceLocationVidmem`.

    Parameters
    ----------
    eglStream : :py:obj:`~.EGLStreamKHR`
        EGLStreamKHR handle
    flags : unsigned int
        Flags denote intended location - system or video.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`
    conn : :py:obj:`~.cudaEglStreamConnection`
        Pointer to the returned connection handle

    See Also
    --------
    :py:obj:`~.cudaEGLStreamConsumerDisconnect`, :py:obj:`~.cudaEGLStreamConsumerAcquireFrame`, :py:obj:`~.cudaEGLStreamConsumerReleaseFrame`, :py:obj:`~.cuEGLStreamConsumerConnectWithFlags`
    """
    cdef ccudart.EGLStreamKHR ceglStream
    if eglStream is None:
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>0
    elif isinstance(eglStream, (EGLStreamKHR)):
        peglStream = int(eglStream)
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>peglStream
    else:
        peglStream = int(EGLStreamKHR(eglStream))
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>peglStream

    cdef cudaEglStreamConnection conn = cudaEglStreamConnection()
    err = ccudart.cudaEGLStreamConsumerConnectWithFlags(<ccudart.cudaEglStreamConnection*>conn._ptr, ceglStream, flags)
    return (cudaError_t(err), conn)

@cython.embedsignature(True)
def cudaEGLStreamConsumerDisconnect(conn):
    """ Disconnect CUDA as a consumer to EGLStream .

    Disconnect CUDA as a consumer to EGLStreamKHR.

    Parameters
    ----------
    conn : :py:obj:`~.cudaEglStreamConnection`
        Conection to disconnect.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaEGLStreamConsumerConnect`, :py:obj:`~.cudaEGLStreamConsumerAcquireFrame`, :py:obj:`~.cudaEGLStreamConsumerReleaseFrame`, :py:obj:`~.cuEGLStreamConsumerDisconnect`
    """
    cdef ccudart.cudaEglStreamConnection *cconn
    if conn is None:
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>NULL
    elif isinstance(conn, (cudaEglStreamConnection,cuda.CUeglStreamConnection)):
        pconn = conn.getPtr()
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>pconn
    elif isinstance(conn, (int)):
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>conn
    else:
        raise TypeError("Argument 'conn' is not instance of type (expected <class 'int, cudart.cudaEglStreamConnection,cuda.CUeglStreamConnection'>, found " + str(type(conn)))

    err = ccudart.cudaEGLStreamConsumerDisconnect(cconn)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEGLStreamConsumerAcquireFrame(conn, pCudaResource, pStream, unsigned int timeout):
    """ Acquire an image frame from the EGLStream with CUDA as a consumer.

    Acquire an image frame from EGLStreamKHR.
    :py:obj:`~.cudaGraphicsResourceGetMappedEglFrame` can be called on
    `pCudaResource` to get :py:obj:`~.cudaEglFrame`.

    Parameters
    ----------
    conn : :py:obj:`~.cudaEglStreamConnection`
        Connection on which to acquire
    pCudaResource : :py:obj:`~.cudaGraphicsResource_t`
        CUDA resource on which the EGLStream frame will be mapped for use.
    pStream : :py:obj:`~.cudaStream_t`
        CUDA stream for synchronization and any data migrations implied by
        :py:obj:`~.cudaEglResourceLocationFlags`.
    timeout : unsigned int
        Desired timeout in usec.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`, :py:obj:`~.cudaErrorLaunchTimeout`

    See Also
    --------
    :py:obj:`~.cudaEGLStreamConsumerConnect`, :py:obj:`~.cudaEGLStreamConsumerDisconnect`, :py:obj:`~.cudaEGLStreamConsumerReleaseFrame`, :py:obj:`~.cuEGLStreamConsumerAcquireFrame`
    """
    cdef ccudart.cudaStream_t *cpStream
    if pStream is None:
        cpStream = <ccudart.cudaStream_t*><void_ptr>NULL
    elif isinstance(pStream, (cudaStream_t,cuda.CUstream)):
        ppStream = pStream.getPtr()
        cpStream = <ccudart.cudaStream_t*><void_ptr>ppStream
    elif isinstance(pStream, (int)):
        cpStream = <ccudart.cudaStream_t*><void_ptr>pStream
    else:
        raise TypeError("Argument 'pStream' is not instance of type (expected <class 'int, cudart.cudaStream_t,cuda.CUstream'>, found " + str(type(pStream)))

    cdef ccudart.cudaGraphicsResource_t *cpCudaResource
    if pCudaResource is None:
        cpCudaResource = <ccudart.cudaGraphicsResource_t*><void_ptr>NULL
    elif isinstance(pCudaResource, (cudaGraphicsResource_t)):
        ppCudaResource = pCudaResource.getPtr()
        cpCudaResource = <ccudart.cudaGraphicsResource_t*><void_ptr>ppCudaResource
    elif isinstance(pCudaResource, (int)):
        cpCudaResource = <ccudart.cudaGraphicsResource_t*><void_ptr>pCudaResource
    else:
        raise TypeError("Argument 'pCudaResource' is not instance of type (expected <class 'int, cudart.cudaGraphicsResource_t'>, found " + str(type(pCudaResource)))

    cdef ccudart.cudaEglStreamConnection *cconn
    if conn is None:
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>NULL
    elif isinstance(conn, (cudaEglStreamConnection,cuda.CUeglStreamConnection)):
        pconn = conn.getPtr()
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>pconn
    elif isinstance(conn, (int)):
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>conn
    else:
        raise TypeError("Argument 'conn' is not instance of type (expected <class 'int, cudart.cudaEglStreamConnection,cuda.CUeglStreamConnection'>, found " + str(type(conn)))

    err = ccudart.cudaEGLStreamConsumerAcquireFrame(cconn, cpCudaResource, cpStream, timeout)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEGLStreamConsumerReleaseFrame(conn, pCudaResource, pStream):
    """ Releases the last frame acquired from the EGLStream.

    Release the acquired image frame specified by `pCudaResource` to
    EGLStreamKHR.

    Parameters
    ----------
    conn : :py:obj:`~.cudaEglStreamConnection`
        Connection on which to release
    pCudaResource : :py:obj:`~.cudaGraphicsResource_t`
        CUDA resource whose corresponding frame is to be released
    pStream : :py:obj:`~.cudaStream_t`
        CUDA stream on which release will be done.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaEGLStreamConsumerConnect`, :py:obj:`~.cudaEGLStreamConsumerDisconnect`, :py:obj:`~.cudaEGLStreamConsumerAcquireFrame`, :py:obj:`~.cuEGLStreamConsumerReleaseFrame`
    """
    cdef ccudart.cudaStream_t *cpStream
    if pStream is None:
        cpStream = <ccudart.cudaStream_t*><void_ptr>NULL
    elif isinstance(pStream, (cudaStream_t,cuda.CUstream)):
        ppStream = pStream.getPtr()
        cpStream = <ccudart.cudaStream_t*><void_ptr>ppStream
    elif isinstance(pStream, (int)):
        cpStream = <ccudart.cudaStream_t*><void_ptr>pStream
    else:
        raise TypeError("Argument 'pStream' is not instance of type (expected <class 'int, cudart.cudaStream_t,cuda.CUstream'>, found " + str(type(pStream)))

    cdef ccudart.cudaGraphicsResource_t cpCudaResource
    if pCudaResource is None:
        cpCudaResource = <ccudart.cudaGraphicsResource_t><void_ptr>0
    elif isinstance(pCudaResource, (cudaGraphicsResource_t)):
        ppCudaResource = int(pCudaResource)
        cpCudaResource = <ccudart.cudaGraphicsResource_t><void_ptr>ppCudaResource
    else:
        ppCudaResource = int(cudaGraphicsResource_t(pCudaResource))
        cpCudaResource = <ccudart.cudaGraphicsResource_t><void_ptr>ppCudaResource

    cdef ccudart.cudaEglStreamConnection *cconn
    if conn is None:
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>NULL
    elif isinstance(conn, (cudaEglStreamConnection,cuda.CUeglStreamConnection)):
        pconn = conn.getPtr()
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>pconn
    elif isinstance(conn, (int)):
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>conn
    else:
        raise TypeError("Argument 'conn' is not instance of type (expected <class 'int, cudart.cudaEglStreamConnection,cuda.CUeglStreamConnection'>, found " + str(type(conn)))

    err = ccudart.cudaEGLStreamConsumerReleaseFrame(cconn, cpCudaResource, cpStream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEGLStreamProducerConnect(eglStream, width, height):
    """ Connect CUDA to EGLStream as a producer.

    Connect CUDA as a producer to EGLStreamKHR specified by `stream`.

    The EGLStreamKHR is an EGL object that transfers a sequence of image
    frames from one API to another.

    Parameters
    ----------
    eglStream : :py:obj:`~.EGLStreamKHR`
        EGLStreamKHR handle
    width : :py:obj:`~.EGLint`
        width of the image to be submitted to the stream
    height : :py:obj:`~.EGLint`
        height of the image to be submitted to the stream

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`
    conn : :py:obj:`~.cudaEglStreamConnection`
        Pointer to the returned connection handle

    See Also
    --------
    :py:obj:`~.cudaEGLStreamProducerDisconnect`, :py:obj:`~.cudaEGLStreamProducerPresentFrame`, :py:obj:`~.cudaEGLStreamProducerReturnFrame`, :py:obj:`~.cuEGLStreamProducerConnect`
    """
    cdef ccudart.EGLint cheight
    if height is None:
        cheight = <ccudart.EGLint><void_ptr>0
    elif isinstance(height, (EGLint)):
        pheight = int(height)
        cheight = <ccudart.EGLint><void_ptr>pheight
    else:
        pheight = int(EGLint(height))
        cheight = <ccudart.EGLint><void_ptr>pheight

    cdef ccudart.EGLint cwidth
    if width is None:
        cwidth = <ccudart.EGLint><void_ptr>0
    elif isinstance(width, (EGLint)):
        pwidth = int(width)
        cwidth = <ccudart.EGLint><void_ptr>pwidth
    else:
        pwidth = int(EGLint(width))
        cwidth = <ccudart.EGLint><void_ptr>pwidth

    cdef ccudart.EGLStreamKHR ceglStream
    if eglStream is None:
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>0
    elif isinstance(eglStream, (EGLStreamKHR)):
        peglStream = int(eglStream)
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>peglStream
    else:
        peglStream = int(EGLStreamKHR(eglStream))
        ceglStream = <ccudart.EGLStreamKHR><void_ptr>peglStream

    cdef cudaEglStreamConnection conn = cudaEglStreamConnection()
    err = ccudart.cudaEGLStreamProducerConnect(<ccudart.cudaEglStreamConnection*>conn._ptr, ceglStream, cwidth, cheight)
    return (cudaError_t(err), conn)

@cython.embedsignature(True)
def cudaEGLStreamProducerDisconnect(conn):
    """ Disconnect CUDA as a producer to EGLStream .

    Disconnect CUDA as a producer to EGLStreamKHR.

    Parameters
    ----------
    conn : :py:obj:`~.cudaEglStreamConnection`
        Conection to disconnect.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaEGLStreamProducerConnect`, :py:obj:`~.cudaEGLStreamProducerPresentFrame`, :py:obj:`~.cudaEGLStreamProducerReturnFrame`, :py:obj:`~.cuEGLStreamProducerDisconnect`
    """
    cdef ccudart.cudaEglStreamConnection *cconn
    if conn is None:
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>NULL
    elif isinstance(conn, (cudaEglStreamConnection,cuda.CUeglStreamConnection)):
        pconn = conn.getPtr()
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>pconn
    elif isinstance(conn, (int)):
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>conn
    else:
        raise TypeError("Argument 'conn' is not instance of type (expected <class 'int, cudart.cudaEglStreamConnection,cuda.CUeglStreamConnection'>, found " + str(type(conn)))

    err = ccudart.cudaEGLStreamProducerDisconnect(cconn)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEGLStreamProducerPresentFrame(conn, eglframe not None : cudaEglFrame, pStream):
    """ Present a CUDA eglFrame to the EGLStream with CUDA as a producer.

    The :py:obj:`~.cudaEglFrame` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    For :py:obj:`~.cudaEglFrame` of type :py:obj:`~.cudaEglFrameTypePitch`,
    the application may present sub-region of a memory allocation. In that
    case, :py:obj:`~.cudaPitchedPtr.ptr` will specify the start address of
    the sub-region in the allocation and :py:obj:`~.cudaEglPlaneDesc` will
    specify the dimensions of the sub-region.

    Parameters
    ----------
    conn : :py:obj:`~.cudaEglStreamConnection`
        Connection on which to present the CUDA array
    eglframe : :py:obj:`~.cudaEglFrame`
        CUDA Eglstream Proucer Frame handle to be sent to the consumer over
        EglStream.
    pStream : :py:obj:`~.cudaStream_t`
        CUDA stream on which to present the frame.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaEGLStreamProducerConnect`, :py:obj:`~.cudaEGLStreamProducerDisconnect`, :py:obj:`~.cudaEGLStreamProducerReturnFrame`, :py:obj:`~.cuEGLStreamProducerPresentFrame`
    """
    cdef ccudart.cudaStream_t *cpStream
    if pStream is None:
        cpStream = <ccudart.cudaStream_t*><void_ptr>NULL
    elif isinstance(pStream, (cudaStream_t,cuda.CUstream)):
        ppStream = pStream.getPtr()
        cpStream = <ccudart.cudaStream_t*><void_ptr>ppStream
    elif isinstance(pStream, (int)):
        cpStream = <ccudart.cudaStream_t*><void_ptr>pStream
    else:
        raise TypeError("Argument 'pStream' is not instance of type (expected <class 'int, cudart.cudaStream_t,cuda.CUstream'>, found " + str(type(pStream)))

    cdef ccudart.cudaEglStreamConnection *cconn
    if conn is None:
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>NULL
    elif isinstance(conn, (cudaEglStreamConnection,cuda.CUeglStreamConnection)):
        pconn = conn.getPtr()
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>pconn
    elif isinstance(conn, (int)):
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>conn
    else:
        raise TypeError("Argument 'conn' is not instance of type (expected <class 'int, cudart.cudaEglStreamConnection,cuda.CUeglStreamConnection'>, found " + str(type(conn)))

    err = ccudart.cudaEGLStreamProducerPresentFrame(cconn, eglframe._ptr[0], cpStream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaEGLStreamProducerReturnFrame(conn, eglframe : cudaEglFrame, pStream):
    """ Return the CUDA eglFrame to the EGLStream last released by the consumer.

    This API can potentially return cudaErrorLaunchTimeout if the consumer
    has not returned a frame to EGL stream. If timeout is returned the
    application can retry.

    Parameters
    ----------
    conn : :py:obj:`~.cudaEglStreamConnection`
        Connection on which to present the CUDA array
    eglframe : :py:obj:`~.cudaEglFrame`
        CUDA Eglstream Proucer Frame handle returned from the consumer over
        EglStream.
    pStream : :py:obj:`~.cudaStream_t`
        CUDA stream on which to return the frame.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorLaunchTimeout`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`

    See Also
    --------
    :py:obj:`~.cudaEGLStreamProducerConnect`, :py:obj:`~.cudaEGLStreamProducerDisconnect`, :py:obj:`~.cudaEGLStreamProducerPresentFrame`, :py:obj:`~.cuEGLStreamProducerReturnFrame`
    """
    cdef ccudart.cudaStream_t *cpStream
    if pStream is None:
        cpStream = <ccudart.cudaStream_t*><void_ptr>NULL
    elif isinstance(pStream, (cudaStream_t,cuda.CUstream)):
        ppStream = pStream.getPtr()
        cpStream = <ccudart.cudaStream_t*><void_ptr>ppStream
    elif isinstance(pStream, (int)):
        cpStream = <ccudart.cudaStream_t*><void_ptr>pStream
    else:
        raise TypeError("Argument 'pStream' is not instance of type (expected <class 'int, cudart.cudaStream_t,cuda.CUstream'>, found " + str(type(pStream)))

    cdef ccudart.cudaEglStreamConnection *cconn
    if conn is None:
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>NULL
    elif isinstance(conn, (cudaEglStreamConnection,cuda.CUeglStreamConnection)):
        pconn = conn.getPtr()
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>pconn
    elif isinstance(conn, (int)):
        cconn = <ccudart.cudaEglStreamConnection*><void_ptr>conn
    else:
        raise TypeError("Argument 'conn' is not instance of type (expected <class 'int, cudart.cudaEglStreamConnection,cuda.CUeglStreamConnection'>, found " + str(type(conn)))

    cdef ccudart.cudaEglFrame* ceglframe_ptr = eglframe._ptr if eglframe != None else NULL
    err = ccudart.cudaEGLStreamProducerReturnFrame(cconn, ceglframe_ptr, cpStream)
    return (cudaError_t(err),)

@cython.embedsignature(True)
def cudaGraphicsResourceGetMappedEglFrame(resource, unsigned int index, unsigned int mipLevel):
    """ Get an eglFrame through which to access a registered EGL graphics resource.

    Returns in `*eglFrame` an eglFrame pointer through which the registered
    graphics resource `resource` may be accessed. This API can only be
    called for EGL graphics resources.

    The :py:obj:`~.cudaEglFrame` is defined as

    **View CUDA Toolkit Documentation for a C++ code example**

    Parameters
    ----------
    resource : :py:obj:`~.cudaGraphicsResource_t`
        Registered resource to access.
    index : unsigned int
        Index for cubemap surfaces.
    mipLevel : unsigned int
        Mipmap level for the subresource to access.

    Returns
    -------
    cudaError_t
        :py:obj:`~.cudaSuccess`, :py:obj:`~.cudaErrorInvalidValue`, :py:obj:`~.cudaErrorUnknown`
    eglFrame : :py:obj:`~.cudaEglFrame`
        Returned eglFrame.

    See Also
    --------
    :py:obj:`~.cudaGraphicsSubResourceGetMappedArray`, :py:obj:`~.cudaGraphicsResourceGetMappedPointer`, :py:obj:`~.cuGraphicsResourceGetMappedEglFrame`

    Notes
    -----
    Note that in case of multiplanar `*eglFrame`, pitch of only first plane (unsigned int :py:obj:`~.cudaEglPlaneDesc.pitch`) is to be considered by the application.
    """
    cdef ccudart.cudaGraphicsResource_t cresource
    if resource is None:
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>0
    elif isinstance(resource, (cudaGraphicsResource_t)):
        presource = int(resource)
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource
    else:
        presource = int(cudaGraphicsResource_t(resource))
        cresource = <ccudart.cudaGraphicsResource_t><void_ptr>presource

    cdef cudaEglFrame eglFrame = cudaEglFrame()
    err = ccudart.cudaGraphicsResourceGetMappedEglFrame(eglFrame._ptr, cresource, index, mipLevel)
    return (cudaError_t(err), eglFrame)

@cython.embedsignature(True)
def cudaEventCreateFromEGLSync(eglSync, unsigned int flags):
    """ Creates an event from EGLSync object.

    Creates an event *phEvent from an EGLSyncKHR eglSync with the flages
    specified via `flags`. Valid flags include:

    - :py:obj:`~.cudaEventDefault`: Default event creation flag.

    - :py:obj:`~.cudaEventBlockingSync`: Specifies that the created event
      should use blocking synchronization. A CPU thread that uses
      :py:obj:`~.cudaEventSynchronize()` to wait on an event created with
      this flag will block until the event has actually been completed.

    :py:obj:`~.cudaEventRecord` and TimingData are not supported for events
    created from EGLSync.

    The EGLSyncKHR is an opaque handle to an EGL sync object. typedef void*
    EGLSyncKHR

    Parameters
    ----------
    eglSync : :py:obj:`~.EGLSyncKHR`
        Opaque handle to EGLSync object
    flags : unsigned int
        